<?php
/**
 * Gateway: Paypal Express - Webhooks
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

use StellarWP\Memberdash\StellarWP\SuperGlobals\SuperGlobals;
use StellarWP\Memberdash\StellarWP\Arrays\Arr;

/**
 * PayPal Express gateway Webhooks class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_Api_Webhooks extends MS_Gateway_Paypalexpress_Api {
	/**
	 * Returns a list of webhooks that are needed for the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	protected function get_webhook_events(): array {
		return [
			'CHECKOUT.ORDER.PROCESSED',
			'PAYMENT.CAPTURE.COMPLETED',
			'PAYMENT.CAPTURE.DENIED',
			'PAYMENT.CAPTURE.REFUNDED',
			'PAYMENT.CAPTURE.REVERSED',
			'PAYMENT.SALE.COMPLETED',
			'PAYMENT.SALE.REFUNDED',
			'PAYMENT.SALE.REVERSED',
			'BILLING.SUBSCRIPTION.PAYMENT.FAILED',
		];
	}

	/**
	 * Returns the parent payment link from the list of Links on the response.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $links The list of links from the response.
	 *
	 * @return array<string,mixed>
	 */
	protected function get_parent_payment_link( array $links ): array {
		return array_filter(
			Arr::wrap(
				current(
					array_filter(
						$links,
						static function ( $link ) {
							return 'parent_payment' === Arr::get( $link, 'rel', '' );
						}
					)
				)
			)
		);
	}

	/**
	 * Returns true if the event type is a subscription event.
	 *
	 * @since 1.5.0
	 *
	 * @param string $event_type The event type to check.
	 *
	 * @return bool
	 */
	protected function is_subscription_event( string $event_type ): bool {
		return in_array(
			$event_type,
			[
				'PAYMENT.SALE.COMPLETED',
				'PAYMENT.SALE.REFUNDED',
				'PAYMENT.SALE.REVERSED',
				'BILLING.SUBSCRIPTION.PAYMENT.FAILED',
			],
			true
		);
	}

	/**
	 * Parses the headers from the PayPal webhook request.
	 *
	 * Used in the signature verification.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,string> $paypal_headers The headers from the PayPal webhook request.
	 *
	 * @return array<string,string>|WP_Error
	 */
	protected function parse_headers( array $paypal_headers ) {
		$header_keys = [
			'transmission_id'   => 'HTTP-PAYPAL-TRANSMISSION-ID',
			'transmission_time' => 'HTTP-PAYPAL-TRANSMISSION-TIME',
			'transmission_sig'  => 'HTTP-PAYPAL-TRANSMISSION-SIG',
			'cert_url'          => 'HTTP-PAYPAL-CERT-URL',
			'auth_algo'         => 'HTTP-PAYPAL-AUTH-ALGO',
			'debug_id'          => 'HTTP-PAYPAL-DEBUG-ID',
		];

		$headers      = [];
		$missing_keys = [];
		foreach ( $header_keys as $property => $key ) {
			// Headers are inconsistent between sandbox and live.
			if ( ! isset( $paypal_headers[ $key ] ) ) {
				$key = str_replace( '-', '_', $key );
				$key = strtoupper( $key );

				if ( ! isset( $paypal_headers[ $key ] ) ) {
					$key = strtolower( $key );
				}
			}

			if ( isset( $paypal_headers[ $key ] ) ) {
				$headers[ $property ] = $paypal_headers[ $key ];
			} else {
				$missing_keys[] = $property;
			}
		}

		// Remove the debug_id from the missing keys.
		if ( ! empty( $missing_keys ) ) {
			$missing_keys = array_diff( $missing_keys, [ 'debug_id' ] );
		}

		if ( ! empty( $missing_keys ) ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-missing-headers',
				__( 'Missing headers from the PayPal webhook request', 'memberdash' ),
				[
					'missing_keys' => $missing_keys,
				]
			);
		}

		return $headers;
	}

	/**
	 * Verifies the identity of the Webhook request, to avoid any security problems.
	 *
	 * @since 1.5.0
	 *
	 * @param string               $webhook_id Which webhook id we have currently stored on the database.
	 * @param array<string,mixed>  $event      The Event received by the endpoint from PayPal.
	 * @param array<string,string> $headers    Headers from the PayPal request that we use to verify the signature.
	 *
	 * @return bool
	 */
	protected function verify_webhook_signature( string $webhook_id, array $event, array $headers ): bool {
		$gateway = $this->get_gateway();

		$body = [
			'transmission_id'   => Arr::get( $headers, 'transmission_id' ),
			'transmission_time' => Arr::get( $headers, 'transmission_time' ),
			'transmission_sig'  => Arr::get( $headers, 'transmission_sig' ),
			'cert_url'          => Arr::get( $headers, 'cert_url' ),
			'auth_algo'         => Arr::get( $headers, 'auth_algo' ),
			'webhook_id'        => $webhook_id,
			'webhook_event'     => $event,
		];

		$args = [
			'headers' => [
				'PayPal-Partner-Attribution-Id' => $gateway->get_partner_attribution_id(),
			],
			'body'    => $body,
		];

		$response = $this->client_post(
			'v1/notifications/verify-webhook-signature',
			[],
			$args,
			$gateway->is_sandbox()
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		return 'SUCCESS' === Arr::get( $response, 'verification_status', false );
	}

	/**
	 * Checks if the given event name is a valid webhook event.
	 *
	 * @since 1.5.0
	 *
	 * @param string $event_name The event name to check.
	 *
	 * @return bool
	 */
	protected function is_valid_event( string $event_name ): bool {
		$events = $this->get_webhook_events();

		return in_array( $event_name, $events, true );
	}

	/**
	 * Processes the PayPal webhook event for a PayPal order.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $order_id The PayPal order ID.
	 * @param array<string,mixed> $event    The PayPal webhook event.
	 *
	 * @return bool|WP_Error
	 */
	protected function process_order_event( string $order_id, array $event ) {
		$subscription = MS_Gateway_Paypalexpress_Helpers_Order::get_ms_subscription( $order_id );

		if ( ! $subscription ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-subscription-not-found',
				sprintf(
					/* translators: %s: PayPal order ID */
					__( 'Subscription not found for PayPal order ID: %s.', 'memberdash' ),
					$order_id
				),
				[
					'order_id' => $order_id,
				]
			);
		}

		$membership = $subscription->get_membership();

		// Stop if the membership is a recurring subscription.
		if ( $membership->supports_recurring_payments() ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-invalid-subscription',
				sprintf(
					/* translators: %s: order ID */
					__( 'Invalid subscription found for PayPal order ID: %s.', 'memberdash' ),
					$order_id
				),
				[
					'order_id' => $order_id,
				]
			);
		}

		$member  = $subscription->get_member();
		$invoice = $subscription->get_current_invoice();

		$event_type  = MS_Helper_Cast::to_string(
			Arr::get( $event, 'event_type', '' )
		);
		$notes       = MS_Helper_Cast::to_string(
			Arr::get( $event, 'summary', '' )
		);
		$external_id = MS_Helper_Cast::to_string(
			Arr::get( $event, 'resource.id', '' )
		);

		switch ( $event_type ) {
			case 'PAYMENT.CAPTURE.COMPLETED':
				$invoice->add_notes( $notes );
				if ( ! $invoice->is_paid() ) {
					$gateway = $this->get_gateway();
					$invoice->pay_it( $gateway->get_id(), $external_id );
					$invoice->save();

					/*
					 * Creates a transaction log entry.
					 *
					 * Documented in /includes/gateways/class-ms-controller-gateway.php.
					 */
					do_action(
						'ms_gateway_transaction_log',
						$gateway->get_id(), // Gateway ID.
						'handle', // Values: request|process|handle.
						true, // Success flag.
						$subscription->get_id(), // Subscription ID.
						$invoice->get_id(), // Invoice ID.
						$invoice->get_invoice_total(), // Charged amount.
						$notes, // Descriptive text.
						$external_id // External ID.
					);
				}
				break;
			case 'PAYMENT.CAPTURE.DENIED':
			case 'PAYMENT.CAPTURE.REFUNDED':
			case 'PAYMENT.CAPTURE.REVERSED':
				$member->cancel_membership( $membership->get_id() );
				break;
		}

		return true;
	}

	/**
	 * Processes the PayPal webhook event for a PayPal subscription.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $subscription_id The PayPal subscription ID.
	 * @param array<string,mixed> $event           The PayPal webhook event.
	 *
	 * @return bool|WP_Error
	 */
	protected function process_subscription_event( string $subscription_id, array $event ) {
		$subscription = MS_Gateway_Paypalexpress_Helpers_Subscription::get_ms_subscription( $subscription_id );

		if ( ! $subscription ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-subscription-not-found',
				sprintf(
					/* translators: %s: PayPal subscription ID */
					__( 'Subscription not found for PayPal subscription ID: %s.', 'memberdash' ),
					$subscription_id
				),
				[
					'subscription_id' => $subscription_id,
				]
			);
		}

		$membership = $subscription->get_membership();

		// Stop if the membership is not a recurring subscription.
		if ( ! $membership->supports_recurring_payments() ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-invalid-subscription',
				sprintf(
					/* translators: %s: PayPal subscription ID */
					__( 'Invalid subscription found for PayPal subscription ID: %s.', 'memberdash' ),
					$subscription_id
				),
				[
					'subscription_id' => $subscription_id,
				]
			);
		}

		$member  = $subscription->get_member();
		$invoice = $subscription->get_current_invoice();

		$event_type = MS_Helper_Cast::to_string(
			Arr::get( $event, 'event_type', '' )
		);

		switch ( $event_type ) {
			case 'PAYMENT.SALE.COMPLETED':
				$notes       = MS_Helper_Cast::to_string(
					Arr::get( $event, 'summary', '' )
				);
				$external_id = MS_Helper_Cast::to_string(
					Arr::get( $event, 'resource.id', '' )
				);
				$total_paid  = MS_Helper_Cast::to_float(
					Arr::get( $event, 'resource.amount.total', 0 )
				);

				// Stop if the payment has already been processed.
				if ( $subscription->is_payment_processed( $external_id ) ) {
					$invoice->add_notes( $notes );

					return true;
				}

				// Only for the first invoice.
				if ( $subscription->get_current_invoice_number() <= 1 ) {
					// Stop if the total paid is the same as the signup fee.
					if (
						$invoice->get_signup_fee() > 0
						&& $total_paid === $invoice->get_signup_fee()
					) {
						$invoice->add_notes(
							sprintf(
								// translators: %1$s: notes, %2$s: external ID.
								__( '%1$s (signup fee). External ID: %2$s', 'memberdash' ),
								$notes,
								$external_id
							)
						);

						return true;
					}

					// Set the external ID and add notes to the first invoice.
					if (
						$invoice->get_external_id() === ''
						&& $invoice->is_paid()
					) {
						$invoice->add_notes( $notes );

						// Update the first payment external ID.
						$payments = $subscription->get_payments();
						$update   = [
							[
								'external_id' => $external_id,
							],
						];
						$updated  = Arr::merge_recursive( $payments, $update );
						update_post_meta( $subscription->get_id(), 'payments', $updated );

						return true;
					}
				}

				if ( $invoice->is_paid() ) {
					$invoice = $subscription->get_next_billable_invoice();
				}

				$gateway = $this->get_gateway();
				$invoice->add_notes( $notes );
				$invoice->pay_it( $gateway->get_id(), $external_id );
				$invoice->save();

				// If the subscription is expired, try to activate it to avoid the user to be stuck in the expired state.
				if ( MS_Model_Relationship::STATUS_EXPIRED === $subscription->get_status() ) {
					$subscription->set_status( MS_Model_Relationship::STATUS_ACTIVE );
					$subscription->save();
				}

				/*
				 * Creates a transaction log entry.
				 *
				 * Documented in /includes/gateways/class-ms-controller-gateway.php.
				 */
				do_action(
					'ms_gateway_transaction_log',
					$gateway->get_id(), // Gateway ID.
					'handle', // Values: request|process|handle.
					true, // Success flag.
					$subscription->get_id(), // Subscription ID.
					$invoice->get_id(), // Invoice ID.
					$invoice->get_invoice_total(), // Charged amount.
					$notes, // Descriptive text.
					$external_id // External ID.
				);
				break;
			case 'PAYMENT.SALE.REFUNDED':
			case 'PAYMENT.SALE.REVERSED':
			case 'BILLING.SUBSCRIPTION.PAYMENT.FAILED':
				$member->cancel_membership( $membership->get_id() );
				break;
		}

		return true;
	}

	/**
	 * Processes the PayPal webhook event.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $event The PayPal webhook event.
	 *
	 * @return bool|WP_Error
	 */
	protected function process_event( array $event ) {
		if (
			empty( $event['event_type'] )
			|| empty( $event['resource'] )
		) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-event-invalid',
				__( 'Invalid PayPal webhook event', 'memberdash' ),
				[
					'event' => $event,
				]
			);
		}

		$event_type = MS_Helper_Cast::to_string(
			Arr::get( $event, 'event_type', '' )
		);

		if ( ! $this->is_valid_event( $event_type ) ) {
			return new WP_Error(
				'memberdash-gateway-paypal-express-webhook-event-invalid',
				sprintf(
					/* translators: %s: event type */
					__( 'Invalid PayPal webhook event type: %s.', 'memberdash' ),
					json_encode( $event )
				),
				[
					'event' => $event,
				]
			);
		}

		$is_subscription = $this->is_subscription_event( $event_type );

		if ( $is_subscription ) {
			$id = MS_Helper_Cast::to_string(
				Arr::get( $event, 'resource.billing_agreement_id', '' )
			);
		} else {
			$id = MS_Helper_Cast::to_string(
				Arr::get( $event, 'resource.supplementary_data.related_ids.order_id', '' )
			);
		}

		$link = $this->get_parent_payment_link(
			Arr::wrap(
				Arr::get( $event, 'resource.links', [] )
			)
		);

		if ( ! empty( $link ) ) {
			$parent_payment = $this->client_request(
				MS_Helper_Cast::to_string(
					Arr::get( $link, 'method' )
				),
				MS_Helper_Cast::to_string(
					Arr::get( $link, 'href' )
				),
				[],
				[],
				$this->get_gateway()->is_sandbox()
			);

			if ( is_wp_error( $parent_payment ) ) {
				return $parent_payment;
			}

			$id = MS_Helper_Cast::to_string(
				Arr::get( $parent_payment, 'id' )
			);
		}

		if ( $is_subscription ) {
			return $this->process_subscription_event( $id, $event );
		}

		return $this->process_order_event( $id, $event );
	}

	/**
	 * Creates all the webhooks needed for the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	public function create_webhooks() {
		$gateway = $this->get_gateway();

		$args = [
			'headers' => [
				'PayPal-Partner-Attribution-Id' => $gateway->get_partner_attribution_id(),
			],
			'body'    => [
				'url'         => $gateway->get_webhook_url(),
				'event_types' => array_map(
					static function ( $event_type ) {
						return [
							'name' => $event_type,
						];
					},
					$this->get_webhook_events()
				),
			],
		];

		$response = $this->client_post(
			'v1/notifications/webhooks',
			[],
			$args,
			$gateway->is_sandbox()
		);

		if (
			is_array( $response )
			&& empty( $response['id'] )
		) {
			$error = $this->json_decode(
				MS_Helper_Cast::to_string(
					Arr::get( $response, 'body', '' )
				)
			);

			$error_name = MS_Helper_Cast::to_string(
				Arr::get( $error, 'name', '' )
			);

			if ( empty( $error_name ) ) {
				$message = __( 'Unexpected PayPal response when creating webhook', 'memberdash' );
				MS_Helper_Debug::debug_log( $message );

				return new WP_Error( 'memberdash-gateway-paypal-express-webhook-unexpected', $message, $response );
			}

			if ( 'WEBHOOK_URL_ALREADY_EXISTS' === $error_name ) {
				return new WP_Error(
					'memberdash-gateway-paypal-express-webhook-url-already-exists',
					MS_Helper_Cast::to_string(
						Arr::get( $error, 'message', '' )
					),
					$response
				);
			}

			if ( 'WEBHOOK_NUMBER_LIMIT_EXCEEDED' === $error_name ) {
				$message = __( 'PayPal webhook limit has been reached, you need to go into your developer.paypal.com account and remove webhooks from the associated account', 'memberdash' );
				// Limit has been reached, we cannot just delete all webhooks without permission.
				MS_Helper_Debug::debug_log( $message );

				return new WP_Error( 'memberdash-gateway-paypal-express-webhook-limit-exceeded', $message, $response );
			}
		}

		return $response;
	}

	/**
	 * Updates a list of webhooks with the given ID.
	 *
	 * @since 1.5.0
	 *
	 * @param string $webhook_id The webhook list ID to update.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	public function update_webhooks( string $webhook_id ) {
		$gateway = $this->get_gateway();

		$args = [
			'headers' => [
				'PayPal-Partner-Attribution-Id' => $gateway->get_partner_attribution_id(),
			],
			'body'    => [
				[
					'op'    => 'replace',
					'path'  => '/url',
					'value' => $gateway->get_webhook_url(),
				],
				[
					'op'    => 'replace',
					'path'  => '/event_types',
					'value' => array_map(
						static function ( $event_type ) {
							return [
								'name' => $event_type,
							];
						},
						$this->get_webhook_events()
					),
				],
			],
		];

		$webhook_id = rawurlencode( $webhook_id );
		$url        = "v1/notifications/webhooks/{$webhook_id}";
		$response   = $this->client_patch(
			$url,
			[],
			$args,
			$gateway->is_sandbox()
		);

		if (
			is_array( $response )
			&& empty( $response['id'] )
		) {
			$error = $this->json_decode(
				MS_Helper_Cast::to_string(
					Arr::get( $response, 'body', '' )
				)
			);

			$error_name = MS_Helper_Cast::to_string(
				Arr::get( $error, 'name', '' )
			);

			if ( empty( $error_name ) ) {
				$message = __( 'Unexpected PayPal response when updating webhook', 'memberdash' );
				MS_Helper_Debug::debug_log( $message );

				return new WP_Error( 'memberdash-gateway-paypal-express-webhook-update-unexpected', $message );
			}

			if ( 'INVALID_RESOURCE_ID' === $error_name ) {
				return new WP_Error(
					'memberdash-gateway-paypal-express-webhook-update-invalid-id',
					MS_Helper_Cast::to_string(
						Arr::get( $error, 'message', '' )
					)
				);
			}
		}

		return $response;
	}

	/**
	 * Updates the webhook with the given ID.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed>|null $webhook The webhook data or null to use data from the database.
	 *
	 * @return bool
	 */
	public function webhooks_needs_update( $webhook = null ): bool {
		if ( ! is_array( $webhook ) ) {
			$webhook = $this->get_webhook_data();
		}

		$webhook = Arr::wrap( $webhook );

		// If these are not valid indexes, we just say we need an update.
		if ( ! isset( $webhook['url'], $webhook['event_types'] ) ) {
			return true;
		}

		$url = MS_Helper_Cast::to_string( $webhook['url'] );

		if ( $url !== $this->get_gateway()->get_webhook_url() ) {
			return true;
		}

		$types_data = is_array( $webhook['event_types'] )
			? $webhook['event_types']
			: [];

		$event_types = wp_list_pluck( $types_data, 'name' );

		$has_diff_events = array_diff( $this->get_webhook_events(), $event_types );
		if ( ! empty( $has_diff_events ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Fetches a list of webhooks.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string,mixed>
	 */
	public function list_webhooks() {
		$gateway = $this->get_gateway();

		$args = [
			'headers' => [
				'PayPal-Partner-Attribution-Id' => $gateway->get_partner_attribution_id(),
				'Prefer'                        => 'return=representation',
			],
			'body'    => [],
		];

		$response = $this->client_get(
			'v1/notifications/webhooks',
			[],
			$args,
			$gateway->is_sandbox()
		);

		if ( is_wp_error( $response ) ) {
			return [];
		}

		$response = Arr::wrap( $response );

		if ( empty( $response['webhooks'] ) ) {
			return [];
		}

		return $response['webhooks'];
	}

	/**
	 * Returns the webhook data from the database.
	 *
	 * @since 1.5.0
	 *
	 * @param string $index The index to fetch from the webhook data.
	 *
	 * @return mixed
	 */
	public function get_webhook_data( string $index = '' ) {
		$webhook_data = get_option( 'ms_paypal_express_webhook_data', [] );

		if ( empty( $index ) ) {
			return $webhook_data;
		}

		$webhook_data = Arr::wrap( $webhook_data );

		return Arr::get( $webhook_data, $index, '' );
	}

	/**
	 * Updates the webhook data in the database.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $webhook_data The webhook data to update.
	 *
	 * @return bool
	 */
	public function update_webhook_data( array $webhook_data ): bool {
		return update_option( 'ms_paypal_express_webhook_data', $webhook_data );
	}

	/**
	 * Deletes the webhook data from the database.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function delete_webhook_data(): bool {
		return delete_option( 'ms_paypal_express_webhook_data' );
	}

	/**
	 * Creates or updates the existing webhooks.
	 *
	 * @since 1.5.0
	 *
	 * @return bool|WP_Error
	 */
	public function create_or_update_existing_webhooks() {
		$existing_id = MS_Helper_Cast::to_string(
			$this->get_webhook_data( 'id' )
		);

		// If we don't have any existing data, we create the webhooks.
		if ( ! $existing_id ) {
			$webhook = $this->create_webhooks();

			// Update the settings if we have a webhook.
			if ( ! is_wp_error( $webhook ) ) {
				return $this->update_webhook_data( $webhook );
			}

			if ( 'memberdash-gateway-paypal-express-webhook-url-already-exists' === $webhook->get_error_code() ) {
				$gateway = $this->get_gateway();

				$error_message     = $webhook->get_error_message();
				$existing_webhooks = $this->list_webhooks();
				$existing_webhooks = array_filter(
					array_map(
						static function ( $webhook ) use ( $gateway ) {
							$webhook = Arr::wrap( $webhook );
							if ( $gateway->get_webhook_url() !== $webhook['url'] ) {
								return null;
							}

							return $webhook;
						},
						$existing_webhooks
					)
				);

				if ( empty( $existing_webhooks ) ) {
					return new WP_Error( 'memberdash-gateway-paypal-express-webhook-unexpected-update-create', $error_message );
				}

				$existing_webhook = current( $existing_webhooks );

				if ( ! $this->webhooks_needs_update( $existing_webhook ) ) {
					// We found a existing webhook that matched the URL but we just save it to the DB since it was up-to-date.
					return $this->update_webhook_data( $existing_webhook );
				}
			}

			// Returns the failed webhook creation or update.
			return $webhook;
		}

		if ( ! $this->webhooks_needs_update() ) {
			return true;
		}

		$webhook = $this->update_webhooks( $existing_id );
		// Update the settings if the webhook was updated.
		if ( ! is_wp_error( $webhook ) ) {
			return $this->update_webhook_data( $webhook );
		}

		$webhook = $this->create_webhooks();
		// Update the settings if a new webhook was created.
		if ( ! is_wp_error( $webhook ) ) {
			return $this->update_webhook_data( $webhook );
		}

		// Returns the failed webhook creation or update.
		return $webhook;
	}

	/**
	 * Returns a list of available webhooks.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	public function get_available_webhooks(): array {
		$events = $this->get_webhook_data( 'event_types' );

		if (
			empty( $events )
			|| ! is_array( $events )
		) {
			return [];
		}

		$webhooks = [];

		$event_names = [
			'CHECKOUT.ORDER.PROCESSED'            => __( 'Checkout order processed', 'memberdash' ),
			'PAYMENT.CAPTURE.COMPLETED'           => __( 'Completed payments', 'memberdash' ),
			'PAYMENT.CAPTURE.DENIED'              => __( 'Denied payments', 'memberdash' ),
			'PAYMENT.CAPTURE.REFUNDED'            => __( 'Refunds', 'memberdash' ),
			'PAYMENT.CAPTURE.REVERSED'            => __( 'Reversed', 'memberdash' ),
			'PAYMENT.SALE.COMPLETED'              => __( 'Completed subscriptions', 'memberdash' ),
			'PAYMENT.SALE.REFUNDED'               => __( 'Refunded subscriptions', 'memberdash' ),
			'PAYMENT.SALE.REVERSED'               => __( 'Reversed subscriptions', 'memberdash' ),
			'BILLING.SUBSCRIPTION.PAYMENT.FAILED' => __( 'Failed subscription payments', 'memberdash' ),
		];

		foreach ( $events as $event ) {
			if ( ! array_key_exists( $event['name'], $event_names ) ) {
				continue;
			}

			$webhooks[] = $event_names[ $event['name'] ];
		}

		return $webhooks;
	}

	/**
	 * Handles the PayPal webhook events.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function handler(): void {
		$raw_data = file_get_contents( 'php://input' );

		if ( empty( $raw_data ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal webhook event', 'memberdash' ),
				]
			);
		}

		$event = $this->json_decode(
			MS_Helper_Cast::to_string( $raw_data )
		);

		$event_type = MS_Helper_Cast::to_string(
			Arr::get( $event, 'event_type', '' )
		);

		if ( ! $this->is_valid_event( $event_type ) ) {
			wp_send_json_error(
				[
					'message' => sprintf(
						/* translators: %s: event type */
						__( 'Invalid PayPal webhook event type: %s.', 'memberdash' ),
						json_encode( $event )
					),
				]
			);
		}

		$webhook_id     = MS_Helper_Cast::to_string(
			$this->get_webhook_data( 'id' )
		);
		$headers        = Arr::wrap(
			SuperGlobals::get_raw_superglobal( 'SERVER' )
		);
		$parsed_headers = $this->parse_headers( $headers );

		if ( is_wp_error( $parsed_headers ) ) {
			wp_send_json_error(
				[
					'message' => $parsed_headers->get_error_message(),
				]
			);
		}

		if ( ! $this->verify_webhook_signature( $webhook_id, $event, $parsed_headers ) ) {
			wp_send_json_error(
				[
					'message'    => __( 'Failed PayPal webhook event verification.', 'memberdash' ),
					'webhook_id' => $webhook_id,
					'event'      => $event,
					'headers'    => $parsed_headers,
				]
			);
		}

		$debug_header = MS_Helper_Cast::to_string(
			Arr::get( $parsed_headers, 'debug_id', '' )
		);
		if ( ! empty( $debug_header ) ) {
			$event = Arr::wrap(
				Arr::add( $event, 'debug_id', $debug_header )
			);
		}

		$response = $this->process_event( $event );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error(
				[
					'message' => $response->get_error_message(),
				]
			);
		}

		wp_send_json_success(
			[
				'message' => __( 'PayPal webhook event processed successfully.', 'memberdash' ),
			]
		);
	}
}
