<?php
/**
 * Gateway: Paypal Express - Settings View
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

/**
 * PayPal Express gateway settings view class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_View_Settings extends MS_View {
	/**
	 * Returns the HTML for the settings page.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function to_html() {
		$fields  = $this->prepare_fields();
		$gateway = $this->data['model'];
		$errors  = $gateway->get_errors_from_on_boarded_data();

		ob_start();

		if (
			$gateway->is_connected()
			&& ! empty( $errors )
		) :
			foreach ( $errors as $error ) :
				?>
				<div class="notice notice-error">
					<p>
						<?php
						echo wp_kses(
							$error,
							[
								'a' => [
									'href'   => [],
									'target' => [],
								],
							]
						);
						?>
					</p>
				</div>
				<?php
			endforeach;
		endif;
		?>
		<form class="ms-gateway-settings-form ms-form">
		<?php
			$description = __( 'PayPal Checkout accepts all payment methods enabled in your PayPal account. This gateway is not suited for recurring payments.', 'memberdash' );

			MS_Helper_Html::settings_box_header( '', $description );

		if ( ! is_ssl() ) :
			?>
				<div class="ms-error-text">
					<p>
					<?php
					$doc_link = sprintf(
						'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
						'https://memberdashwp.com/docs/memberdash/billing-and-payments/paypal-checkout-payment-gateway/',
						esc_html__( 'Learn more', 'memberdash' )
					);

					echo wp_kses(
						sprintf(
							/* translators: %s - URL to the documentation */
							__( 'A valid SSL certificate and secure (https) connection are required to connect with PayPal. %s.', 'memberdash' ),
							$doc_link
						),
						[
							'a' => [
								'href'   => [],
								'target' => [],
								'rel'    => [],
							],
						]
					);
					?>
					</p>
				</div>
			<?php
			else :
				foreach ( $fields as $field ) {
					MS_Helper_Html::html_element( $field );
				}
			endif;

			MS_Helper_Html::settings_box_footer();
			?>
		</form>

		<?php
		return MS_Helper_Cast::to_string( ob_get_clean() );
	}

	/**
	 * Prepares the fields for the settings page.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string,mixed>
	 */
	protected function prepare_fields(): array {
		$gateway       = $this->data['model'];
		$action        = MS_Controller_Gateway::AJAX_ACTION_UPDATE_GATEWAY;
		$nonce         = wp_create_nonce( $action );
		$country_codes = MS_Gateway::get_country_codes();
		$modes         = $gateway->get_mode_types();

		$logo_url = MS_Plugin::instance()->get_url() . 'app/assets/images/gateways/paypalexpress/logo-white.svg';
		$logo     = sprintf(
			'<img src="%s" alt="%s" class="ms-gateway-logo">',
			esc_url( $logo_url ),
			esc_attr__( 'PayPal Checkout', 'memberdash' )
		);

		$fields = [
			'mode'            => [
				'id'            => 'mode',
				'title'         => esc_html__( 'PayPal Mode', 'memberdash' ),
				'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
				'value'         => $gateway->get_mode(),
				'field_options' => $modes,
				'class'         => 'ms-text-large',
				'ajax_data'     => [
					'field'      => 'mode',
					'_wpnonce'   => $nonce,
					'action'     => $action,
					'gateway_id' => $gateway->get_id(),
				],
			],
			'account_country' => [
				'id'            => 'account_country',
				'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
				'title'         => esc_html__( 'Account Country', 'memberdash' ),
				'field_options' => $country_codes,
				'value'         => $gateway->get_account_country(),
				'desc'          => esc_html__( 'Select the country where your PayPal account is registered.', 'memberdash' ),
				'class'         => 'ms-text-large',
				'ajax_data'     => [
					'field'      => 'account_country',
					'_wpnonce'   => $nonce,
					'action'     => $action,
					'gateway_id' => $gateway->get_id(),
				],
			],
			'connect_button'  => [
				'id'     => 'connect_button',
				'type'   => MS_Helper_Html::TYPE_HTML_LINK,
				'title'  => esc_html__( 'Connect PayPal Checkout', 'memberdash' ),
				'value'  => '<span class="ms-text-lg">' . $logo . esc_html__( 'Connect PayPal Checkout', 'memberdash' ) . '</span>',
				'url'    => '',
				'target' => 'PPFrame',
				'class'  => 'button button-primary ms-bg-black ms-border-black ms-text-white ms-shadow-none ms-text-center disabled',
				'data'   => [
					'paypal-onboard-complete' => 'msGatewayPayPalExpressSignupCallback',
					'paypal-button'           => true,
				],
			],
		];

		if ( $gateway->is_connected() ) {
			unset( $fields['connect_button'] );

			$fields['mode']['type']  = MS_Helper_Html::TYPE_HTML_TEXT;
			$fields['mode']['desc']  = esc_html__( 'Disconnect your PayPal account to change the mode.', 'memberdash' );
			$fields['mode']['value'] = $modes[ $gateway->get_mode() ];

			$fields['account_country']['type']  = MS_Helper_Html::TYPE_HTML_TEXT;
			$fields['account_country']['desc']  = '';
			$fields['account_country']['value'] = $country_codes[ $gateway->get_account_country() ];

			$fields['connected_as'] = [
				'id'    => 'connected_as',
				'type'  => MS_Helper_Html::TYPE_HTML_TEXT,
				'title' => esc_html__( 'Connected As', 'memberdash' ),
				'value' => '<strong>' . esc_html( $gateway->get_merchant_id() ) . '</strong>',
			];

			$fields['account_id'] = [
				'id'    => 'account_id',
				'type'  => MS_Helper_Html::TYPE_HTML_TEXT,
				'title' => esc_html__( 'PayPal Account ID', 'memberdash' ),
				'value' => esc_html( $gateway->get_account_id() ),
			];

			$fields['granted_scopes'] = [
				'id'    => 'granted_scopes',
				'type'  => MS_Helper_Html::TYPE_HTML_TEXT,
				'title' => esc_html__( 'API Granted Scopes', 'memberdash' ),
				'value' => $this->generate_list( $this->get_granted_scopes() ),
			];

			$fields['webhooks'] = [
				'id'    => 'webhooks',
				'type'  => MS_Helper_Html::TYPE_HTML_TEXT,
				'title' => esc_html__( 'Webhooks', 'memberdash' ),
				'value' => $this->generate_list( $gateway->get_available_webhooks() ),
			];

			$fields['reconnect_button'] = [
				'id'    => 'reconnect_button',
				'type'  => MS_Helper_Html::TYPE_HTML_LINK,
				'value' => esc_html__( 'Resync Payment Connection and Recreate Webhooks', 'memberdash' ),
				'title' => esc_html__( 'Resync your PayPal account with MemberDash.', 'memberdash' ),
				'url'   => wp_nonce_url( add_query_arg( 'action', 'ms_paypal_express_reconnect', admin_url( 'admin-ajax.php' ) ), 'reconnect_account' ),
				'class' => 'button button-primary ms-text-center',
			];

			$fields['disconnect_button'] = [
				'id'    => 'disconnect_button',
				'type'  => MS_Helper_Html::TYPE_HTML_LINK,
				'value' => $logo . esc_html__( 'Disconnect PayPal Checkout', 'memberdash' ),
				'title' => esc_html__( 'Disconnect your PayPal Checkout account.', 'memberdash' ),
				'url'   => wp_nonce_url( add_query_arg( 'action', 'ms_paypal_express_disconnect', admin_url( 'admin-ajax.php' ) ), 'disconnect_account' ),
				'class' => 'button button-primary ms-bg-black ms-border-black ms-text-white ms-shadow-none ms-text-center',
			];
		}

		return $fields;
	}

	/**
	 * Generates a list of items.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string> $items The items to list.
	 *
	 * @return string
	 */
	private function generate_list( array $items ): string {
		if ( empty( $items ) ) {
			return '';
		}

		$list = array_map(
			function ( $item ) {
				return '<li>' . $item . '</li>';
			},
			$items
		);

		return '<ul>' . implode( '', $list ) . '</ul>';
	}

	/**
	 * Returns the granted scopes in a human-readable format.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	private function get_granted_scopes(): array {
		$gateway = $this->data['model'];

		$scopes = $gateway->get_granted_scopes();

		if ( empty( $scopes ) ) {
			return [];
		}

		$features = [
			'PAYMENT' => esc_html__( 'Payment', 'memberdash' ),
			'REFUND'  => esc_html__( 'Refund', 'memberdash' ),
		];

		return array_map(
			function ( $scope ) use ( $features ) {
				return isset( $features[ $scope ] )
					? $features[ $scope ]
					: $scope;
			},
			$scopes
		);
	}
}
