<?php
/**
 * WooCommerce Helper Utility.
 *
 * @since 1.4.0
 *
 * @package MemberDash
 */

/**
 * WooCommerce Helper Utility class.
 *
 * @since 1.4.0
 */
class MS_Helper_WooCommerce extends MS_Helper {
	/**
	 * Checks if WooCommerce is activated.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	public static function is_woocommerce_activated(): bool {
		return class_exists( 'WooCommerce' );
	}

	/**
	 * Checks if WooCommerce Subscriptions is activated.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	public static function is_subscriptions_activated(): bool {
		return class_exists( 'WC_Subscriptions' );
	}

	/**
	 * Checks if WooCommerce and WooCommerce Subscriptions are activated.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	public static function is_activated(): bool {
		return self::is_woocommerce_activated() && self::is_subscriptions_activated();
	}

	/**
	 * Returns a list of order statuses by access status.
	 *
	 * @since 1.4.0
	 *
	 * @param bool $granted True if access is granted, false otherwise.
	 *
	 * @return array<string> Order statuses.
	 */
	public static function get_order_statuses_by_access_status( bool $granted ): array {
		$granted_statuses = [
			'completed',
			'processing',
		];

		if ( $granted ) {
			return $granted_statuses;
		}

		$statuses = array_map(
			function ( $status ) {
				return str_replace( 'wc-', '', $status );
			},
			array_keys(
				wc_get_order_statuses()
			)
		);

		return array_values(
			array_diff( $statuses, $granted_statuses )
		);
	}

	/**
	 * Returns a list of subscription statuses by access status.
	 *
	 * @since 1.4.0
	 *
	 * @param bool $granted True if access is granted, false otherwise.
	 *
	 * @return array<string> Subscription statuses.
	 */
	public static function get_subscription_statuses_by_access_status( bool $granted ): array {
		if ( ! function_exists( 'wcs_get_subscription_statuses' ) ) {
			return [];
		}

		$granted_statuses = [
			'active',
			'pending-cancel',
		];

		if ( $granted ) {
			return $granted_statuses;
		}

		$statuses = array_map(
			function ( $status ) {
				return str_replace( 'wc-', '', $status );
			},
			array_keys(
				wcs_get_subscription_statuses()
			)
		);

		return array_values(
			array_diff( $statuses, $granted_statuses )
		);
	}

	/**
	 * Returns the WooCommerce order from the invoice.
	 *
	 * @since 1.4.0
	 *
	 * @param MS_Model_Invoice $invoice The invoice instance.
	 *
	 * @return WC_Order|WC_Order_Refund|WC_Subscription|null
	 */
	public static function get_order_from_invoice( MS_Model_Invoice $invoice ) {
		$external_id = $invoice->get_external_id();

		if ( ! $external_id ) {
			return null;
		}

		// Stop if isn't a WooCommerce order.
		if ( strpos( $external_id, 'wc_' ) !== 0 ) {
			return null;
		}

		$order_id = MS_Helper_Cast::to_int(
			str_replace( 'wc_', '', $external_id )
		);

		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return null;
		}

		return $order;
	}

	/**
	 * Returns a list of membership IDs from the WooCommerce order.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Order|WC_Order_Refund|WC_Subscription $order    The WooCommerce order.
	 * @param bool                                     $is_order True if the order is a WC_Order instance, false otherwise.
	 *
	 * @return array<int>
	 */
	public static function get_memberships_from_order( $order, bool $is_order = false ): array {
		$items = $order->get_items();

		$memberships = [];

		foreach ( $items as $item ) {
			if ( ! $item instanceof WC_Order_Item_Product ) {
				continue;
			}

			$product = $item->get_product();

			if ( ! is_object( $product ) ) {
				continue;
			}

			// If is an order filter out subscriptions.
			if (
				$is_order
				&& $product->get_type() === 'subscription'
			) {
				continue;
			}

			$data = $product->get_meta( '_ms_memberships', true );

			if (
				empty( $data )
				|| ! is_array( $data )
			) {
				continue;
			}

			$memberships = array_merge( $memberships, $data );
		}

		$memberships = array_map(
			'MS_Helper_Cast::to_int',
			array_filter(
				array_unique( $memberships )
			)
		);

		return $memberships;
	}

	/**
	 * Checks if the cart contains at least one membership.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	public static function cart_contains_membership(): bool {
		$cart = WC()->cart;

		// Skip if the cart is not available.
		if ( ! $cart instanceof WC_Cart ) {
			return false;
		}

		foreach ( $cart->get_cart() as $item ) {
			$product = $item['data'];

			if (
				! is_object( $product )
				|| ! $product instanceof WC_Product
			) {
				continue;
			}

			$memberships = $product->get_meta( '_ms_memberships', true );

			if (
				empty( $memberships )
				|| ! is_array( $memberships )
			) {
				continue;
			}

			return true;
		}

		return false;
	}
}
