<?php
/**
 * Dialog: Member Payment Infos
 *
 * Extends MS_View for rendering methods and magic methods.
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Member Payment Infos dialog class.
 *
 * @since 1.0.0
 */
class MS_View_Member_Payment extends MS_Dialog {

	/**
	 * Generate/Prepare the dialog attributes.
	 *
	 * @since 1.0.0
	 */
	public function prepare() {
		$subscription_id = $_POST['subscription_id'];
		$subscription    = MS_Factory::load( 'MS_Model_Relationship', $subscription_id );

		$data = array(
			'model' => $subscription,
		);

		$data = apply_filters( 'ms_view_member_payment_data', $data );

		// Dialog Title
		$this->title = sprintf(
			__( 'Subscription Details: %1$s', 'memberdash' ),
			esc_html( $subscription->get_membership()->name )
		);

		// Dialog Size
		$this->width  = 940;
		$this->height = 600;

		// Contents
		$this->content = $this->get_contents( $data );

		// Make the dialog modal
		$this->modal = true;
	}

	/**
	 * Save the dialog details.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function submit() {
		// Does nothing...
	}

	/**
	 * Returns the content of the dialog
	 *
	 * @param array $data The data.
	 *
	 * @since 1.0.0
	 *
	 * @return object
	 */
	public function get_contents( $data ) {
		$subscription = $data['model'];
		$gateways     = MS_Model_Gateway::get_gateway_names();
		$invoices     = $subscription->get_invoices();

		$pay_details = array();
		$inv_details = array();

		foreach ( $subscription->get_payments() as $payment ) {
			if ( isset( $gateways[ $payment['gateway'] ] ) ) {
				$gateway = $gateways[ $payment['gateway'] ];
			} else {
				$gateway = '(' . $payment['gateway'] . ')';
			}

			$pay_details[] = array(
				'title'         => __( 'Recorded Payment', 'memberdash' ),
				'type'          => MS_Helper_Html::TYPE_HTML_TABLE,
				'value'         => array(
					0 => array( 'Payment Date', $payment['date'] ),
					1 => array( 'Payment Gateway', $gateway ),
					2 => array( 'Amount', $payment['amount'] ),
					3 => array( 'External ID', $payment['external_id'] ),
				),
				'field_options' => array(
					'head_col' => true,
				),
			);
		}

		$invoice_list = array();
		foreach ( $invoices as $invoice ) {
			if ( isset( $gateways[ $invoice->gateway_id ] ) ) {
				$gateway = $gateways[ $invoice->gateway_id ];
			} else {
				$gateway = '(' . $invoice->gateway_id . ')';
			}

			$transaction_log = sprintf(
				' <small>- <a href="%s" target="_blank">%s</a></small>',
				MS_Controller_Plugin::get_admin_url(
					'billing',
					array(
						'show'    => 'logs',
						'invoice' => $invoice->id,
					)
				),
				__( 'Show Transaction', 'memberdash' )
			);
			$invoice_list[]  = $invoice->id;

			$coupon_discount = '0';

			if ( $invoice->get_discount() ) {
				$coupon_discount = sprintf(
					'-%s',
					MS_Helper_Billing::format_price( $invoice->get_discount() )
				);

				if ( $invoice->get_coupon_id() ) {
					$coupon_type = MS_Addon_Coupon_Model::DURATION_ONCE === $invoice->get_coupon_duration()
						? __( 'Once', 'memberdash' )
						: __( 'Forever', 'memberdash' );

					$coupon_discount = sprintf(
						'%s (%s)',
						$coupon_discount,
						$coupon_type
					);
				}
			}

			$inv_details[ $invoice->id ] = [
				'title'         => sprintf(
					// Translators: %s is the invoice ID.
					__( 'Invoice %s', 'memberdash' ),
					$invoice->id
				),
				'type'          => MS_Helper_Html::TYPE_HTML_TABLE,
				'value'         => [
					[
						__( 'Invoice ID', 'memberdash' ),
						$invoice->get_id() . $transaction_log,
					],
					[
						__( 'Create Date', 'memberdash' ),
						$invoice->invoice_date,
					],
					[
						__( 'Payment Gateway', 'memberdash' ),
						$gateway,
					],
					[
						__( 'Due Date', 'memberdash' ),
						$invoice->due_date,
					],
					[
						__( 'Regular amount', 'memberdash' ),
						$invoice->amount,
					],
					[
						__( 'Signup fee', 'memberdash' ),
						$invoice->get_signup_fee(),
					],
					[
						__( 'Coupon discount', 'memberdash' ),
						$coupon_discount,
					],
					[
						__( 'Total billed', 'memberdash' ),
						$invoice->total,
					],
					[
						__( 'Status', 'memberdash' ),
						$invoice->status,
					],
					[
						__( 'Notes', 'memberdash' ),
						$invoice->description,
					],
					[
						__( 'Checkout IP', 'memberdash' ),
						$invoice->checkout_ip,
					],
					[
						__( 'Checkout Date', 'memberdash' ),
						$invoice->checkout_date,
					],
				],
				'field_options' => [
					'head_col' => true,
				],
			];

			if ( $invoice->amount == $invoice->total ) {
				unset( $inv_details[ $invoice->id ]['value'][3] );
				$inv_details[ $invoice->id ]['value'] = array_values( $inv_details[ $invoice->id ]['value'] );
			}
		}

		$transaction_url = MS_Controller_Plugin::get_admin_url(
			'billing',
			array(
				'show'    => 'logs',
				'invoice' => implode( ',', $invoice_list ),
			)
		);
		$transaction_log = array(
			'type'   => MS_Helper_Html::TYPE_HTML_LINK,
			'value'  => __( 'Show all Transactions for this subscription', 'memberdash' ),
			'url'    => $transaction_url,
			'target' => '_blank',
		);

		ob_start();
		?>
		<div class="memberdash-grid-8 ms-payment-infos">
			<div class="col-5">
				<?php
				foreach ( $inv_details as $detail ) {
					MS_Helper_Html::html_element( $detail );
				}
				?>
			</div>
			<div class="col-3">
				<?php
				MS_Helper_Html::html_element( $transaction_log );
				MS_Helper_Html::html_separator();
				foreach ( $pay_details as $detail ) {
					MS_Helper_Html::html_element( $detail );
				}
				?>
			</div>
		</div>
		<?php
		$html = ob_get_clean();
		return apply_filters( 'ms_view_member_payment_to_html', $html );
	}

};
