<?php
/**
 * Payment Settings Page
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Payment Settings View Class
 *
 * @since 1.0.0
 */
class MS_View_Settings_Page_Payment extends MS_View_Settings_Edit {
	/**
	 * Overrides parent's to_html() method.
	 *
	 * Creates an output buffer, outputs the HTML and grabs the buffer content before releasing it.
	 * HTML contains the list of available payment gateways.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function to_html() {
		$show_global_payment_settings = isset( $this->data['show_global_payment_settings'] ) ?
			$this->data['show_global_payment_settings'] : true;

		ob_start();
		?>
		<div id="ms-payment-settings-wrapper">
			<div class="ms-global-payment-wrapper">
				<div class="ms-list-table-wrapper">
					<?php
					if ( $show_global_payment_settings ) {
						MS_Helper_Html::settings_tab_header(
							[
								'title' => __( 'Global Payment Settings', 'memberdash' ),
								'desc'  => __( 'These are shared across all memberships.', 'memberdash' ),
							]
						);

						MS_Helper_Html::settings_box( $this->get_global_payment_fields() );
					} else {
						// Required to close the tab header later.
						MS_Helper_Html::settings_tab_header();
					}

					MS_Helper_Html::settings_box_header(
						__( 'Payment Gateways', 'memberdash' ),
						sprintf(
							/* translators: %s: docs link */
							__( 'MemberDash requires at least one payment gateway to be set up in order to collect payments. %s', 'memberdash' ),
							'<a href="https://memberdashwp.com/docs/memberdash/billing-and-payments/" target="_blank">' . __( 'Learn More', 'memberdash' ) . '</a>'
						)
					);
					?>
						<div class="gateways ms-mt-6">
							<?php $this->gateway_settings(); ?>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
		$html = ob_get_clean();

		return $html;
	}

	/**
	 * Prepares a list with field definitions that are required to render the
	 * payment list/global options (i.e. currency and sender name)
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_global_payment_fields() {
		$settings = MS_Factory::load( 'MS_Model_Settings' );
		$action   = MS_Controller_Settings::AJAX_ACTION_UPDATE_SETTING;
		$nonce    = wp_create_nonce( $action );

		$fields = array(
			'currency'               => array(
				'id'            => 'currency',
				'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
				'title'         => __( 'Payment currency', 'memberdash' ),
				'value'         => $settings->currency,
				'field_options' => $settings->get_currencies(),
				'class'         => 'ms-select ms-w-full',
				'data_ms'       => array(
					'field' => 'currency',
				),
			),

			'invoice_sender_name'    => array(
				'id'      => 'invoice_sender_name',
				'type'    => MS_Helper_Html::INPUT_TYPE_TEXT,
				'title'   => __( 'Invoice sender name', 'memberdash' ),
				'value'   => $settings->invoice_sender_name,
				'class'   => 'ms-w-full',
				'data_ms' => array(
					'field' => 'invoice_sender_name',
				),
			),

			'billing_address'        => array(
				'id'      => 'billing_address',
				'type'    => MS_Helper_Html::INPUT_TYPE_TEXT,
				'title'   => __( 'Billing address', 'memberdash' ),
				'value'   => $settings->billing_address,
				'class'   => 'ms-w-full',
				'data_ms' => array(
					'field' => 'billing_address',
				),
			),

			'company_name'           => array(
				'id'      => 'company_name',
				'type'    => MS_Helper_Html::INPUT_TYPE_TEXT,
				'title'   => __( 'Company name', 'memberdash' ),
				'value'   => $settings->company_name,
				'class'   => 'ms-w-full',
				'data_ms' => array(
					'field' => 'company_name',
				),
			),

			'company_vax_tax_number' => array(
				'id'      => 'company_vax_tax_number',
				'type'    => MS_Helper_Html::INPUT_TYPE_TEXT,
				'title'   => __( 'Company VAT/TAX number', 'memberdash' ),
				'value'   => $settings->company_vax_tax_number,
				'class'   => 'ms-w-full',
				'data_ms' => array(
					'field' => 'company_vax_tax_number',
				),
			),
		);

		foreach ( $fields as $key => $field ) {
			if ( is_array( $field['data_ms'] ) ) {
				$fields[ $key ]['data_ms']['_wpnonce'] = $nonce;
				$fields[ $key ]['data_ms']['action']   = $action;
			}
		}

		return apply_filters( 'ms_gateway_view_get_global_payment_fields', $fields );
	}

	/**
	 * Displays the edit form for all payment gateways.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	protected function gateway_settings() {
		$gateways = MS_Model_Gateway::get_gateways();
		$groups   = [];

		// Sort gateways by priority.
		usort(
			$gateways,
			function ( $a, $b ) {
				return MS_Helper_Cast::to_int( $a->get_priority() >= $b->get_priority() );
			}
		);

		foreach ( $gateways as $gateway ) {
			$group = $gateway->get_group();
			if ( empty( $group ) ) {
				continue;
			}

			// Hide Stripe Plan Settings - Merged to Stripe
			if ( MS_Gateway_Stripeplan::ID === $gateway->get_id() ) {
				continue;
			}
			$groups[ $group ]                       = mslib3()->array->get( $groups[ $group ] );
			$groups[ $group ][ $gateway->get_id() ] = $gateway;
		}

		/**
		 * Allows to filter the payment gateway settings groups.
		 *
		 * @since 1.5.0
		 *
		 * @param array<string,array<string,MS_Gateway>> $groups The payment gateway settings groups.
		 *
		 * @return array<string,array<string,MS_Gateway>>
		 */
		$groups = apply_filters(
			'ms_gateway_settings_groups',
			$groups
		);

		foreach ( $groups as $label => $group ) :
			?>
			<div>
				<?php
				foreach ( $group as $gateway ) {
					$this->gateway_item_settings( $gateway );
				}
				?>
			</div>
			<?php
		endforeach;
	}

	/**
	 * Displays the edit form for a single payment gateway.
	 *
	 * @since 1.0.0
	 *
	 * @param MS_Gateway $gateway The gateway model.
	 *
	 * @return void
	 */
	protected function gateway_item_settings( $gateway ) {
		$is_online    = mslib3()->net->is_online( MS_Helper_Utility::home_url( '/' ) );
		$row_class    = 'gateway-' . $gateway->get_id();
		$active_class = 'ms-gateway-not-configured';

		if ( $gateway->is_configured() ) {
			$row_class   .= ' is-configured';
			$active_class = 'ms-gateway-configured';
		} else {
			$row_class .= ' not-configured';
		}

		if ( $gateway->is_live_mode() ) {
			$row_class .= ' is-live';
		} else {
			$row_class .= ' is-sandbox';
		}

		if ( ! $is_online ) {
			$row_class .= ' is-offline';
		} else {
			$row_class .= ' is-online';
		}

		$actions = array(
			sprintf(
				'<a href="%1$s">%2$s</a>',
				MS_Controller_Plugin::get_admin_url(
					'billing',
					[ 'gateway_id' => $gateway->get_id() ]
				),
				__( 'View Transactions', 'memberdash' )
			),
			sprintf(
				'<a href="%1$s">%2$s</a>',
				MS_Controller_Plugin::get_admin_url(
					'billing',
					array(
						'show'       => 'logs',
						'gateway_id' => $gateway->get_id(),
					)
				),
				__( 'View Logs', 'memberdash' )
			),
		);

		$actions = apply_filters(
			'gateway_settings_actions',
			$actions,
			$gateway
		);

		$action_tag = array();
		foreach ( $actions as $action => $link ) {
			$action_tag[] = "<span class='$action'>$link</span>";
		}

		$toggle = array(
			'id'      => 'ms-toggle-' . $gateway->get_id(),
			'type'    => MS_Helper_Html::INPUT_TYPE_RADIO_SLIDER,
			'value'   => $gateway->is_activated(),
			'data_ms' => array(
				'action'     => MS_Controller_Gateway::AJAX_ACTION_TOGGLE_GATEWAY,
				'gateway_id' => $gateway->get_id(),
			),
		);

		$settings_class = 'MS_Gateway_' . ucwords( esc_attr( $gateway->get_id() ) ) . '_View_Settings';
		$settings       = MS_Factory::create( $settings_class );
		$settings->data = array( 'model' => $gateway );

		// -- Output the form --

		?>
		<div class="ms-gateway-item ms-mt-4 <?php echo esc_attr( $row_class ); ?>">
			<span class="gateway-title ms-flex ms-justify-between ms-w-full ms-bg-gray-50 ms-py-3 ms-px-5 ms-rounded-md">
				<label class="gateway-toggle gateway-name" for="settings-<?php echo esc_attr( $gateway->get_id() ); ?>">
					<i class="row-status-close memberdash-fa memberdash-fa-caret-right"></i>
					<i class="row-status-open memberdash-fa memberdash-fa-caret-down"></i>
					<?php
					if ( $gateway->get_admin_logo_url() ) :
						?>
						<img src="<?php echo esc_url( $gateway->get_admin_logo_url() ); ?>" alt="<?php echo esc_html( $gateway->get_name() ); ?>" class="gateway-logo">
						<?php
					endif;

					echo esc_html( $gateway->get_name() );
					?>
				</label>
				<span class="gateway-description">
					<?php echo esc_html( $gateway->get_description() ); ?>
				</span>
				<span class="memberdash-fa offline-flag" title="<?php echo esc_html__( 'Website seems to be not publicly available. This payment method might not work.', 'memberdash' ); ?>">
				</span>
			</span>

			<div class="ms-gateway-status <?php echo esc_attr( $active_class ); ?> ms-active-wrapper-<?php echo esc_attr( $gateway->get_id() ); ?>">
				<?php MS_Helper_Html::html_element( $toggle ); ?>
			</div>

			<input type="checkbox" class="show-settings" id="settings-<?php echo esc_attr( $gateway->get_id() ); ?>"/>
			<div class="ms-gateway-settings">
				<?php echo $settings->to_html(); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
			</div>

		</div>
		<?php
	}

}
