<?php
/**
 * WooCommerce Product Options View.
 *
 * @since 1.4.0
 *
 * @package MemberDash
 */

/**
 * WooCommerce Product Options View class.
 *
 * @since 1.4.0
 */
class MS_View_WooCommerce_Product_Options extends MS_View {
	/**
	 * Renders the product options.
	 *
	 * @since 1.4.0
	 *
	 * @return string
	 */
	public function to_html(): string {
		$memberships = $this->get_current_memberships();

		ob_start();
		?>

		<div class="options_group show_if_simple show_if_subscription ms-product-options">
			<h2><?php esc_html_e( 'MemberDash', 'memberdash' ); ?></h2>

			<?php
			$attributes = [
				'data-sortable'    => 'true',
				'data-placeholder' => __( 'Search for a membership&hellip;', 'memberdash' ),
				'data-action'      => 'memberdash_woocommerce_json_search_memberships',
			];

			if ( MS_Model_Addon::is_enabled( 'multi_memberships' ) ) {
				$attributes['multiple'] = 'multiple';
			}

			woocommerce_wp_select(
				[
					'id'                => '_ms_memberships',
					'label'             => __( 'Membership', 'memberdash' ),
					'name'              => '_ms_memberships[]',
					'value'             => array_keys( $memberships ),
					'options'           => $memberships,
					'class'             => 'ms-membership-search wc-product-search',
					'style'             => 'width: 50%;',
					'custom_attributes' => $attributes,
				]
			);

			woocommerce_wp_checkbox(
				[
					'id'          => '_ms_sold_individually',
					'label'       => __( 'Sold individually', 'memberdash' ),
					'description' => __( 'Limit purchases to 1 item per order (this setting also exists in the Inventory tab.)', 'memberdash' ),
					'value'       => $this->is_sold_individually() ? 'yes' : 'no',
				]
			);
			?>
		</div>

		<?php
		return MS_Helper_Cast::to_string( ob_get_clean() );
	}

	/**
	 * Gets the sold individually value.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	protected function is_sold_individually(): bool {
		$product = wc_get_product( $this->data['post']->ID );

		if ( ! $product ) {
			return false;
		}

		return $product->is_sold_individually();
	}

	/**
	 * Gets the current memberships.
	 *
	 * @since 1.4.0
	 *
	 * @return array<int,string> The current memberships.
	 */
	protected function get_current_memberships(): array {
		$product = wc_get_product( $this->data['post']->ID );

		if ( ! $product ) {
			return [];
		}

		$ids = $product->meta_exists( '_ms_memberships' )
			? $product->get_meta( '_ms_memberships', true )
			: [];

		if ( ! is_array( $ids ) ) {
			$ids = [];
		}

		$memberships = [];

		foreach ( $ids as $id ) {
			$membership = MS_Factory::load(
				'MS_Model_Membership',
				MS_Helper_Cast::to_int( $id )
			);

			if ( $membership->is_valid() ) {
				$memberships[ $membership->get_id() ] = $membership->get_name();
			}
		}

		return $memberships;
	}
}
