<?php
/**
 * Settings view of the Add-on
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Profile fields settings view class.
 *
 * @since 1.0.0
 */
class MS_Addon_Profilefields_View_Settings extends MS_View {

	/**
	 * Return the Form HTML code.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function render_tab() {
		$fields = $this->prepare_fields();
		ob_start();
		?>
		<div class="ms-addon-wrap">
			<form method="post">
			<?php
			MS_Helper_Html::settings_tab_header(
				array( 'title' => __( 'Profile Fields Settings', 'memberdash' ) )
			);

			foreach ( $fields as $field ) {
				MS_Helper_Html::html_element( $field );
			}
			?>
			</form>
		</div>
		<?php
		$html = ob_get_clean();
		echo $html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Returns the field specifications for the form.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public function prepare_fields() {
		$settings = $this->data['settings'];
		$action   = 'save';

		$action         = MS_Controller_Settings::AJAX_ACTION_UPDATE_CUSTOM_SETTING;
		$profile_fields = MS_Addon_Profilefields::list_fields();

		$reg_config      = $settings->get_custom_setting( 'profilefields', 'register' );
		$reg_config      = mslib3()->array->get( $reg_config );
		$edit_config     = $settings->get_custom_setting( 'profilefields', 'profile' );
		$edit_config     = mslib3()->array->get( $edit_config );
		$was_initialized = false;

		$registration_options = array(
			'off'      => '-',
			'optional' => __( 'Optional', 'memberdash' ),
			'required' => __( 'Required', 'memberdash' ),
		);

		$profile_options = array(
			'off'      => '-',
			'readonly' => __( 'Read-only', 'memberdash' ),
			'optional' => __( 'Optional', 'memberdash' ),
			'required' => __( 'Required', 'memberdash' ),
		);

		$fields = array();
		$rows   = array();
		$rows[] = array(
			__( 'Field Name', 'memberdash' ),
			__( 'Registration Form', 'memberdash' ),
			__( 'Profile Form', 'memberdash' ),
		);

		$default_labels = MS_Addon_Profilefields::get_default_labels();

		// Prepare the rows inside the table. Each row is a profile field.
		foreach ( $profile_fields as $id => $details ) {
			// Registration form options.
			if ( empty( $reg_config[ $id ] ) && ! empty( $details['default_reg'] ) ) {
				$reg_config[ $id ] = $details['default_reg'];
				$was_initialized   = true;
			}

			if ( isset( $reg_config[ $id ] ) ) {
				$value_reg = $reg_config[ $id ];
			} else {
				$value_reg = '';
			}

			if ( isset( $details['allowed_reg'] ) && is_array( $details['allowed_reg'] ) ) {
				$reg_options = array();
				foreach ( $details['allowed_reg'] as $key ) {
					$reg_options[ $key ] = $registration_options[ $key ];
				}
			} else {
				$reg_options = $registration_options;
			}

			$field_reg = MS_Helper_Html::html_element(
				array(
					'id'            => 'register[' . $id . ']',
					'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
					'value'         => $value_reg,
					'field_options' => $reg_options,
				),
				true
			);

			// Profile form options.
			if ( empty( $edit_config[ $id ] ) && ! empty( $details['default_edit'] ) ) {
				$edit_config[ $id ] = $details['default_edit'];
				$was_initialized    = true;
			}

			if ( isset( $edit_config[ $id ] ) ) {
				$value_edit = $edit_config[ $id ];
			} else {
				$value_edit = '';
			}

			if ( isset( $details['allowed_edit'] ) && is_array( $details['allowed_edit'] ) ) {
				$edit_options = array();
				foreach ( $details['allowed_edit'] as $key ) {
					$edit_options[ $key ] = $profile_options[ $key ];
				}
			} else {
				$edit_options = $profile_options;
			}

			$field_edit = MS_Helper_Html::html_element(
				array(
					'id'            => 'profile[' . $id . ']',
					'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
					'value'         => $value_edit,
					'field_options' => $edit_options,
				),
				true
			);

			// Include link to revert back to the default.
			$has_default_link = ! empty( $default_labels[ $id ] )
				&& $default_labels[ $id ] !== $details['label'];

			// Build the row: Label [edit] [default] | registration form | profile form.
			$rows[] = [
				sprintf(
					'%1$s%2$s%3$s',
					$details['label'],
					$this->get_label_button( $id, __( '[edit]', 'memberdash' ), 'edit' ),
					$has_default_link
						? $this->get_label_button( $id, __( '[default]', 'memberdash' ), 'default' )
						: '',
				),
				$field_reg,
				$field_edit,
			];
		}

		$fields[] = array(
			'id'            => 'fieldlist',
			'type'          => MS_Helper_Html::TYPE_HTML_TABLE,
			'value'         => $rows,
			'field_options' => array(
				'head_row' => true,
			),
		);

		$fields[] = array(
			'id'    => 'action',
			'type'  => MS_Helper_Html::INPUT_TYPE_HIDDEN,
			'value' => $action,
		);

		$fields[] = array(
			'id'    => '_wpnonce',
			'type'  => MS_Helper_Html::INPUT_TYPE_HIDDEN,
			'value' => wp_create_nonce( $action ),
		);

		$fields[] = array(
			'id'    => 'save',
			'type'  => MS_Helper_Html::INPUT_TYPE_SUBMIT,
			'value' => __( 'Save Changes', 'memberdash' ),
		);

		// Save changes in case fields were initialized.
		if ( $was_initialized ) {
			$settings->set_custom_setting( 'profilefields', 'register', $reg_config );
			$settings->set_custom_setting( 'profilefields', 'profile', $edit_config );
			$settings->save();
		}

		return $fields;
	}

	/**
	 * Returns a label button for the profile field.
	 *
	 * @since 1.3.0
	 *
	 * @param string $id     The field ID.
	 * @param string $label  The label of the button.
	 * @param string $action The action of the button.
	 *
	 * @return string
	 */
	private function get_label_button( string $id, string $label, string $action ): string {
		return sprintf(
			'<button type="button" class="button-link profile-field-label-edit" data-id="%1$s" data-action="%3$s">%2$s</button>',
			esc_attr( $id ),
			esc_html( $label ),
			esc_attr( $action )
		);
	}
}
