<?php
/**
 * Controller to manage notices.
 *
 * @since 1.3.1
 *
 * @package MemberDash
 */

/**
 * Controller to manage notices.
 *
 * @since 1.3.1
 */
class MS_Controller_Notices extends MS_Controller {

	/**
	 * Notice for first sale.
	 *
	 * @since 1.3.1
	 *
	 * @var string
	 */
	const NOTICE_FIRST_SALE = 'notice_first_sale';

	/**
	 * Notice for X members.
	 *
	 * @since 1.3.1
	 *
	 * @var string
	 */
	const NOTICE_X_MEMBERS = 'notice_x_members';

	/**
	 * Notice for first membership.
	 *
	 * @since 1.3.1
	 *
	 * @var string
	 */
	const NOTICE_FIRST_MEMBERSHIP = 'notice_first_membership';

	/**
	 * AJAX action to disable notice.
	 *
	 * @since 1.3.1
	 *
	 * @var string
	 */
	const AJAX_DISABLE_NOTICE = 'ms_ajax_disable_notice';

	/**
	 * Nonce action.
	 *
	 * @since 1.3.1
	 *
	 * @var string
	 */
	const NONCE_ACTION = 'ms_notice';

	/**
	 * Notice permalink setup.
	 *
	 * @since 1.4.4
	 *
	 * @var string
	 */
	private static $notice_permalink_setup = 'notice_permalink_setup';

	/**
	 * Notices model.
	 *
	 * @since 1.3.1
	 *
	 * @var MS_Model_Notices
	 */
	private $model = null;

	/**
	 * Constructor.
	 *
	 * @since 1.3.1
	 */
	public function __construct() {
		parent::__construct();

		$this->model = MS_Factory::load( 'MS_Model_Notices' );

		$this->add_ajax_action( self::AJAX_DISABLE_NOTICE, 'ajax_disable_notice' );
	}

	/**
	 * Init admin actions.
	 *
	 * @since 1.3.1
	 *
	 * @return void
	 */
	public function admin_init() {
		$this->add_action(
			'admin_init',
			'register_admin_notices'
		);

		$this->add_action(
			'admin_notices',
			'display_admin_notices'
		);

		$this->add_action(
			'admin_enqueue_scripts',
			'admin_enqueue_scripts'
		);
	}

	/**
	 * Disable notice.
	 *
	 * @since 1.3.1
	 *
	 * @return void
	 */
	public function ajax_disable_notice(): void {
		$nonce = sanitize_key( wp_unslash( $_GET['nonce'] ?? '' ) );

		// Bail if the user is not an admin or the nonce is invalid.
		if (
			! $this->is_admin_user()
			|| ! wp_verify_nonce( $nonce, self::NONCE_ACTION )
		) {
			MS_Helper_Debug::debug_log(
				__( 'Invalid nonce verification', 'memberdash' ),
			);
			wp_send_json_error();
		}

		$notice = sanitize_key( wp_unslash( $_GET['notice'] ?? '' ) );

		// Bail if the notice is invalid.
		if ( ! in_array(
			$notice,
			$this->get_notice_keys(),
			true
		) ) {
			wp_send_json_error();
		}

		$user_id = get_current_user_id();

		// Update user meta to disable the notice.
		$notices = $this->model->get_user_notices( $user_id );

		$notices[ $notice ] = 'disabled';

		$this->model->update_user_notices( $user_id, $notices );

		wp_send_json_success();
	}

	/**
	 * Register admin notices.
	 *
	 * @since 1.3.1
	 *
	 * @return void
	 */
	public function register_admin_notices(): void {
		// Only show notices to admin users.
		if ( ! $this->is_admin_user() ) {
			return;
		}

		$user_id = get_current_user_id();
		$notices = $this->model->get_user_notices( $user_id );

		// Register first sale notice.
		if (
			! isset( $notices[ self::NOTICE_FIRST_SALE ] )
			&& ! is_plugin_active( 'backupbuddy/backupbuddy.php' )
		) {
			$total = MS_Model_Invoice::get_invoice_count(
				[
					'meta_query' => [
						[
							'key'     => 'status',
							'value'   => MS_Model_Invoice::STATUS_PAID,
							'compare' => '=',
						],
					],
				]
			);

			if ( $total > 0 ) {
				$notices[ self::NOTICE_FIRST_SALE ] = 'enabled';
			}
		}

		// Register X members notice.
		if (
			! isset( $notices[ self::NOTICE_X_MEMBERS ] )
			&& ! is_plugin_active( 'backupbuddy/backupbuddy.php' )
		) {
			$total = MS_Model_Member::get_active_members_count();

			if ( $total > 0 ) {
				$notices[ self::NOTICE_X_MEMBERS ] = 'enabled';
			}
		}

		// Register first membership notice.
		if (
			! isset( $notices[ self::NOTICE_FIRST_MEMBERSHIP ] )
			&& ! is_plugin_active( 'backupbuddy/backupbuddy.php' )
		) {
			$total = MS_Model_Membership::get_membership_count(
				[
					'include_base'  => false,
					'include_guest' => false,
				]
			);

			if ( $total === 1 ) {
				$notices[ self::NOTICE_FIRST_MEMBERSHIP ] = 'enabled';
			} elseif ( $total > 1 ) {
				$notices[ self::NOTICE_FIRST_MEMBERSHIP ] = 'disabled';
			}
		}

		// Register permalink setup notice.
		$permalink_structure = get_option( 'permalink_structure' );
		if (
			'' === $permalink_structure
		) {
			$notices[ self::$notice_permalink_setup ] = 'enabled';
		} elseif (
			isset( $notices[ self::$notice_permalink_setup ] )
			&& '' !== $permalink_structure
		) {
			$notices[ self::$notice_permalink_setup ] = 'disabled';
		}

		$this->model->update_user_notices( $user_id, $notices );
	}

	/**
	 * Display admin notices.
	 *
	 * @since 1.3.1
	 *
	 * @return void
	 */
	public function display_admin_notices(): void {
		// Only show notices to admin users.
		if ( ! $this->is_admin_user() ) {
			return;
		}

		$user_id = get_current_user_id();
		$notices = $this->model->get_user_notices( $user_id );

		// Bail if there are no notices.
		if ( empty( $notices ) ) {
			return;
		}

		// Display notices.
		foreach ( $notices as $id => $status ) {
			if ( 'disabled' === $status ) {
				continue;
			}

			$notice = $this->get_notice( $id );

			// If it's empty, disable this notice.
			if ( empty( $notice ) ) {
				$notices[ $id ] = 'disabled';
				$this->model->update_user_notices( $user_id, $notices );
				continue;
			}

			$class = $this->get_notice_class( $id );

			?>
			<div
				class="<?php echo esc_attr( $class ); ?>"
				data-dismiss="<?php echo esc_attr( $id ); ?>"
			>
				<p>
					<?php echo wp_kses_post( $notice ); ?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Enqueue admin scripts.
	 *
	 * @since 1.3.1
	 *
	 * @return void
	 */
	public function admin_enqueue_scripts(): void {
		$data = [
			'ms_init' => [
				'disable_admin_notice',
			],
			'notices' => [
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'action'   => self::AJAX_DISABLE_NOTICE,
				'nonce'    => wp_create_nonce( self::NONCE_ACTION ),
			],
		];

		mslib3()->ui->data( 'ms_data', $data );
	}

	/**
	 * Gets list of notice keys.
	 *
	 * @since 1.3.1
	 *
	 * @return array<int,string>
	 */
	private function get_notice_keys(): array {
		return [
			self::NOTICE_FIRST_SALE,
			self::NOTICE_X_MEMBERS,
			self::NOTICE_FIRST_MEMBERSHIP,
			self::$notice_permalink_setup,
		];
	}

	/**
	 * Returns a list of class names as a string for a notice.
	 *
	 * @since 1.4.4
	 *
	 * @param string $notice_id Notice ID.
	 *
	 * @return string
	 */
	private function get_notice_class( string $notice_id ): string {
		$classes = [
			self::NOTICE_FIRST_SALE       => 'notice notice-success is-dismissible',
			self::NOTICE_X_MEMBERS        => 'notice notice-success is-dismissible',
			self::NOTICE_FIRST_MEMBERSHIP => 'notice notice-success is-dismissible',
			self::$notice_permalink_setup => 'notice notice-error',
		];

		return $classes[ $notice_id ] ?? 'notice notice-info is-dismissible';
	}

	/**
	 * Get notice.
	 *
	 * @since 1.3.1
	 *
	 * @param string $notice Notice key.
	 *
	 * @return string
	 */
	private function get_notice( string $notice ): string {
		$notices = [
			self::$notice_permalink_setup => sprintf(
				// translators: %s: Permalink settings URL.
				__( 'In order for the Protection Rules to be fully functional, it is required to change your site\'s permalink structure. To do this, please go to %s, and ensure that your permalink structure is set to something like "Post name", and not "Plain". Once this is done, save your changes.', 'memberdash' ),
				'<a href="' . esc_url( admin_url( 'options-permalink.php' ) ) . '">' . esc_html__( 'Settings > Permalinks', 'memberdash' ) . '</a>'
			),
		];

		return $notices[ $notice ] ?? '';
	}
}
