<?php
/**
 * WooCommerce Controller.
 *
 * @since 1.4.0
 *
 * @package MemberDash
 */

/**
 * WooCommerce Controller class.
 *
 * @since 1.4.0
 */
class MS_Controller_WooCommerce extends MS_Controller {
	/**
	 * Control WooCommerce hooks.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		parent::__construct();

		// Stop if WooCommerce is not activated.
		if ( ! MS_Helper_WooCommerce::is_woocommerce_activated() ) {
			return;
		}

		// Sets external product search and URL.
		$this->add_filter(
			'ms_controller_membership_ajax_action_search_external_product_data',
			'search_products',
			10,
			2
		);
		$this->add_filter(
			'ms_view_membership_tab_payment_get_external_product_data_options',
			'get_external_product_data_options',
			10,
			2
		);
		$this->add_filter(
			'ms_membership_price',
			'get_product_price',
			10,
			2
		);
		$this->add_filter(
			'ms_view_shortcode_membershipsignup_button',
			'get_product_url',
			10,
			2
		);
		$this->add_filter(
			'ms_view_shortcode_membershipsignup_cancel_button',
			'hide_cancel_button',
			10,
			2
		);
		$this->add_filter(
			'ms_helper_listtable_membership_column_price',
			'membership_column_price',
			10,
			2
		);

		// Declare WooCommerce custom table compatibility.
		$this->add_action(
			'before_woocommerce_init',
			'declare_custom_order_tables_compatibility'
		);

		// Manage product options.
		$this->add_action(
			'woocommerce_product_options_general_product_data',
			'render_product_options'
		);
		$this->add_ajax_action(
			'memberdash_woocommerce_json_search_memberships',
			'ajax_search_memberships'
		);
		$this->add_action(
			'woocommerce_admin_process_product_object',
			'save_product_options'
		);
		$this->add_action(
			'admin_enqueue_scripts',
			'enqueue_product_scripts'
		);

		// Manage product variation options.
		$this->add_action(
			'woocommerce_product_after_variable_attributes',
			'render_product_variation_options',
			10,
			3
		);
		$this->add_action(
			'woocommerce_admin_process_variation_object',
			'save_product_variation_options',
			10,
			2
		);

		// Orders and subscriptions.
		$this->add_action(
			'init',
			'handle_order_status'
		);
		$this->add_action(
			'init',
			'handle_subscription_status'
		);

		$this->add_action(
			'woocommerce_before_trash_order',
			'handle_deleted_order',
			10,
			2
		);
		$this->add_action(
			'woocommerce_before_delete_order',
			'handle_deleted_order',
			10,
			2
		);

		$this->add_action(
			'woocommerce_process_shop_order_meta',
			'handle_update_order_meta'
		);
		$this->add_action(
			'woocommerce_process_shop_order_meta',
			'handle_update_subscription_meta'
		);

		$this->add_action(
			'woocommerce_new_order_item',
			'handle_new_order_item',
			10,
			3
		);
		$this->add_action(
			'woocommerce_before_delete_order_item',
			'handle_delete_order'
		);

		$this->add_action(
			'woocommerce_subscription_renewal_payment_complete',
			'handle_subscription_on_billing_cycle_completion'
		);

		$this->add_action(
			'woocommerce_subscription_checkout_switch_order_processed',
			'handle_subscription_switch',
			10,
			2
		);

		$this->add_action(
			'woocommerce_checkout_registration_enabled',
			'maybe_require_registration',
			20
		);
		$this->add_action(
			'woocommerce_checkout_registration_required',
			'maybe_require_registration',
			20
		);
		$this->add_action(
			'woocommerce_before_checkout_process',
			'force_registration_during_checkout'
		);

		// Validate cart items when multiple memberships aren't allowed.
		$this->add_filter(
			'woocommerce_add_to_cart_validation',
			'validate_add_to_cart_action',
			10,
			3
		);

		// Calculate subscription expire date based on the WooCommerce Subscription next payment date.
		$this->add_filter(
			'ms_model_relationship_calc_expire_date',
			'get_subscription_expire_date',
			10,
			2
		);
	}

	/**
	 * Search products.
	 *
	 * @since 1.4.0
	 *
	 * @param array{items:array<int,array{id:string,text:string}>,more:bool} $response Response object.
	 * @param string                                                         $term     Search term.
	 *
	 * @return array{
	 *     items: array<int,array{
	 *         id: string,
	 *         text: string
	 *     }>,
	 *     more: bool
	 * }
	 */
	public function search_products( array $response, string $term ): array {
		$products = wc_get_products(
			[
				'status' => 'publish',
				's'      => $term,
				'limit'  => 30,
				'order'  => 'ASC',
			]
		);

		// Stop if no products are found.
		if (
			empty( $products )
			|| ! is_iterable( $products )
		) {
			return $response;
		}

		foreach ( $products as $product ) {
			$response['items'][] = [
				'id'   => MS_Helper_Cast::to_string( $product->get_id() ),
				'text' => $product->get_formatted_name(),
			];
		}

		return $response;
	}

	/**
	 * Returns an array of options for the external product data option.
	 *
	 * @since 1.4.0
	 *
	 * @param array<string,string> $options The options list.
	 * @param string               $current The current value.
	 *
	 * @return array<string,string>
	 */
	public function get_external_product_data_options( array $options, string $current ): array {
		$product_id = MS_Helper_Cast::to_int( $current );
		$product    = wc_get_product( $product_id );

		// Stop if product is not found.
		if ( ! $product ) {
			return $options;
		}

		// Include the current product to the options list.
		$options[ MS_Helper_Cast::to_string( $product->get_id() ) ] = $product->get_formatted_name();

		return $options;
	}

	/**
	 * Returns the product price.
	 *
	 * @since 1.4.0
	 *
	 * @param string              $price      The price.
	 * @param MS_Model_Membership $membership The membership instance.
	 *
	 * @return string
	 */
	public function get_product_price( string $price, MS_Model_Membership $membership ): string {
		$product = $this->get_product( $membership );

		// Stop if product is not found.
		if ( ! $product ) {
			return $price;
		}

		// Remove any HTML tags from the price.
		return sanitize_text_field( $product->get_price_html() );
	}

	/**
	 * Returns the product URL.
	 *
	 * @since 1.4.0
	 *
	 * @param array<string,string> $button     The button.
	 * @param MS_Model_Membership  $membership The membership instance.
	 *
	 * @return array<string,string>
	 */
	public function get_product_url( array $button, MS_Model_Membership $membership ): array {
		$product = $this->get_product( $membership );

		// Stop if product is not found.
		if ( ! $product ) {
			return $button;
		}

		$button['type']  = MS_Helper_Html::TYPE_HTML_LINK;
		$button['url']   = $product->get_permalink();
		$button['class'] = 'memberdash-field-input button ms-signup-button membership_signup memberdash-submit button-primary';

		return $button;
	}

	/**
	 * Hides the cancel button.
	 *
	 * @since 1.4.0
	 *
	 * @param array<string,string>  $button       The button data.
	 * @param MS_Model_Relationship $subscription The subscription instance.
	 *
	 * @return array<string,string>
	 */
	public function hide_cancel_button( array $button, MS_Model_Relationship $subscription ): array {
		$membership = $subscription->get_membership();

		if ( ! $membership->has_external_cart_support() ) {
			return $button;
		}

		$button['type'] = MS_Helper_Html::INPUT_TYPE_HIDDEN;
		return $button;
	}

	/**
	 * Updates price in the memberships list.
	 *
	 * @since 1.4.0
	 *
	 * @param string              $html       The HTML content.
	 * @param MS_Model_Membership $membership The membership instance.
	 *
	 * @return string
	 */
	public function membership_column_price( string $html, MS_Model_Membership $membership ): string {
		// Stop if membership doesn't have external cart support.
		if ( ! $membership->has_external_cart_support() ) {
			return $html;
		}

		$product = $this->get_product( $membership );

		// Stop if product is not found.
		if ( ! $product ) {
			return esc_html__( 'N/A', 'memberdash' );
		}

		return sprintf(
			'<span class="ms-bold">%s</span> (<a class="ms-external-product" href="%s">%s</a>)',
			$product->get_price_html(),
			esc_url(
				MS_Helper_Cast::to_string(
					get_edit_post_link( $product->get_id() )
				)
			),
			esc_html__( 'See product', 'memberdash' )
		);
	}

	/**
	 * Declare custom order tables compatibility.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function declare_custom_order_tables_compatibility(): void {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', MEMBERDASH_PLUGIN_FILE, true );
		}
	}

	/**
	 * Ajax search memberships.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function ajax_search_memberships(): void {
		check_ajax_referer( 'search-products', 'security' );

		if ( ! $this->is_admin_user() ) {
			wp_send_json( [] );
		}

		$term = sanitize_text_field(
			wp_unslash(
				MS_Helper_Cast::to_string(
					self::get_request_field( 'term', '', 'GET' )
				)
			)
		);

		$memberships = MS_Model_Membership::get_memberships(
			[
				's'             => $term,
				'post_per_page' => 30, // The same quantity used in the WooCommerce product search.
				'meta_query'    => [
					[
						'key'     => 'external_cart_support',
						'value'   => '1',
						'compare' => '=',
					],
				],
			]
		);

		$results = [];

		foreach ( $memberships as $membership ) {
			$results[ $membership->get_id() ] = $membership->get_name();
		}

		wp_send_json( $results );
	}

	/**
	 * Renders the product options.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function render_product_options(): void {
		global $post;

		$view = MS_Factory::create( 'MS_View_WooCommerce_Product_Options' );
		$view->set_data(
			[
				'post' => $post,
			]
		);
		$view->render();
	}

	/**
	 * Renders the product variation options.
	 *
	 * @since 1.4.0
	 *
	 * @param int                 $loop Position in the loop.
	 * @param array<string,mixed> $data Array of variation data.
	 * @param WP_Post             $post Post data.
	 *
	 * @return void
	 */
	public function render_product_variation_options( int $loop, array $data, WP_Post $post ): void {
		$view = MS_Factory::create( 'MS_View_WooCommerce_Product_Variation_Options' );
		$view->set_data(
			[
				'post'  => $post,
				'index' => $loop,
			]
		);
		$view->render();
	}

	/**
	 * Saves product options.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Product $product The product object.
	 *
	 * @return void
	 */
	public function save_product_options( WC_Product $product ): void {
		// Update memberships list.
		$memberships = self::get_request_field( '_ms_memberships', [], 'POST' );

		if ( ! is_array( $memberships ) ) {
			return;
		}

		$memberships = array_map( 'intval', $memberships );
		$product->update_meta_data( '_ms_memberships', $memberships );
		$this->update_membership_external_product_data( $memberships, $product->get_id() );
	}

	/**
	 * Enqueues product scripts.
	 *
	 * @since 1.4.0
	 *
	 * @param string $hook The current hook.
	 *
	 * @return void
	 */
	public function enqueue_product_scripts( $hook ): void {
		if (
			'post.php' !== $hook
			&& 'post-new.php' !== $hook
		) {
			return;
		}

		$screen = get_current_screen();

		if (
			! $screen
			|| 'product' !== $screen->post_type
		) {
			return;
		}

		$data = [
			'ms_init' => [
				'control_woo_product_sold_individually_option',
			],
		];

		mslib3()->ui->data( 'ms_data', $data );
	}

	/**
	 * Saves product variation options.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Product_Variation $variation The variation object.
	 * @param int                  $index     The index of the variation.
	 *
	 * @return void
	 */
	public function save_product_variation_options( WC_Product_Variation $variation, int $index ): void {
		$memberships = self::get_request_field( 'ms_variable_memberships', [], 'POST' );

		if ( ! is_array( $memberships ) ) {
			return;
		}

		$memberships = array_map(
			'intval',
			$memberships[ $index ] ?? []
		);

		$variation->update_meta_data( '_ms_memberships', $memberships );
		$this->update_membership_external_product_data( $memberships, $variation->get_parent_id() );
	}

	/**
	 * Handles order status updates.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function handle_order_status(): void {
		$granted_statuses = MS_Helper_WooCommerce::get_order_statuses_by_access_status( true );
		$denied_statuses  = MS_Helper_WooCommerce::get_order_statuses_by_access_status( false );

		// Add membership access.
		foreach ( $granted_statuses as $status ) {
			$this->add_action(
				'woocommerce_order_status_' . $status,
				'add_order_membership_access',
				10,
				2
			);
		}

		// Remove membership access.
		foreach ( $denied_statuses as $status ) {
			if ( $status === 'refunded' ) {
				continue;
			}

			$this->add_action(
				'woocommerce_order_status_' . $status,
				'remove_order_membership_access',
				10,
				2
			);
		}

		/*
		 * Refunded status requires a separate action `woocommerce_order_refunded` to support partial refund.
		 * Partial refund order still has a status of `completed`, hence the need for another action to
		 * process course/group access update.
		 */
		$this->add_action(
			'woocommerce_order_refunded',
			'remove_order_membership_access_on_refund'
		);
	}

	/**
	 * Handles subscription status updates.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function handle_subscription_status(): void {
		$granted_statuses = MS_Helper_WooCommerce::get_subscription_statuses_by_access_status( true );
		$denied_statuses  = MS_Helper_WooCommerce::get_subscription_statuses_by_access_status( false );

		// Add membership access.
		foreach ( $granted_statuses as $status ) {
			$this->add_action(
				'woocommerce_subscription_status_' . $status,
				'add_membership_access_to_subscription'
			);
		}

		// Remove membership access.
		foreach ( $denied_statuses as $status ) {
			if ( $status === 'on-hold' ) {
				continue;
			}

			$this->add_action(
				'woocommerce_subscription_status_' . $status,
				'remove_subscription_membership_access'
			);
		}
	}

	/**
	 * Adds membership access to an order.
	 *
	 * @since 1.4.0
	 *
	 * @param int      $order_id The order ID.
	 * @param WC_Order $order    The order instance.
	 *
	 * @return void
	 */
	public function add_order_membership_access( int $order_id, WC_Order $order ): void {
		// Create a new order instance if it is not provided.
		if ( ! $order instanceof WC_Order ) {
			$order = wc_get_order( $order_id );
		}

		$user = $order->get_user();

		// Stop if user is not found.
		if ( ! $user ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order );

		// Stop if no memberships are found.
		if ( empty( $memberships ) ) {
			return;
		}

		// Add memberships to the user.
		$this->add_memberships_to_user( $user, $memberships, $order );
	}

	/**
	 * Removes membership access from an order.
	 *
	 * @since 1.4.0
	 *
	 * @param int      $order_id The order ID.
	 * @param WC_Order $order    The order instance.
	 *
	 * @return void
	 */
	public function remove_order_membership_access( int $order_id, WC_Order $order ): void {
		// Create a new order instance if it is not provided.
		if ( ! $order instanceof WC_Order ) {
			$order = wc_get_order( $order_id );
		}

		$user = $order->get_user();

		// Stop if user is not found.
		if ( ! $user ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order );

		// Stop if no memberships are found.
		if ( empty( $memberships ) ) {
			return;
		}

		// Add memberships to the user.
		$this->remove_user_memberships( $user, $memberships );
	}

	/**
	 * Removes membership access from an refunded order.
	 *
	 * @since 1.4.0
	 *
	 * @param int $order_id The order ID.
	 *
	 * @return void
	 */
	public function remove_order_membership_access_on_refund( int $order_id ): void {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		$user = $order->get_user();

		// Stop if user is not found.
		if ( ! $user ) {
			return;
		}

		if (
			$order->get_status() === 'refunded'
		) {
			$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order );

			// Stop if no memberships are found.
			if ( empty( $memberships ) ) {
				return;
			}

			// Add memberships to the user.
			$this->remove_user_memberships( $user, $memberships );
		} elseif (
			$order->get_status() !== 'refunded'
		) {
			$this->remove_user_membership_on_partial_refund( $user, $order );
		}
	}

	/**
	 * Adds membership access to a subscription.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Subscription $subscription The subscription instance.
	 *
	 * @return void
	 */
	public function add_membership_access_to_subscription( WC_Subscription $subscription ): void {
		$user = $subscription->get_user();

		if ( ! $user ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $subscription, false );

		if ( empty( $memberships ) ) {
			return;
		}

		$this->add_memberships_to_user( $user, $memberships, $subscription );
	}

	/**
	 * Removes membership access from a subscription.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Subscription $subscription The subscription instance.
	 *
	 * @return void
	 */
	public function remove_subscription_membership_access( WC_Subscription $subscription ): void {
		$user = $subscription->get_user();

		if ( ! $user ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $subscription, false );

		if ( empty( $memberships ) ) {
			return;
		}

		$this->remove_user_memberships( $user, $memberships );
	}

	/**
	 * Handles deleted orders.
	 *
	 * @since 1.4.0
	 *
	 * @param int                      $order_id The order ID.
	 * @param WC_Order|WC_Subscription $order    The order instance.
	 *
	 * @return void
	 */
	public function handle_deleted_order( int $order_id, $order ): void {
		if ( $order->get_type() === 'shop_order' ) {
			$this->remove_order_membership_access( $order_id, $order );
		} elseif (
			$order->get_type() === 'shop_subscription'
			&& $order instanceof WC_Subscription
		) {
			$this->remove_subscription_membership_access( $order );
		}
	}

	/**
	 * Handles updating order meta.
	 *
	 * @since 1.4.0
	 *
	 * @param int $order_id The order ID.
	 *
	 * @return void
	 */
	public function handle_update_order_meta( int $order_id ): void {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		$old_user = $order->get_user();
		$new_user = self::get_request_field( '_customer_user', false, 'POST' );

		if ( $new_user === false ) {
			return;
		}

		$new_user = get_user_by( 'ID', MS_Helper_Cast::to_int( $new_user ) );

		if (
			! $new_user
			|| ! $old_user
			|| $old_user->ID === $new_user->ID
		) {
			return;
		}

		$granted_statuses = MS_Helper_WooCommerce::get_order_statuses_by_access_status( true );

		if ( ! in_array( $order->get_status(), $granted_statuses, true ) ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order );

		if ( empty( $memberships ) ) {
			return;
		}

		$this->remove_user_memberships( $old_user, $memberships );
		$this->add_memberships_to_user( $new_user, $memberships, $order );
	}

	/**
	 * Handles updating subscription meta.
	 *
	 * @since 1.4.0
	 *
	 * @param int $subscription_id The subscription ID.
	 *
	 * @return void
	 */
	public function handle_update_subscription_meta( int $subscription_id ): void {
		if ( ! function_exists( 'wcs_get_subscription' ) ) {
			return;
		}

		$subscription = wcs_get_subscription( $subscription_id );

		if ( ! $subscription instanceof WC_Subscription ) {
			return;
		}

		if ( ! $subscription->get_id() ) {
			return;
		}

		$old_user = $subscription->get_user();
		$new_user = self::get_request_field( '_customer_user', false, 'POST' );

		if ( $new_user === false ) {
			return;
		}

		$new_user = get_user_by( 'ID', MS_Helper_Cast::to_int( $new_user ) );

		if (
			! $new_user
			|| ! $old_user
			|| $old_user->ID === $new_user->ID
		) {
			return;
		}

		$granted_statuses = MS_Helper_WooCommerce::get_subscription_statuses_by_access_status( true );

		if ( ! in_array( $subscription->get_status(), $granted_statuses, true ) ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $subscription, false );

		if ( empty( $memberships ) ) {
			return;
		}

		$this->remove_user_memberships( $old_user, $memberships );
		$this->add_memberships_to_user( $new_user, $memberships, $subscription );
	}

	/**
	 * Handles new order item action.
	 *
	 * @since 1.4.0
	 *
	 * @param int           $item_id  The item ID.
	 * @param WC_Order_Item $item     The item instance.
	 * @param int           $order_id The order ID.
	 *
	 * @return void
	 */
	public function handle_new_order_item( int $item_id, WC_Order_Item $item, int $order_id ): void {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		if (
			function_exists( 'wcs_is_subscription' )
			&& wcs_is_subscription( $order )
		) {
			$subscription = wcs_get_subscription( $order );

			if ( ! $subscription instanceof WC_Subscription ) {
				return;
			}

			$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $subscription, false );
		} else {
			$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order );
		}

		if ( empty( $memberships ) ) {
			return;
		}

		$user = $order->get_user();

		if ( ! $user ) {
			return;
		}

		$this->add_memberships_to_user( $user, $memberships, $order );
	}

	/**
	 * Handles delete order item action.
	 *
	 * @since 1.4.0
	 *
	 * @param int $item_id The item ID.
	 *
	 * @return void
	 */
	public function handle_delete_order( int $item_id ): void {
		try {
			$order_id = wc_get_order_id_by_order_item_id( $item_id );
			$order    = wc_get_order( $order_id );
		} catch ( Exception $e ) {
			return;
		}

		if ( ! $order instanceof WC_Order ) {
			return;
		}

		// Load the order item.
		$order_item = $order->get_item( $item_id, false );

		if ( ! $order_item ) {
			return;
		}

		// Get order item memberships.
		$memberships = $order_item->get_meta( '_ms_memberships', true );

		// Stop if no memberships are found.
		if (
			empty( $memberships )
			|| ! is_array( $memberships )
		) {
			return;
		}

		$memberships = array_map(
			'MS_Helper_Cast::to_int',
			$memberships
		);

		$user = $order->get_user();

		if ( ! $user ) {
			return;
		}

		$this->remove_user_memberships( $user, $memberships );
	}

	/**
	 * Handles subscription on billing cycle completion.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Subscription $subscription The subscription instance.
	 *
	 * @return void
	 */
	public function handle_subscription_on_billing_cycle_completion( WC_Subscription $subscription ): void {
		$next_payment = $subscription->calculate_date( 'next_payment' );

		// Stop if next payment exists.
		if ( $next_payment !== 0 ) {
			return;
		}

		$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $subscription, false );

		if ( empty( $memberships ) ) {
			return;
		}

		$user = $subscription->get_user();

		if ( ! $user ) {
			return;
		}

		$this->remove_subscription_membership_access( $subscription );
	}

	/**
	 * Handles subscription switch.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Order                                                   $order             The order instance.
	 * @param array<int,array<string,array<string,array<string,mixed>>>> $switch_order_data The data.
	 *
	 * @return void
	 */
	public function handle_subscription_switch( WC_Order $order, array $switch_order_data ): void {
		$subscriptions = wcs_get_subscriptions_for_switch_order( $order );

		foreach ( $switch_order_data as $subscription_id => $subscription_data ) {
			foreach ( $subscription_data['switches'] as $switch_item_id => $switch_data ) {
				if ( ! empty( $switch_data['remove_line_item'] ) ) {
					$old_order_id = wc_get_order_id_by_order_item_id(
						MS_Helper_Cast::to_int( $switch_data['remove_line_item'] )
					);

					if ( ! $old_order_id ) {
						continue;
					}

					$old_order = wc_get_order( $old_order_id );

					if ( ! $old_order instanceof WC_Order ) {
						continue;
					}

					$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $old_order, false );

					if ( empty( $memberships ) ) {
						continue;
					}

					foreach ( $subscriptions as $subscription ) {
						$user = $subscription->get_user();

						if ( ! $user ) {
							continue;
						}

						$this->remove_user_memberships( $user, $memberships );
					}
				}

				if ( ! empty( $switch_data['add_line_item'] ) ) {
					$memberships = MS_Helper_WooCommerce::get_memberships_from_order( $order, false );
					$user        = $order->get_user();

					if ( ! $user ) {
						continue;
					}

					if ( empty( $memberships ) ) {
						continue;
					}

					$this->add_memberships_to_user( $user, $memberships, $order );
				}
			}
		}
	}

	/**
	 * Maybe enables and require registration during checkout.
	 *
	 * @since 1.4.0
	 *
	 * @param bool $enabled The registration status.
	 *
	 * @return bool
	 */
	public function maybe_require_registration( bool $enabled ): bool {
		if (
			MS_Helper_WooCommerce::cart_contains_membership()
			&& ! is_user_logged_in()
		) {
			return true;
		}

		return $enabled;
	}

	/**
	 * Forces registration during checkout.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function force_registration_during_checkout(): void {
		if (
			MS_Helper_WooCommerce::cart_contains_membership()
			&& ! is_user_logged_in()
		) {
			$_POST['createaccount'] = '1';
		}
	}

	/**
	 * Validates add to cart action when multiple memberships aren't allowed.
	 *
	 * @since 1.4.0
	 *
	 * @param bool      $valid        The validation status.
	 * @param int       $product_id   The product ID.
	 * @param int|float $quantity     The quantity.
	 *
	 * @return bool
	 */
	public function validate_add_to_cart_action( bool $valid, int $product_id, $quantity ): bool {
		if ( MS_Model_Addon::is_enabled( 'multi_memberships' ) ) {
			return $valid;
		}

		$product = wc_get_product( $product_id );

		if ( ! $product instanceof WC_Product ) {
			return $valid;
		}

		// Get the variation ID from the request.
		$variation_id = MS_Helper_Cast::to_int(
			self::get_request_field( 'variation_id', 0, 'REQUEST' )
		);

		// Handle variable products.
		if ( $variation_id ) {
			$product = wc_get_product( $variation_id );

			if ( ! $product instanceof WC_Product_Variation ) {
				return $valid;
			}
		}

		$memberships = $product->get_meta( '_ms_memberships', true );

		if (
			empty( $memberships )
			|| ! is_array( $memberships )
		) {
			return $valid;
		}

		// If user is logged in, check if they already have a membership.
		if ( MS_Model_Member::is_logged_in() ) {
			$member = MS_Model_Member::get_current_member();

			// Stop if member is not found.
			if ( ! $member->get_id() ) {
				return $valid;
			}

			// Stop if member already has a membership.
			$membership = MS_Helper_Cast::to_int( reset( $memberships ) );
			if ( $member->can_subscribe_to( $membership ) ) {
				wc_add_notice( __( 'You already have an active membership in your account. Please cancel it before purchasing a new one.', 'memberdash' ), 'error' );
				return false;
			}
		}

		// Validate cart items.
		if (
			MS_Helper_WooCommerce::cart_contains_membership()
			|| $quantity > 1
		) {
			wc_add_notice( __( 'Multiple memberships can not be purchased at the same time.', 'memberdash' ), 'error' );
			return false;
		}

		return $valid;
	}

	/**
	 * Returns the subscription expire date.
	 *
	 * This method is used to calculate the expire date for a subscription
	 * when the membership is purchased via WooCommerce Subscriptions.
	 *
	 * The expire date is the next payment date of the subscription.
	 *
	 * @since 1.4.0
	 *
	 * @param string|null           $expire_date The expire date.
	 * @param MS_Model_Relationship $subscription The subscription instance.
	 *
	 * @return string|null
	 */
	public function get_subscription_expire_date( $expire_date, MS_Model_Relationship $subscription ) {
		$membership = $subscription->get_membership();

		// Stop if membership doesn't have external cart support.
		if ( ! $membership->has_external_cart_support() ) {
			return $expire_date;
		}

		$invoice = $subscription->get_current_invoice( false );

		// Stop if invoice is not found.
		if ( ! $invoice instanceof MS_Model_Invoice ) {
			return $expire_date;
		}

		$order = MS_Helper_WooCommerce::get_order_from_invoice( $invoice );

		if ( ! $order instanceof WC_Subscription ) {
			return $expire_date;
		}

		return gmdate(
			'Y-m-d',
			strtotime( $order->get_date( 'next_payment' ) )
		);
	}

	/**
	 * Adds memberships to the user.
	 *
	 * @since 1.4.0
	 *
	 * @param WP_User                  $user        The user instance.
	 * @param array<int>               $memberships List of membership IDs.
	 * @param WC_Order|WC_Subscription $order       The order instance.
	 *
	 * @return void
	 */
	protected function add_memberships_to_user( WP_User $user, array $memberships, $order ): void {
		$member = MS_Factory::load( 'MS_Model_Member', $user->ID );

		foreach ( $memberships as $membership_id ) {
			if ( $member->has_membership( $membership_id ) ) {
				continue;
			}

			$relationship = $member->add_membership( $membership_id, 'woocommerce' );

			if ( ! $relationship ) {
				continue;
			}

			$gateway = $relationship->get_gateway();

			if ( ! $gateway instanceof MS_Gateway_WooCommerce ) {
				continue;
			}

			$member->set_is_member( true );
			$member->save();

			$gateway->process_order( $relationship, $order );
		}
	}

	/**
	 * Removes memberships from the user.
	 *
	 * @since 1.4.0
	 *
	 * @param WP_User    $user        The user instance.
	 * @param array<int> $memberships List of membership IDs.
	 *
	 * @return void
	 */
	protected function remove_user_memberships( WP_User $user, array $memberships ): void {
		$member = MS_Factory::load( 'MS_Model_Member', $user->ID );

		foreach ( $memberships as $membership_id ) {
			$member->drop_membership( $membership_id );
		}
	}

	/**
	 * Removes memberships from the user on partial refund.
	 *
	 * @since 1.4.0
	 *
	 * @param WP_User  $user  The user instance.
	 * @param WC_Order $order The order instance.
	 *
	 * @return void
	 */
	protected function remove_user_membership_on_partial_refund( WP_User $user, WC_Order $order ): void {
		$products = [];
		$refunds  = $order->get_refunds();

		if ( ! is_iterable( $refunds ) ) {
			return;
		}

		foreach ( $refunds as $refund ) {
			$refunded = $refund->get_items();

			if (
				empty( $refunded )
				|| ! is_array( $refunded )
			) {
				continue;
			}

			$products = array_merge( $products, $refunded );
		}

		if ( empty( $products ) ) {
			return;
		}

		$memberships = [];

		foreach ( $products as $product ) {
			$data = $product->get_meta( '_ms_memberships', true );

			if (
				empty( $data )
				|| ! is_array( $data )
			) {
				continue;
			}

			$memberships = array_merge( $memberships, $data );
		}

		$memberships = array_map(
			'MS_Helper_Cast::to_int',
			array_filter(
				array_unique( $memberships )
			)
		);

		$this->remove_user_memberships( $user, $memberships );
	}

	/**
	 * Returns the product.
	 *
	 * @since 1.4.0
	 *
	 * @param MS_Model_Membership $membership The membership instance.
	 *
	 * @return WC_Product|null
	 */
	protected function get_product( MS_Model_Membership $membership ) {
		// Stop if membership does not have support for external shop carts.
		if ( ! $membership->has_external_cart_support() ) {
			return null;
		}

		$product_id = MS_Helper_Cast::to_int( $membership->get_external_product_data() );

		// Stop if product ID is not set.
		if ( ! $product_id ) {
			return null;
		}

		$product = wc_get_product( $product_id );

		// Stop if product is not found.
		if ( ! $product ) {
			return null;
		}

		return $product;
	}

	/**
	 * Updates the membership external product data.
	 *
	 * @since 1.4.0
	 *
	 * @param array<int> $memberships List of membership IDs.
	 * @param int        $product_id  The product ID.
	 *
	 * @return void
	 */
	protected function update_membership_external_product_data( array $memberships, int $product_id ): void {
		foreach ( $memberships as $membership_id ) {
			$membership = MS_Factory::load( 'MS_Model_Membership', $membership_id );

			if (
				! $membership->has_external_cart_support()
				|| $membership->get_external_product_data() !== ''
			) {
				continue;
			}

			$membership->set_external_product_data( MS_Helper_Cast::to_string( $product_id ) );
			$membership->save();
		}
	}
}
