<?php
/**
 * Gateway: Paypal Express
 *
 * Process single and recurring paypal purchases/payments.
 *
 * Persisted by parent class MS_Model_Option. Singleton.
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

use StellarWP\Memberdash\StellarWP\Arrays\Arr;

/**
 * PayPal Express gateway class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress extends MS_Gateway {
	/**
	 * Gateway ID.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	const ID = 'paypalexpress';

	/**
	 * PayPal API instance.
	 *
	 * @since 1.5.0
	 *
	 * @var MS_Gateway_Paypalexpress_Api_Client
	 */
	protected $api = null;

	/**
	 * Whodat instance.
	 *
	 * @since 1.5.0
	 *
	 * @var MS_Gateway_Paypalexpress_Api_Whodat
	 */
	protected $whodat = null;

	/**
	 * Webhooks instance.
	 *
	 * @since 1.5.0
	 *
	 * @var MS_Gateway_Paypalexpress_Api_Webhooks
	 */
	protected $webhooks = null;

	/**
	 * Gateway mode.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $mode = 'sandbox';

	/**
	 * Account country.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $account_country = '';

	/**
	 * Connection status.
	 *
	 * @since 1.5.0
	 *
	 * @var bool
	 */
	protected $is_connected = false;

	/**
	 * Signup hash.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $signup_hash = '';

	/**
	 * Merchant ID.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $merchant_id = '';

	/**
	 * Merchant ID in PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $merchant_id_in_paypal = '';

	/**
	 * Merchant account is ready for receiving payments.
	 *
	 * @since 1.5.0
	 *
	 * @var bool
	 */
	protected $merchant_account_is_ready = false;

	/**
	 * Supports custom payments.
	 *
	 * @since 1.5.0
	 *
	 * @var bool
	 */
	protected $supports_custom_payments = false;

	/**
	 * Client ID.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $client_id = '';

	/**
	 * Client secret.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $client_secret = '';

	/**
	 * Account ID.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $account_id = '';

	/**
	 * Granted scopes.
	 *
	 * @since 1.5.0
	 *
	 * @var array<string>
	 */
	protected $granted_scopes = [];

	/**
	 * Partner Attribution ID.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	private $partner_attribution_id = 'MemberDash_SP_PPCP';

	/**
	 * Hook to add custom transaction status.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function after_load() {
		parent::after_load();

		// APIs.
		$this->api      = MS_Factory::load( 'MS_Gateway_Paypalexpress_Api_Client' );
		$this->whodat   = MS_Factory::load( 'MS_Gateway_Paypalexpress_Api_Whodat' );
		$this->webhooks = MS_Factory::load( 'MS_Gateway_Paypalexpress_Api_Webhooks' );

		$this->id             = self::ID;
		$this->name           = __( 'PayPal Checkout', 'memberdash' );
		$this->group          = 'PayPalConnect';
		$this->manual_payment = false; // Recurring charged automatically.
		$this->pro_rate       = false;
		$this->priority       = 1; // Second gateway to be displayed.
		$this->admin_logo_url = MS_Plugin::instance()->get_url() . 'app/assets/images/gateways/paypalexpress/logo.svg';

		// Admin.
		$this->add_ajax_action(
			'ms_paypal_express_refresh_connect_url',
			'ajax_refresh_connect_url'
		);

		$this->add_ajax_action(
			'ms_paypal_express_onboarding_complete',
			'ajax_fetch_access_token'
		);

		$this->add_ajax_action(
			'ms_paypal_express_reconnect',
			'ajax_reconnect_account'
		);

		$this->add_ajax_action(
			'ms_paypal_express_disconnect',
			'ajax_account_disconnect'
		);

		$this->add_action(
			'admin_enqueue_scripts',
			'admin_enqueue_scripts'
		);

		$this->add_action(
			'admin_notices',
			'admin_notices'
		);

		$this->add_action(
			'wp_loaded',
			'handle_signup_redirect',
			20 // After the gateway controller webhook.
		);

		$this->add_filter(
			'ms_model_clean_metadata',
			'keep_metadata',
			10
		);

		// Frontend.
		$this->add_action(
			'wp_enqueue_scripts',
			'enqueue_scripts'
		);

		$this->add_ajax_action(
			'ms_gateway_paypal_express_create_order',
			'ajax_create_order',
			true,
			true
		);
		$this->add_ajax_action(
			'ms_gateway_paypal_express_handle_order_approve',
			'ajax_handle_order_approve',
			true,
			true
		);
		$this->add_ajax_action(
			'ms_gateway_paypal_express_handle_order_cancel',
			'ajax_handle_order_cancel',
			true,
			true
		);
		$this->add_ajax_action(
			'ms_gateway_paypal_express_create_subscription',
			'ajax_create_subscription',
			true,
			true
		);
		$this->add_ajax_action(
			'ms_gateway_paypal_express_handle_subscription_approve',
			'ajax_handle_subscription_approve',
			true,
			true
		);
		$this->add_ajax_action(
			'ms_gateway_paypal_express_handle_subscription_cancel',
			'ajax_handle_subscription_cancel',
			true,
			true
		);

		$this->add_filter(
			'ms_gateway_view_button_data',
			'view_button_data',
			10,
			2
		);

		$this->add_filter(
			'ms_controller_shortcode_thank_you_page',
			'thank_you_page_display_transaction_id',
		);

		$this->add_filter(
			'ms_gateway_button-' . MS_Gateway_Paypalstandard::ID,
			'hide_paypal_standard_button',
			10
		);
	}

	/**
	 * Returns if the gateway is configured.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_configured() {
		return $this->is_connected
			&& ! empty( $this->client_id )
			&& ! empty( $this->client_secret )
			&& $this->merchant_account_is_ready;
	}

	/**
	 * Handles the signup redirect.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function handle_signup_redirect(): void {
		$required = [
			'paypal-express-signup',
			'merchantId',
			'productIntentId',
		];

		if ( ! MS_Controller::validate_required( $required, 'REQUEST' ) ) {
			return;
		}

		$request_type      = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'paypal-express-signup', '', 'REQUEST' )
		);
		$product_intent_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'productIntentId', '', 'REQUEST' )
		);

		if (
			$request_type !== 'return'
			|| $product_intent_id !== 'addipmt'
		) {
			return;
		}

		// Get seller referral data.
		$seller_data = $this->whodat->get_seller_referral_data(
			$this->whodat->get_referral_data_link(),
			$this->is_sandbox()
		);

		if ( empty( $seller_data ) ) {
			return;
		}

		$referral_data = [];

		if ( isset( $seller_data['referral_data'] ) ) {
			$referral_data = is_array( $seller_data['referral_data'] )
				? $seller_data['referral_data']
				: [];
		}

		$integration_details = [];

		if ( isset( $referral_data['operations'][0]['api_integration_preference']['rest_api_integration']['first_party_details'] ) ) {
			$integration_details = $referral_data['operations'][0]['api_integration_preference']['rest_api_integration']['first_party_details'];
		}

		$hash = isset( $integration_details['seller_nonce'] )
			? MS_Helper_Cast::to_string( $integration_details['seller_nonce'] )
			: '';

		// Validate hash.
		if ( $hash !== $this->whodat->get_transient_hash() ) {
			return;
		}

		$this->signup_hash           = $hash;
		$this->merchant_id           = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'merchantId', '', 'REQUEST' )
		);
		$this->merchant_id_in_paypal = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'merchantIdInPayPal', '', 'REQUEST' )
		);
		$this->granted_scopes        = isset( $integration_details['features'] )
			? $integration_details['features']
			: [];

		// PayPal partner JS will call the ajax_fetch_access_token() method to generate the access token.
		// This happens after the user completes the onboarding process.
		$access_token = $this->api->get_access_token();
		$credentials  = $this->whodat->get_seller_credentials(
			$access_token,
			$this->is_sandbox()
		);

		if (
			! isset( $credentials['client_id'] )
			&& ! isset( $credentials['client_secret'] )
		) {
			// Save what we have before moving forward.
			$this->save();

			$message = isset( $credentials['message'] )
				? MS_Helper_Cast::to_string( $credentials['message'] )
				: '';

			wp_safe_redirect(
				esc_url_raw(
					add_query_arg(
						[
							'tab'                   => 'payment',
							'paypal-express-notice' => 'signup-error',
							'msg'                   => rawurlencode( $message ),
						],
						MS_Controller_Plugin::get_admin_url( 'settings' )
					)
				)
			);
			exit;
		}

		// Save the credentials.
		$this->client_id     = MS_Helper_Cast::to_string( $credentials['client_id'] );
		$this->client_secret = MS_Helper_Cast::to_string( $credentials['client_secret'] );
		$this->account_id    = MS_Helper_Cast::to_string( $credentials['payer_id'] );

		$this->supports_custom_payments = in_array(
			'PPCP',
			is_array( $referral_data['products'] )
				? $referral_data['products']
				: [],
			true
		);

		// Enable the gateway.
		$this->is_connected = true;
		$this->active       = true;
		$this->save();

		// Pull access token data.
		$token_data = $this->api->get_access_token_from_client_credentials(
			$this->client_id,
			$this->client_secret,
			$this->is_sandbox()
		);
		$this->api->save_access_token_data( $token_data );

		// Create webhooks.
		$this->webhooks->create_or_update_existing_webhooks();

		// Required to set the global payment settings after the first connection.
		$settings = MS_Plugin::get_settings();
		$settings->set_is_global_payments_set( true );
		$settings->save();

		wp_safe_redirect(
			esc_url_raw(
				add_query_arg(
					[
						'tab'                   => 'payment',
						'paypal-express-notice' => 'connected',
					],
					MS_Controller_Plugin::get_admin_url( 'settings' )
				)
			)
		);
		exit;
	}

	/**
	 * Manage what metadata to keep for subscriptions.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string> $metadata The metadata to keep.
	 *
	 * @return array<string>
	 */
	public function keep_metadata( array $metadata ): array {
		return array_diff(
			$metadata,
			[
				'ms_gateway_paypal_payload',
				'ms_gateway_paypal_order_id',
				'ms_gateway_paypal_subscription_id',
				'ms_gateway_paypal_sandbox_product_id',
				'ms_gateway_paypal_live_product_id',
				'ms_gateway_paypal_sandbox_plan_id',
				'ms_gateway_paypal_live_plan_id',
			]
		);
	}

	/**
	 * Ajax callback to refresh the connect URL.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_refresh_connect_url(): void {
		check_ajax_referer( 'refresh_connect_url', 'nonce' );

		if ( ! MS_Model_Member::is_admin_user() ) {
			wp_send_json_error(
				[
					'message' => __( 'You do not have permission to perform this action.', 'memberdash' ),
				]
			);
		}

		// Update the country in the database.
		$this->account_country = sanitize_text_field(
			MS_Helper_Cast::to_string(
				MS_Controller::get_request_field( 'account_country', 'US', 'GET' )
			)
		);
		$this->save();

		// Get the connect URL.
		$url = $this->whodat->generate_connect_url(
			$this->account_country,
			$this->is_sandbox(),
			true
		);

		if ( empty( $url ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not generate the connect URL.', 'memberdash' ),
				]
			);
		}

		$data = [
			'connect_url' => add_query_arg(
				[
					'displayMode' => 'minibrowser',
				],
				$url
			),
		];

		wp_send_json_success( $data );
	}

	/**
	 * Ajax callback to fetch the access token.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_fetch_access_token(): void {
		check_ajax_referer( 'fetch_access_token', 'nonce' );

		if ( ! MS_Model_Member::is_admin_user() ) {
			wp_send_json_error(
				[
					'message' => __( 'You do not have permission to perform this action.', 'memberdash' ),
				]
			);
		}

		$shared_id = sanitize_text_field(
			MS_Helper_Cast::to_string(
				MS_Controller::get_request_field( 'shared_id', '', 'REQUEST' )
			)
		);

		$auth_code = sanitize_text_field(
			MS_Helper_Cast::to_string(
				MS_Controller::get_request_field( 'auth_code', '', 'REQUEST' )
			)
		);

		$token_data = $this->api->get_access_token_from_authorization_code(
			$shared_id,
			$auth_code,
			'',
			$this->is_sandbox()
		);

		if (
			empty( $token_data )
			|| array_key_exists( 'error', $token_data )
		) {
			wp_send_json_error(
				[
					'message' => __( 'Unexpected response from PayPal when on boarding.', 'memberdash' ),
				]
			);
		}

		$this->api->save_access_token_data( $token_data );

		wp_send_json_success();
	}

	/**
	 * Ajax callback to reconnect the account.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_reconnect_account(): void {
		check_ajax_referer( 'reconnect_account', 'nonce' );

		if ( ! MS_Model_Member::is_admin_user() ) {
			wp_send_json_error(
				[
					'message' => __( 'You do not have permission to perform this action.', 'memberdash' ),
				]
			);
		}

		$access_token = $this->api->get_access_token_from_client_credentials(
			$this->client_id,
			$this->client_secret,
			$this->is_sandbox()
		);

		if ( ! isset( $access_token['access_token'] ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not get the access token.', 'memberdash' ),
				]
			);
		}

		$saved = $this->api->save_access_token_data( $access_token );

		if ( ! $saved ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not save the access token data.', 'memberdash' ),
				]
			);
		}

		// Create webhooks.
		$this->webhooks->create_or_update_existing_webhooks();

		wp_send_json_success(
			[
				'url' => esc_url_raw(
					add_query_arg(
						[
							'tab'                   => 'payment',
							'paypal-express-notice' => 'reconnected',
						],
						MS_Controller_Plugin::get_admin_url( 'settings' )
					)
				),
			]
		);
	}

	/**
	 * Ajax callback to disconnect the account.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_account_disconnect(): void {
		check_ajax_referer( 'disconnect_account', 'nonce' );

		if ( ! MS_Model_Member::is_admin_user() ) {
			wp_send_json_error(
				[
					'message' => __( 'You do not have permission to perform this action.', 'memberdash' ),
				]
			);
		}

		$this->is_connected             = false;
		$this->active                   = false;
		$this->client_id                = '';
		$this->client_secret            = '';
		$this->account_id               = '';
		$this->granted_scopes           = [];
		$this->merchant_id              = '';
		$this->merchant_id_in_paypal    = '';
		$this->supports_custom_payments = false;

		$this->save();

		// Remove access token data.
		$this->api->delete_access_token_data();

		// Remove client access token.
		$this->api->delete_client_token();

		// Remove webhook data.
		$this->webhooks->delete_webhook_data();

		// Remove product and plan IDs.
		$this->api->delete_all_membership_meta_data();

		wp_send_json_success(
			[
				'url' => esc_url_raw(
					add_query_arg(
						[
							'tab'                   => 'payment',
							'paypal-express-notice' => 'disconnected',
						],
						MS_Controller_Plugin::get_admin_url( 'settings' )
					)
				),
			]
		);
	}

	/**
	 * Enqueue scripts for the admin settings page.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function admin_enqueue_scripts(): void {
		$screen = get_current_screen();

		if ( is_null( $screen ) ) {
			return;
		}

		if ( $screen->id !== 'memberdash_page_membership-settings' ) {
			return;
		}

		$tab = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'tab', '', 'GET' )
		);

		if ( $tab !== 'payment' ) {
			return;
		}

		wp_enqueue_script(
			'ms-gateway-paypalexpress-admin',
			MS_Plugin::instance()->get_url() . 'app/assets/dist/js/admin/gateway/paypalexpress/admin.js',
			[
				'jquery',
				'ms-admin',
			],
			MEMBERDASH_VERSION,
			true
		);

		$logo_url = MS_Plugin::instance()->get_url() . 'app/assets/images/gateways/paypalexpress/logo.svg';
		$logo     = sprintf(
			'<img src="%s" alt="%s" class="ms-paypal-gateway-logo" style="width: 75px; vertical-align: middle; margin-right: 10px;">',
			esc_url( $logo_url ),
			esc_attr__( 'PayPal Express', 'memberdash' )
		);

		wp_localize_script(
			'ms-gateway-paypalexpress-admin',
			'msPayPalExpressAdmin',
			[
				'refresh_connect_url' => [
					'action' => 'ms_paypal_express_refresh_connect_url',
					'nonce'  => wp_create_nonce( 'refresh_connect_url' ),
				],
				'onboarding_complete' => [
					'action' => 'ms_paypal_express_onboarding_complete',
					'nonce'  => wp_create_nonce( 'fetch_access_token' ),
				],
				'actions'             => [
					'disconnect' => [
						'action'         => 'ms_paypal_express_disconnect',
						'nonce'          => wp_create_nonce( 'disconnect_account' ),
						'message'        => esc_html__( 'Disconnecting your PayPal account will prevent you from offering PayPal at the checkout on your website. Do you wish to continue?', 'memberdash' ),
						'btn_disconnect' => esc_html__( 'Disconnect', 'memberdash' ),
						'btn_cancel'     => esc_html__( 'Cancel', 'memberdash' ),
					],
					'reconnect'  => [
						'action' => 'ms_paypal_express_reconnect',
						'nonce'  => wp_create_nonce( 'reconnect_account' ),
					],
					'connected'  => [
						'title' => $logo . esc_html__( 'PayPal Checkout Connected', 'memberdash' ),
					],
				],
			]
		);

		// Enqueue the partner JS if isn't connected.
		if ( $this->is_connected ) {
			return;
		}

		wp_enqueue_script(
			'memberdash-paypal-express-partner-js',
			$this->api->get_partner_js_url(),
			[],
			MEMBERDASH_VERSION,
			true
		);
	}

	/**
	 * Displays admin notices.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function admin_notices(): void {
		$notice = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'paypal-express-notice', '', 'GET' )
		);

		if ( empty( $notice ) ) {
			return;
		}

		$messages = [
			'connected'    => [
				'message' => esc_html__( 'PayPal Checkout account connected.', 'memberdash' ),
				'type'    => 'success',
			],
			'disconnected' => [
				'message' => esc_html__( 'PayPal Checkout account disconnected.', 'memberdash' ),
				'type'    => 'info',
			],
			'signup-error' => [
				'message' => esc_html__( 'There was an error connecting to your PayPal Checkout account:', 'memberdash' ),
				'type'    => 'error',
			],
			'reconnected'  => [
				'message' => esc_html__( 'Your PayPal Checkout account was reconnected, and webhooks were recreated.', 'memberdash' ),
				'type'    => 'success',
			],
		];

		if ( ! array_key_exists( $notice, $messages ) ) {
			return;
		}

		$view = MS_Factory::load( 'MS_Gateway_Paypalexpress_View_Notice' );

		// Check if there is a return message from PayPal.
		$return_msg = sanitize_text_field(
			urldecode(
				MS_Helper_Cast::to_string(
					MS_Controller::get_request_field( 'msg', '', 'GET' )
				)
			)
		);

		$message = $messages[ $notice ];
		if ( ! empty( $return_msg ) ) {
			$message['message'] = sprintf(
				'%s %s',
				$message['message'],
				$return_msg
			);
		}

		$view->set_data( $message );
		$view->render();

		// Show the connected dialog.
		if ( $notice === 'connected' ) {
			$closed = MS_Helper_Cast::to_string(
				MS_Controller::get_request_field( 'closed', '', 'GET' )
			);

			if ( $closed === 'true' ) {
				return;
			}

			$dialog = MS_Factory::load( 'MS_Gateway_Paypalexpress_View_Connected' );
			$dialog->set_data(
				[
					'sandbox' => $this->is_sandbox(),
				]
			);
			$dialog->render();
		}
	}

	/**
	 * Returns if the gateway is in sandbox mode.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_sandbox(): bool {
		return ! $this->is_live_mode();
	}

	/**
	 * Returns the gateway ID.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->id;
	}

	/**
	 * Returns the gateway mode.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_mode(): string {
		if ( empty( $this->mode ) ) {
			$this->mode = self::MODE_SANDBOX;
		}

		return $this->mode;
	}

	/**
	 * Returns the merchant account country.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_account_country(): string {
		return $this->account_country;
	}

	/**
	 * Returns the connection status.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function is_connected(): bool {
		return $this->is_connected;
	}

	/**
	 * Returns the PayPal API client ID.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_client_id(): string {
		return $this->client_id;
	}

	/**
	 * Returns the PayPal API client secret.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_client_secret(): string {
		return $this->client_secret;
	}

	/**
	 * Returns the PayPal account ID.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_account_id(): string {
		return $this->account_id;
	}

	/**
	 * Returns the granted scopes.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	public function get_granted_scopes(): array {
		return $this->granted_scopes;
	}

	/**
	 * Returns the PayPal merchant ID.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_merchant_id(): string {
		return $this->merchant_id;
	}

	/**
	 * Returns the PayPal merchant ID in PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_merchant_id_in_paypal(): string {
		return $this->merchant_id_in_paypal;
	}

	/**
	 * Returns if the merchant account is ready for receiving payments.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function get_merchant_account_is_ready(): bool {
		return $this->merchant_account_is_ready;
	}

	/**
	 * Returns the PayPal partner attribution ID.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_partner_attribution_id(): string {
		return $this->partner_attribution_id;
	}

	/**
	 * Returns a list of available webhooks.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	public function get_available_webhooks(): array {
		return $this->webhooks->get_available_webhooks();
	}

	/**
	 * Enqueues public scripts.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function enqueue_scripts(): void {
		$page_type = MS_Model_Pages::get_page_type( MS_Model_Pages::current_page() );
		$pages     = [
			MS_Model_Pages::MS_PAGE_REGISTER,
			MS_Model_Pages::MS_PAGE_MEMBERSHIPS,
		];

		if ( ! in_array( $page_type, $pages, true ) ) {
			return;
		}

		wp_enqueue_script(
			'ms-gateway-paypalexpress-checkout',
			MS_Plugin::instance()->get_url() . 'app/assets/dist/js/public/gateway/paypalexpress/checkout.js',
			[ 'jquery' ],
			MEMBERDASH_VERSION,
			true
		);

		wp_localize_script(
			'ms-gateway-paypalexpress-checkout',
			'msPayPalExpress',
			[
				'ajax_url'                => admin_url( 'admin-ajax.php' ),
				'nonce'                   => wp_create_nonce( 'ms-paypal-express-checkout' ),
				'card_fields'             => [
					'ineligible_card_vendor' => __( 'The card vendor is not supported.', 'memberdash' ),
					'invalid_name'           => __( 'Please enter a valid card holder name.', 'memberdash' ),
					'invalid_number'         => __( 'Please enter a valid card number.', 'memberdash' ),
					'invalid_expiry'         => __( 'Please enter a valid expiry date.', 'memberdash' ),
					'invalid_cvv'            => __( 'Please enter a valid CVV.', 'memberdash' ),
					'unknown_error'          => __( 'An invalid card was provided. Please review the card details and try again.', 'memberdash' ),
				],
				'card_fields_processing'  => __( 'Processing payment...', 'memberdash' ),
				'card_fields_redirecting' => __( 'Redirecting...', 'memberdash' ),
				'card_fields_submit'      => __( 'Pay Now', 'memberdash' ),
			]
		);
	}

	/**
	 * Ajax callback to create an order in PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_create_order(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$subscription_id = MS_Helper_Cast::to_int(
			MS_Controller::get_request_field( 'subscription_id', 0, 'REQUEST' )
		);

		if ( empty( $subscription_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid subscription ID.', 'memberdash' ),
				]
			);
		}

		$subscription = MS_Factory::load( 'MS_Model_Relationship', $subscription_id );
		$subscription->set_gateway( self::ID );

		if ( ! $subscription->is_valid() ) {
			wp_send_json_error(
				[
					'message' => __( 'Subscription not found.', 'memberdash' ),
				]
			);
		}

		if ( MS_Model_Relationship::STATUS_CANCELED === $subscription->get_status() ) {
			wp_send_json_error(
				[
					'message' => sanitize_text_field( $subscription->get_status_description() ),
				]
			);
		}

		$invoice    = $subscription->get_next_billable_invoice();
		$member     = $subscription->get_member();
		$membership = $subscription->get_membership();

		$data = [
			'reference_id'  => sprintf(
				'ms-order-%s-%s-%s',
				$subscription->get_id(),
				$invoice->get_id(),
				time()
			),
			'amount'          => MS_Helper_Cast::to_string( $invoice->get_invoice_total() ),
			'currency_code'   => $invoice->get_currency(),
			'first_name'      => $member->get_user()->first_name,
			'last_name'       => $member->get_user()->last_name,
			'email'           => $member->get_user()->user_email,
			'items'           => [
				[
					'name'        => MS_Gateway_Paypalexpress_Helpers::trim_text( $membership->get_name() ),
					'quantity'    => 1,
					'unit_amount' => [
						'value'         => MS_Helper_Cast::to_string( $invoice->get_invoice_total() ),
						'currency_code' => $invoice->get_currency(),
					],
				],
			],
			'description'     => MS_Gateway_Paypalexpress_Helpers::trim_text( $membership->get_name() ),
			'merchant_id'     => $this->merchant_id_in_paypal,
			'invoice_id'      => sprintf(
				'ms-invoice-%s-%s',
				$invoice->get_id(),
				time()
			),
			'return_url'      => MS_Gateway::get_success_url( $subscription->get_id() ),
			'cancel_url'      => MS_Gateway_Paypalexpress_Helpers::get_cancel_url( $membership->get_id() ),
			'use_card_fields' => MS_Helper_Cast::to_bool(
				MS_Controller::get_request_field( 'card_fields', false, 'REQUEST' )
			),
		];

		$response = $this->api->create_order( $data, $this->is_sandbox() );

		if ( empty( $response['id'] ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not create the PayPal order.', 'memberdash' ),
				]
			);
		}

		$debug_id = $this->api->get_debug_id();
		if ( ! empty( $debug_id ) ) {
			$response['debug_id'] = $debug_id;
		}

		update_post_meta( $subscription->get_id(), 'ms_gateway_paypal_payload', $response );
		update_post_meta( $subscription->get_id(), 'ms_gateway_paypal_order_id', $response['id'] );

		wp_send_json_success(
			[
				'order_id' => $response['id'],
			]
		);
	}

	/**
	 * Ajax callback to handle the order approve action.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_handle_order_approve(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$order_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'order_id', '', 'REQUEST' )
		);

		if ( empty( $order_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal order ID.', 'memberdash' ),
				]
			);
		}

		$payer_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'payer_id', '', 'REQUEST' )
		);

		// Find pending subscription by PayPal order ID.
		$subscription = MS_Gateway_Paypalexpress_Helpers_Order::get_ms_subscription( $order_id );

		// Make sure the subscription is valid before proceeding.
		if ( empty( $subscription ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Membership subscription not found.', 'memberdash' ),
				]
			);
		}

		$recheck = MS_Helper_Cast::to_bool(
			MS_Controller::get_request_field( 'recheck', false, 'REQUEST' )
		);

		// Recheck the order to confirm the status.
		if ( $recheck ) {
			$order_response = $this->api->get_order(
				$order_id,
				$this->is_sandbox()
			);

			$status  = MS_Gateway_Paypalexpress_Helpers_Order::get_order_status( $order_response );
			$invoice = $subscription->get_current_invoice( false, false );

			// Stop if invoice is paid.
			if ( $invoice->is_paid() ) {
				// If the subscription is pending, set it to active.
				if ( MS_Model_Relationship::STATUS_PENDING === $subscription->get_status() ) {
					$subscription->set_status( MS_Model_Relationship::STATUS_ACTIVE );
					$subscription->save();
				}

				wp_send_json_success(
					[
						'order_id'     => $order_id,
						'redirect_url' => MS_Gateway::get_success_url( $subscription->get_id() ),
					]
				);
			}

			if ( in_array( $status, [ 'FAILED', 'DECLINED', 'VOIDED' ], true ) ) {
				// Cancel the subscription.
				$membership = $subscription->get_membership();
				$member     = $subscription->get_member();

				$member->cancel_membership( $membership->get_id() );
				$member->save();

				wp_send_json_error(
					[
						'message' => __( 'Your payment was declined.', 'memberdash' ),
					]
				);
			} elseif ( in_array( $status, [ 'APPROVED', 'COMPLETED' ], true ) ) {
				$notes = __( 'Payment approved', 'memberdash' );
				$invoice->add_notes( $notes );

				$external_id = MS_Helper_Cast::to_string(
					Arr::get( $order_response, 'id', '' )
				);

				// Mark the invoice as paid.
				$invoice->pay_it( self::ID, $external_id );
				$invoice->save();

				/*
				 * Creates a transaction log entry.
				 *
				 * Documented in /includes/gateways/class-ms-controller-gateway.php.
				 */
				do_action(
					'ms_gateway_transaction_log',
					self::ID, // Gateway ID.
					'process', // Values: request|process|handle.
					true, // Success flag.
					$subscription->get_id(), // Subscription ID.
					$invoice->get_id(), // Invoice ID.
					$invoice->get_invoice_total(), // Charged amount.
					$notes, // Descriptive text.
					$external_id // External ID.
				);
			}

			wp_send_json_success(
				[
					'order_id'     => $order_id,
					'redirect_url' => MS_Gateway::get_success_url( $subscription->get_id() ),
				]
			);
		}

		// Capture the order.
		$capture_response = $this->api->capture_order(
			$order_id,
			$payer_id,
			$this->is_sandbox()
		);

		$status = isset( $capture_response['status'] )
			? $capture_response['status']
			: '';

		if ( 'COMPLETED' !== $status ) {
			wp_send_json_error(
				[
					$capture_response,
				]
			);
		}

		wp_send_json_success(
			[
				'order_id' => $order_id,
			]
		);
	}

	/**
	 * Ajax callback to handle the order cancel action.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_handle_order_cancel(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$order_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'order_id', '', 'REQUEST' )
		);

		if ( empty( $order_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal order ID.', 'memberdash' ),
				]
			);
		}

		// Find pending subscription by PayPal order ID.
		$subscription = MS_Gateway_Paypalexpress_Helpers_Order::get_ms_subscription( $order_id );

		// Make sure the subscription is valid before proceeding.
		if ( empty( $subscription ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Membership subscription not found.', 'memberdash' ),
				]
			);
		}

		$membership = $subscription->get_membership();

		wp_send_json_success(
			[
				'order_id'     => $order_id,
				'redirect_url' => MS_Gateway_Paypalexpress_Helpers::get_cancel_url( $membership->get_id() ),
			]
		);
	}

	/**
	 * Ajax callback to create a subscription in PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_create_subscription(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$subscription_id = MS_Helper_Cast::to_int(
			MS_Controller::get_request_field( 'subscription_id', 0, 'REQUEST' )
		);

		if ( empty( $subscription_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid subscription ID.', 'memberdash' ),
				]
			);
		}

		$subscription = MS_Factory::load( 'MS_Model_Relationship', $subscription_id );
		$subscription->set_gateway( self::ID );

		if ( ! $subscription->is_valid() ) {
			wp_send_json_error(
				[
					'message' => __( 'Subscription not found.', 'memberdash' ),
				]
			);
		}

		$member     = $subscription->get_member();
		$membership = $subscription->get_membership();
		$invoice    = $subscription->get_current_invoice( false, false );

		$plan_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'plan_id', 0, 'REQUEST' )
		);

		if ( empty( $plan_id ) ) {
			$plan_id = MS_Gateway_Paypalexpress_Helpers_Subscription::get_plan_id( $membership, $this, $invoice );
		}

		$data = [
			'plan_id'    => $plan_id,
			'custom_id'  => $subscription->get_id(),
			'first_name' => $member->get_user()->first_name,
			'last_name'  => $member->get_user()->last_name,
			'email'      => $member->get_user()->user_email,
			'return_url' => MS_Gateway::get_success_url( $subscription->get_id() ),
			'cancel_url' => MS_Gateway_Paypalexpress_Helpers::get_cancel_url( $membership->get_id() ),
		];

		$response = $this->api->create_subscription( $data, $this->is_sandbox() );

		if ( empty( $response['id'] ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not create the PayPal subscription.', 'memberdash' ),
				]
			);
		}

		$debug_id = $this->api->get_debug_id();
		if ( ! empty( $debug_id ) ) {
			$response['debug_id'] = $debug_id;
		}

		update_post_meta( $subscription->get_id(), 'ms_gateway_paypal_payload', $response );
		update_post_meta( $subscription->get_id(), 'ms_gateway_paypal_subscription_id', $response['id'] );

		wp_send_json_success(
			[
				'subscription_id' => $response['id'],
			]
		);
	}

	/**
	 * Ajax callback to handle the subscription approve action.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_handle_subscription_approve(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$subscription_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'subscription_id', '', 'REQUEST' )
		);

		if ( empty( $subscription_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal subscription ID.', 'memberdash' ),
				]
			);
		}

		$order_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'order_id', '', 'REQUEST' )
		);

		if ( empty( $order_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal order ID.', 'memberdash' ),
				]
			);
		}

		// Find pending subscription by PayPal subscription ID.
		$subscription = MS_Gateway_Paypalexpress_Helpers_Subscription::get_ms_subscription( $subscription_id );

		// Make sure the subscription is valid before proceeding.
		if ( empty( $subscription ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Membership subscription not found.', 'memberdash' ),
				]
			);
		}

		// Get the subscription details.
		$subscription_response = $this->api->get_subscription(
			$subscription_id,
			$this->is_sandbox()
		);

		$status = MS_Helper_Cast::to_string(
			Arr::get( $subscription_response, 'status', '' )
		);

		if ( empty( $status ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Could not get the subscription details.', 'memberdash' ),
				]
			);
		}

		$recheck = MS_Helper_Cast::to_bool(
			MS_Controller::get_request_field( 'recheck', false, 'REQUEST' )
		);

		// Recheck the order to confirm the status.
		if ( $recheck ) {
			if ( 'ACTIVE' === $status ) {
				wp_send_json_success(
					[
						'subscription_id' => $subscription_id,
						'order_id'        => $order_id,
						'redirect_url'    => MS_Gateway::get_success_url( $subscription->get_id() ),
					]
				);
			}
		}

		$outstanding_balance = Arr::wrap(
			Arr::get(
				$subscription_response,
				'billing_info.outstanding_balance',
				[
					'value' => 0.0,
				]
			)
		);

		if ( $outstanding_balance['value'] > 0 ) {
			$capture_response = $this->api->capture_subscription(
				[
					'subscription_id' => $subscription_id,
					'currency_code'   => $outstanding_balance['currency_code'],
					'amount'          => $outstanding_balance['value'],
				],
				$this->is_sandbox()
			);

			if ( ! $capture_response ) {
				wp_send_json_error(
					[
						'message' => __( 'Could not capture the subscription.', 'memberdash' ),
					]
				);
			}
		}

		// Stop if the subscription is not active.
		if ( 'ACTIVE' !== $status ) {
			wp_send_json_success(
				[
					'recheck'         => true,
					'subscription_id' => $subscription_id,
					'order_id'        => $order_id,
				]
			);
		}

		// The subscription ID is the external ID for the first payment.
		$external_id = MS_Helper_Cast::to_string(
			Arr::get( $subscription_response, 'id', '' )
		);

		// Mark the invoice as paid.
		$invoice = $subscription->get_current_invoice( false, false );
		$notes   = __( 'Payment approved', 'memberdash' );
		$invoice->add_notes( $notes );
		$invoice->pay_it( self::ID ); // No external ID at this point.
		$invoice->save();

		/*
		 * Creates a transaction log entry.
		 *
		 * Documented in /includes/gateways/class-ms-controller-gateway.php.
		 */
		do_action(
			'ms_gateway_transaction_log',
			self::ID, // Gateway ID.
			'process', // Values: request|process|handle.
			true, // Success flag.
			$subscription->get_id(), // Subscription ID.
			$invoice->get_id(), // Invoice ID.
			$invoice->get_invoice_total(), // Charged amount.
			$notes, // Descriptive text.
			$external_id // External ID.
		);

		wp_send_json_success(
			[
				'subscription_id' => $subscription_id,
				'order_id'        => $order_id,
				'redirect_url'    => MS_Gateway::get_success_url( $subscription->get_id() ),
			]
		);
	}

	/**
	 * Ajax callback to handle the subscription cancel action.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function ajax_handle_subscription_cancel(): void {
		check_ajax_referer( 'ms-paypal-express-checkout', 'nonce' );

		$order_id = MS_Helper_Cast::to_string(
			MS_Controller::get_request_field( 'order_id', '', 'REQUEST' )
		);

		if ( empty( $order_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid PayPal order ID.', 'memberdash' ),
				]
			);
		}

		$subscription_id = MS_Helper_Cast::to_int(
			MS_Controller::get_request_field( 'subscription_id', '', 'REQUEST' )
		);

		if ( empty( $subscription_id ) ) {
			wp_send_json_error(
				[
					'message' => __( 'Invalid subscription ID.', 'memberdash' ),
				]
			);
		}

		// Get the subscription.
		$subscription = MS_Factory::load( 'MS_Model_Relationship', $subscription_id );
		$membership   = $subscription->get_membership();

		wp_send_json_success(
			[
				'order_id'     => $order_id,
				'redirect_url' => MS_Gateway_Paypalexpress_Helpers::get_cancel_url( $membership->get_id() ),
			]
		);
	}

	/**
	 * Updates the button view data.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data       The view data.
	 * @param string              $gateway_id The gateway ID.
	 *
	 * @return array<string,mixed>
	 */
	public function view_button_data( array $data, string $gateway_id ): array {
		// Stop here if the gateway is not PayPal Express.
		if ( $gateway_id !== self::ID ) {
			return $data;
		}

		$client_token_data = $this->api->get_client_token(
			$this->is_sandbox()
		);

		if ( ! $data['ms_relationship'] instanceof MS_Model_Relationship ) {
			return $data;
		}

		$membership = $data['ms_relationship']->get_membership();
		$invoice    = $data['ms_relationship']->get_current_invoice( false, false );
		$sdk_args   = [
			'client-id'   => $this->client_id,
			'merchant-id' => $this->merchant_id_in_paypal,
		];

		// Setup recurring payments.
		$plan_id = '';
		if ( $membership->supports_recurring_payments() ) {
			$sdk_args['intent'] = 'subscription';
			$sdk_args['vault']  = 'true';
			$plan_id            = MS_Gateway_Paypalexpress_Helpers_Subscription::get_plan_id( $membership, $this, $invoice );
		}

		$data['js_sdk_url'] = $this->api->get_js_sdk_url( $sdk_args );
		$data['plan_id']    = $plan_id;

		$data['client_token'] = isset( $client_token_data['client_token'] )
			? $client_token_data['client_token']
			: '';

		$data['client_token_expires_in'] = isset( $client_token_data['expires_in'] )
			? MS_Helper_Cast::to_int( $client_token_data['expires_in'] ) - 60 // 60 seconds before expiration.
			: '';

		$data['supports_custom_payments'] = $this->supports_custom_payments;
		$data['active_custom_payments']   = $this->supports_custom_payments;

		return $data;
	}

	/**
	 * Updates the thank you page content.
	 *
	 * @since 1.5.0
	 *
	 * @param string $html The HTML content.
	 *
	 * @return string
	 */
	public function thank_you_page_display_transaction_id( string $html ): string {
		$subscription_id = MS_Helper_Cast::to_int(
			MS_Controller::get_request_field( 'ms_relationship_id', 0, 'REQUEST' )
		);

		if ( empty( $subscription_id ) ) {
			return $html;
		}

		$subscription = MS_Factory::load( 'MS_Model_Relationship', $subscription_id );
		if ( ! $subscription instanceof MS_Model_Relationship ) {
			return $html;
		}

		if ( $subscription->get_gateway()->get_id() !== self::ID ) {
			return $html;
		}

		$membership = $subscription->get_membership();

		if ( $membership->supports_recurring_payments() ) {
			$transaction_id = MS_Helper_Cast::to_string(
				get_post_meta( $subscription->get_id(), 'ms_gateway_paypal_subscription_id', true )
			);
		} else {
			$transaction_id = MS_Helper_Cast::to_string(
				get_post_meta( $subscription->get_id(), 'ms_gateway_paypal_order_id', true )
			);
		}

		if ( empty( $transaction_id ) ) {
			return $html;
		}

		return sprintf(
			'<p class="ms-alert-box ms-alert-success">%s<br /><br />%s</p>',
			wp_strip_all_tags( $html ),
			sprintf(
				// translators: %s: PayPal transaction ID.
				__( 'Your PayPal transaction ID is %s.', 'memberdash' ),
				'<strong>' . esc_html( $transaction_id ) . '</strong>'
			)
		);
	}

	/**
	 * Returns the PayPal API instance.
	 *
	 * @since 1.5.0
	 *
	 * @return MS_Gateway_Paypalexpress_Api_Client
	 */
	public function get_api(): MS_Gateway_Paypalexpress_Api_Client {
		return $this->api;
	}

	/**
	 * Handles PayPal webhooks.
	 *
	 * @since 1.5.0
	 *
	 * @return void
	 */
	public function handle_webhook(): void {
		$this->webhooks->handler();
	}

	/**
	 * Cancels a subscription in PayPal when the membership is canceled.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.5.0
	 *
	 * @param MS_Model_Relationship $subscription The subscription.
	 *
	 * @return void
	 */
	public function cancel_membership( $subscription ) {
		parent::cancel_membership( $subscription );

		$membership = $subscription->get_membership();

		// Stop here if the membership does not support recurring payments.
		if ( ! $membership->supports_recurring_payments() ) {
			return;
		}

		$subscription_id = MS_Helper_Cast::to_string(
			get_post_meta( $subscription->get_id(), 'ms_gateway_paypal_subscription_id', true )
		);

		if ( empty( $subscription_id ) ) {
			return;
		}

		$this->api->cancel_subscription(
			$subscription_id,
			$this->is_sandbox()
		);

		delete_post_meta( $subscription->get_id(), 'ms_gateway_paypal_subscription_id' );
	}

	/**
	 * Returns the onboarded status of the merchant.
	 *
	 * @since 1.5.0
	 *
	 * @return array<string>
	 */
	public function get_errors_from_on_boarded_data(): array {
		$error_messages    = [];
		$saved_merchant_id = $this->get_merchant_id_in_paypal();

		if (
			! $saved_merchant_id
			|| $this->merchant_account_is_ready
		) {
			return [];
		}

		$seller_status = $this->api->get_seller_status(
			$this->get_merchant_id_in_paypal(),
			$this->is_sandbox()
		);

		if (
			! Arr::exists( $seller_status, 'payments_receivable' )
			|| ! Arr::exists( $seller_status, 'primary_email_confirmed' )
		) {
			$error_messages[] = esc_html__( 'There was a problem with the status check for your PayPal account. Please try disconnecting and connecting again. If the problem persists, please contact support.', 'memberdash' );

			// Return here since the rest of the validations will definitely fail.
			return $error_messages;
		}

		$payments_receivable     = MS_Helper_Cast::to_bool( Arr::get( $seller_status, 'payments_receivable' ) );
		$primary_email_confirmed = MS_Helper_Cast::to_bool( Arr::get( $seller_status, 'primary_email_confirmed' ) );

		if ( ! $payments_receivable ) {
			$error_messages[] = esc_html__( 'Your account has been limited by PayPal - please check your PayPal account email inbox for details and next steps.', 'memberdash' );
		}

		if ( ! $primary_email_confirmed ) {
			$error_messages[] = wp_kses(
				sprintf(
					// translators: %s: PayPal email confirmation docs.
					__( 'Your PayPal account email is unconfirmed - please <a href="%s" target="_blank">confirm</a> your PayPal account email to start accepting payments.', 'memberdash' ),
					'https://www.paypal.com/us/cshelp/article/how-do-i-confirm-my-email-address-help138'
				),
				[
					'a' => [
						'href'   => [],
						'target' => [],
					],
				]
			);
		}

		$this->merchant_account_is_ready = $payments_receivable && $primary_email_confirmed;
		$this->save();

		if ( ! $this->supports_custom_payments ) {
			return $error_messages;
		}

		if ( array_diff( [ 'products', 'capabilities' ], array_keys( $seller_status ) ) ) {
			$error_messages[] = esc_html__( 'Your account was expected to be able to accept custom payments, but is not. Please make sure your account country matches the country setting. If the problem persists, please contact PayPal.', 'memberdash' );

			// Return here since the rest of the validations will definitely fail.
			return $error_messages;
		}

		// Grab the PPCP_CUSTOM product from the status data.
		$products = Arr::wrap(
			Arr::get( $seller_status, 'products', [] )
		);
		$custom_product = current(
			array_filter(
				$products,
				static function ( $product ) {
					return 'PPCP_CUSTOM' === Arr::get( $product, 'name' );
				}
			)
		);

		$has_custom_payments = 'SUBSCRIBED' === MS_Helper_Cast::to_string(
			Arr::get( $custom_product, 'vetting_status' )
		);
		if ( ! $has_custom_payments ) {
			$error_messages[] = esc_html__( 'Reach out to PayPal to enable PPCP_CUSTOM for your account', 'memberdash' );
		}

		// Loop through the capabilities and see if any are not active.
		$invalid_capabilities = [];
		$capabilities         = Arr::wrap(
			Arr::get( $seller_status, 'capabilities', [] )
		);
		foreach ( $capabilities as $capability ) {
			if ( $capability['status'] !== 'ACTIVE' ) {
				$invalid_capabilities[] = $capability['name'];
			}
		}

		if ( ! empty( $invalid_capabilities ) ) {
			$error_messages[] = esc_html__( 'Reach out to PayPal to resolve the following capabilities:', 'memberdash' ) . ' ' . implode( ', ', $invalid_capabilities );
		}

		// If there were errors then redirect the user with notices.
		return $error_messages;
	}

	/**
	 * Hides the PayPal Standard button if the PayPal Checkout is configured.
	 *
	 * @since 1.6.1
	 *
	 * @param string $html The HTML to hide the PayPal Standard button.
	 *
	 * @return string The HTML to hide the PayPal Standard button.
	 */
	public function hide_paypal_standard_button( string $html ): string {
		if ( $this->is_configured() ) {
			return '';
		}

		return $html;
	}
}
