<?php
/**
 * WooCommerce gateway.
 *
 * @since 1.4.0
 *
 * @package MemberDash
 */

/**
 * WooCommerce gateway class.
 *
 * @since 1.4.0
 */
class MS_Gateway_WooCommerce extends MS_Gateway {
	/**
	 * Gateway ID.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.4.0
	 */
	const ID = 'woocommerce';

	/**
	 * Hook to show payment info.
	 *
	 * This is called by the MS_Factory. Inherited from MS_Gateway.
	 *
	 * @since 1.4.0
	 *
	 * @return void
	 */
	public function after_load() {
		parent::after_load();

		$this->id     = self::ID;
		$this->name   = 'WooCommerce';
		$this->group  = '';
		$this->active = true;

		// Make this a manual payment gateway since WooCommerce is handling the payment and no automation is needed.
		$this->manual_payment = true;

		/*
		 * No sandbox option for manual payment gateway.
		 * The mode is always set to live.
		 */
		$this->mode = 'live';
	}

	/**
	 * Return status if all fields are configured.
	 *
	 * Inherited from MS_Gateway.
	 *
	 * @since 1.4.0
	 *
	 * @return bool
	 */
	public function is_configured() {
		return MS_Helper_WooCommerce::is_woocommerce_activated();
	}

	/**
	 * Processes the order.
	 *
	 * @since 1.4.0
	 *
	 * @param MS_Model_Relationship    $subscription The related membership relationship.
	 * @param WC_Order|WC_Subscription $order        The WooCommerce order.
	 *
	 * @return void
	 */
	public function process_order( MS_Model_Relationship $subscription, $order ): void {
		if (
			function_exists( 'wcs_order_contains_subscription' )
			&& wcs_order_contains_subscription( $order )
		) {
			return;
		}

		$invoice = $subscription->get_current_invoice( true, false );
		$total   = $this->get_order_total( $order );

		// Set invoice data and pay it.
		$invoice->set_total( $total );
		$invoice->set_amount( $order->get_total( 'edit' ) );
		$invoice->set_discount( $order->get_total_discount() );
		$invoice->pay_it(
			self::ID,
			sprintf(
				'wc_%s',
				$order->get_id()
			)
		);
		$invoice->add_notes(
			sprintf(
				// translators: %1$s: Order number, %2$s: Payment method name.
				__( 'Payment processed by WooCommerce. Order number: %1$s. Payment method: %2$s.', 'memberdash' ),
				$order->get_order_number(),
				$order->get_payment_method_title()
			)
		);

		// Set checkout IP and date.
		$invoice->set_checkout_ip( $order->get_customer_ip_address() );
		$date = $order->get_date_created();
		if ( $date instanceof WC_DateTime ) {
			$invoice->set_checkout_date( $date->format( 'Y-m-d H:i:s' ) );
		}

		// Update invoice signup fee.
		if ( $order instanceof WC_Subscription ) {
			$invoice->set_signup_fee( MS_Helper_Cast::to_float( $order->get_sign_up_fee() ) );
		}

		$invoice->save();

		// Activate subscription.
		$subscription->set_status( MS_Model_Relationship::STATUS_ACTIVE );
		$subscription->save();

		$this->maybe_update_payments( $subscription->get_id() );
		MS_Model_Event::save_event( MS_Model_Event::TYPE_MS_SIGNED_UP, $subscription );

		/*
		 * Creates a transaction log entry.
		 *
		 * Documented in /includes/gateways/class-ms-controller-gateway.php.
		 */
		do_action(
			'ms_gateway_transaction_log',
			$subscription->get_gateway_id(),
			'handle', // Always handle for WooCommerce.
			true, // Success flag.
			$subscription->get_id(), // Subscription ID.
			$invoice->get_id(), // Invoice ID.
			$total, // Charged amount.
			sprintf(
				// translators: %s: Order ID.
				__( 'Payment processed by WooCommerce. Order ID: %s.', 'memberdash' ),
				$order->get_id()
			), // Descriptive text.
			$order->get_id() // External ID.
		);
	}

	/**
	 * Maybe update payments.
	 *
	 * Remove free payments from the subscription that were created because the
	 * membership does not have a price.
	 *
	 * @since 1.4.0
	 *
	 * @param int $subscription_id The subscription ID.
	 *
	 * @return void
	 */
	private function maybe_update_payments( int $subscription_id ): void {
		$payments = get_post_meta( $subscription_id, 'payments', true );

		if ( ! is_array( $payments ) ) {
			return;
		}

		// Remove free payments.
		$payments = array_filter(
			$payments,
			function( $payment ) {
				return $payment['gateway'] !== 'free';
			}
		);

		update_post_meta( $subscription_id, 'payments', $payments );
	}

	/**
	 * Get the order total.
	 *
	 * @since 1.4.0
	 *
	 * @param WC_Order|WC_Subscription $order The WooCommerce order.
	 *
	 * @return float
	 */
	private function get_order_total( $order ): float {
		$total = $order->get_total( 'edit' );

		// Get the parent order if the order is a subscription.
		if ( $order instanceof WC_Subscription ) {
			$parent = $order->get_parent();

			if ( $parent instanceof WC_Order ) {
				$total = $parent->get_total( 'edit' );
			}
		}

		return MS_Helper_Cast::to_float( $total );
	}
}
