<?php
/**
 * Billing Helper class
 *
 * @since 1.0.0
 * @package MemberDash
 * @subpackage Helper
 */

/**
 * Billing Helper class.
 *
 * @since 1.0.0
 */
class MS_Helper_Billing extends MS_Helper {

	const BILLING_MSG_ADDED        = 1;
	const BILLING_MSG_DELETED      = 2;
	const BILLING_MSG_UPDATED      = 3;
	const BILLING_MSG_NOT_ADDED    = -1;
	const BILLING_MSG_NOT_DELETED  = -2;
	const BILLING_MSG_NOT_UPDATED  = -3;
	const BILLING_MSG_NOT_A_MEMBER = -5;

	public static function get_admin_message( $msg = 0 ) {
		$messages = apply_filters(
			'ms_helper_billing_get_admin_messages',
			array(
				self::BILLING_MSG_ADDED        => __( 'Added new invoice.', 'memberdash' ),
				self::BILLING_MSG_DELETED      => __( 'Removed invoice(s).', 'memberdash' ),
				self::BILLING_MSG_UPDATED      => __( 'Invoice details updated.', 'memberdash' ),
				self::BILLING_MSG_NOT_ADDED    => __( 'Could not add invoice.', 'memberdash' ),
				self::BILLING_MSG_NOT_DELETED  => __( 'Could not remove invoice(s).', 'memberdash' ),
				self::BILLING_MSG_NOT_UPDATED  => __( 'Could not update invoice.', 'memberdash' ),
				self::BILLING_MSG_NOT_A_MEMBER => __( 'Did not add the invoice: User is not a member of selected Membership.', 'memberdash' ),
			)
		);

		if ( array_key_exists( $msg, $messages ) ) {
			return $messages[ $msg ];
		}
		return null;
	}

	public static function print_admin_message() {
		$msg      = ! empty( $_GET['msg'] ) ? (int) $_GET['msg'] : 0;
		$class    = ( $msg > 0 ) ? 'updated' : 'error';
		$contents = self::get_admin_message( $msg );

		if ( $contents ) {
			mslib3()->ui->admin_message( $contents, $class );
		}
	}

	/**
	 * Formats a number to display a valid price.
	 *
	 * @since 1.0.0
	 * @param  numeric $price
	 * @return numeric
	 */
	public static function format_price( $price ) {
		$formatted = number_format( (float) $price, 2, '.', '' );

		return apply_filters(
			'ms_format_price',
			$formatted,
			$price
		);
	}

	/**
	 * Gets currency symbol.
	 *
	 * @since 1.3.0
	 *
	 * @param string $currency_code Currency code. Default is empty.
	 *
	 * @return string Currency symbol.
	 */
	public static function get_currency_symbol( string $currency_code = '' ): string {
		$currency_code = ! empty( $currency_code )
			? $currency_code
			: MS_Plugin::get_settings()->get_currency();

		// Default to currency code if symbol is not found.
		$symbol = $currency_code;

		// Get currency symbol from NumberFormatter if available.
		if (
			! empty( $symbol )
			&& class_exists( 'NumberFormatter' )
		) {
			$number_format = new NumberFormatter(
				get_locale() . '@currency=' . $symbol,
				NumberFormatter::CURRENCY
			);

			$symbol = $number_format->getSymbol( NumberFormatter::CURRENCY_SYMBOL );
		} elseif ( ! empty( $currency_code ) ) {
			// Fallback for hosts lacking NumberFormatter support.
			switch ( $currency_code ) {
				case 'USD':
					$symbol = '$';
					break;
				case 'EUR':
					$symbol = '€';
					break;
				case 'JPY':
					$symbol = '¥';
					break;
			}
		}

		/**
		 * Filters the currency symbol.
		 *
		 * @since 1.3.0
		 *
		 * @param string $symbol        Currency symbol.
		 * @param string $currency_code Currency code.
		 *
		 * @return string Currency symbol.
		 */
		return apply_filters(
			'ms_get_currency_symbol',
			$symbol,
			$currency_code
		);
	}

	/**
	 * Gets the price formatted based on the MemberDash global currency settings.
	 *
	 * @since 1.3.0
	 *
	 * @param float  $price                The price to format.
	 * @param string $currency_code        The currency code. Default is empty.
	 * @param bool   $use_number_formatter Whether to use NumberFormatter. Default is true. If set to false, a fallback will be used.
	 *
	 * @return string The formatted price.
	 */
	public static function get_formatted_price(
		float $price,
		string $currency_code = '',
		bool $use_number_formatter = true
	): string {
		// Get currency code.
		$currency_code = ! empty( $currency_code )
			? $currency_code
			: MS_Plugin::get_settings()->get_currency();

		// Price is shown as is if no currency set.
		if ( empty( $currency_code ) ) {
			return "$price";
		}

		// Format price.
		if ( $use_number_formatter && class_exists( 'NumberFormatter' ) ) {
			$number_format = new NumberFormatter(
				get_locale() . '@currency=' . $currency_code,
				NumberFormatter::CURRENCY
			);

			// Returns string if successful or false if failed.
			$formatted = $number_format->format( $price );

			// Return formatted price if available.
			if ( $formatted ) {
				return $formatted;
			}
		}

		// Fallback for hosts lacking NumberFormatter support.
		return sprintf(
			'%s %s',
			self::get_currency_symbol( $currency_code ),
			self::format_price( $price )
		);
	}

}
