<?php
/**
 * Membership Page Rule class.
 *
 * Persisted by Membership class.
 *
 * @since 1.0.0
 *
 * @package MemberDash
 * @subpackage Model
 */
class MS_Rule_Page_Model extends MS_Rule {

	/**
	 * Rule type.
	 *
	 * @since 1.0.0
	 *
	 * @var string $rule_type
	 */
	protected $rule_type = MS_Rule_Page::RULE_ID;

	/**
	 * Initialize the rule.
	 *
	 * @since 1.0.0
	 * @param int $membership_id
	 */
	public function __construct( $membership_id ) {
		parent::__construct( $membership_id );

		$this->add_filter(
			'ms_rule_exclude_items-' . $this->rule_type,
			'exclude_items',
			10,
			2
		);
	}

	/**
	 * Set initial protection (front-end only)
	 *
	 * @since 1.0.0
	 */
	public function protect_content() {
		parent::protect_content();

		$this->add_filter( 'get_pages', 'protect_pages', 99, 2 );
		$this->add_filter( 'pre_get_posts', 'hide_pages_from_search', 99 );
	}

	/**
	 * Excludes pages from site search by filtering the WP query.
	 *
	 * Related Action Hooks:
	 * - pre_get_posts
	 *
	 * @param WP_Query $wp_query The WP_Query object to filter.
	 */
	public function hide_pages_from_search( $wp_query ) {
		if ( ! $this->is_frontend_search( $wp_query ) || ! MS_Model_Addon::is_enabled( MS_Model_Addon::ADDON_HIDE_PAGES_FROM_SEARCH ) ) {
			return;
		}

		$denied_ids = array();
		foreach ( $this->rule_value as $id => $value ) {
			if ( ! $this->has_access( $id ) ) {
				$denied_ids[] = $id;
			}
		}

		$denied_ids = array_unique( $denied_ids, SORT_NUMERIC );

		// Tell the WP query which pages are actually off limit for the user.
		$wp_query->query_vars['post__not_in'] = array_merge(
			$wp_query->query_vars['post__not_in'],
			$denied_ids
		);

		do_action(
			'ms_rule_page_model_hide_pages_from_search',
			$wp_query,
			$this
		);
	}

	/**
	 * Examines the passed WP query object to check if it is for frontend search.
	 *
	 * @param $wp_query \WP_Query The query object to examine.
	 * @return bool
	 */
	private function is_frontend_search( $wp_query ) {
		return ! $wp_query->is_admin && $wp_query->is_search();
	}

	/**
	 * Filters protected pages.
	 *
	 * @since 1.0.0
	 *
	 * Related action hook:
	 * - get_pages
	 *
	 * @param  array $pages The array of pages to filter.
	 * @param  array $args Array of arguments used by WP to filter the posts.
	 * @return array Filtered array which doesn't include prohibited pages.
	 */
	public function protect_pages( $pages, $args ) {
		// Do not filter custom post types with this rule!
		if ( 'page' != $args['post_type'] ) {
			return $pages;
		}

		$rule_value = apply_filters(
			'ms_rule_page_model_protect_pages_rule_value',
			$this->rule_value
		);
		$membership = $this->get_membership();

		if ( ! is_array( $pages ) ) {
			$pages = (array) $pages;
		}

		foreach ( $pages as $key => $page ) {
			if ( ! self::has_access( $page->ID ) ) {
				unset( $pages[ $key ] );
			}
		}

		return apply_filters(
			'ms_rule_page_model_protect_pages',
			$pages,
			$this
		);
	}

	/**
	 * Get the current page id.
	 *
	 * @since 1.0.0
	 *
	 * @return int The page id, or null if it is not a page.
	 */
	private function get_current_page_id() {
		$page_id = null;
		$post    = get_queried_object();

		if ( $post instanceof WP_Post && 'page' == $post->post_type ) {
			$page_id = $post->ID;
		}

		return apply_filters(
			'ms_rule_page_model_get_current_page_id',
			$page_id,
			$this
		);
	}

	/**
	 * Verify access to the current page.
	 *
	 * @since 1.0.0
	 *
	 * @param int $id The page_id to verify access.
	 * @return bool|null True if has access, false otherwise.
	 *     Null means: Rule not relevant for current page.
	 */
	public function has_access( $id, $admin_has_access = true ) {
		$has_access = null;

		if ( empty( $id ) ) {
			$id = $this->get_current_page_id();
		} else {
			$post = get_post( $id );
			if ( ! $post instanceof WP_Post || 'page' != $post->post_type ) {
				$id = 0;
			}
		}

		if ( ! empty( $id ) ) {
			$has_access = false;
			// Membership special pages has access
			if ( MS_Model_Pages::is_membership_page( $id ) ) {
				$has_access = true;
			} else {
				$has_access = parent::has_access( $id, $admin_has_access );
			}
		}

		return apply_filters(
			'ms_rule_page_model_has_access',
			$has_access,
			$id,
			$this
		);
	}

	/**
	 * Verify if has dripped rules.
	 *
	 * @since 1.0.0
	 *
	 * @param string $id The content id to verify.
	 * @return boolean True if has dripped rules.
	 */
	public function has_dripped_rules( $page_id = null ) {
		if ( empty( $page_id ) ) {
			$page_id = $this->get_current_page_id();
		}

		return parent::has_dripped_rules( $page_id );
	}

	/**
	 * Get the total content count.
	 *
	 * @since 1.0.0
	 *
	 * @param $args The query post args
	 *     @see @link http://codex.wordpress.org/Function_Reference/get_pages
	 * @return int The total content count.
	 */
	public function get_content_count( $args = null ) {
		unset( $args['number'] );
		$args  = $this->get_query_args( $args );
		$posts = get_pages( $args );

		$count = count( $posts );

		return apply_filters(
			'ms_rule_page_model_get_content_count',
			$count,
			$args
		);
	}

	/**
	 * Get content to protect.
	 *
	 * @since 1.0.0
	 * @param $args The query post args
	 *     @see @link http://codex.wordpress.org/Function_Reference/get_pages
	 * @return array The contents array.
	 */
	public function get_contents( $args = null ) {
		/**
		 * The 'hierarchial' flag messes up the offset by skipping some children
		 * in some cases (i.e. it will always skip pages until the first result
		 * is a top-level page). We have to get all pages from 0 and paginate
		 * manually...
		 */
		$offset         = is_array( $args ) && array_key_exists( 'offset', $args ) ? absint( $args['offset'] ) : 0;
		$limit          = $offset + ( is_array( $args ) && array_key_exists( 'number', $args ) ? absint( $args['number'] ) : 0 );
		$args['offset'] = 0;
		$args['number'] = 0;

		$args = $this->get_query_args( $args );

		$pages    = get_pages( $args );
		$contents = array();
		if ( 0 == $limit ) {
			$limit = count( $pages ); }

		for ( $num = $offset; $num < $limit; $num++ ) {
			if ( ! isset( $pages[ $num ] ) ) {
				continue; }

			$content = $pages[ $num ];
			$name    = $content->post_title;

			$parent = get_post( $content->post_parent );
			for ( $level = 0; $level < 5 && $parent; $level++ ) {
				$name   = '&mdash; ' . $name;
				$parent = get_post( $parent->post_parent );
			}

			$content->id     = $content->ID;
			$content->type   = MS_Rule_Page::RULE_ID;
			$content->name   = $name;
			$content->access = $this->get_rule_value( $content->id );

			$contents[ $content->id ] = $content;
		}

		return apply_filters(
			'ms_rule_page_model_get_contents',
			$contents,
			$this
		);
	}

	/**
	 * Get the default query args.
	 *
	 * @since 1.0.0
	 *
	 * @param string $args The query post args.
	 *     @see @link http://codex.wordpress.org/Function_Reference/get_pages
	 * @return array The parsed args.
	 */
	public function get_query_args( $args = null ) {
		return parent::prepare_query_args( $args, 'get_pages' );
	}

	/**
	 * Exclude the special Membership pages from the results as they
	 * cannot be protected.
	 *
	 * @since 1.0.0
	 * @param  array $excluded
	 * @param  array $args
	 * @return array
	 */
	public function exclude_items( $excluded, $args ) {
		static $Page_Ids = null;

		if ( null === $Page_Ids ) {
			$Page_Ids = array();
			$types    = MS_Model_Pages::get_page_types();
			foreach ( $types as $type => $title ) {
				$Page_Ids[] = MS_Model_Pages::get_setting( $type );
			}
		}

		return array_merge( $excluded, $Page_Ids );
	}

}
