<?php
/**
 * Protection API hook
 *
 * Manages all Protection API actions
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Protection API hook
 *
 * Manages all Protection API actions
 *
 * @since 1.0.0
 */
class MS_Api_Protection extends MS_Api {
	const BASE_API_ROUTE = '/protection/';

	/**
	 * Singleton instance of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var MS_Plugin
	 */
	private static $instance = null;

	/**
	 * Returns singleton instance of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @static
	 * @access public
	 *
	 * @return MS_Api
	 */
	public static function instance() {
		if ( ! self::$instance ) {
			self::$instance = new MS_Api_Protection();
		}

		return self::$instance;
	}

	/**
	 * Set up the api routes
	 *
	 * @since 1.0.0
	 *
	 * @param string $endpoint The parent endpoint.
	 *
	 * @return void
	 */
	public function set_up_route( $endpoint ) {
		register_rest_route(
			$endpoint,
			self::BASE_API_ROUTE . 'check',
			[
				'method'              => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'check_access' ],
				'permission_callback' => [ $this, 'validate_request' ],
				'args'                => [
					'content_id' => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'Content ID to check access for', 'memberdash' ),
					],
					'rule_type'  => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'string',
						'description'       => __( 'Rule type (page, post, url, etc.)', 'memberdash' ),
					],
					'user_id'    => [
						'required'          => false,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'User ID to check access for. Defaults to current user.', 'memberdash' ),
					],
				],
			]
		);

		register_rest_route(
			$endpoint,
			self::BASE_API_ROUTE . 'protect',
			[
				'method'              => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'protect_content' ],
				'permission_callback' => [ $this, 'validate_request' ],
				'args'                => [
					'content_id'     => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'Content ID to protect', 'memberdash' ),
					],
					'rule_type'      => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'string',
						'description'       => __( 'Rule type (page, post, url, etc.)', 'memberdash' ),
					],
					'membership_ids' => [
						'required'          => true,
						'sanitize_callback' => [ $this, 'sanitize_membership_ids' ],
						'type'              => 'array',
						'description'       => __( 'Array of membership IDs that should have access', 'memberdash' ),
					],
				],
			]
		);

		register_rest_route(
			$endpoint,
			self::BASE_API_ROUTE . 'unprotect',
			[
				'method'              => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'unprotect_content' ],
				'permission_callback' => [ $this, 'validate_request' ],
				'args'                => [
					'content_id' => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'Content ID to unprotect', 'memberdash' ),
					],
					'rule_type'  => [
						'required'          => true,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'string',
						'description'       => __( 'Rule type (page, post, url, etc.)', 'memberdash' ),
					],
				],
			]
		);

		register_rest_route(
			$endpoint,
			self::BASE_API_ROUTE . 'list',
			[
				'method'              => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'list_protected_content' ],
				'permission_callback' => [ $this, 'validate_request' ],
				'args'                => [
					'rule_type' => [
						'required'          => false,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'string',
						'description'       => __( 'Rule type filter (page, post, url, etc.)', 'memberdash' ),
					],
					'per_page'  => [
						'required'          => false,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'Results per page. Defaults to 10', 'memberdash' ),
					],
					'page'      => [
						'required'          => false,
						'sanitize_callback' => 'sanitize_text_field',
						'type'              => 'integer',
						'description'       => __( 'Current page. Defaults to 1', 'memberdash' ),
					],
				],
			]
		);
	}

	/**
	 * Sanitize membership IDs array.
	 *
	 * @param int[] $membership_ids Array of membership IDs.
	 *
	 * @return int[]
	 */
	public function sanitize_membership_ids( $membership_ids ) {
		if ( ! is_array( $membership_ids ) ) {
			return [];
		}

		return array_map( 'absint', $membership_ids );
	}

	/**
	 * Check if user has access to content.
	 *
	 * @param WP_REST_Request<array<string, mixed>> $request The request object.
	 *
	 * @return array<string, mixed>|WP_Error
	 */
	public function check_access( $request ) {
		$content_id = absint( MS_Helper_Cast::to_int( $request->get_param( 'content_id' ) ) );
		$rule_type  = sanitize_text_field( MS_Helper_Cast::to_string( $request->get_param( 'rule_type' ) ) );
		$user_id    = MS_Helper_Cast::to_int( $request->get_param( 'user_id' ) );

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		} else {
			$user_id = absint( $user_id );
		}

		$valid_rule_types = MS_Model_Rule::get_rule_types();
		if ( ! in_array( $rule_type, $valid_rule_types, true ) ) {
			return new WP_Error( 'invalid_rule_type', __( 'Invalid rule type specified', 'memberdash' ), [ 'status' => 400 ] );
		}

		$member     = MS_Factory::load( 'MS_Model_Member', $user_id );
		$has_access = false;

		if ( $member->is_normal_admin() ) {
			$has_access = true;
		} else {
			$subscriptions = $member->get_subscriptions();
			foreach ( $subscriptions as $subscription ) {
				$status          = $subscription->get_status();
				$active_statuses = [
					MS_Model_Relationship::STATUS_ACTIVE,
					MS_Model_Relationship::STATUS_TRIAL,
					MS_Model_Relationship::STATUS_CANCELED,
				];
				if ( in_array( $status, $active_statuses, true ) ) {
					$membership = $subscription->get_membership();
					$rule       = $membership->get_rule( (string) $rule_type );

					if ( $rule->has_access( (string) $content_id ) ) {
						$has_access = true;
						break;
					}
				}
			}
		}

		return [
			'has_access' => $has_access,
			'content_id' => $content_id,
			'rule_type'  => $rule_type,
			'user_id'    => $user_id,
		];
	}

	/**
	 * Protect content.
	 *
	 * @param WP_REST_Request<array<string, mixed>> $request The request object.
	 *
	 * @return array<string, mixed>|WP_Error
	 */
	public function protect_content( $request ) {
		$content_id     = absint( MS_Helper_Cast::to_int( $request->get_param( 'content_id' ) ) );
		$rule_type      = sanitize_text_field( MS_Helper_Cast::to_string( $request->get_param( 'rule_type' ) ) );
		$membership_ids = $request->get_param( 'membership_ids' );

		$valid_rule_types = MS_Model_Rule::get_rule_types();
		if ( ! in_array( $rule_type, $valid_rule_types, true ) ) {
			return new WP_Error( 'invalid_rule_type', __( 'Invalid rule type specified', 'memberdash' ), [ 'status' => 400 ] );
		}

		if ( empty( $membership_ids ) || ! is_array( $membership_ids ) ) {
			return new WP_Error( 'invalid_membership_ids', __( 'Valid membership IDs array is required', 'memberdash' ), [ 'status' => 400 ] );
		}

		try {
			$base_membership = MS_Model_Membership::get_base();
			$base_rule       = $base_membership->get_rule( (string) $rule_type );

			$base_rule->give_access( (string) $content_id );
			$base_membership->set_rule( $rule_type, $base_rule );
			$base_membership->save();

			foreach ( $membership_ids as $membership_id ) {
				$membership = MS_Factory::load( 'MS_Model_Membership', $membership_id );
				if ( $membership->is_valid() ) {
					$rule = $membership->get_rule( (string) $rule_type );
					$rule->give_access( (string) $content_id );
					$membership->set_rule( $rule_type, $rule );
					$membership->save();
				}
			}

			return [
				'success'        => true,
				'content_id'     => $content_id,
				'rule_type'      => $rule_type,
				'membership_ids' => $membership_ids,
				'message'        => __( 'Content protected successfully', 'memberdash' ),
			];
		} catch ( Exception $e ) {
			return new WP_Error( 'protection_error', $e->getMessage(), [ 'status' => 500 ] );
		}
	}

	/**
	 * Unprotect content.
	 *
	 * @param WP_REST_Request<array<string, mixed>> $request The request object.
	 *
	 * @return array<string, mixed>|WP_Error
	 */
	public function unprotect_content( $request ) {
		$content_id = absint( MS_Helper_Cast::to_int( $request->get_param( 'content_id' ) ) );
		$rule_type  = sanitize_text_field( MS_Helper_Cast::to_string( $request->get_param( 'rule_type' ) ) );

		$valid_rule_types = MS_Model_Rule::get_rule_types();
		if ( ! in_array( $rule_type, $valid_rule_types, true ) ) {
			return new WP_Error( 'invalid_rule_type', __( 'Invalid rule type specified', 'memberdash' ), [ 'status' => 400 ] );
		}

		try {
			$base_membership = MS_Model_Membership::get_base();
			$base_rule       = $base_membership->get_rule( (string) $rule_type );

			$base_rule->remove_access( (string) $content_id );
			$base_membership->set_rule( $rule_type, $base_rule );
			$base_membership->save();

			$memberships = MS_Model_Membership::get_memberships();
			foreach ( $memberships as $membership ) {
				if ( ! $membership->is_base() ) {
					$rule = $membership->get_rule( (string) $rule_type );
					$rule->remove_access( (string) $content_id );
					$membership->set_rule( $rule_type, $rule );
					$membership->save();
				}
			}

			return [
				'success'    => true,
				'content_id' => $content_id,
				'rule_type'  => $rule_type,
				'message'    => __( 'Content unprotected successfully', 'memberdash' ),
			];
		} catch ( Exception $e ) {
			return new WP_Error( 'unprotected_error', $e->getMessage(), [ 'status' => 500 ] );
		}
	}

	/**
	 * List protected content.
	 *
	 * @param WP_REST_Request<array<string, mixed>> $request The request object.
	 *
	 * @return array<string, mixed>|WP_Error
	 */
	public function list_protected_content( $request ) {
		$rule_type = $request->get_param( 'rule_type' );
		$per_page  = absint( MS_Helper_Cast::to_int( $request->get_param( 'per_page' ) ) );
		$page      = absint( MS_Helper_Cast::to_int( $request->get_param( 'page' ) ) );

		if ( empty( $per_page ) ) {
			$per_page = 10;
		}
		if ( empty( $page ) ) {
			$page = 1;
		}

		try {
			$base_membership   = MS_Model_Membership::get_base();
			$protected_content = [];

			if ( $rule_type ) {
				$valid_rule_types = MS_Model_Rule::get_rule_types();
				if ( ! in_array( $rule_type, $valid_rule_types, true ) ) {
					return new WP_Error( 'invalid_rule_type', __( 'Invalid rule type specified', 'memberdash' ), [ 'status' => 400 ] );
				}

				$rule                            = $base_membership->get_rule( MS_Helper_Cast::to_string( $rule_type ) );
				$args                            = [
					'per_page' => $per_page,
					'page'     => $page,
				];
				$content                         = $rule->get_contents( $args );
				$protected_content[ $rule_type ] = $content;
			} else {
				$rule_types = MS_Model_Rule::get_rule_types();
				foreach ( $rule_types as $type ) {
					$rule    = $base_membership->get_rule( (string) $type );
					$args    = [
						'per_page' => $per_page,
						'page'     => $page,
					];
					$content = $rule->get_contents( $args );
					if ( ! empty( $content ) ) {
						$protected_content[ $type ] = $content;
					}
				}
			}

			return [
				'success'           => true,
				'protected_content' => $protected_content,
				'per_page'          => $per_page,
				'page'              => $page,
			];
		} catch ( Exception $e ) {
			return new WP_Error( 'list_error', $e->getMessage(), [ 'status' => 500 ] );
		}
	}
}
