/*global msPayPalExpress:false */
/*global paypal:false */

jQuery(function ($) {
	const script = $('.ms-gateway-paypalexpress-script'),
		gateway = $('.ms-gateway-paypalexpress-checkout');

	// Stop if no gateway or script.
	if (!gateway.length || !script.length) {
		return;
	}

	const container = $('#ms-gateway-paypalexpress-buttons');

	let timeouts = [],
		isGenericError = true;

	if (typeof paypal !== 'undefined') {
		// Render the PayPal button.
		setupButtons();

		// Render the card fields.
		setupCardFields();

		// Refresh the page when the token expires.
		if (script.length && script.is('[data-client-token-expires-in]')) {
			const timeout =
				parseInt(script.data('clientTokenExpiresIn'), 10) * 1000;

			timeouts.push(
				setTimeout(function () {
					window.location.replace(window.location.href);
				}, timeout)
			);
		}
	}

	function showNotice(message) {
		const wrap = $('.ms-purchase-table'),
			notice = $(
				'<div id="ms-paypal-express-notice" class="ms-alert-box ms-alert-error" role="alert"></div>'
			);

		notice.text(message);
		wrap.before(notice);
	}

	function showCardFieldsNotice(id) {
		// Workaround for a bug in PayPal's CardFields library.
		if (id === 'p is not a function') {
			id = 'unknown_error';
		}

		const wrap = $('#ms-gateway-paypalexpress-card-fields-title'),
			notice = $(
				'<div id="ms-paypal-express-card-fields-notice" class="ms-alert-box ms-alert-error" role="alert"></div>'
			),
			message = msPayPalExpress.card_fields[id.toLowerCase()] ?? id;

		$('#ms-paypal-express-card-fields-notice').remove();

		notice.text(message);
		wrap.after(notice);
	}

	function hideNotice() {
		$('#ms-paypal-express-notice').remove();
	}

	function createOrder(cardFields) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_create_order',
				nonce: msPayPalExpress.nonce,
				subscription_id: container.data('subscription-id'),
				card_fields: cardFields,
			},
		}).then(function (res) {
			if (res.success) {
				return res.data.order_id;
			} else {
				isGenericError = false;
				return showNotice(res.data.message);
			}
		});
	}

	function createSubscription() {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_create_subscription',
				nonce: msPayPalExpress.nonce,
				subscription_id: container.data('subscription-id'),
				plan_id: container.data('plan-id'),
			},
		}).then(function (res) {
			if (res.success) {
				return res.data.subscription_id;
			} else {
				isGenericError = false;
				return showNotice(res.data.message);
			}
		});
	}

	function handleApproveFail(data) {
		if ('UNPROCESSABLE_ENTITY' === data.data[0].name) {
			if ('INSTRUMENT_DECLINED' === data.data[0].details[0].issue) {
				showNotice(data.data[0].details[0].description);
			} else {
				console.error(data.data[0].details[0].description);
				showNotice(data.data[0].message);
			}
		} else {
			showNotice(data.data[0].message);
		}
	}

	function handleOrderCheckSuccess(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_order_approve',
				nonce: msPayPalExpress.nonce,
				order_id: data.orderID,
				payer_id: data.payerID,
				recheck: true,
			},
		})
			.then(function (res) {
				if (!res.success) {
					return handleApproveFail(res);
				}

				return window.location.replace(res.data.redirect_url);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleOrderApprove(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_order_approve',
				nonce: msPayPalExpress.nonce,
				order_id: data.orderID,
				payer_id: data.payerID,
			},
		})
			.then(function (res) {
				if (!res.success) {
					return handleApproveFail(res);
				}

				return handleOrderCheckSuccess(data);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleSubscriptionCheckSuccess(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_subscription_approve',
				nonce: msPayPalExpress.nonce,
				subscription_id: data.subscriptionID,
				order_id: data.orderID,
				recheck: true,
			},
		})
			.then(function (res) {
				if (!res.success) {
					return showNotice(res.data.message);
				}

				return window.location.replace(res.data.redirect_url);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleSubscriptionApprove(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_subscription_approve',
				nonce: msPayPalExpress.nonce,
				subscription_id: data.subscriptionID,
				order_id: data.orderID,
			},
		})
			.then(function (res) {
				if (!res.success) {
					return showNotice(res.data.message);
				}

				if (res.data.recheck) {
					return handleSubscriptionCheckSuccess(data);
				}

				return window.location.replace(res.data.redirect_url);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleOrderCancel(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_order_cancel',
				nonce: msPayPalExpress.nonce,
				order_id: data.orderID,
			},
		})
			.then(function (res) {
				if (!res.success) {
					showNotice(res.data.message);
				}

				return window.location.replace(res.data.redirect_url);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleSubscriptionCancel(data) {
		return $.ajax({
			url: msPayPalExpress.ajax_url,
			type: 'POST',
			dataType: 'json',
			data: {
				action: 'ms_gateway_paypal_express_handle_subscription_cancel',
				nonce: msPayPalExpress.nonce,
				subscription_id: container.data('subscription-id'),
				order_id: data.orderID,
			},
		})
			.then(function (res) {
				if (!res.success) {
					showNotice(res.data.message);
				}

				return window.location.replace(res.data.redirect_url);
			})
			.catch(function (err) {
				console.error(err);
			});
	}

	function handleGenericError(data) {
		// Only show the generic error message once.
		if (!isGenericError) {
			isGenericError = true;
			return;
		}

		return showNotice(data);
	}

	function setupButtons() {
		const args = {
			style: {
				layout: 'vertical',
				shape: 'rect',
				label: 'paypal',
			},
			onError: function (data) {
				return handleGenericError(data);
			},
			onClick: function () {
				return hideNotice();
			},
		};

		// Create the order.
		if (container.data('plan-id')) {
			args.createSubscription = function () {
				return createSubscription();
			};
			args.onApprove = function (data) {
				return handleSubscriptionApprove(data);
			};
			args.onCancel = function (data) {
				return handleSubscriptionCancel(data);
			};
		} else {
			args.createOrder = function () {
				return createOrder(false);
			};
			args.onApprove = function (data) {
				return handleOrderApprove(data);
			};
			args.onCancel = function (data) {
				return handleOrderCancel(data);
			};
		}

		paypal.Buttons(args).render('#ms-gateway-paypalexpress-buttons');
	}

	function setupCardFields() {
		const args = {};

		// Card fields are not supported for subscriptions.
		if (container.data('plan-id')) {
			// We should use createVaultSetupToken instead, but it isn't supported yet in MemberDash.
			return;
		}

		// Handle orders for one-time payments.
		args.createOrder = function () {
			return createOrder(true);
		};
		args.onApprove = function (data) {
			return handleOrderApprove(data);
		};

		const cardField = window.paypal.CardFields(args);

		if (!cardField.isEligible()) {
			return;
		}

		// Display the card form.
		$('#ms-gateway-paypalexpress-card-fields').show();

		const nameField = cardField.NameField(),
			numberField = cardField.NumberField(),
			expiryField = cardField.ExpiryField(),
			cvvField = cardField.CVVField(),
			submitButton = $('#ms-gateway-paypalexpress-card-submit');

		nameField.render('#ms-gateway-paypalexpress-card-name-field');
		numberField.render('#ms-gateway-paypalexpress-card-number-field');
		expiryField.render('#ms-gateway-paypalexpress-card-expiry-field');
		cvvField.render('#ms-gateway-paypalexpress-card-cvv-field');

		submitButton.on('click', function (evt) {
			evt.preventDefault();

			submitButton
				.text(msPayPalExpress.card_fields_processing)
				.attr('disabled', 'disabled');

			cardField
				.submit()
				.then(function () {
					submitButton.text(msPayPalExpress.card_fields_redirecting);
				})
				.catch(function (err) {
					console.log(err);

					submitButton
						.text(msPayPalExpress.card_fields_submit)
						.removeAttr('disabled');

					// Error codes: "INELIGIBLE_CARD_VENDOR","INVALID_NAME", "INVALID_NUMBER", "INVALID_EXPIRY" or "INVALID_CVV".
					showCardFieldsNotice(err.message);
				});
		});
	}
});
