<?php
/**
 * Gateway: Paypal Express - WhoDat API
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

/**
 * PayPal Express gateway WhoDat API class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_Api_Whodat extends MS_Model_Option {
	/**
	 * WhoDat connect base URL.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	private static $base_url = 'https://whodat.stellarwp.com/md/commerce/v1/paypal';

	/**
	 * Returns the transient hash.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public static function get_transient_hash(): string {
		return MS_Helper_Cast::to_string(
			get_transient( 'ms_paypal_express_unique_signup_hash' )
		);
	}

	/**
	 * Returns the referral data link.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_referral_data_link(): string {
		$links = get_transient( 'ms_paypal_express_signup_data' );

		if ( ! is_array( $links ) ) {
			return '';
		}

		return $links['links'][0]['href'];
	}

	/**
	 * Updates the transient hash.
	 *
	 * @since 1.5.0
	 *
	 * @param string $value The value to store in the transient.
	 *
	 * @return bool
	 */
	public function update_transient_hash( string $value ): bool {
		return set_transient( 'ms_paypal_express_unique_signup_hash', $value, DAY_IN_SECONDS );
	}

	/**
	 * Fetches the signup link from PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @param string $hash       Which unique hash we are passing to the PayPal system.
	 * @param string $country    Which country code we are using.
	 * @param bool   $is_sandbox Whether we are using the sandbox mode. Defaults to false.
	 * @param bool   $force      It prevents the use of the cached version of the signup data.
	 *
	 * @return array<string,array<int,array<string,string>>>
	 */
	public function get_seller_signup_data(
		string $hash,
		string $country = 'US',
		bool $is_sandbox = false,
		bool $force = false
	): array {
		if ( empty( $hash ) ) {
			$hash = $this->generate_unique_signup_hash();
		}

		$signup_data = get_transient( 'ms_paypal_express_signup_data' );

		// If we have a cached version of the signup data, return it.
		if (
			! $force
			&& ! empty( $signup_data )
			&& is_array( $signup_data )
		) {
			return $signup_data;
		}

		$return_url = MS_Helper_Utility::home_url( '?paypal-express-signup=return' );

		$query_args = [
			'is_sandbox'  => $is_sandbox,
			'nonce'       => $hash,
			'tracking_id' => rawurlencode( $this->generate_unique_tracking_id() ),
			'return_url'  => esc_url( $return_url ),
			'country'     => $country,
			'vaulting'    => true,
		];

		$request = $this->client_get( 'seller/signup', $query_args );

		if (
			empty( $request )
			|| empty( $request['links'] )
		) {
			return [];
		}

		// Cache the signup data.
		set_transient( 'ms_paypal_express_signup_data', $request, DAY_IN_SECONDS );

		return $request;
	}

	/**
	 * Fetches the referral data from WhoDat/PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @param string $url        The URL to fetch the referral data for.
	 * @param bool   $is_sandbox Whether we are using the sandbox mode. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_seller_referral_data(
		string $url,
		bool $is_sandbox = false
	): array {
		if ( empty( $url ) ) {
			return [];
		}

		$query_args = [
			'is_sandbox' => $is_sandbox,
			'url'        => $url,
		];

		$request = $this->client_get( 'seller/referral-data', $query_args );

		if ( empty( $request ) ) {
			return [];
		}

		return $request;
	}

	/**
	 * Generates the signup link that redirects the seller to PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @param string $country    Which country code we are generating the URL for.
	 * @param bool   $is_sandbox Whether we are using the sandbox mode. Defaults to false.
	 * @param bool   $force      It prevents the system from using the cached version of the URL.
	 *
	 * @return string
	 */
	public function generate_connect_url(
		string $country = 'US',
		bool $is_sandbox = false,
		bool $force = false
	): string {
		$transient = 'ms_paypal_express_connect_url_' . $country;

		if ( ! $force ) {
			$url = get_transient( $transient );

			if ( ! empty( $url ) ) {
				return MS_Helper_Cast::to_string( $url );
			}
		}

		$hash = $this->generate_unique_signup_hash();
		$this->update_transient_hash( $hash );

		$signup_data = $this->get_seller_signup_data(
			$hash,
			$country,
			$is_sandbox,
			$force
		);

		if ( ! $signup_data ) {
			return '';
		}

		$signup_url = $signup_data['links'][1]['href'];

		set_transient( $transient, $signup_url, DAY_IN_SECONDS );

		return $signup_url;
	}

	/**
	 * Fetches the seller credentials from PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @param string $access_token The access token to use.
	 * @param bool   $is_sandbox    Whether we are using the sandbox mode. Defaults to false.
	 *
	 * @return array<string,array<int,array<string,string>>>
	 */
	public function get_seller_credentials(
		string $access_token,
		bool $is_sandbox = false
	): array {
		$query_args = [
			'is_sandbox'   => $is_sandbox,
			'access_token' => $access_token,
		];

		return $this->client_post( 'seller/credentials', $query_args );
	}

	/**
	 * Verifies if the seller is connected to PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @param string $saved_merchant_id The ID we are looking at Paypal with.
	 * @param bool   $is_sandbox        Whether we are using the sandbox mode. Defaults to false.
	 *
	 * @return array<string,array<int,array<string,string>>>
	 */
	public function get_seller_status(
		string $saved_merchant_id = '',
		bool $is_sandbox = false
	): array {
		$query_args = [
			'is_sandbox'  => $is_sandbox,
			'merchant_id' => $saved_merchant_id,
		];

		return $this->client_post( 'seller/status', $query_args );
	}

	/**
	 * Returns the base URL.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	protected function get_base_url(): string {
		return self::$base_url;
	}

	/**
	 * Returns the connect URL.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint The endpoint to connect to.
	 * @param array<string,mixed> $args     The arguments to pass to the endpoint.
	 *
	 * @return string
	 */
	protected function get_api_url( string $endpoint, array $args = [] ): string {
		$url = trailingslashit( $this->get_base_url() ) . $endpoint;

		if ( ! empty( $args ) ) {
			$url = add_query_arg( $args, $url );
		}

		return $url;
	}

	/**
	 * Fetches data from the WhoDat API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint The endpoint to connect to.
	 * @param array<string,mixed> $args     The arguments to pass to the endpoint.
	 *
	 * @return array<string,array<int,array<string,string>>>
	 */
	protected function client_get( string $endpoint, array $args = [] ): array {
		$url = $this->get_api_url( $endpoint, $args );

		$request = wp_remote_get( $url );

		if ( is_wp_error( $request ) ) {
			MS_Helper_Debug::debug_log( 'WhoDat request error: ' . $request->get_error_message() );

			return [];
		}

		$body = wp_remote_retrieve_body( $request );
		$body = json_decode( $body, true );

		if ( ! is_array( $body ) ) {
			MS_Helper_Debug::debug_log( 'WhoDat request error: ' . $body );

			return [];
		}

		return $body;
	}

	/**
	 * Posts data to the WhoDat API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint The endpoint to connect to.
	 * @param array<string,mixed> $args     The arguments to pass to the endpoint.
	 *
	 * @return array<string,array<int,array<string,string>>>
	 */
	protected function client_post( string $endpoint, array $args = [] ): array {
		$url = $this->get_api_url( $endpoint );

		$request = wp_remote_post(
			$url,
			[
				'body' => $args,
			]
		);

		if ( is_wp_error( $request ) ) {
			MS_Helper_Debug::debug_log( 'WhoDat request error: ' . $request->get_error_message() );

			return [];
		}

		$body = wp_remote_retrieve_body( $request );
		$body = json_decode( $body, true );

		if ( ! is_array( $body ) ) {
			MS_Helper_Debug::debug_log( 'WhoDat request error: ' . $body );

			return [];
		}

		return $body;
	}

	/**
	 * Generates a Tracking ID for this website.
	 *
	 * The Tracking ID is a site-specific identifier that links the client and platform accounts in the Payment Gateway
	 * without exposing sensitive data. By default, the identifier generated is a URL in the format:
	 *
	 * {SITE_URL}?v={GATEWAY_VERSION}-{RANDOM_6_CHAR_HASH}
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	protected function generate_unique_tracking_id(): string {
		$id        = wp_generate_password( 6, false, false );
		$url_frags = wp_parse_url( home_url() );
		$scheme    = isset( $url_frags['scheme'] ) ? $url_frags['scheme'] : 'http';
		$host      = isset( $url_frags['host'] ) ? $url_frags['host'] : 'localhost';
		$url       = $scheme . '://' . $host;
		$url       = add_query_arg(
			[
				'v' => MEMBERDASH_VERSION . '-' . $id,
			],
			$url
		);

		// Always limit it to 127 chars.
		return substr( $url, 0, 127 );
	}

	/**
	 * Generates a Unique Hash for signup. It will always be 20 characters long.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	protected function generate_unique_signup_hash(): string {
		$nonce_key  = defined( 'NONCE_KEY' ) ? NONCE_KEY : uniqid( '', true );
		$nonce_salt = defined( 'NONCE_SALT' ) ? NONCE_SALT : uniqid( '', true );

		$unique = uniqid( '', true );

		$keys = [ $nonce_key, $nonce_salt, $unique ];
		$keys = array_map( 'md5', $keys );

		return substr( str_shuffle( implode( '-', $keys ) ), 0, 45 );
	}
}
