<?php
/**
 * Gateway: Paypal Express - API
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

use StellarWP\Memberdash\StellarWP\Arrays\Arr;

/**
 * PayPal Express gateway API class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_Api extends MS_Model_Option {
	/**
	 * Debug ID from PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	protected $debug_id = '';

	/**
	 * Returns the last stored debug ID from PayPal.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_debug_id(): string {
		return $this->debug_id;
	}

	/**
	 * Saves the access token data.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data The data to save.
	 *
	 * @return bool
	 */
	public function save_access_token_data( array $data ): bool {
		if ( empty( $data['access_token'] ) ) {
			return false;
		}

		update_option( 'ms_paypal_express_access_token', $data['access_token'] );

		if ( ! empty( $data['expires_in'] ) ) {
			$expires_in = new DateInterval( 'PT' . $data['expires_in'] . 'S' );

			// Store date related data in readable formats.
			$data['token_retrieval_time']  = ( new DateTime() )->format( 'Y-m-d H:i:s' );
			$data['token_expiration_time'] = ( new DateTime() )->add( $expires_in )->format( 'Y-m-d H:i:s' );
		}

		return update_option( 'ms_paypal_express_access_token_data', $data );
	}

	/**
	 * Fetches an access token from the PayPal API using the client credentials.
	 *
	 * @since 1.5.0
	 *
	 * @param string $client_id     The client ID to use.
	 * @param string $client_secret The client secret to use.
	 * @param bool   $is_sandbox    Whether to use the sandbox or not. Defaults to false.
	 *
	 * @return array<string,mixed>
	 */
	public function get_access_token_from_client_credentials(
		string $client_id,
		string $client_secret,
		bool $is_sandbox = false
	): array {
		$auth       = base64_encode( "$client_id:$client_secret" ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- We need to encode the client ID and secret.
		$query_args = [];

		$args = [
			'headers' => [
				'Authorization' => sprintf( 'Basic %s', $auth ),
				'Content-Type'  => 'application/x-www-form-urlencoded',
			],
			'body'    => [
				'grant_type' => 'client_credentials',
			],
		];

		$response = $this->client_post( 'v1/oauth2/token', $query_args, $args, $is_sandbox );

		if ( is_wp_error( $response ) ) {
			return [];
		}

		return $response;
	}

	/**
	 * Returns the access token data.
	 *
	 * @since 1.5.0
	 *
	 * @return bool
	 */
	public function delete_access_token_data(): bool {
		$token_data   = delete_option( 'ms_paypal_express_access_token_data' );
		$access_token = delete_option( 'ms_paypal_express_access_token' );

		return $token_data && $access_token;
	}

	/**
	 * Returns the access token.
	 *
	 * @since 1.5.0
	 *
	 * @return string
	 */
	public function get_access_token(): string {
		return MS_Helper_Cast::to_string(
			get_option( 'ms_paypal_express_access_token', '' )
		);
	}

	/**
	 * Stores the debug ID from a given PayPal request, which allows for us to store it with the gateway payload.
	 *
	 * @since 1.5.0
	 *
	 * @param string $debug_id The debug header to store.
	 *
	 * @return void
	 */
	protected function set_debug_id( string $debug_id ): void {
		$this->debug_id = $debug_id;
	}

	/**
	 * Get the PayPal API URL.
	 *
	 * @since 1.5.0
	 *
	 * @param bool $is_sandbox Whether to use the sandbox API URL. Defaults to false.
	 *
	 * @return string
	 */
	protected function get_environment_url( bool $is_sandbox = false ): string {
		if ( $is_sandbox ) {
			return 'https://api.sandbox.paypal.com';
		}

		return 'https://api.paypal.com';
	}

	/**
	 * Get the API URL.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint   The endpoint to connect to.
	 * @param bool                $is_sandbox Whether to use the sandbox API URL. Defaults to false.
	 * @param array<string,mixed> $query_args The query arguments to pass to the endpoint.
	 *
	 * @return string
	 */
	protected function get_api_url(
		string $endpoint,
		bool $is_sandbox = false,
		array $query_args = []
	): string {
		$base_url = $this->get_environment_url( $is_sandbox );
		$endpoint = ltrim( $endpoint, '/' );

		return add_query_arg( $query_args, trailingslashit( $base_url ) . $endpoint );
	}

	/**
	 * Builds a request URL.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $url        The URL to connect to.
	 * @param bool                $is_sandbox Whether to use the sandbox API URL. Defaults to false.
	 * @param array<string,mixed> $query_args The query arguments to pass to the endpoint.
	 *
	 * @return string
	 */
	protected function build_request_url(
		string $url,
		bool $is_sandbox = false,
		array $query_args = []
	): string {
		return 0 !== strpos( $url, 'https://' )
			? $this->get_api_url( $url, $is_sandbox, $query_args )
			: add_query_arg( $query_args, $url );
	}

	/**
	 * Builds request arguments.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $method The method to use for the request.
	 * @param array<string,mixed> $args   The arguments to pass to the endpoint.
	 *
	 * @return array<string,mixed>
	 */
	protected function build_request_args( string $method, array $args ): array {
		$default_args = [
			'headers' => [
				'Accept'        => 'application/json',
				'Authorization' => sprintf( 'Bearer %s', $this->get_access_token() ),
				'Content-Type'  => 'application/json',
			],
		];

		if ( 'GET' !== $method ) {
			$default_args['body'] = [];
		}

		$args = Arr::merge_recursive(
			$default_args,
			$args
		);

		if ( 'GET' !== $method ) {
			$content_type = MS_Helper_Cast::to_string(
				Arr::get( $args, 'headers.Content-Type', '' )
			);

			$body = Arr::get( $args, 'body', [] );

			if (
				! empty( $body )
				&& 'application/json' === strtolower( $content_type )
			) {
				$args['body'] = is_string( $body )
					? $body
					: wp_json_encode( $body );
			}
		}

		return $args;
	}

	/**
	 * Performs an API request to the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $method             The method to use for the request.
	 * @param string              $url                The URL to connect to.
	 * @param array<string,mixed> $query_args         The query arguments to pass to the endpoint.
	 * @param array<string,mixed> $request_arguments  The request arguments to pass to the endpoint.
	 * @param bool                $is_sandbox         Whether to use the sandbox API URL. Defaults to false.
	 * @param int                 $retries            The number of retries to attempt.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	protected function client_request(
		string $method,
		string $url,
		array $query_args = [],
		array $request_arguments = [],
		bool $is_sandbox = false,
		int $retries = 0
	) {
		$method            = strtoupper( $method );
		$url               = $this->build_request_url( $url, $is_sandbox, $query_args );
		$request_arguments = $this->build_request_args( $method, $request_arguments );

		if ( 'GET' === $method ) {
			// @phpstan-ignore-next-line -- We are using the correct array structure.
			$response = wp_remote_get( $url, $request_arguments );
		} elseif ( 'POST' === $method ) {
			// @phpstan-ignore-next-line -- We are using the correct array structure.
			$response = wp_remote_post( $url, $request_arguments );
		} else {
			$request_arguments['method'] = $method;

			// @phpstan-ignore-next-line -- We are using the correct array structure.
			$response = wp_remote_request( $url, $request_arguments );
		}

		if ( is_wp_error( $response ) ) {
			MS_Helper_Debug::debug_log(
				sprintf(
					'[%s] PayPal "%s" request error: %s',
					$method,
					$url,
					$response->get_error_message()
				)
			);

			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		// If the debug header was set we pass it or reset it.
		$this->set_debug_id( '' );
		if ( ! empty( $response['headers']['Paypal-Debug-Id'] ) ) {
			$this->set_debug_id( $response['headers']['Paypal-Debug-Id'] );
		}

		// When we get specifically a 401 and we are not trying to generate a token we try once more.
		if (
			401 === $response_code
			&& 2 >= $retries
			&& false === strpos( $url, 'v1/oauth2/token' )
		) {
			$gateway    = $this->get_gateway();
			$token_data = $this->get_access_token_from_client_credentials(
				$gateway->get_client_id(),
				$gateway->get_client_secret(),
				$is_sandbox
			);
			$saved      = $this->save_access_token_data( $token_data );

			// If we properly saved, just re-try the request.
			if ( $saved ) {
				$arguments = func_get_args(); // phpcs:ignore PHPCompatibility.FunctionUse.ArgumentFunctionsReportCurrentValue.Changed -- Method is converted to uppercase.

				// Set the Authorization header with the new token to avoid multiple retries.
				$arguments = Arr::set(
					$arguments,
					'3.headers.Authorization',
					sprintf(
						'Bearer %s',
						MS_Helper_Cast::to_string( $token_data['access_token'] )
					)
				);

				// Increase the number of retries.
				$arguments = Arr::set(
					$arguments,
					'5',
					$retries + 1
				);

				// @phpstan-ignore-next-line -- We are calling the same method.
				return call_user_func_array( [ $this, 'client_request' ], $arguments );
			}
		}

		$response_body = wp_remote_retrieve_body( $response );
		$response_body = $this->json_decode( $response_body );
		if ( empty( $response_body ) ) {
			return $response;
		}

		if ( ! is_array( $response_body ) ) {
			MS_Helper_Debug::debug_log(
				sprintf(
					'[%s] Unexpected PayPal %s response',
					$url,
					$method
				)
			);

			return new WP_Error(
				'ms-gateway-paypalexpress-api-unexpected-response',
				'',
				[
					'method'            => $method,
					'url'               => $url,
					'query_args'        => $query_args,
					'request_arguments' => $request_arguments,
					'response'          => $response,
				]
			);
		}

		return $response_body;
	}

	/**
	 * Performs a GET request to the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint           The endpoint to connect to.
	 * @param array<string,mixed> $query_args         The query arguments to pass to the endpoint.
	 * @param array<string,mixed> $request_arguments  The request arguments to pass to the endpoint.
	 * @param bool                $is_sandbox         Whether to use the sandbox API URL. Defaults to false.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	protected function client_get(
		string $endpoint,
		array $query_args = [],
		array $request_arguments = [],
		bool $is_sandbox = false
	) {
		return $this->client_request( 'GET', $endpoint, $query_args, $request_arguments, $is_sandbox );
	}

	/**
	 * Performs a POST request to the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint           The endpoint to connect to.
	 * @param array<string,mixed> $query_args         The query arguments to pass to the endpoint.
	 * @param array<string,mixed> $request_arguments  The request arguments to pass to the endpoint.
	 * @param bool                $is_sandbox         Whether to use the sandbox API URL. Defaults to false.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	protected function client_post(
		string $endpoint,
		array $query_args = [],
		array $request_arguments = [],
		bool $is_sandbox = false
	) {
		return $this->client_request( 'POST', $endpoint, $query_args, $request_arguments, $is_sandbox );
	}

	/**
	 * Performs a PATCH request to the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint           The endpoint to connect to.
	 * @param array<string,mixed> $query_args         The query arguments to pass to the endpoint.
	 * @param array<string,mixed> $request_arguments  The request arguments to pass to the endpoint.
	 * @param bool                $is_sandbox         Whether to use the sandbox API URL. Defaults to false.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	protected function client_patch(
		string $endpoint,
		array $query_args = [],
		array $request_arguments = [],
		bool $is_sandbox = false
	) {
		return $this->client_request( 'PATCH', $endpoint, $query_args, $request_arguments, $is_sandbox );
	}

	/**
	 * Performs a DELETE request to the PayPal API.
	 *
	 * @since 1.5.0
	 *
	 * @param string              $endpoint           The endpoint to connect to.
	 * @param array<string,mixed> $query_args         The query arguments to pass to the endpoint.
	 * @param array<string,mixed> $request_arguments  The request arguments to pass to the endpoint.
	 * @param bool                $is_sandbox         Whether to use the sandbox API URL. Defaults to false.
	 *
	 * @return array<string,mixed>|WP_Error
	 */
	protected function client_delete(
		string $endpoint,
		array $query_args = [],
		array $request_arguments = [],
		bool $is_sandbox = false
	) {
		return $this->client_request( 'DELETE', $endpoint, $query_args, $request_arguments, $is_sandbox );
	}

	/**
	 * Decodes a JSON string into an array.
	 *
	 * If the JSON string is invalid, an empty array is returned.
	 *
	 * @since 1.5.0
	 *
	 * @param string $json The JSON string to decode.
	 *
	 * @return array<string,mixed>
	 */
	protected function json_decode( string $json ): array {
		if ( empty( $json ) ) {
			return [];
		}

		try {
			$data = json_decode( $json, true, 512, JSON_THROW_ON_ERROR );

			return Arr::wrap( $data );
		} catch ( JsonException $e ) {
			MS_Helper_Debug::debug_log( 'Error decoding JSON: ' . $e->getMessage() );

			return [];
		}
	}

	/**
	 * Returns the gateway instance.
	 *
	 * @since 1.5.0
	 *
	 * @return MS_Gateway_Paypalexpress
	 */
	protected function get_gateway(): MS_Gateway_Paypalexpress {
		return MS_Factory::load( 'MS_Gateway_Paypalexpress' );
	}

	/**
	 * Generates a string from request data.
	 *
	 * @since 1.5.0
	 *
	 * @param array<string,mixed> $data The data to generate the string from.
	 * @param string              $separator The separator to use. Defaults to '&'.
	 *
	 * @return string
	 */
	protected function request_data_to_string( array $data, string $separator = '&' ): string {
		return implode(
			$separator,
			array_map(
				function ( $value ) {
					if ( is_array( $value ) ) {
						return $this->request_data_to_string( $value, '~' );
					}

					return $value;
				},
				$data
			)
		);
	}
}
