<?php
/**
 * Gateway: Paypal Express - Subscription Helper
 *
 * @since 1.5.0
 *
 * @package MemberDash
 */

/**
 * PayPal Express gateway Subscription helper class.
 *
 * @since 1.5.0
 */
class MS_Gateway_Paypalexpress_Helpers_Subscription extends MS_Helper {
	/**
	 * Returns the meta key for the PayPal product ID.
	 *
	 * @since 1.5.0
	 *
	 * @param bool $is_sandbox Whether the gateway is in sandbox mode.
	 *
	 * @return string
	 */
	protected static function get_membership_product_id_key( bool $is_sandbox ): string {
		return $is_sandbox
			? 'ms_gateway_paypal_sandbox_product_id'
			: 'ms_gateway_paypal_live_product_id';
	}

	/**
	 * Returns the meta key for the PayPal plan ID.
	 *
	 * @since 1.5.0
	 *
	 * @param bool $is_sandbox Whether the gateway is in sandbox mode.
	 *
	 * @return string
	 */
	protected static function get_membership_plan_id_key( bool $is_sandbox ): string {
		return $is_sandbox
			? 'ms_gateway_paypal_sandbox_plan_id'
			: 'ms_gateway_paypal_live_plan_id';
	}

	/**
	 * Returns the PayPal product ID of a membership.
	 *
	 * @since 1.5.0
	 *
	 * @param MS_Model_Membership      $membership The membership.
	 * @param MS_Gateway_Paypalexpress $gateway    The gateway.
	 *
	 * @return string
	 */
	public static function get_product_id(
		MS_Model_Membership $membership,
		MS_Gateway_Paypalexpress $gateway
	): string {
		$product_id = '';

		if ( ! $membership->supports_recurring_payments() ) {
			return $product_id;
		}

		$product_id = MS_Helper_Cast::to_string(
			get_post_meta(
				$membership->get_id(),
				self::get_membership_product_id_key(
					$gateway->is_sandbox()
				),
				true
			)
		);

		if ( ! empty( $product_id ) ) {
			return $product_id;
		}

		$product = $gateway->get_api()->create_product(
			[
				'name'        => $membership->get_name(),
				'description' => $membership->get_description(),
			],
			$gateway->is_sandbox()
		);

		if ( empty( $product['id'] ) ) {
			return $product_id;
		}

		$product_id = MS_Helper_Cast::to_string( $product['id'] );

		update_post_meta(
			$membership->get_id(),
			self::get_membership_product_id_key(
				$gateway->is_sandbox()
			),
			$product_id
		);

		return $product_id;
	}

	/**
	 * Returns the PayPal plan ID of a membership.
	 *
	 * @since 1.5.0
	 *
	 * @param MS_Model_Membership      $membership The membership.
	 * @param MS_Gateway_Paypalexpress $gateway    The gateway.
	 * @param MS_Model_Invoice         $invoice    The invoice.
	 *
	 * @return string
	 */
	public static function get_plan_id(
		MS_Model_Membership $membership,
		MS_Gateway_Paypalexpress $gateway,
		MS_Model_Invoice $invoice
	): string {
		$plan_id = '';

		if ( ! $membership->supports_recurring_payments() ) {
			return $plan_id;
		}

		$key = self::get_membership_plan_id_key(
			$gateway->is_sandbox()
		);

		// Check if the invoice has a discount or pro rating amount.
		$has_discount = $invoice->get_discount() > 0
			|| $invoice->get_pro_rate() > 0;

		// PayPal makes it hard to update existing plans, limiting the number of cycles that you may edit based on the first plan created.
		// So we add a suffix to the plan ID to differentiate between different invoices.
		if ( $has_discount ) {
			$key .= '_' . $invoice->get_id();
		}

		$is_first_invoice = true;

		// If already paid, it means we're creating a renewal plan for
		// a customer who already paid the signup fee but switched to a different
		// payment gateway.
		if ( $invoice->is_paid() ) {
			$key .= '_renewal_' . $invoice->get_id();

			$is_first_invoice = false;
		}

		// Fetches the plan ID from the custom data.
		$plan_id = MS_Helper_Cast::to_string(
			get_post_meta(
				$membership->get_id(),
				$key,
				true
			)
		);

		// Stop here if the plan ID is already set.
		if ( ! empty( $plan_id ) ) {
			return $plan_id;
		}

		// Fetches or creates the product ID.
		$product_id = self::get_product_id( $membership, $gateway );

		if ( empty( $product_id ) ) {
			return $plan_id;
		}

		// Create a new plan.
		$billing_cycles = [];
		$sequence       = 1;
		$currency       = MS_Plugin::get_settings()->get_currency();

		$first_cycle_is_free = $has_discount && $invoice->get_calculated_total() === 0.0;

		// Add the trial cycle if the membership has a trial or if the first cycle is free.
		if (
			$membership->has_trial()
			|| $first_cycle_is_free
		) {
			$billing_cycles[] = [
				'frequency'    => [
					'interval_unit'  => rtrim( strtoupper( $membership->trial_period_type ), 'S' ),
					'interval_count' => $membership->trial_period_unit,
				],
				'tenure_type'  => 'TRIAL',
				'sequence'     => $sequence++,
				'total_cycles' => $membership->has_trial() && $first_cycle_is_free
					? 2 // Duplicate the trial cycle if the first cycle is free.
					: 1,
			];
		}

		// Add the first regular cycle if the membership has a discount..
		if (
			$has_discount
			&& ! $first_cycle_is_free
		) {
			$billing_cycles[] = [
				'frequency'      => [
					'interval_unit'  => rtrim( strtoupper( $membership->pay_cycle_period_type ), 'S' ),
					'interval_count' => $membership->pay_cycle_period_unit,
				],
				'tenure_type'    => 'TRIAL',
				'sequence'       => $sequence++,
				'total_cycles'   => 1,
				'pricing_scheme' => [
					'fixed_price' => [
						'value'         => MS_Helper_Cast::to_string( $invoice->get_calculated_total() ),
						'currency_code' => $currency,
					],
				],
			];
		}

		$repetitions = $membership->get_pay_cycle_repetitions();

		$fixed_price = $invoice->get_coupon_id() > 0
			&& $invoice->get_coupon_duration() === MS_Addon_Coupon_Model::DURATION_ALWAYS
				? round( $membership->get_price() - $invoice->get_discount(), 2 )
				: $membership->get_price();

		// Remaining cycles.
		$billing_cycles[] = [
			'frequency'      => [
				'interval_unit'  => rtrim( strtoupper( $membership->pay_cycle_period_type ), 'S' ),
				'interval_count' => $membership->pay_cycle_period_unit,
			],
			'tenure_type'    => 'REGULAR',
			'sequence'       => $sequence++,
			'total_cycles'   => $has_discount && $repetitions >= 2
				? $repetitions - 1 // If the membership has a discount and the repetitions are greater than 1, subtract 1 from the repetitions.
				: $repetitions,
			'pricing_scheme' => [
				'fixed_price' => [
					'value'         => MS_Helper_Cast::to_string( $fixed_price ),
					'currency_code' => $currency,
				],
			],
		];

		$plan = $gateway->get_api()->create_plan(
			[
				'product_id'     => $product_id,
				'name'           => $membership->get_name(),
				'description'    => $membership->get_payment_type_desc(),
				'billing_cycles' => $billing_cycles,
				'setup_fee'      => MS_Helper_Cast::to_string(
					$is_first_invoice
						? $membership->get_signup_fee()
						: 0
				),
				'currency_code'  => $currency,
			],
			$gateway->is_sandbox()
		);

		if ( empty( $plan['id'] ) ) {
			return $plan_id;
		}

		$plan_id = MS_Helper_Cast::to_string( $plan['id'] );

		// Store the plan ID.
		update_post_meta(
			$membership->get_id(),
			$key,
			$plan_id
		);

		return $plan_id;
	}

	/**
	 * Returns a MemberDash subscription from a PayPal subscription ID.
	 *
	 * @since 1.5.0
	 *
	 * @param string $paypal_subscription_id The PayPal subscription ID.
	 * @param string $status                 The MemberDash subscription status. Default 'valid'.
	 *
	 * @return MS_Model_Relationship|null
	 */
	public static function get_ms_subscription( string $paypal_subscription_id, string $status = 'valid' ) {
		$ids = MS_Model_Relationship::get_subscription_ids(
			[
				'status'     => $status,
				'meta_query' => [
					[
						'key'     => 'ms_gateway_paypal_subscription_id',
						'value'   => $paypal_subscription_id,
						'compare' => '=',
					],
				],
			]
		);

		if ( ! empty( $ids[0] ) ) {
			return MS_Factory::load(
				'MS_Model_Relationship',
				$ids[0]
			);
		}

		return null;
	}
}
