<?php
/**
 * Membership Helper List Table.
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Memberships Helper List Table class.
 *
 * @since 1.0.0
 */
class MS_Helper_ListTable_Membership extends MS_Helper_ListTable {

	protected $id = 'membership';

	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'membership',
				'plural'   => 'memberships',
				'ajax'     => false,
			)
		);
	}

	public function get_columns() {
		$columns = array(
			'cb'               => '<input type="checkbox" />',
			'priority'         => sprintf(
				'<span title="%s">#</span>',
				__( 'Membership Order', 'memberdash' )
			),
			'name'             => __( 'Membership Name', 'memberdash' ),
			'active'           => __( 'Active', 'memberdash' ),
			'type_description' => __( 'Type of Membership', 'memberdash' ),
			'members'          => __( 'Members', 'memberdash' ),
			'price'            => __( 'Payment', 'memberdash' ),
			'shortcode'        => __( 'Shortcodes', 'memberdash' ),
		);

		return apply_filters(
			'membership_helper_listtable_membership_columns',
			$columns
		);
	}

	public function column_cb( $item ) {
		return sprintf(
			'<input class="del_membership_ids" type="checkbox" name="memberships[]" value="%s" />',
			$item->id
		);
	}

	public function get_hidden_columns() {
		return apply_filters(
			'membership_helper_listtable_membership_hidden_columns',
			array()
		);
	}

	public function get_sortable_columns() {
		return apply_filters(
			'membership_helper_listtable_membership_sortable_columns',
			array(
				'priority'         => array( 'menu_order', true ),
				'name'             => array( 'name', true ),
				'type_description' => array( 'type', true ),
				'active'           => array( 'active', true ),
			)
		);
	}

	public function prepare_items() {
		$this->_column_headers = array(
			$this->get_columns(),
			$this->get_hidden_columns(),
			$this->get_sortable_columns(),
		);

		$args = array();

		if ( ! empty( $_REQUEST['orderby'] ) && ! empty( $_REQUEST['order'] ) ) {
			$args['orderby'] = $_REQUEST['orderby'];
			$args['order']   = $_REQUEST['order'];
		}

		// Prepare order by statement.
		if ( ! empty( $args['orderby'] )
			&& property_exists( 'MS_Model_Membership', $args['orderby'] )
		) {
			$args['meta_key'] = $args['orderby'];
			$args['orderby']  = 'meta_value';
		}

		$this->items = apply_filters(
			'membership_helper_listtable_membership_items',
			MS_Model_Membership::get_memberships( $args )
		);
	}

	public function column_priority( $item ) {
		$result = '-';

		if ( ! $item->is_system() ) {
			$result = $item->priority;
		}

		return $result;
	}

	/**
	 * Generates the HTML for the "Name" column in the membership listing table.
	 *
	 * @since 1.0.0
	 *
	 * @param MS_Model_Membership $item The MS_Model_Relationship object.
	 *
	 * @return string The generated HTML for the "Name" column with actions and the membership name.
	 */
	public function column_name( $item ) {
		$actions = [];

		$membership_name = $item->name;

		$membership_page_slug = MS_Controller_Plugin::MENU_SLUG;

		// Prepare the Membership actions.
		$actions['edit'] = sprintf(
			'<a href="?page=%1$s&step=%2$s&tab=%3$s&membership_id=%4$s">%5$s</a>',
			$membership_page_slug,
			MS_Controller_Membership::STEP_EDIT,
			MS_Controller_Membership::TAB_DETAILS,
			esc_attr( MS_Helper_Cast::to_string( $item->id ) ),
			__( 'Edit', 'memberdash' )
		);

		if ( ! $item->is_system() ) {
			$actions['payment'] = sprintf(
				'<a href="?page=%1$s&step=%2$s&tab=%3$s&membership_id=%4$s">%5$s</a>',
				$membership_page_slug,
				MS_Controller_Membership::STEP_EDIT,
				MS_Controller_Membership::TAB_PAYMENT,
				esc_attr( MS_Helper_Cast::to_string( $item->id ) ),
				$item->is_free ? __( 'Access options', 'memberdash' ) : __( 'Payment options', 'memberdash' )
			);
		}

		$actions['delete'] = sprintf(
			'<span class="delete"><a href="%s">%s</a></span>',
			wp_nonce_url(
				sprintf(
					'?page=%1$s&membership_id=%2$s&action=%3$s',
					$membership_page_slug,
					esc_attr( MS_Helper_Cast::to_string( $item->id ) ),
					'delete'
				),
				'delete'
			),
			__( 'Delete', 'memberdash' )
		);

		/**
		 * We need to verify whether the membership is configured properly.
		 */
		if ( $item->form_requires_completion() ) {
			$complete_setup_url = admin_url(
				sprintf(
					'admin.php?page=%1$s&step=%2$s&membership_id=%3$s&need_complete=1',
					$membership_page_slug,
					MS_Controller_Membership::STEP_PAYMENT,
					esc_attr( MS_Helper_Cast::to_string( $item->id ) )
				)
			);

			// If the membership is controlled by an external shop cart, the edit link should point to the payment tab.
			if ( $item->has_external_cart_support() ) {
				$complete_setup_url = admin_url(
					sprintf(
						'admin.php?page=%1$s&step=%2$s&tab=%3$s&membership_id=%4$s',
						$membership_page_slug,
						MS_Controller_Membership::STEP_EDIT,
						MS_Controller_Membership::TAB_PAYMENT,
						esc_attr( MS_Helper_Cast::to_string( $item->id ) )
					)
				);
			}

			// Point the edit link to the last step off.
			$actions['edit'] = sprintf(
				'<a href="%1$s">%2$s</a>',
				esc_url( $complete_setup_url ),
				__( 'Complete Setup', 'memberdash' )
			);

			// we also dont want to include the payment here.
			unset( $actions['payment'] );

			// translators: %s is the membership name.
			$membership_name = sprintf( __( 'Draft: %s', 'memberdash' ), $membership_name );
		}

		$actions = apply_filters(
			'ms_helper_listtable_membership_column_name_actions',
			$actions,
			$item
		);

		// Add the badge to special memberships.
		$badge = apply_filters(
			'ms_helper_listtable_memberships_name_badge',
			'',
			$item,
			$this
		);

		// if ( $item->is_guest() ) {
		// $badge = sprintf(
		// '<span class="ms-badge ms-guest-badge" data-memberdash-tooltip="%2$s" data-width="180">%1$s</span>',
		// __( 'Guest', 'memberdash' ),
		// __( 'All Logged-Out users are considered guests', 'memberdash' )
		// );
		// } elseif ( $item->is_user() ) {
		// $badge = sprintf(
		// '<span class="ms-badge ms-user-badge" data-memberdash-tooltip="%2$s" data-width="180">%1$s</span>',
		// __( 'Default', 'memberdash' ),
		// __( 'All logged-in users that have not signed up for any membership', 'memberdash' )
		// );
		// } else {
		// $badge = apply_filters(
		// 'ms_helper_listtable_memberships_name_badge',
		// '',
		// $item,
		// $this
		// );
		// }

		return sprintf(
			'<span class="ms-color ms-ml-0.5" style="background-color:%4$s">&nbsp;</span> ' .
			'<a href="?page=%5$s&step=%6$s&membership_id=%7$s" class="the-name">%1$s</a> ' .
			'%3$s%2$s',
			esc_html( $membership_name ),
			$this->row_actions( $actions ),
			$badge,
			$item->get_color(),
			esc_attr( $membership_page_slug ),
			MS_Controller_Membership::STEP_OVERVIEW,
			esc_attr( MS_Helper_Cast::to_string( $item->id ) ),
		);
	}

	public function column_active( $item ) {
		$toggle = array(
			'id'      => 'ms-toggle-' . $item->id,
			'type'    => MS_Helper_Html::INPUT_TYPE_RADIO_SLIDER,
			'value'   => $item->active,
			'class'   => 'ms--mt-0.5',
			'data_ms' => array(
				'action'        => MS_Controller_Membership::AJAX_ACTION_TOGGLE_MEMBERSHIP,
				'field'         => 'active',
				'membership_id' => $item->id,
			),
		);

		$html = MS_Helper_Html::html_element( $toggle, true );

		return $html;
	}

	public function column_members( $item, $column_name ) {
		$html = '';

		if ( ! $item->is_system() ) {
			$count = $item->get_members_count();

			$url = MS_Controller_Plugin::get_admin_url(
				'members',
				array( 'membership_id' => $item->id )
			);

			$html = sprintf(
				'<a href="%2$s">%1$s</a>',
				intval( $count ),
				$url
			);
		}

		return $html;
	}

	public function column_type_description( $item, $column_name ) {
		$html = '';

		// $html .= sprintf(
		// '<span class="ms-img-type ms-img-type-%1$s small"></span> ',
		// esc_attr( $item->type )
		// );

		$desc = $item->type_description;

		if ( ! empty( $desc ) ) {
			$html .= sprintf(
				'<span class="ms-type-desc ms-ml-1 ms-inline-block ms-%1$s">%2$s</span>',
				esc_attr( $item->type ),
				$desc
			);
		}

		if ( ! $item->is_system() && $item->private ) {
			$html .= sprintf(
				'<span class="ms-is-private">, <span>%1$s</span></span>',
				__( 'Private', 'memberdash' )
			);
		}

		return $html;
	}

	/**
	 * Renders the price column.
	 *
	 * @param MS_Model_Membership $item The membership object.
	 *
	 * @return string
	 */
	public function column_price( $item ): string {
		$html = '';

		if ( ! $item->is_system() ) {
			if ( ! $item->is_free() ) {
				$html = sprintf(
					'<span class="ms-currency">%1$s</span> <span class="ms-price">%2$s</span> (<span class="ms-payment">%3$s</span>)',
					MS_Plugin::instance()->settings->currency_symbol,
					MS_Helper_Billing::format_price( $item->get_price() ),
					$item->get_payment_type_desc()
				);
				$html = '<span class="ms-bold">' . $html . '</span>';
			} else {
				$html = sprintf(
					'<span class="ms-bold">%1$s</span> (<span class="ms-payment">%2$s</span>)',
					__( 'Free', 'memberdash' ),
					$item->get_payment_type_desc()
				);
			}

			// Add signup fee description.
			if ( $item->get_signup_fee() > 0 ) {
				$html .= sprintf(
					'<div class="ms-signup-fee">%1$s: <span class="ms-currency ms-bold">%2$s</span> <span class="ms-price ms-bold">%3$s</span></div>',
					__( 'Signup Fee', 'memberdash' ),
					MS_Helper_Billing::get_currency_symbol(),
					MS_Helper_Billing::format_price( $item->get_signup_fee() )
				);
			}

			$followup = MS_Factory::load(
				'MS_Model_Membership',
				$item->get_on_end_membership_id()
			);

			if ( $followup->is_valid() ) {
				$html .= '<div class="ms-followup">' . sprintf(
					__( 'Follow with: %1$s', 'memberdash' ),
					'<span class="ms-color" style="background:' . $followup->get_color() . '">&nbsp;</span>' . $followup->name
				) . '</div>';
			}
		}

		/**
		 * Allows to filter the HTML output of the price column.
		 *
		 * @param string              $html The HTML output.
		 * @param MS_Model_Membership $item The membership object.
		 *
		 * @return string
		 */
		$html = apply_filters(
			'ms_helper_listtable_membership_column_price',
			$html,
			$item
		);

		return '<div class="ms-pl-1.5">' . $html . '</div>';
	}

	public function column_shortcode( $item, $column_name ) {
		$shortcodes = array(
			MS_Rule_Shortcode_Model::PROTECT_CONTENT_SHORTCODE => array(
				'tag'   => '[%1$s id="%2$s"][/%1$s]',
				'label' => __( 'Protect content', 'memberdash' ),
			),
			MS_Helper_Shortcode::SCODE_MS_TITLE      => array(
				'tag'   => '[%1$s id="%2$s"]',
				'label' => __( 'Membership Title', 'memberdash' ),
			),
			MS_Helper_Shortcode::SCODE_MS_DETAILS    => array(
				'tag'   => '[%1$s id="%2$s"]',
				'label' => __( 'Membership Description', 'memberdash' ),
			),
			MS_Helper_Shortcode::SCODE_MS_PRICE      => array(
				'tag'   => '[%1$s id="%2$s"]',
				'label' => __( 'Membership Price', 'memberdash' ),
			),
			MS_Helper_Shortcode::SCODE_MS_BUY        => array(
				'tag'   => '[%1$s id="%2$s"]',
				'label' => __( 'Subscribe Button', 'memberdash' ),
			),
			MS_Helper_Shortcode::SCODE_REGISTER_USER => array(
				'tag'   => '[%1$s membership_id="%2$s"]',
				'label' => __( 'Registration form', 'memberdash' ),
			),
		);
		if ( $item->is_system() ) {
			unset( $shortcodes[ MS_Helper_Shortcode::SCODE_MS_TITLE ] );
			unset( $shortcodes[ MS_Helper_Shortcode::SCODE_MS_DETAILS ] );
			unset( $shortcodes[ MS_Helper_Shortcode::SCODE_MS_PRICE ] );
			unset( $shortcodes[ MS_Helper_Shortcode::SCODE_MS_BUY ] );
			unset( $shortcodes[ MS_Helper_Shortcode::SCODE_REGISTER_USER ] );
		}

		$shortcodes = apply_filters(
			'ms_helper_listtable_membership_shortcodes',
			$shortcodes,
			$item
		);

		$lines = array();
		foreach ( $shortcodes as $code => $details ) {
			$lines[] = sprintf(
				'<div><span class="ms-shortcode-description">%s</span> <span class="ms-code">%s</span></div>',
				$details['label'],
				sprintf(
					$details['tag'],
					$code,
					esc_attr( $item->id )
				)
			);
		}

		$html = sprintf(
			'<div class="ms-shortcodes"><span class="ms-trigger"><span class="label">%s</span><div class="inner">%s</div></span></div>',
			__( 'Show', 'memberdash' ) . ' <i class="memberdash-fa memberdash-fa-caret-down"></i>',
			implode( '', $lines )
		);

		return $html;
	}

	public function get_bulk_actions() {
		return apply_filters(
			'ms_helper_listtable_membership_bulk_actions',
			array()
		);
	}
}
