<?php
/**
 * Membership Custom Post Type Item List Table
 *
 * @since 1.0.0
 *
 * @package MemberDash
 */

/**
 * Membership Custom Post Type Item List Table class.
 *
 * @since 1.0.0
 */
class MS_Rule_CptItem_ListTable extends MS_Helper_ListTable_Rule {

	/**
	 * The rule ID.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $id = MS_Rule_CptItem::RULE_ID;

	/**
	 * Initialize the list table.
	 *
	 * @since 1.0.0
	 *
	 * @param MS_Rule $model The rule model.
	 *
	 * @return void
	 */
	public function __construct( $model ) {
		parent::__construct( $model );
		$this->name['singular'] = __( 'Custom Post', 'memberdash' );
		$this->name['plural']   = __( 'Custom Posts', 'memberdash' );
	}

	/**
	 * Returns a list of columns.
	 *
	 * @since 1.0.0
	 *
	 * @return array<string,bool|string> The list of columns.
	 */
	public function get_columns() {
		return apply_filters(
			"membership_helper_listtable_{$this->id}_columns",
			array(
				'cb'         => true,
				'post_title' => __( 'Custom Post Title', 'memberdash' ),
				'post_type'  => __( 'Post Type', 'memberdash' ),
				'access'     => true,
				'dripped'    => true,
			)
		);
	}

	/**
	 * Returns a list of sortable columns.
	 *
	 * @since 1.0.0
	 *
	 * @return array<string,string> The list of sortable columns.
	 */
	public function get_sortable_columns() {
		return apply_filters(
			"membership_helper_listtable_{$this->id}_sortable_columns",
			array(
				'post_title' => 'post_title',
				'post_type'  => 'post_type',
				'access'     => 'access',
			)
		);
	}

	/**
	 * Returns the post title column content.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Post $item The current post object.
	 *
	 * @return string The post title column content.
	 */
	public function column_post_title( $item ) {
		$actions = array(
			sprintf(
				'<a href="%s">%s</a>',
				get_edit_post_link( $item->ID, true ),
				__( 'Edit', 'memberdash' )
			),
			sprintf(
				'<a href="%s">%s</a>',
				get_permalink( $item->ID ),
				__( 'View', 'memberdash' )
			),
		);

		$actions = apply_filters(
			'ms_rule_' . $this->id . '_column_actions',
			$actions,
			$item
		);

		return sprintf(
			'%1$s %2$s',
			$item->post_title,
			$this->row_actions( $actions )
		);
	}

	/**
	 * Returns the post type column content.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Post $item        The current post object.
	 * @param string  $column_name The current column name.
	 *
	 * @return string The post type column content.
	 */
	public function column_post_type( $item, $column_name ) {
		return $item->post_type;
	}

	/**
	 * Displayed above the views.
	 *
	 * In the rule list-tables the list-head is used to display a filter for
	 * membership-ID. Combined with the views (below) users can filter all rules
	 * by membership + protection status independently
	 *
	 * @since 1.2.0
	 *
	 * @return void Echoes the HTML.
	 */
	public function list_head() {
		$membership_id = ! empty( $_GET['membership_id'] )
			? absint( wp_unslash( $_GET['membership_id'] ) )
			: 0;

		$status = ! empty( $_GET['status'] )
			? sanitize_key( wp_unslash( $_GET['status'] ) )
			: '';

		?>
		<form id="ms-protection-cpt-filter" method="GET">
			<?php
			MS_Helper_Html::html_element(
				[
					'id'            => 'cpt',
					'class'         => 'item-post-type',
					'type'          => MS_Helper_Html::INPUT_TYPE_SELECT,
					'title'         => __( 'Showing:', 'memberdash' ),
					'field_options' => $this->get_public_post_type_list(),
					'value'         => ! empty( $_GET['cpt'] ) ? sanitize_key( wp_unslash( $_GET['cpt'] ) ) : 'all',
				]
			);

			echo wp_kses(
				$this->get_view_suffix( $membership_id, $status ),
				[
					'span' => [
						'class' => [],
						'style' => [],
					],
					'b'    => [],
				]
			);
			?>
		</form>
		<?php
	}

	/**
	 * Returns a list of public post types.
	 *
	 * @since 1.2.0
	 *
	 * @return array<int|string,string> The list of public post types.
	 */
	protected function get_public_post_type_list(): array {
		$post_types = [];

		// Get custom post types.
		foreach ( MS_Rule_CptGroup_Model::get_custom_post_types() as $cpt ) {
			// Skip post types with no items.
			if ( $this->get_total_items( $cpt ) <= 0 ) {
				continue;
			}

			// Get the post type object.
			$post_type = get_post_type_object( $cpt );
			if ( ! $post_type ) {
				continue;
			}

			// Add the post type to the list.
			$post_types[ $post_type->name ] = sprintf(
				/* translators: %s: Post type label */
				__( 'All %s', 'memberdash' ),
				$post_type->label
			);
		}

		// Set a default option to display all post types.
		$all = [
			'all' => sprintf(
				/* translators: %s: Post type label */
				__( 'All %s', 'memberdash' ),
				$this->name['plural']
			),
		];

		// Add the default option to the list of post types and return it.
		return array_merge( $all, $post_types );
	}

	/**
	 * Returns a suffix for the view based in the membership ID and status.
	 *
	 * @since 1.2.0
	 *
	 * @param int    $membership_id The membership ID.
	 * @param string $status        The status.
	 *
	 * @return string The suffix for the view.
	 */
	protected function get_view_suffix( $membership_id, $status ): string {
		$suffix = '';

		if ( empty( $membership_id ) ) {
			if ( MS_Model_Rule::FILTER_NOT_PROTECTED === $status ) {
				$suffix = __( 'that are <b>not protected</b>', 'memberdash' );
			} elseif ( MS_Model_Rule::FILTER_PROTECTED === $status ) {
				$suffix = __( 'that are <b>protected</b>', 'memberdash' );
			}
		} else {
			$membership = MS_Factory::load( 'MS_Model_Membership', $membership_id );

			if ( empty( $status ) ) {
				/* translators: %s: membership name */
				$suffix = __( 'for %s', 'memberdash' );
			} elseif ( MS_Model_Rule::FILTER_NOT_PROTECTED === $status ) {
				/* translators: %s: membership name */
				$suffix = __( 'that are <b>not protected</b> by %s', 'memberdash' );
			} elseif ( MS_Model_Rule::FILTER_PROTECTED === $status ) {
				/* translators: %s: membership name */
				$suffix = __( 'that are <b>protected</b> by %s', 'memberdash' );
			}

			$membership_name  = $membership->get_name();
			$membership_color = $membership->get_color();

			$suffix = sprintf(
				$suffix,
				sprintf(
					'<span class="ms-membership" style="background-color:%1$s">%2$s</span>',
					esc_attr( $membership_color ),
					esc_html( $membership_name )
				)
			);
		}

		// Wrap the suffix in a span.
		if ( ! empty( $suffix ) ) {
			$suffix = sprintf(
				'<span class="ms-view-protection-cpt">%s</span>',
				$suffix
			);
		}

		return $suffix;
	}

	/**
	 * Returns the total number of items in a post type.
	 *
	 * @since 1.2.0
	 *
	 * @param string $post_type The post type.
	 *
	 * @return int The total number of items in the post type.
	 */
	protected function get_total_items( string $post_type ): int {
		$values = (array) wp_count_posts( $post_type );
		$total  = 0;

		// Exclude auto-drafts and trashed posts.
		$exclude_statuses = [
			'auto-draft',
			'trash',
		];

		foreach ( $values as $status => $num_posts ) {
			if ( in_array( $status, $exclude_statuses, true ) ) {
				continue;
			}

			$total += $num_posts;
		}

		return $total;
	}
}
