<?php
/**
 * Class MEC_FES_Builder Core
 *
 * @since           1.0.0
 */

namespace MEC_FES_Builder;

defined( 'ABSPATH' ) || exit;

/**
 * Core Class in plugin
 */
final class Base {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public static $version = '1.0.0';

	/**
	 * Session instance
	 *
	 * @since 1.0.0
	 *
	 * @var bool
	 */
	protected static $instance;

	/**
	 * @since 1.0.0
	 *
	 * Constructor
	 */
	public function __construct() {

		$this->define();
		$this->includes();
		$this->init_hooks();
		$this->admin();
		$this->enqueue_scripts();
	}

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @return self()
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Set Constants
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function define() {

		define( 'MEC_FES_BUILDER_PD', plugin_dir_path( MEC_FES_BUILDER_FILE ) );
		define( 'MEC_FES_BUILDER_PDI', plugin_dir_path( MEC_FES_BUILDER_FILE ) . 'src/' );
		define( 'MEC_FES_BUILDER_PU_JS', plugins_url( 'assets/js/', MEC_FES_BUILDER_FILE ) );
		define( 'MEC_FES_BUILDER_PU_CSS', plugins_url( 'assets/css/', MEC_FES_BUILDER_FILE ) );
		define( 'MEC_FES_BUILDER_PU_IMG', plugins_url( 'assets/img/', MEC_FES_BUILDER_FILE ) );
		define( 'MEC_FES_BUILDER_PU_FONTS', plugins_url( 'assets/fonts/', MEC_FES_BUILDER_FILE ) );
		define( 'MEC_FES_BUILDER_TEMPLATES', plugin_dir_path( MEC_FES_BUILDER_FILE ) . 'templates/' );

		define( 'MEC_FES_BUILDER_VERSION', self::$version );
	}

	/**
	 * Include Files
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function includes() {

		require_once MEC_FES_BUILDER_PDI . 'functions.php';
	}

	/**
	 * Include Files If is Admin
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function admin() {

		if ( !is_admin() ) {

			return;
		}

		Admin::getInstance()->init();
	}

	/**
	 * @since 1.0.0
	 *
	 * Load Plugin Text Domain
	 */
	public function load_plugin_textdomain() {

		load_plugin_textdomain(
			MECFBTEXTDOMAIN,
			false,
			dirname( plugin_basename( MEC_FES_BUILDER_FILE ) ) . '/languages'
		);
	}

	/**
	 * Register actions enqueue scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

	}

	/**
	 * Add Hooks - Actions and Filters
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init_hooks() {

		add_action( 'init', array( $this, 'init' ) );

		if( CheckPlugins::check_plugins() ){

			Elementor::getInstance()->init();

			add_action( 'init', array( __CLASS__, 'register_post_type' ) );

			add_filter( 'mec_fes_form_template_path', array( __CLASS__, 'filter_fes_form_template_path' ) );

			add_filter( 'mec_fes_form_is_required_fields', array( __CLASS__, 'validate_fes_form' ), 10, 2 );
		}

		do_action( 'mec_fes_builder_init' );
	}

	/**
	 * Init
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init() {

		$this->load_plugin_textdomain();
	}

	/**
	 * Register post type
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function register_post_type(){


		$labels = array(
			'name'                => _x('Events FES Builder', 'Post Type General Name', 'mec-single-builder'),
			'singular_name'       => _x('Events FES Page', 'Post Type Singular Name', 'mec-single-builder'),
			'menu_name'           => __('Events FES Builder', 'mec-single-builder'),
			'parent_item_colon'   => __('Parent FES Builder', 'mec-single-builder'),
			'all_items'           => __('Events FES Builder', 'mec-single-builder'),
			'view_item'           => __('View FES Builder', 'mec-single-builder'),
			'add_new_item'        => __('Add New FES Builder', 'mec-single-builder'),
			'add_new'             => __('Add New', 'mec-single-builder'),
			'edit_item'           => __('Edit FES Builder', 'mec-single-builder'),
			'update_item'         => __('Update FES Builder', 'mec-single-builder'),
			'search_items'        => __('Search FES Builder', 'mec-single-builder'),
			'not_found'           => __('Not Found', 'mec-single-builder'),
			'not_found_in_trash'  => __('Not found in Trash', 'mec-single-builder'),
		);

		$args = array(
			'label'               => __('Events FES Page Builder', 'mec-single-builder'),
			'description'         => __('FES Builder news and reviews', 'mec-single-builder'),
			'public'             => current_user_can( 'edit_posts' ) ? true : false,
			'publicly_queryable' => true,
			'show_ui'            => true,
			'show_in_menu'       => false,
			'query_var'          => true,
			'has_archive'        => false,
			'hierarchical'       => false,
			'menu_position'      => 49,
			'supports'           => ['title', 'editor', 'elementor'],
			'labels'              => $labels,
			'exclude_from_search' => true
		);

		register_post_type( 'mec-fes-builder', $args );
	}

	/**
	 * Return fes form template id
	 *
	 * @since 1.0.0
	 *
	 * @return int|false
	 */
	public static function get_fes_form_template_id(){

		$fes_template_id = \MEC\Settings\Settings::getInstance()->get_settings('fes_template_id');

		return $fes_template_id;
	}

	/**
	 * Filter FES form template path
	 *
	 * @param string $path
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function filter_fes_form_template_path( $path ){

		if( static::get_fes_form_template_id() ){

			$path = MEC_FES_BUILDER_TEMPLATES . 'fes-form.php';
		}

		return $path;
	}

	/**
	 * Return widgets
	 *
	 * @param array $elements_data
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public static function get_widgets( $elements_data ){

		$widgets = array();
		foreach( $elements_data as $data ){

			if( isset( $data['elements'] ) && !empty( $data['elements'] ) ){

				$widgets = array_merge(
					static::get_widgets( $data['elements'] ),
					$widgets
				);
			}elseif( 'widget' == $data['elType'] ) {

				$widgets[ $data['id'] ] = $data;
			}
		}

		return $widgets;
	}

	/**
	 * Return FES form widgets
	 *
	 * @param int $post_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public static function get_fes_form_widgets( $post_id ){

		$document = \Elementor\Plugin::instance()->documents->get( $post_id, false );
		$elements_data = $document->get_elements_raw_data( null, false );

		return static::get_widgets( $elements_data );
	}

	/**
	 * Return fes form widget settings
	 *
	 * @param int $post_id
	 * @param string $widget_type
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public static function get_fes_form_widget_settings( $post_id, $widget_type ){

		$widgets_data = static::get_fes_form_widgets( $post_id );

		foreach( $widgets_data as $widget_data ){

			if( $widget_type == $widget_data['widgetType'] ){

				return $widget_data['settings'] ?? array();
			}
		}

		return array();
	}

	/**
	 * Validate fes form fields
	 *
	 * @param bool $is_required
	 * @param string $field_type
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public static function validate_fes_form( $is_required, $field_type ){

		$custom_form_id = absint( $_POST['mec-fes-template-id'] ?? 0 );

		if( $custom_form_id ){

			$required_key = 'required';
			switch( $field_type ){
				case 'description':
				case 'content':

					$widget_type = 'mec-fes-description';
					break;
				case 'excerpt':

					$widget_type = 'mec-fes-excerpt';
					break;
				case 'category':

					$widget_type = 'mec-fes-categories';
					break;
				case 'label':

					$widget_type = 'mec-fes-labels';
					break;
				case 'featured_image':

					$widget_type = 'mec-fes-featured-image';
					break;
				case 'cost':

					$widget_type = 'mec-fes-cost';
					break;
				case 'event_link':

					$widget_type = 'mec-fes-links';
					$required_key = 'event_link_required';
					break;
				case 'more_info':

					$widget_type = 'mec-fes-links';
					$required_key = 'more_info_required';
					break;
			}

			if( !empty( $widget_type ) ){

				$settings = static::get_fes_form_widget_settings( $custom_form_id, $widget_type );

				return isset( $settings[ $required_key ] ) && 'yes' === $settings[ $required_key ] ? true : false;
			}
		}

		return $is_required;
	}

	/**
	 * Get MEC_Main
	 *
	 * @since 1.0.0
	 *
	 * @return \MEC_Main
	 */
	public static function get_main() {

		global $MEC_Main;
		if ( !is_a( $MEC_Main, \MEC_main::class ) ) {

			$MEC_Main = new \MEC_Main();
		}

		return $MEC_Main;
	}
}
