<?php
/**
 * Widget Base.
 *
 * @since 1.0.0
 *
 * @package MEC_FES_Builder
 */

namespace MEC_FES_Builder\Widgets;

defined( 'ABSPATH' ) || exit;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Css_Filter;
use Elementor\Controls_Manager;

/**
 * WidgetBase class
 *
 * @version 1.0.0
 */
abstract class WidgetBase extends Widget_Base {

	/**
	 * Is Edit Mode
	 *
	 * @return boolean
	 */
	public function is_edit_mode() {

		return Plugin::$instance->editor->is_edit_mode()
			||
			( isset( $_POST['action'] ) && 'elementor_ajax' === $_POST['action'] );
	}

	/**
	 * Prepare Args For Register Control
	 *
	 * @param array  $atts
	 * @param array  $args
	 * @param string $css_property_pattern
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function prepare_args( $atts, $args, $css_property_pattern = '' ) {

		/** Default Begin */
		if ( isset( $atts['default'] ) && ! empty( $atts['default'] ) ) {

			$args['default'] = $atts['default'];
		}
		/** Default End */

		/** Condition Begin */
		if ( isset( $atts['condition'] ) && ! empty( $atts['condition'] ) ) {

			$args['condition'] = $atts['condition'];
		}
		/** Condition End */

		/** CSS Selector Begin */
		$selector  = isset( $atts['selector'] ) ? $atts['selector'] : $this->fes_base_selector;
		$selectors = isset( $atts['selectors'] ) ? $atts['selectors'] : '';

		if ( ! empty( $css_property_pattern ) ) {

			$selectors = array(
				$selector => $css_property_pattern,
			);
		}

		if ( ! empty( $selectors ) ) {

			$args['selectors'] = $selectors;
		} else {

			$args['selector'] = $selector;
		}
		/** CSS Selector End */

		return apply_filters( 'mec_fes_builder_elementor_widget_prepare_args', $args );
	}

	/**
	 * Register Controls
	 *
	 * @param array $controls
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function fes_register_controls( $controls ) {

		foreach ( $controls as $atts ) {

			$this->fes_register_control( $atts );
		}
	}

	/**
	 * Initialize Array of Controls
	 *
	 * @since 1.0.0
	 *
	 * @access Public
	 *
	 * @param array  $controls
	 * @param array  $rewrite_controls
	 * @param string $group_id
	 *
	 * @return array
	 */
	public function fes_init_controls( $controls, $rewrite_controls, $group_id, $selector, $hover_type = '' ) {

		if ( empty( $rewrite_controls ) || ! is_array( $rewrite_controls ) ) {

			return $controls;
		}

		// Add Custom Controls
		foreach ( $rewrite_controls as $r_control_id => $r_control ) {

			if ( false === $r_control ) {

				unset( $controls[ $r_control_id ] );
				continue;
			}

			$find_hover = false !== strpos( $r_control_id, 'hover_' );
			if ( ( $hover_type && ! $find_hover ) || ( ! $hover_type && $find_hover ) ) {

				continue;
			}

			$type = isset( $r_control['type'] ) ? $r_control['type'] : false;
			$name = $group_id . $type;

			if ( ! $type ) {

				continue;
			}

			switch ( $type ) {

				case 'box_width':
				case 'image_width':
				case 'width':
				case 'height':
				case 'min_width':
				case 'min_height':
				case 'max_width':
				case 'max_height':
				case 'opacity':
				case 'transition':
				case 'font_size':
				case 'css_filter':
				case 'hover_animation':
				case 'float':
				case 'html_tag':
				case 'display':
				case 'switcher':
				case 'show_hide':
				case 'position':
				case 'position_space':
				case 'position_top':
				case 'position_bottom':
				case 'position_right':
				case 'position_left':
				case 'icon_align':
				case 'text_align':
				case 'align':
				case 'table_layout':
					$labels = array(
						'box_width'       => __( 'Box Width', MECFBTEXTDOMAIN ),
						'image_width'     => __( 'Image Width', MECFBTEXTDOMAIN ),
						'width'           => __( 'Width', MECFBTEXTDOMAIN ),
						'height'          => __( 'Height', MECFBTEXTDOMAIN ),
						'min_width'       => __( 'Min Width', MECFBTEXTDOMAIN ),
						'min_height'      => __( 'Min Height', MECFBTEXTDOMAIN ),
						'max_width'       => __( 'Max Width', MECFBTEXTDOMAIN ),
						'max_height'      => __( 'Max Height', MECFBTEXTDOMAIN ),
						'opacity'         => __( 'Opacity', MECFBTEXTDOMAIN ),
						'transition'      => __( 'Transition', MECFBTEXTDOMAIN ),
						'font_size'       => __( 'Font Size', MECFBTEXTDOMAIN ),

						'switcher'        => __( 'On/Off', MECFBTEXTDOMAIN ),
						'show_hide'       => __( 'Show/Hide', MECFBTEXTDOMAIN ),

						'css_filter'      => __( 'CSS Filter', MECFBTEXTDOMAIN ),

						'float'           => __( 'Float', MECFBTEXTDOMAIN ),
						'hover_animation' => __( 'Hover Animation', MECFBTEXTDOMAIN ),
						'html_tag'        => __( 'HTML Tag', MECFBTEXTDOMAIN ),
						'display'         => __( 'Display', MECFBTEXTDOMAIN ),
						'switcher'        => __( 'Switcher', MECFBTEXTDOMAIN ),

						'position'        => __( 'Position', MECFBTEXTDOMAIN ),
						'position_space'  => __( 'Position Space', MECFBTEXTDOMAIN ),
						'position_top'    => __( 'Position Top', MECFBTEXTDOMAIN ),
						'position_bottom' => __( 'Position Bottom', MECFBTEXTDOMAIN ),
						'position_right'  => __( 'Position Right', MECFBTEXTDOMAIN ),
						'position_left'   => __( 'Position Left', MECFBTEXTDOMAIN ),

						'table_layout'    => __( 'Table Layout', MECFBTEXTDOMAIN ),
					);

					if ( $r_control ) {

						$controls[ $r_control_id ] = array(
							'name'     => $name,
							'selector' => $selector,
						);

						if ( isset( $labels[ $type ] ) ) {

							$controls[ $r_control_id ]['label'] = $labels[ $type ];
						}
					}

					break;
			}
		}

		// Rewrite Controls
		foreach ( $controls as $control_id => $control_args ) {

			if ( ! isset( $rewrite_controls[ $control_id ] ) ) {

				continue;
			}

			$settings_field_args = $rewrite_controls[ $control_id ];

			if ( is_numeric( $settings_field_args ) || is_bool( $settings_field_args ) ) {

				continue;
			} elseif ( is_array( $settings_field_args ) ) {

				$control_args = wp_parse_args( $settings_field_args, $control_args );
			} else {

				$control_args['selector'] = $settings_field_args;
			}

			$controls[ $control_id ] = $control_args;
		}

		return $controls;
	}

	/**
	 * Register Share Links Icon Style Controls
	 *
	 * @since 1.0.0
	 *
	 * @access Public
	 *
	 * @return void
	 */
	public function fes_register_styles_controls( $group_id, $section_label, $description, $primary_selector, $primary_hover_selector, $rewrite_settings_fields = array(), $condition = array() ) {

		$this->start_controls_section(
			$group_id . 'section',
			array(
				'label'     => $section_label,
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => $condition,
			)
		);

		if ( ! empty( $description ) ) {

			$this->add_control(
				$group_id . 'description',
				array(
					'label'     => $description,
					'type'      => Controls_Manager::HEADING,
					'separator' => 'before',
				)
			);
		}

		switch( $group_id ){
			case 'radio_':
			case 'checkbox_':
			case 'color_':

				$tabs = array(
					'normal' => __( 'Normal', MECFBTEXTDOMAIN ),
					'hover'  => __( 'Selected', MECFBTEXTDOMAIN ),
				);

				break;
			default:

				$tabs = array(
					'normal' => __( 'Normal', MECFBTEXTDOMAIN ),
					'hover'  => __( 'Hover', MECFBTEXTDOMAIN ),
				);
		}

		$this->start_controls_tabs(
			$group_id . 'style_tabs'
		);

		foreach ( $tabs as $tab_id => $label ) {

			$this->start_controls_tab(
				$group_id . $tab_id . '_tab',
				array(
					'label' => $label,
				)
			);

			$type       = ( 'hover' === $tab_id ) ? 'hover_' : '';
			$selector_2 = ( 'hover' === $tab_id ) ? $primary_hover_selector : $primary_selector;
			$t_group_id = $group_id . $type;

			$controls = array(
				$t_group_id . 'show_hide'      => array(
					'name'		  => $t_group_id . 'show_hide',
				),
				$t_group_id . 'text_align'      => array(
					'name'		  => $t_group_id . 'text_align',
				),
				$t_group_id . 'checkbox_color'      => array(
					'name'		  => $t_group_id . 'checkbox_color',
				),
				$t_group_id . 'radio_color'      => array(
					'name'		  => $t_group_id . 'radio_color',
				),
				$t_group_id . 'typography' => array(
					'name'        => $t_group_id . 'typography',
					'type'        => 'typography_2',
					'label'       => __( 'Typography', MECFBTEXTDOMAIN ),
					'default'     => '',
					'selector'    => $selector_2,
					'text_align'  => true,
					'text_shadow' => true,
				),
				$t_group_id . 'color'      => array(
					'name'     => $t_group_id . 'color',
					'type'     => 'font_color',
					'label'    => __( 'Color', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
				),
				$t_group_id . 'background' => array(
					'name'     => $t_group_id . 'background',
					'type'     => 'background',
					'label'    => __( 'Background', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
				),
				$t_group_id . 'margin'     => array(
					'name'     => $t_group_id . 'margin',
					'type'     => 'margin',
					'label'    => __( 'Margin', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
				),
				$t_group_id . 'padding'    => array(
					'name'     => $t_group_id . 'padding',
					'type'     => 'padding',
					'label'    => __( 'Padding', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
				),
				$t_group_id . 'border'     => array(
					'name'     => $t_group_id . 'border',
					'type'     => 'border',
					'label'    => __( 'Border', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
					'radius'   => true,
				),
				$t_group_id . 'box_shadow' => array(
					'name'     => $t_group_id . 'box_shadow',
					'type'     => 'box_shadow',
					'label'    => __( 'Box Shadow', MECFBTEXTDOMAIN ),
					'default'  => '',
					'selector' => $selector_2,
				),
			);

			$controls = $this->fes_init_controls(
				$controls,
				$rewrite_settings_fields,
				$t_group_id,
				$selector_2,
				$type
			);

			if ( ! empty( $condition ) ) {

				foreach ( $controls as $control_id => $control ) {

					if ( ! isset( $controls[ $control_id ]['condition'] ) ) {

						$controls[ $control_id ]['condition'] = $condition;
					}
				}
			}

			$this->fes_register_controls( $controls );

			$this->end_controls_tab();
		}

		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Register Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function fes_register_control( $atts ) {

		$defaults = array(
			'type'      => '',
			// 'label'     => '',
			'selector'  => '',
			'selectors' => '',
			'condition' => '',
		);

		$atts = wp_parse_args( $atts, $defaults );

		$type = $atts['type'];
		if ( empty( $type ) ) {

			return;
		}

		switch ( $type ) {
			case 'typography':
			case 'typography_1':
			case 'typography_2':
			case 'typography_3':
			case 'typography_4':
				$this->typography( $atts );

				break;
			case 'text_shadow':
				$this->text_shadow( $atts );

				break;
			case 'icon_align':
			case 'text_align':
			case 'align':
				$this->align( $atts );

				break;
			case 'radio_color':
			case 'checkbox_color':
			case 'font_color':
			case 'color':
				$this->color( $atts );

				break;
			case 'background':
				$this->background( $atts );

				break;
			case 'margin':
				$this->margin( $atts );

				break;
			case 'padding':
				$this->padding( $atts );

				break;
			case 'border':
				$this->border( $atts );

				break;
			case 'border_radius':
				$this->border_radius( $atts );

				break;
			case 'box_shadow':
				$this->box_shadow( $atts );

				break;
			case 'box_width':
			case 'image_width':
			case 'width':
			case 'height':
			case 'min_width':
			case 'min_height':
			case 'max_width':
			case 'max_height':
			case 'opacity':
			case 'transition':
			case 'font_size':
			case 'position_top':
			case 'position_bottom':
			case 'position_left':
			case 'position_right':
				$this->slider( $atts );

				break;
			case 'switcher':
			case 'show_hide':
				$this->switcher( $atts );

				break;

			case 'css_filter':
				$this->css_filter( $atts );

				break;
			case 'hover_animation':
				$this->hover_animation( $atts );

				break;
			case 'position':
				$this->position( $atts );

				break;
			case 'position_space':
				$this->position_space( $atts );

				break;

			case 'html_tag':
				$this->html_tag( $atts );

				break;
			case 'display':
				$this->fes_display( $atts );

				break;
			case 'table_layout':
				$this->table_layout( $atts );

				break;

			case 'float':
				$this->float( $atts );

				break;

		}
	}

	/**
	 * Register Typography Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function typography( $atts ) {

		$defaults = array(
			'name'        => 'typography',
			'label'       => __( 'Typography', MECFBTEXTDOMAIN ),
			'text_align'  => true,
			'text_shadow' => true,
		);
		$atts     = wp_parse_args( $atts, $defaults );

		switch ( $atts['type'] ) {

			case 'typography_1':
				$scheme = \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY;
				break;
			case 'typography_2':
				$scheme = \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_SECONDARY;
				break;
			case 'typography_3':
				$scheme = \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT;
				break;
			case 'typography_4':
				$scheme = \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_ACCENT;
				break;
			default:
				$scheme = \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_SECONDARY;
		}

		$args = array(
			'name'        => $atts['name'],
			'label'       => $atts['label'],
            'global' => [
                    'default' => $scheme,
             		],
			'text_align'  => $atts['text_align'],
			'text_shadow' => $atts['text_shadow'],
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			$args
		);

		if ( $args['text_align'] ) {

			$align_atts = array(
				'name'      => $atts['name'] . '_text_align',
				'type'      => 'text_align',
				'selector'  => isset( $args['selector'] ) ? $args['selector'] : $this->fes_base_selector,
				'selectors' => isset( $args['selectors'] ) ? $args['selectors'] : array(),
				'condition' => isset( $args['condition'] ) ? $args['condition'] : array(),
			);

			$this->fes_register_control( $align_atts );
		}

		if ( $args['text_shadow'] ) {

			$text_shadow_atts = array(
				'name'      => $atts['name'] . '_text_shadow',
				'type'      => 'text_shadow',
				'selector'  => isset( $args['selector'] ) ? $args['selector'] : $this->fes_base_selector,
				'selectors' => isset( $args['selectors'] ) ? $args['selectors'] : array(),
				'condition' => isset( $args['condition'] ) ? $args['condition'] : array(),
			);

			$this->fes_register_control( $text_shadow_atts );
		}
	}


	/**
	 * Register Text Shadow Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function text_shadow( $atts ) {

		$defaults = array(
			'name'  => 'text_shadow',
			'label' => __( 'Text Shadow', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'  => $atts['name'],
			'label' => $atts['label'],
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			$args
		);
	}

	/**
	 * Register Align Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function align( $atts ) {

		$defaults = array(
			'name'    => 'align',
			'label'   => __( 'Align', MECFBTEXTDOMAIN ),
			'default' => '',
			'toggle'  => true,
			'options' => array(
				'left'   => array(
					'title' => __( 'Left', MECFBTEXTDOMAIN ),
					'icon'  => 'eicon-text-align-left',
				),
				'center' => array(
					'title' => __( 'Center', MECFBTEXTDOMAIN ),
					'icon'  => 'eicon-text-align-center',
				),
				'right'  => array(
					'title' => __( 'Right', MECFBTEXTDOMAIN ),
					'icon'  => 'eicon-text-align-right',
				),
			),
		);

		if ( in_array( $atts['type'], array( 'icon_align' ) ) ) {

			unset( $defaults['options']['center'] );
		}

		$atts = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::CHOOSE,
			'toggle'  => $atts['toggle'],
			'options' => $atts['options'],
		);

		$pattern = 'text-align: {{VALUE}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Color Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function color( $atts ) {

		$defaults = array(
			'name'  => 'typography',
			'label' => __( 'Color', MECFBTEXTDOMAIN ),
			'important' => false,
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$scheme =  \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY;

		$args = array(
			'label'  => $atts['label'],
			'type'   => Controls_Manager::COLOR,
//            'global' => [
//                			'default' => $scheme,
//                		],
		);

		$pattern = '';
		switch ( $atts['type'] ) {

			case 'radio_color':
			case 'checkbox_color':
				$pattern = 'background: {{VALUE}} !important;';
				break;
			case 'font_color':
			case 'color':
			default:
				if( !$atts['important'] ){

					$pattern = 'color: {{VALUE}};fill: {{VALUE}};';
				}else{

					$pattern = 'color: {{VALUE}} !important;fill: {{VALUE}} !important;';
				}
		}
		$args = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Background Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function background( $atts ) {

		$defaults = array(
			'name'  => 'background',
			'label' => __( 'Background', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'  => $atts['name'],
			'label' => $atts['label'],
			'types' => array(
				'classic',
				'gradient',
			),
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_group_control(
			Group_Control_Background::get_type(),
			$args
		);
	}

	/**
	 * Register Margin Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function margin( $atts ) {

		$defaults = array(
			'name'  => 'margin',
			'label' => __( 'Margin', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'       => $atts['name'],
			'label'      => $atts['label'],
			'type'       => Controls_Manager::DIMENSIONS,
			'size_units' => array( 'px', 'em', '%' ),
			'devices'    => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Padding Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function padding( $atts ) {

		$defaults = array(
			'name'  => 'padding',
			'label' => __( 'Padding', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'       => $atts['name'],
			'label'      => $atts['label'],
			'type'       => Controls_Manager::DIMENSIONS,
			'size_units' => array( 'px', 'em', '%' ),
			'devices'    => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Border Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function border( $atts ) {

		$defaults = array(
			'name'         => 'border',
			'label'        => __( 'Border', MECFBTEXTDOMAIN ),
			'radius'       => true,
			'label_radius' => __( 'Border Radius', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$d_args = array(
			'name'       => $atts['name'],
			'label'      => $atts['label'],
			'type'       => Controls_Manager::DIMENSIONS,
			'size_units' => array( 'px', 'em', '%' ),
			'devices'    => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $d_args, $pattern );

		$this->add_group_control(
			Group_Control_Border::get_type(),
			$args
		);

		if ( $atts['radius'] ) {

			$atts['type']  = 'border_radius';
			$atts['name']  = $atts['name'] . '_radius';
			$atts['label'] = $atts['label_radius'];

			$this->fes_register_control( $atts );
		}
	}

	/**
	 * Register Border Radius Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function border_radius( $atts ) {

		$defaults = array(
			'name'  => 'border_radius',
			'label' => __( 'Border Radius', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'       => $atts['name'],
			'label'      => $atts['label'],
			'type'       => Controls_Manager::DIMENSIONS,
			'size_units' => array( 'px', 'em', '%' ),
			'devices'    => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}


	/**
	 * Register Box Shadow Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function box_shadow( $atts ) {

		$defaults = array(
			'name'  => 'box_shadow',
			'label' => __( 'Box Shadow', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'  => $atts['name'],
			'label' => $atts['label'],
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			$args
		);
	}

	/**
	 * Register Range Slider Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function slider( $atts ) {

		$defaults = array(
			'name'       => 'slider',
			'label'      => __( 'Size', MECFBTEXTDOMAIN ),
			'size_units' => array( 'px', '%', 'vw' ),
			'range'      => array(
				'px' => array(
					'min'  => 0,
					'max'  => 1000,
					'step' => 5,
				),
				'%'  => array(
					'min' => 0,
					'max' => 100,
				),
				'vw' => array(
					'min' => 1,
					'max' => 100,
				),
			),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'       => $atts['name'],
			'label'      => $atts['label'],
			'type'       => Controls_Manager::SLIDER,
			'size_units' => $atts['size_units'],
			'range'      => $atts['range'],
		);

		$pattern = '';
		switch ( $atts['type'] ) {

			case 'box_width':
			case 'image_width':
			case 'width':
				$pattern = 'width: {{SIZE}}{{UNIT}};';

				break;
			case 'max_width':
				$pattern = 'max-width: {{SIZE}}{{UNIT}};';

				break;
			case 'max_height':
				$pattern = 'max-height: {{SIZE}}{{UNIT}};';

				$args['size_units'] = array( 'px', 'vh' );
				$args['range']      = array(
					'px' => array(
						'min' => 1,
						'max' => 500,
					),
					'vh' => array(
						'min' => 1,
						'max' => 100,
					),
				);

				break;
			case 'min_width':
				$pattern = 'min-width: {{SIZE}}{{UNIT}};';

				break;
			case 'height':
				$pattern = 'height: {{SIZE}}{{UNIT}};';

				$args['size_units'] = array( 'px', 'vh' );
				$args['range']      = array(
					'px' => array(
						'min' => 1,
						'max' => 500,
					),
					'vh' => array(
						'min' => 1,
						'max' => 100,
					),
				);

				break;
			case 'min_height':
				$pattern = 'min-height: {{SIZE}}{{UNIT}};';

				$args['size_units'] = array( 'px', 'vh' );
				$args['range']      = array(
					'px' => array(
						'min' => 1,
						'max' => 500,
					),
					'vh' => array(
						'min' => 1,
						'max' => 100,
					),
				);

				break;
			case 'opacity':
				$pattern = 'opacity: {{SIZE}};';

				$args['size_units'] = array( 'px' );
				$args['range']      = array(
					'px' => array(
						'max'  => 1,
						'min'  => 0.10,
						'step' => 0.01,
					),
				);

				break;
			case 'position_top':
			case 'position_bottom':
			case 'position_right':
			case 'position_left':
				$args['size_units'] = array( 'px', '%' );
				$args['range']      = array(
					'px' => array(
						'min'  => -1000,
						'max'  => 1000,
						'step' => 5,
					),
					'%'  => array(
						'min' => -100,
						'max' => 100,
					),
				);

				$default_labels = array(
					'position_top'    => __( 'Position Top', MECFBTEXTDOMAIN ),
					'position_bottom' => __( 'Position Bottom', MECFBTEXTDOMAIN ),
					'position_right'  => __( 'Position Right', MECFBTEXTDOMAIN ),
					'position_left'   => __( 'Position Left', MECFBTEXTDOMAIN ),
				);

				if ( empty( $args['label'] ) ) {

					$args['label'] = $default_labels[ $atts['type'] ];
				}

				$css_k   = str_replace( 'position_', '', $atts['type'] );
				$pattern = "{$css_k}: {{SIZE}}{{UNIT}};";

				break;

			case 'transition':
				$pattern = 'transition: {{SIZE}}s;';

				$args['size_units'] = array( '' );
				$args['range']      = array(
					'px' => array(
						'max'  => 3,
						'step' => 0.01,
					),
				);

				break;

			case 'font_size':
			default:
				$pattern = 'font-size: {{SIZE}}{{UNIT}};';

				$args['size_units'] = array( 'px' );
				$args['range']      = array(
					'px' => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
				);

		}
		$args = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Switcher Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function switcher( $atts ) {

		$defaults = array(
			'name'         => 'status',
			'label'        => __( 'On/Off', MECFBTEXTDOMAIN ),
			'label_on'     => __( 'On', MECFBTEXTDOMAIN ),
			'label_off'    => __( 'Off', MECFBTEXTDOMAIN ),
			'return_value' => 'yes',
		);

		$pattern = '';
		switch ( $atts['type'] ) {

			case 'show_hide':
				$defaults['label']        = __( 'Show/Hide', MECFBTEXTDOMAIN );
				$defaults['label_on']     = __( 'Hide', MECFBTEXTDOMAIN );
				$defaults['label_off']    = __( 'Show', MECFBTEXTDOMAIN );
				$defaults['return_value'] = 'none';

				$pattern = 'display: {{VALUE}};';

				break;
		}

		$atts = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'         => $atts['name'],
			'label'        => $atts['label'],
			'type'         => Controls_Manager::SWITCHER,
			'label_on'     => $atts['label_on'],
			'label_off'    => $atts['label_off'],
			'return_value' => $atts['return_value'],
			'default'      => 'yes',
		);

		$args = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Display Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function fes_display( $atts ) {

		$defaults = array(
			'name'    => 'display',
			'label'   => __( 'Display', MECFBTEXTDOMAIN ),
			'default' => '',
			'options' => array(
				''             => __( 'Default', MECFBTEXTDOMAIN ),
				'inherit'      => __( 'Inherit', MECFBTEXTDOMAIN ),
				'inline'       => __( 'inline', MECFBTEXTDOMAIN ),
				'inline-block' => __( 'inline block', MECFBTEXTDOMAIN ),
				'block'        => __( 'block', MECFBTEXTDOMAIN ),
				'none'         => __( 'none', MECFBTEXTDOMAIN ),
			),
		);

		$atts = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::SELECT,
			'default' => $atts['default'],
			'options' => $atts['options'],
			'devices' => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'display: {{VALUE}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Table Layout Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function table_layout( $atts ) {

		$defaults = array(
			'name'  => 'table_layout',
			'label' => __( 'Table Layout', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::SELECT,
			'default' => '',
			'options' => array(
				''      => __( 'Default', MECFBTEXTDOMAIN ),
				'fixed' => __( 'Fixed', MECFBTEXTDOMAIN ),
				'auto'  => __( 'Auto', MECFBTEXTDOMAIN ),
			),
			'devices' => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'table-layout: {{VALUE}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Float Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function float( $atts ) {

		$defaults = array(
			'name'  => 'float',
			'label' => __( 'Float', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::SELECT,
			'default' => '',
			'options' => array(
				''      => __( 'Default', MECFBTEXTDOMAIN ),
				'right' => __( 'Right', MECFBTEXTDOMAIN ),
				'left'  => __( 'Left', MECFBTEXTDOMAIN ),
				'unset' => __( 'Unset', MECFBTEXTDOMAIN ),
			),
			'devices' => array( 'desktop', 'tablet', 'mobile' ),
		);

		$pattern = 'float: {{VALUE}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_responsive_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register HTML Tag Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function html_tag( $atts ) {

		$defaults = array(
			'name'  => 'html_tag',
			'label' => __( 'HTML Tag', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::SELECT,
			'default' => 'h1',
			'options' => array(
				'h1' => __( 'h1', MECFBTEXTDOMAIN ),
				'h2' => __( 'h2', MECFBTEXTDOMAIN ),
				'h3' => __( 'h3', MECFBTEXTDOMAIN ),
				'h4' => __( 'h4', MECFBTEXTDOMAIN ),
				'h5' => __( 'h5', MECFBTEXTDOMAIN ),
				'h6' => __( 'h6', MECFBTEXTDOMAIN ),
			),
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register CSS Filter Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function css_filter( $atts ) {

		$defaults = array(
			'name'  => 'css_filter',
			'label' => __( 'CSS Filter', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'  => $atts['name'],
			'label' => $atts['label'],
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			$args
		);
	}

	/**
	 * Register Hover Animation Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function hover_animation( $atts ) {

		$defaults = array(
			'name'  => 'hover_animation',
			'label' => __( 'Hover Animation', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'  => $atts['name'],
			'label' => $atts['label'],
			'type'  => Controls_Manager::HOVER_ANIMATION,
		);

		$pattern = '';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Position Type Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function position( $atts ) {

		$defaults = array(
			'name'  => 'position',
			'label' => __( 'Hover Animation', MECFBTEXTDOMAIN ),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		$args = array(
			'name'    => $atts['name'],
			'label'   => $atts['label'],
			'type'    => Controls_Manager::SELECT,
			'options' => array(
				''         => __( 'Default', MECFBTEXTDOMAIN ),
				'relative' => __( 'Relative', MECFBTEXTDOMAIN ),
				'absolute' => __( 'Absolute', MECFBTEXTDOMAIN ),
				'fixed'    => __( 'Fixed', MECFBTEXTDOMAIN ),
				'static'   => __( 'Static', MECFBTEXTDOMAIN ),
			),
		);

		$pattern = 'position: {{VALUE}};';
		$args    = $this->prepare_args( $atts, $args, $pattern );

		$this->add_control(
			$atts['name'],
			$args
		);
	}

	/**
	 * Register Position Space Control
	 *
	 * @param array $atts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function position_space( $atts ) {

		$defaults = array(
			'name'      => 'position_space',
			'label'     => __( 'Position Space', MECFBTEXTDOMAIN ),
			'positions' => array(
				'top'    => array(),
				'bottom' => array(),
				'right'  => array(),
				'left'   => array(),
			),
		);
		$atts     = wp_parse_args( $atts, $defaults );

		foreach ( (array) $atts['positions'] as $position => $position_args ) {

			$group_id = "{$atts['name']}_{$position}";
			$p_atts   = array(
				'name'     => $group_id,
				'type'     => 'position_' . $position,
				'selector' => $atts['selector'],
				'label'    => '', // Auto Set Default
			);

			$atts = wp_parse_args( $position_args, $p_atts );

			$p_atts = $this->prepare_args( $atts, $p_atts, '' );

			$this->fes_register_control( $p_atts );
		}
	}
}
