<?php

namespace MEC_Zapier\Core\SetRoutes;

use DateInterval;

// don't load directly.
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SetRoutes.
 *
 * @author      author
 * @package     package
 * @since       2.0.0
 */
class SetRoutesV2 {


	/**
	 * Instance of this class.
	 *
	 * @since   2.0.0
	 * @access  public
	 * @var     MEC_Zapier
	 */
	public static $instance;

	/**
	 * The directory of the file.
	 *
	 * @access  public
	 * @var     string
	 */
	public static $dir;

	/**
	 * The Args
	 *
	 * @access  public
	 * @var     array
	 */
	public static $args;

	/**
	 * Used for set settings
	 *
	 * @var int
	 */
	public $max_attendees         = 10;
	public $max_tickets           = 10;
	public $max_event_occurrences = 10;
	public $max_terms             = 2;


	/**
	 * Used for set settings
	 *
	 * @var bool
	 */
	public $set_settings_status = false;

	public $request;
	public $task;
	public $datetimeFormat;
	public $dateFormat;
	public $timeFormat;

	/**
	 * Provides access to a single instance of a module using the singleton pattern.
	 *
	 * @since   2.0.0
	 * @return  object
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}


	public function __construct() {
		self::settingUp();
		self::setHooks( $this );
		self::init();
	}

	/**
	 * Set Hooks.
	 *
	 * @since   2.0.0
	 */
	public static function setHooks( $This ) {
		add_action( 'rest_api_init', array( $This, 'mecZapierRoute' ) );
	}

	/**
	 * Global Variables.
	 *
	 * @since   2.0.0
	 */
	public static function settingUp() {
	}

	/**
	 * API Route
	 *
	 * @since     2.0.0
	 */
	public function mecZapierRoute() {
		$namespace = 'mec-zapier/v2';
		$apiPath   = 'api/';

		register_rest_route(
			$namespace,
			$apiPath,
			array(
				'methods'  =>  \WP_REST_Server::READABLE,
				'permission_callback' => '__return_true',
				'callback' => array( $this, 'mecZapierCallback' ),
			)
		);

	}

	/**
	 * @since     2.0.0
	 */
	public function get_request_param( $key ) {

		return $this->request->get_param( $key );
	}

	public function get_last_response_key(){

		$key = '';
		switch ( $this->task ) {

			case 'new_event':
			case 'modified_event':
			case 'canceled_event':

				$categories = $this->get_request_param( 'event_category' );

				$key = $this->task . '_c_' . trim( $categories );

				break;
			case 'new_booking':

				$event_ids = $this->get_request_param( 'event_id' );

				$key = $this->task . '_e_' . trim( $event_ids );

				break;

		}

		return $key;
	}

	public function get_last_response_item_id(){

		$key = $this->get_last_response_key();
		return get_option( 'mec_zapier_'. $key .'_last_item_id', 0 );
	}

	public function update_last_response_item_id( $item_id ){

		$key = $this->get_last_response_key();
		update_option( 'mec_zapier_'. $key .'_last_item_id', $item_id );
	}

	public static function filter_last_item( $where, $query ){

		global $wpdb;
		global $mec_zapier_last_item_id;

		if( !$mec_zapier_last_item_id ){

			return $where;
		}

		return $where . $wpdb->prepare( " AND {$wpdb->posts}.ID > %d", $mec_zapier_last_item_id );
	}

	/**
	 * @since     2.0.0
	 */
	public function get_query_args( $q_args ) {

		global $mec_zapier_last_item_id;
		$mec_zapier_last_item_id = $this->get_last_response_item_id();

		add_filter( 'posts_where', array( __CLASS__, 'filter_last_item' ), 10, 2 );

		switch ( $this->task ) {

			case 'list_events':
			case 'all_events':
				$limit                    = $this->get_request_param( 'limit' );
				$limit                    = (int) $limit > 0 ? $limit : 5;
				$q_args['posts_per_page'] = $limit;

				$paged           = $this->get_request_param( 'paged' );
				$paged           = (int) $paged > 0 ? $paged : 1;
				$q_args['paged'] = $paged;

				$categories = $this->get_request_param( 'event_category' );
				if ( ! empty( $categories ) && ! is_array( $categories ) ) {
					$q_args['tax_query'][] = array(
						'taxonomy' => 'mec_category',
						'field'    => 'term_id',
						'terms'    => explode( ' ', $categories ),
						'operator' => 'IN',
					);
				}
				break;
			case 'new_event':
			case 'modified_event':
			case 'canceled_event':

				$q_args['suppress_filters'] = false;
				if( $mec_zapier_last_item_id ){

					$q_args['orderby'] = 'ID';
					$q_args['order'] = 'ASC';
				}

				$categories = $this->get_request_param( 'event_category' );
				if ( ! empty( $categories ) && ! is_array( $categories ) ) {
					$q_args['tax_query'][] = array(
						'taxonomy' => 'mec_category',
						'field'    => 'term_id',
						'terms'    => explode( ' ', $categories ),
						'operator' => 'IN',
					);
				}
				break;
			case 'new_booking':

				$q_args['suppress_filters'] = false;
				if( $mec_zapier_last_item_id ){

					$q_args['orderby'] = 'ID';
					$q_args['order'] = 'ASC';
				}

				$event_ids = $this->get_request_param( 'event_id' );
				if ( ! empty( $event_ids ) && ! is_array( $event_ids ) ) {
					$q_args['meta_query'][] = array(
						'key'     => 'mec_event_id',
						'value'   => explode( ' ', $event_ids ),
						'compare' => 'IN',
					);
				}
				break;
			case 'event_occurrences':
				$event_ids = $this->get_request_param( 'event_id' );
				if ( ! empty( $event_ids ) && ! is_array( $event_ids ) ) {
					$q_args['meta_query'][] = array(
						'key'     => 'mec_event_id',
						'value'   => explode( ' ', $event_ids ),
						'compare' => 'IN',
					);
				}
				break;

		}

		return $q_args;
	}

	/**
	 * API Callback
	 *
	 * @since     2.0.0
	 */
	public function mecZapierCallback( $request ) {

		$this->request = $request;
		$api_key       = $request->get_param( 'api_key' );
		$task          = $request->get_param( 'task' );
		$this->task    = $task;
		if ( $api_key != get_option( 'mecZapierKey' ) & empty( $task ) ) {

			return new \WP_Error( 'invalid_key', 'Invalid API Key', array( 'status' => 404 ) );

		} elseif ( $api_key == get_option( 'mecZapierKey' ) & empty( $task ) ) {

			return new \WP_REST_Response(
				array(
					'url'         => site_url(),
					'zapier_desc' => 'MEC Zapier Integration',
				),
				200
			);

		} elseif ( $api_key != get_option( 'mecZapierKey' ) & ! empty( $task ) ) {

			return new \WP_Error( 'invalid_key', 'Invalid API Key', array( 'status' => 404 ) );

		} elseif ( $api_key == get_option( 'mecZapierKey' ) & ! empty( $task ) ) {

			$main                        = new \MEC_Main();
			$settings                    = $main->get_settings();
			$this->set_settings_status   = isset( $settings['api_v2']['set_settings_status'] ) ? 'yes' === $settings['api_v2']['set_settings_status'] : false;
			$this->max_attendees         = isset( $settings['api_v2']['max_attendees'] ) && is_numeric( $settings['api_v2']['max_attendees'] ) ? (int) $settings['api_v2']['max_attendees'] : 10;
			$this->max_tickets           = isset( $settings['api_v2']['max_tickets'] ) && is_numeric( $settings['api_v2']['max_tickets'] ) ? (int) $settings['api_v2']['max_tickets'] : 10;
			$this->max_event_occurrences = isset( $settings['api_v2']['max_event_occurrences'] ) && is_numeric( $settings['api_v2']['max_event_occurrences'] ) ? (int) $settings['api_v2']['max_event_occurrences'] : 10;
			$this->max_terms             = isset( $settings['api_v2']['max_terms'] ) && is_numeric( $settings['api_v2']['max_terms'] ) ? (int) $settings['api_v2']['max_terms'] : 2;

			$this->dateFormat     = 'Y-m-d';
			$this->timeFormat     = 'H:i:s';
			$this->datetimeFormat = DATE_ISO8601;
			switch ( $task ) {
				case 'new_event':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_new_event_info();
					} else {

						$data = $this->get_new_event_info_for_set_setting();
					}

					break;
				case 'modified_event':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_modified_event_info();
					} else {

						$data = $this->get_modified_event_info_for_set_setting();
					}

					break;
				case 'new_booking':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_new_booking_info();
					} else {

						$data = $this->get_new_booking_info_for_set_setting();
					}

					break;
				case 'canceled_event':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_cancelled_event_info();
					} else {

						$data = $this->get_cancelled_event_info_for_set_setting();
					}

					break;
				case 'all_events':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_all_event_info();
					} else {

						$data = $this->get_all_event_info_for_set_setting();
					}

					return apply_filters('mec_zapier_rest_api_data', $data, $task, $this->set_settings_status, $this);

					break;
				case 'list_events':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_list_event_info();
					} else {

						$data = $this->get_list_event_info_for_set_setting();
					}

					break;
				case 'event_occurrences':
					if ( true !== $this->set_settings_status ) {

						$data = $this->get_event_occurrences_info();
					} else {

						$event_id = 1;
						$data     = $this->get_event_occurrences_for_set_setting( $event_id );
					}

					break;
				default:
					$data = array( 'Please enter valid task.' );
					break;
			}

			$data = apply_filters('mec_zapier_rest_api_data', $data, $task, $this->set_settings_status, $this);

			return array( $data );

		}
	}

	/**
	 * @since     2.0.0
	 */
	public function get_new_event_info() {

		$data        = array();
		$q_args      = $this->get_query_args(
			array(
				'posts_per_page' => 1, // Number of recent posts thumbnails to display
				'post_type'      => 'mec-events',
				'post_status'    => 'publish', // Show only the published posts
				'no_found_rows'  => true,
			)
		);
		$recentEvent = new \WP_Query( $q_args );

		if ( $recentEvent->have_posts() ) :
			while ( $recentEvent->have_posts() ) :
				$recentEvent->the_post();

				$data = $this->get_event_detail();

			endwhile;
			wp_reset_postdata();
		endif;

		$this->update_last_response_item_id( $data['id'] );

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_modified_event_info() {

		$data                 = array();
		$q_args               = $this->get_query_args(
			array(
				'post_type'      => 'mec-events',
				'posts_per_page' => 1,
				'orderby'        => 'modified',
				'no_found_rows'  => true, // speed up query when we don't need pagination
			)
		);
		$recentlyUpdatedEvent = new \WP_Query( $q_args );
		if ( $recentlyUpdatedEvent->have_posts() ) :
			while ( $recentlyUpdatedEvent->have_posts() ) :
				$recentlyUpdatedEvent->the_post();
				if ( get_the_modified_date() . get_the_modified_time() == get_the_date() . get_the_time() ) {

					return array( $data );
				}

				$data = $this->get_event_detail();
			endwhile;
			wp_reset_postdata();
		endif;

		$this->update_last_response_item_id( $data['id'] );

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_new_booking_info() {

		$data   = array();
		$q_args = $this->get_query_args(
			array(
				'numberposts' => 1, // Number of recent posts thumbnails to display
				'post_type'   => 'mec-books',
				'post_status' => 'publish', // Show only the published posts
				'orderby'     => 'ID',
				'order'       => 'DESC',
			)
		);

		$recentBooking = wp_get_recent_posts( $q_args );
		foreach ( $recentBooking as $booking ) :
			$mecFeaturesBook = new \MEC_feature_books();
			$mecMain         = new \MEC_main();
			// Event
			$eventId    = get_post_meta( $booking['ID'], 'mec_event_id', true );
			$price      = get_post_meta( $booking['ID'], 'mec_price', true );
			$eventTitle = get_the_title( $eventId );
			// Price
			$bookPrice = get_post_meta( $booking['ID'], 'mec_price', true );

			$confirmStatus = get_post_meta( $booking['ID'], 'mec_confirmed', true );
			if ( $confirmStatus == '1' ) {
				$confirmStatus = __( 'Confirmed', 'mec-zapier' );
			} elseif ( $confirmStatus == '0' ) {
				$confirmStatus = __( 'Pending', 'mec-zapier' );
			} elseif ( $confirmStatus == '-1' ) {
				$confirmStatus = __( 'Rejected', 'mec-zapier' );
			}

			$verifyStatus = get_post_meta( $booking['ID'], 'mec_verified', true );
			if ( $verifyStatus == '1' ) {
				$verifyStatus = __( 'Verified', 'mec-zapier' );
			} elseif ( $verifyStatus == '0' ) {
				$verifyStatus = __( 'Waiting', 'mec-zapier' );
			} elseif ( $verifyStatus == '-1' ) {
				$verifyStatus = __( 'Canceled', 'mec-zapier' );
			}

			$attendees = $mecFeaturesBook->book->get_attendees( $booking['ID'] );

			$book_id 							 = $booking['ID'];
			$datetime 							 = get_post_datetime( $book_id );
			$event_date 						 = get_post_meta( $book_id, 'mec_attention_time', true );
			$ex_event_date 					 = explode( ':', $event_date );

			$data['id']                          = $book_id;
			$data['booking_id']                  = $book_id;
			$data['booking_title']               = $attendees[0]['name'] . ' - ' . $attendees[0]['email'];
			$data['booking_email']               = $attendees[0]['email'];
			$data['booking_name']                = $attendees[0]['name'];
			$data['booking_date']                = date_i18n( $this->datetimeFormat, strtotime( $datetime->format('Y-m-d H:i') ) );
			$data['booking_order_time']          = date_i18n( $this->datetimeFormat, strtotime( get_post_meta( $booking['ID'], 'mec_booking_time', true ) ) );
			$data['booking_event_title']         = $eventTitle;
			$data['booking_event_id']            = $eventId;
			$data['booking_attendees']           = $mecFeaturesBook->book->get_total_attendees( $booking['ID'] );
			$data['booking_price']               = $mecMain->render_price( ( $price ? $price : 0 ) );
			$data['booking_price_label']         = get_post_meta( $booking['ID'], 'mec_gateway_label', true );
			$data['booking_confirmation_status'] = $confirmStatus;
			$data['booking_verification_status'] = $verifyStatus;
			$data['booking_transaction_id']      = get_post_meta( $booking['ID'], 'mec_transaction_id', true );
			$data['event_start_on']              = date_i18n( $this->datetimeFormat, $ex_event_date[0] );
			$data['event_end_on']                = date_i18n( $this->datetimeFormat, $ex_event_date[1] );

			$book = new \MEC_Core\Books\Book( $booking );

			// Ticket detail start
			$book_tickets = $book->get_tickets_ids();
			$tickets      = \MEC_Core\Tickets\Tickets::getInstance()->get_event_tickets( $eventId );
			$this->get_tickets_detail( $book_tickets, $tickets, $data );
			// Ticket detail end

			// Attendees detail start
			$this->get_attendees_detail( $eventId, $attendees, $tickets, $data );
			// Attendees detail end

			// Booking Fixed Custom Fields detail start
			$this->get_booking_fixed_detail( $booking['ID'], $data );
			// Booking Fixed Custom Fields detail start

		endforeach;
		wp_reset_query();

		$this->update_last_response_item_id( $data['id'] );

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_cancelled_event_info() {

		$data   = array();
		$q_args = $this->get_query_args(
			array(
				'post_type'      => 'mec-events',
				'posts_per_page' => 1,
				'orderby'        => 'modified',
				'no_found_rows'  => true, // speed up query when we don't need pagination
				'meta_query'     => array(
					array(
						'key'   => 'mec_event_status',
						'value' => 'EventCancelled',
					),
				),
			)
		);

		$recentlyUpdatedEvent = new \WP_Query( $q_args );

		if ( $recentlyUpdatedEvent->have_posts() ) :
			while ( $recentlyUpdatedEvent->have_posts() ) :
				$recentlyUpdatedEvent->the_post();
				if ( get_the_modified_date() . get_the_modified_time() == get_the_date() . get_the_time() ) {
					return array( $data );
				}
				$event_id                          = get_the_ID();
				$data                              = $this->get_event_detail();
				$data['event_reason_cancellation'] = get_post_meta( $event_id, 'mec_cancelled_reason', true );
			endwhile;
			wp_reset_postdata();
		endif;

		$this->update_last_response_item_id( $data['id'] );

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_all_event_info() {

		$data        = array();
		$q_args      = $this->get_query_args(
			array(
				'posts_per_page' => 5, // Number of recent posts thumbnails to display
				'post_type'      => 'mec-events',
				'post_status'    => 'publish', // Show only the published posts
				'orderby'        => 'date',
				'no_found_rows'  => true,
			)
		);
		$recentEvent = new \WP_Query( $q_args );

		if ( $recentEvent->have_posts() ) :
			while ( $recentEvent->have_posts() ) :
				$recentEvent->the_post();

				$data[] = $this->get_event_detail();

			endwhile;
			wp_reset_postdata();
		endif;

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_list_event_info() {
		$data        = array();
		$q_args      = $this->get_query_args(
			array(
				'posts_per_page' => 5, // Number of recent posts thumbnails to display
				'post_type'      => 'mec-events',
				'post_status'    => 'publish', // Show only the published posts
				'orderby'        => 'date',
				'no_found_rows'  => true,
			)
		);
		$recentEvent = new \WP_Query( $q_args );

		$i = 0;
		if ( $recentEvent->have_posts() ) :
			while ( $recentEvent->have_posts() ) :
				$recentEvent->the_post();
				$i++;
				$d = $this->get_event_detail(
					array(
						'terms' => true,
					)
				);

				$data['id'] .= $d['id'];
				foreach ( $d as $k => $v ) {

					$data[ $k . '_' . $i ] = $v;
				}

			endwhile;
			wp_reset_postdata();
		endif;

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_event_occurrences_info( $event_id = false ) {

		if ( ! $event_id ) {

			$event_id = $this->get_request_param( 'event_id' );
		}

		if ( ! $event_id ) {

			return array();
		}

		$start = strtotime( $this->get_request_param( 'start_datetime' ) );
		if ( empty( $start ) ) {
			$start = current_time( 'timestamp' );
		}

		$max_event_occurrences = $this->get_request_param( 'limit' );
		if ( empty( $max_event_occurrences ) ) {
			$max_event_occurrences = $this->max_event_occurrences;
		}

		$occurrences = $this->get_event_occurrences( $event_id, $start, $max_event_occurrences );
		$data        = array();
		$i           = 1;
		foreach ( $occurrences as $k => $occurrence ) {
			$occurrence_detail = array(
				'id'                               => $event_id,
				'occurrences_start_date_' . $i     => date_i18n( $this->dateFormat, $occurrence->tstart ),
				'occurrences_end_date_' . $i       => date_i18n( $this->dateFormat, $occurrence->tend ),
				'occurrences_start_datetime_' . $i => date_i18n( $this->datetimeFormat, $occurrence->tstart ),
				'occurrences_end_datetime_' . $i   => date_i18n( $this->datetimeFormat, $occurrence->tend ),
			);

			$data = array_merge( $data, $occurrence_detail );
			$i++;
		}

		for ( $j = $i;$j <= $this->max_event_occurrences;$j++ ) {

			$data[ 'occurrences_start_date_' . $j ]     = '';
			$data[ 'occurrences_end_date_' . $j ]       = '';
			$data[ 'occurrences_start_datetime_' . $j ] = '';
			$data[ 'occurrences_end_datetime_' . $j ]   = '';
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	private function get_event_detail( $info = array(
		'terms'       => true,
		'occurrences' => true,
	) ) {

		$event_id       = get_the_ID();
		$startHour      = get_post_meta( $event_id, 'mec_start_time_hour', true );
		$startMinutes   = get_post_meta( $event_id, 'mec_start_time_minutes', true );
		$startMinutes   = ( $startMinutes == '0' || $startMinutes == '5' ) ? '0' . $startMinutes : $startMinutes;
		$startAMPM      = get_post_meta( $event_id, 'mec_start_time_ampm', true );
		$eventStartTime = $startHour . ':' . $startMinutes . $startAMPM;
		$eventStartDate = get_post_meta( $event_id, 'mec_start_date', true );

		$eventStart = $eventStartDate . ' ' . $eventStartTime;

		// End Time
		$endHour      = get_post_meta( $event_id, 'mec_end_time_hour', true );
		$endMinutes   = get_post_meta( $event_id, 'mec_end_time_minutes', true );
		$endMinutes   = ( $endMinutes == '0' || $endMinutes == '5' ) ? '0' . $endMinutes : $endMinutes;
		$endAMPM      = get_post_meta( $event_id, 'mec_end_time_ampm', true );
		$eventEndTime = $endHour . ':' . $endMinutes . $endAMPM;
		$eventEndDate = get_post_meta( $event_id, 'mec_end_date', true );

		$eventEnd = $eventEndDate . ' ' . $eventEndTime;

		$data['id']                   = $event_id;
		$data['event_id']             = $event_id;
		$data['event_title']          = html_entity_decode(get_the_title( $event_id ),ENT_HTML5,'UTF-8');
		$data['event_link']           = get_the_permalink( $event_id );
		$data['event_thumbnail_link'] = get_the_post_thumbnail_url( $event_id ) ? get_the_post_thumbnail_url( $event_id ) : 'null';
		$data['event_start']          = date_i18n( $this->datetimeFormat, strtotime( $eventStart ) );
		$data['event_end']            = date_i18n( $this->datetimeFormat, strtotime( $eventEnd ) );
		$data['event_read_more']      = get_post_meta( $event_id, 'mec_read_more', true );
		$data['event_more_info']      = get_post_meta( $event_id, 'mec_more_info', true );
		$data['event_cost']           = get_post_meta( $event_id, 'mec_cost', true );
		$data['event_excerpt']        = html_entity_decode(get_the_excerpt( $event_id ),ENT_HTML5,'UTF-8');

		$this->get_event_fixed_detail( $event_id, $data );

		if ( isset( $info['terms'] ) && $info['terms'] ) {

			$data = array_merge(
				$data,
				$this->get_event_terms_detail( $event_id )
			);
		}

		if ( isset( $info['occurrences'] ) && $info['occurrences'] ) {

			$data = array_merge(
				$data,
				$this->get_event_occurrences_info( $event_id )
			);
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	private function get_tickets_detail( $book_tickets, $tickets, &$data ) {
		$i = 1;
		foreach ( $book_tickets as $book_ticket_id ) {
			if ( empty( $book_ticket_id ) ) {

				continue;
			}
			$ticket = isset( $tickets[ $book_ticket_id ] ) ? $tickets[ $book_ticket_id ] : new \MEC_Core\Tickets\Ticket( array() );

			if ( $ticket ) {
				$data[ 'ticket_name_' . $i ]              = $ticket->get_data( 'name' );
				$data[ 'ticket_start_time_hour_' . $i ]   = $ticket->get_data( 'ticket_start_time_hour' );
				$data[ 'ticket_start_time_minute_' . $i ] = $ticket->get_data( 'ticket_start_time_minute' );
				$data[ 'ticket_start_time_ampm_' . $i ]   = $ticket->get_data( 'ticket_start_time_ampm' );

				$data[ 'ticket_end_time_hour_' . $i ]   = $ticket->get_data( 'ticket_end_time_hour' );
				$data[ 'ticket_end_time_minute_' . $i ] = $ticket->get_data( 'ticket_end_time_minute' );
				$data[ 'ticket_end_time_ampm_' . $i ]   = $ticket->get_data( 'ticket_end_time_ampm' );

				$data[ 'ticket_start_time_' . $i ]  = $ticket->get_time( 'start' );
				$data[ 'ticket_end_time_' . $i ]    = $ticket->get_time( 'end' );
				$data[ 'ticket_time_' . $i ]        = $ticket->get_time( 'all' );
				$data[ 'ticket_description_' . $i ] = $ticket->get_data( 'description' );
				$data[ 'ticket_price_' . $i ]       = $ticket->get_data( 'price' );
				$i++;
			}
		}

		for ( $j = $i;$j <= $this->max_tickets;$j++ ) {
			$data[ 'ticket_name_' . $j ]              = '';
			$data[ 'ticket_start_time_hour_' . $j ]   = '';
			$data[ 'ticket_start_time_minute_' . $j ] = '';
			$data[ 'ticket_start_time_ampm_' . $j ]   = '';

			$data[ 'ticket_end_time_hour_' . $j ]   = '';
			$data[ 'ticket_end_time_minute_' . $j ] = '';
			$data[ 'ticket_end_time_ampm_' . $j ]   = '';

			$data[ 'ticket_start_time_' . $j ]  = '';
			$data[ 'ticket_end_time_' . $j ]    = '';
			$data[ 'ticket_time_' . $j ]        = '';
			$data[ 'ticket_description_' . $j ] = '';
			$data[ 'ticket_price_' . $j ]       = '';
		}
	}

	/**
	 * @since     2.0.0
	 */
	private function get_attendees_detail( $event_id, $attendees, $tickets, &$data ) {

		$i          = 1;
		$r_f        = array();
		$reg_fields = $this->get_reg_fields( $event_id );
		foreach ( $attendees as $attendee ) {
			$data[ 'attendee_name_' . $i ]         = isset( $attendee['name'] ) ? $attendee['name'] : '';
			$data[ 'attendee_email_' . $i ]        = isset( $attendee['email'] ) ? $attendee['email'] : '';
			$ticket_id                             = isset( $attendee['id'] ) ? $attendee['id'] : '';
			$data[ 'attendee_ticket_name_' . $i ]  = isset( $tickets[ $ticket_id ] ) && is_a( $tickets[ $ticket_id ], '\MEC_Core\Tickets\Ticket' ) ? $tickets[ $ticket_id ]->get_data( 'name' ) : '';
			$data[ 'attendee_ticket_count_' . $i ] = isset( $attendee['count'] ) ? $attendee['count'] : 0;
			if ( isset( $attendee['reg'] ) && is_array( $attendee['reg'] ) ) {

				foreach ( $attendee['reg'] as $field => $value ) {

					$field_args = isset( $reg_fields[ $field ] ) ? $reg_fields[ $field ] : false;
					if ( ! $field_args || ! isset( $field_args['label'] ) ) {

						continue;
					}

					$label = strtolower( trim( $field_args['label'] ) );

					$label                                    = str_replace( ' ', '_', $label );
					$data[ 'attendee__' . $label . '_' . $i ] = is_array( $value ) ? implode(', ',$value) : $value;
					$r_f[ 'attendee__' . $label ]             = 'attendee__' . $label;
				}
			}
			$i++;
		}

		for ( $j = $i;$j <= $this->max_attendees;$j++ ) {

			$data[ 'attendee_name_' . $j ]         = '';
			$data[ 'attendee_email_' . $j ]        = '';
			$data[ 'attendee_ticket_name_' . $j ]  = '';
			$data[ 'attendee_ticket_count_' . $j ] = '';

			foreach ( (array) $r_f as $field_id ) {

				$data[ $field_id . '_' . $j ] = '';
			}
		}

	}

	private function get_booking_fixed_detail( $booking_id, &$data ) {

		$fields         = $this->get_bfixed_fields( $booking_id );
		$transaction_id = get_post_meta( $booking_id, 'mec_transaction_id', true );
		$transaction    = get_option( $transaction_id, array() );
		$first          = isset( $transaction['fields'] ) && ! empty( $transaction['fields'] ) && is_array( $transaction['fields'] ) ? current( $transaction['fields'] ) : false;
		if ( is_array( $first ) ) {

			foreach ( $transaction['fields'] as $field_id => $value ) {

				$field_args = isset( $fields[ $field_id ] ) ? $fields[ $field_id ] : false;
				if ( ! $field_args || ! isset( $field_args['label'] ) ) {

					continue;
				}

				$label = strtolower( trim( $field_args['label'] ) );

				$label = str_replace( ' ', '_', $label );

				$data[ 'booking_fixed_field__' . $label ] = $value;
			}
		} else {
			$this->get_booking_fixed_fields_for_set_setting( $data, '' );
		}

	}

	private function get_event_fixed_detail( $event_id, &$data ) {

		global $MEC_Main;
		if ( empty( $MEC_Main ) ) {

			$MEC_Main = new \MEC_main();
		}

		$event_fields = get_post_meta( $event_id, 'mec_fields', true );
		if ( empty( $event_fields ) || ! is_array( $event_fields ) ) {

			return $data;
		}

		$event_fixed_fields = $MEC_Main->get_event_fields();
		$event_fixed_fields = is_array( $event_fixed_fields ) ? $event_fixed_fields : false;

		if ( $event_fixed_fields ) {

			foreach ( $event_fields as $field_id => $value ) {

				$field_args = isset( $event_fixed_fields[ $field_id ] ) ? $event_fixed_fields[ $field_id ] : false;
				if ( ! $field_args || ! isset( $field_args['label'] ) ) {

					continue;
				}

				$label = strtolower( trim( $field_args['label'] ) );

				$label = str_replace( ' ', '_', $label );

				$data[ 'event_fixed_field__' . $label ] = $value;
			}
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	private function get_event_terms_detail( $event_id, $taxonomies = false ) {

		if ( ! $taxonomies ) {

			$taxonomies = array(
				'category',
				'location',
				'organizer',
				'speaker',
			);
		}

		$data = array();
		foreach ( (array) $taxonomies as $tax ) {

			$taxonomy = 'mec_' . $tax;
			$terms    = wp_get_post_terms( $event_id, $taxonomy );

			if(is_wp_error($terms)){

				continue;
			}

			/**
			 * TODO:Convert to Class
			 */
			switch ( $tax ) {
				case 'location':
					$i = 1;
					foreach ( $terms as $term ) {
						$term_id = $term->term_id;
						// $data[ $tax . '_id_' . $i ]    = $term_id;
						$data[ $tax . '_name_' . $i ]      = $term->name;
						$data[ $tax . '_address_' . $i ]   = get_term_meta( $term_id, 'address', true );
						$data[ $tax . '_latitude_' . $i ]  = get_term_meta( $term_id, 'latitude', true );
						$data[ $tax . '_longitude_' . $i ] = get_term_meta( $term_id, 'longitude', true );
						$data[ $tax . '_url_' . $i ]       = get_term_meta( $term_id, 'url', true );
						$data[ $tax . '_thumbnail_' . $i ] = get_term_meta( $term_id, 'thumbnail', true );
						$i++;
					}

					for ( $j = $i;$j <= $this->max_terms;$j++ ) {
						// $data[ $tax . '_id_' . $j ]    = '';
						$data[ $tax . '_name_' . $j ]      = '';
						$data[ $tax . '_address_' . $j ]   = '';
						$data[ $tax . '_latitude_' . $j ]  = '';
						$data[ $tax . '_longitude_' . $j ] = '';
						$data[ $tax . '_url_' . $j ]       = '';
						$data[ $tax . '_thumbnail_' . $j ] = '';
					}
					break;
				case 'organizer':
					$i = 1;
					foreach ( $terms as $term ) {
						$term_id = $term->term_id;
						// $data[ $tax . '_id_' . $i ]    = $term_id;
						$data[ $tax . '_name_' . $i ]      = $term->name;
						$data[ $tax . '_tel_' . $i ]       = get_term_meta( $term_id, 'tel', true );
						$data[ $tax . '_email_' . $i ]     = get_term_meta( $term_id, 'email', true );
						$data[ $tax . '_url_' . $i ]       = get_term_meta( $term_id, 'url', true );
						$data[ $tax . '_thumbnail_' . $i ] = get_term_meta( $term_id, 'thumbnail', true );
						$i++;
					}

					for ( $j = $i;$j <= $this->max_terms;$j++ ) {
						$data[ $tax . '_name_' . $j ]      = '';
						$data[ $tax . '_tel_' . $j ]       = '';
						$data[ $tax . '_email_' . $j ]     = '';
						$data[ $tax . '_url_' . $j ]       = '';
						$data[ $tax . '_thumbnail_' . $j ] = '';
					}
					break;
				case 'speaker':
					$i = 1;
					foreach ( $terms as $term ) {
						$term_id = $term->term_id;
						// $data[ $tax . '_id_' . $i ]    = $term_id;
						$data[ $tax . '_name_' . $i ]      = $term->name;
						$data[ $tax . '_job_title_' . $i ] = get_term_meta( $term_id, 'job_title', true );
						$data[ $tax . '_tel_' . $i ]       = get_term_meta( $term_id, 'tel', true );
						$data[ $tax . '_email_' . $i ]     = get_term_meta( $term_id, 'email', true );
						$data[ $tax . '_facebook_' . $i ]  = get_term_meta( $term_id, 'facebook', true );
						$data[ $tax . '_twitter_' . $i ]   = get_term_meta( $term_id, 'twitter', true );
						$data[ $tax . '_gplus_' . $i ]     = get_term_meta( $term_id, 'gplus', true );
						$data[ $tax . '_thumbnail_' . $i ] = get_term_meta( $term_id, 'thumbnail', true );
						$i++;
					}

					for ( $j = $i;$j <= $this->max_terms;$j++ ) {
						$data[ $tax . '_name_' . $j ]      = '';
						$data[ $tax . '_job_title_' . $j ] = '';
						$data[ $tax . '_tel_' . $j ]       = '';
						$data[ $tax . '_email_' . $j ]     = '';
						$data[ $tax . '_facebook_' . $j ]  = '';
						$data[ $tax . '_twitter_' . $j ]   = '';
						$data[ $tax . '_gplus_' . $j ]     = '';
						$data[ $tax . '_thumbnail_' . $j ] = '';
					}
					break;
				case 'category':
				default:
					$i = 1;
					foreach ( $terms as $term ) {
						// $data[ $tax . '_id_' . $i ]    = $term_id;
						$data[ $tax . '_name_' . $i ] = $term->name;
						$i++;
					}

					for ( $j = $i;$j <= $this->max_terms;$j++ ) {
						// $data[ $tax . '_id_' . $j ]    = '';
						$data[ $tax . '_name_' . $j ] = '';
					}
					break;
			}
		}

		return $data;
	}

	// methods for set setting start
	/**
	 * @since     2.0.0
	 */
	public function get_new_event_info_for_set_setting() {

		$data = $this->get_event_detail_for_set_setting();

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_modified_event_info_for_set_setting() {

		$data = $this->get_event_detail_for_set_setting();

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_new_booking_info_for_set_setting() {

		$data = array(
			'id'                          => 99,
			'booking_title'               => 'Webnus - user1@gmail.com',
			'booking_email'               => 'user1@gmail.com',
			'booking_name'                => 'Webnus',
			'booking_date'                => '2021-03-03T00:00:00UTC',
			'booking_order_time'          => '2021-03-02T14:58:13UTC',
			'booking_event_title'         => '2 days',
			'booking_attendees'           => 3,
			'booking_price'               => 'Free',
			'booking_price_label'         => 'Free',
			'booking_confirmation_status' => 'Pending',
			'booking_verification_status' => 'Waiting',
			'booking_transaction_id'      => 'EKW28379',
			'event_start_on'              => '2021-03-03T08:00:00UTC',
			'event_end_on'                => '2021-03-03T18:00:00UTC',
		);

		$this->get_booking_fixed_fields_for_set_setting( $data, 'label' );

		$max_attendees = $this->max_attendees;
		$max_tickets   = $this->max_tickets;

		for ( $i = 1;$i <= $max_tickets;$i++ ) {

			$ticket = array(
				'ticket_name_' . $i              => 'Ticket Name',
				'ticket_start_time_hour_' . $i   => '08',
				'ticket_start_time_minute_' . $i => '0',
				'ticket_start_time_ampm_' . $i   => 'AM',
				'ticket_end_time_hour_' . $i     => '06',
				'ticket_end_time_minute_' . $i   => '0',
				'ticket_end_time_ampm_' . $i     => 'PM',
				'ticket_start_time_' . $i        => '08:00',
				'ticket_end_time_' . $i          => '18:00',
				'ticket_time_' . $i              => '08:00 18:00',
				'ticket_description_' . $i       => 'Description',
				'ticket_price_' . $i             => '99',
			);

			$data = array_merge( $data, $ticket );
		}

		$reg_fields = $this->get_reg_fields( 0 );

		for ( $i = 1;$i <= $max_attendees;$i++ ) {

			$attendees = array(
				'attendee_name_' . $i         => 'Attendee Name',
				'attendee_email_' . $i        => 'user' . $i . '@gmail.com',
				'attendee_ticket_name_' . $i  => 'Ticket Name',
				'attendee_ticket_count_' . $i => '1',
			);

			foreach ( $reg_fields as $field_id => $field_args ) {

				$type = isset( $field_args['type'] ) ? $field_args['type'] : false;

				if ( ! $type || ! is_numeric( $field_id ) || in_array( $type, array( 'mec_email', 'name' ) ) ) {
					continue;
				}

				if ( ! isset( $field_args['label'] ) ) {
					continue;
				}

				$label = str_replace( ' ', '_', trim( $field_args['label'] ) );
				$attendees[ 'attendee__' . $label . '_' . $i ] = $label;
			}

			$data = array_merge( $data, $attendees );
		}

		return $data;
	}

	public function get_booking_fixed_fields_for_set_setting( &$data, $v = 'label' ) {

		$fixed_fields = $this->get_bfixed_fields( 0 );
		if ( ! empty( $fixed_fields ) && is_array( $fixed_fields ) ) {
			foreach ( $fixed_fields as $field_id => $field_args ) {

				$type = isset( $field_args['type'] ) ? $field_args['type'] : false;

				if ( ! $type || ! is_numeric( $field_id ) || in_array( $type, array( 'mec_email', 'name' ) ) ) {
					continue;
				}

				if ( ! isset( $field_args['label'] ) ) {
					continue;
				}

				$label                                    = strtolower( trim( $field_args['label'] ) );
				$label                                    = str_replace( ' ', '_', $field_args['label'] );
				$data[ 'booking_fixed_field__' . $label ] = $v === 'label' ? $label : $v;
			}
		}
	}

	/**
	 * @since     2.0.0
	 */
	public function get_cancelled_event_info_for_set_setting() {

		$data                              = $this->get_event_detail_for_set_setting();
		$data['event_reason_cancellation'] = __( 'This Event Cancelled Because ...', 'mec-zapier' );

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	private function get_event_detail_for_set_setting( $info = array(
		'terms'       => true,
		'occurrences' => true,
	) ) {

		$data = array(
			'id'                   => 1,
			'event_title'          => 'Event Name',
			'event_link'           => 'https://webnus.net/',
			'event_thumbnail_link' => 'null',
			'event_start'          => date_i18n( $this->datetimeFormat ),
			'event_end'            => date_i18n( $this->datetimeFormat, current_time( 'timestamp' ) + 3600 ),
			'event_read_more'      => '',
			'event_more_info'      => '',
			'event_cost'           => '100',
			'event_excerpt'        => 'Excerpt Text',
		);

		if ( isset( $info['terms'] ) && $info['terms'] ) {

			$data = array_merge(
				$data,
				$this->get_event_terms_detail_for_set_settings()
			);
		}

		if ( isset( $info['occurrences'] ) && $info['occurrences'] ) {

			$data = array_merge(
				$data,
				$this->get_event_occurrences_for_set_setting()
			);
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_event_terms_detail_for_set_settings() {

		$taxonomies = array(
			'category',
			'location',
			'organizer',
			'speaker',
		);

		$data = array();
		foreach ( (array) $taxonomies as $tax ) {

			$thumbnail_url = 'http://webnus.net/wp-content/uploads/2018/07/webnus-logo.svg';
			$email         = 'info@webnus.net';
			$url           = 'http://webnus.net';

			switch ( $tax ) {
				case 'location':
					for ( $i = 1; $i <= $this->max_terms;$i++ ) {

						$data[ $tax . '_name_' . $i ]      = 'Location Name ' . $i;
						$data[ $tax . '_address_' . $i ]   = 'Address ' . $i;
						$data[ $tax . '_latitude_' . $i ]  = '123456789';
						$data[ $tax . '_longitude_' . $i ] = '123456789';
						$data[ $tax . '_url_' . $i ]       = $url;
						$data[ $tax . '_thumbnail_' . $i ] = $thumbnail_url;
					}
					break;
				case 'organizer':
					for ( $i = 1; $i <= $this->max_terms;$i++ ) {

						$data[ $tax . '_name_' . $i ]      = 'Organizer Name ' . $i;
						$data[ $tax . '_tel_' . $i ]       = '+123456789';
						$data[ $tax . '_email_' . $i ]     = $email;
						$data[ $tax . '_url_' . $i ]       = $url;
						$data[ $tax . '_thumbnail_' . $i ] = $thumbnail_url;
					}
					break;
				case 'speaker':
					for ( $i = 1; $i <= $this->max_terms;$i++ ) {

						$data[ $tax . '_name_' . $i ]      = 'Speaker Name ' . $i;
						$data[ $tax . '_job_title_' . $i ] = 'Job Title ' . $i;
						$data[ $tax . '_tel_' . $i ]       = '+123456789';
						$data[ $tax . '_email_' . $i ]     = $email;
						$data[ $tax . '_facebook_' . $i ]  = 'https://www.facebook.com/Webnus/';
						$data[ $tax . '_twitter_' . $i ]   = 'https://twitter.com/webnus/';
						$data[ $tax . '_gplus_' . $i ]     = 'https://www.google.com/';
						$data[ $tax . '_thumbnail_' . $i ] = $thumbnail_url;
					}
					break;
				case 'category':
				default:
					for ( $i = 1; $i <= $this->max_terms;$i++ ) {

						$data[ $tax . '_name_' . $i ] = 'Category Name ' . $i;
					}
					break;
			}
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_all_event_info_for_set_setting() {

		$args = $this->get_query_args( array() );

		$posts_per_page = isset( $args['posts_per_page'] ) ? $args['posts_per_page'] : 5;
		$paged          = isset( $args['paged'] ) ? $args['paged'] : 1;

		$end   = $posts_per_page * $paged;
		$start = $end - ( $posts_per_page - 1 );
		for ( $i = $start;$i <= $end;$i++ ) {

			$d                 = $this->get_event_detail_for_set_setting();
			$d['id']           = $i;
			$d['event_title'] .= " $i";
			$data[]            = $d;
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	public function get_list_event_info_for_set_setting() {

		$args = $this->get_query_args( array() );

		$posts_per_page = isset( $args['posts_per_page'] ) ? $args['posts_per_page'] : 5;
		$paged          = isset( $args['paged'] ) ? $args['paged'] : 1;

		$end   = $posts_per_page * $paged;
		$start = $end - ( $posts_per_page - 1 );
		$data  = array(
			'id' => $posts_per_page . $paged,
		);
		$j     = 0;
		for ( $i = $start;$i <= $end;$i++ ) {
			$j++;
			$d                 = $this->get_event_detail_for_set_setting( array( 'terms' => true ) );
			$d['id']           = $i;
			$d['event_title'] .= " $i";
			foreach ( $d as $k => $v ) {

				$data[ $k . '_' . $j ] = $v;
			}
		}

		return $data;
	}

	/**
	 * @since     2.0.0
	 */
	private function get_event_occurrences_for_set_setting( $event_id = false ) {

		$data                  = array();
		$max_event_occurrences = $this->get_request_param( 'limit' );
		if ( ! $max_event_occurrences ) {

			$max_event_occurrences = $this->max_event_occurrences;
		}

		$start = strtotime( $this->get_request_param( 'start_datetime' ) );
		if ( empty( $start ) ) {
			$start = current_time( 'timestamp' );
		}
		$start = date_i18n( 'Y-m-d H:i:s', $start );

		$datetime = new \DateTime( $start );
		for ( $i = 1;$i <= $max_event_occurrences;$i++ ) {

			$occurrence = array(
				'occurrences_start_date_' . $i     => $datetime->format( $this->dateFormat ),
				'occurrences_end_date_' . $i       => $datetime->format( $this->dateFormat ),
				'occurrences_start_datetime_' . $i => $datetime->format( $this->datetimeFormat ),
				'occurrences_end_datetime_' . $i   => $datetime->format( $this->datetimeFormat ),
			);

			$datetime->add( new DateInterval( 'P1D' ) );

			$data = array_merge( $data, $occurrence );
		}

		if ( $event_id ) {
			$data['id'] = $event_id;
		}

		return $data;
	}
	// methods for set setting end

	/**
	 * @since     2.0.0
	 */
	private function get_bfixed_fields( $event_id ) {

		global $MEC_Main;
		if ( empty( $MEC_Main ) ) {

			$MEC_Main = new \MEC_main();
		}

		return $MEC_Main->get_bfixed_fields( $event_id );
	}

	/**
	 * @since     2.0.0
	 */
	private function get_reg_fields( $event_id ) {

		global $MEC_Main;
		if ( empty( $MEC_Main ) ) {

			$MEC_Main = new \MEC_main();
		}

		return $MEC_Main->get_reg_fields( $event_id );
	}

	/**
	 * @since     2.0.0
	 */
	private function get_event_occurrences( $event_id, $start, $limit = 10 ) {

		$k           = 'mec-occurrences-' . $start . $limit;
		$occurrences = wp_cache_get( $k );
		if ( empty( $occurrences ) ) {

			$feature     = new \MEC_feature_occurrences();
			$occurrences = $feature->get_dates( $event_id, $start, $limit );
			wp_cache_set( $k, $occurrences, 'mec-occurrences', 120 );
		}

		return $occurrences;
	}

	/**
	 * Register Autoload Files
	 *
	 * @since     2.0.0
	 */
	public function init() {
		if ( ! class_exists( '\MEC_Zapier\Autoloader' ) ) {
			return;
		}
	}
} //SetRoutes
// add_filter('rest_request_after_callbacks',function($response, $handler, $request){

// 	wp_die(print_r($response));
// },10,3);