<?php
/**
 * Class MEC_RSVP Core
 *
 * @since           1.0.0
 */

namespace MEC_RSVP;

use MEC\Settings\Settings;
use MEC_RSVP\Conversation\Conversation;
use MEC_RSVP\RSVP\AjaxProcessHandler;
use MEC_RSVP\RSVP\DisplayForm;
use MEC_RSVP\RSVP\EventMetaBox;
use MEC_RSVP\RSVP\EventRSVP;
use MEC_RSVP\RSVP\PostTypeFactory;
use MEC_RSVP\RSVP\RSVP;
use MEC_RSVP\RSVP\RSVPNotifications;
use MEC_RSVP\RSVP\RSVPs;
use MEC_RSVP\RSVP\ShortcodeRSVPForm;
use MEC_RSVP\RSVP\ShortcodeRSVPList;

/**
 * Core Class in plugin
 */
final class Base {

	/**
	 * Plugin Version
	 *
	 * @var string
	 */
	public static $version = '1.0.0';

	/**
	 * Session instance
	 *
	 * @var bool
	 */
	protected static $instance;

	/**
	 * Constructor
	 */
	public function __construct() {

		$this->define();
		$this->includes();
		$this->init_hooks();
		$this->admin();
		$this->enqueue_scripts();
	}

	/**
	 * Instance
	 *
	 * @return self()
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Set Constants
	 *
	 * @return void
	 */
	public function define() {

		define( 'MEC_RSVP_PD', plugin_dir_path( MEC_RSVP_FILE ) );
		define( 'MEC_RSVP_PDI', plugin_dir_path( MEC_RSVP_FILE ) . 'src/' );
		define( 'MEC_RSVP_PU_JS', plugins_url( 'assets/js/', MEC_RSVP_FILE ) );
		define( 'MEC_RSVP_PU_CSS', plugins_url( 'assets/css/', MEC_RSVP_FILE ) );
		define( 'MEC_RSVP_PU_IMG', plugins_url( 'assets/img/', MEC_RSVP_FILE ) );
		define( 'MEC_RSVP_PU_FONTS', plugins_url( 'assets/fonts/', MEC_RSVP_FILE ) );
		define( 'MEC_RSVP_TEMPLATES', plugin_dir_path( MEC_RSVP_FILE ) . 'templates/' );

		define( 'MEC_RSVP_VERSION', self::$version );

	}

	/**
	 * Include Files
	 *
	 * @return void
	 */
	public function includes() {

		require_once MEC_RSVP_PDI . 'functions.php';
	}

	/**
	 * Include Files If is Admin
	 *
	 * @return void
	 */
	public function admin() {

		if ( !is_admin() ) {

			return;
		}

		Admin::getInstance()->init();
	}

	/**
	 * Load Plugin Text Domain
	 */
	public function load_plugin_textdomain() {

		load_plugin_textdomain(
			'mec-rsvp',
			false,
			dirname( plugin_basename( MEC_RSVP_FILE ) ) . '/languages'
		);
	}

	/**
	 * Register actions enqueue scripts
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		// add_action( 'wp_enqueue_scripts', '\MEC_RSVP\Scripts::enqueue', 99 );
	}

	/**
	 * Add Hooks - Actions and Filters
	 *
	 * @return void
	 */
	public function init_hooks(): void {

		add_action( 'init', array( $this, 'init' ) );
		if(mec_rsvp_is_enabled()){

			add_action( 'mec_before_booking_form', array( $this, 'display_rsvp_form' ) );//TODO:change action
			add_filter('mec_get_custom_form_fields',[__CLASS__,'get_rsvp_reg_fields_form'],10,6);

			add_action( 'init', array( $this, 'register_endpoints' ) );
			add_action( 'mec_before_main_content', array( $this, 'do_endpoints' ) );

			PostTypeFactory::getInstance()->init();
			Conversation::getInstance()->init();
			RSVPNotifications::getInstance()->init();

			// fes form
			EventMetaBox::getInstance()->fes_init();

			if(wp_doing_ajax()){

				AjaxProcessHandler::getInstance()->init();
			}

			add_shortcode( 'mec-my-rsvps', [ ShortcodeRSVPList::class, 'render' ] );
			add_shortcode( 'mec-rsvp', [ ShortcodeRSVPForm::class, 'render' ] );

			add_action('wp_enqueue_scripts',[ DisplayForm::getInstance(),'enqueue' ]);

		}

		do_action( 'MEC_RSVP_init' );
	}


	public static function get_rsvp_reg_fields_form($fields, $event_id,$form_id, $group_id, $primary_group_id,$translated_event_id){

		if(!$event_id){

			return $fields;
		}

		if(false === strpos($form_id,'rsvp')){

			return $fields;
		}

		$global_inheritance = get_post_meta( $event_id, 'mec_rsvp_form_fields_global_inheritance', true );
		if ( (bool)$global_inheritance ) {

			return $fields;
		}

		$primary_group_id_status_key = 'form_'. str_replace(['_fields','_bfixed','_reg'],'',$primary_group_id) . '_status';
		$group_id_status_key = 'form_' . str_replace(['_fields','_bfixed','_reg'],'',$primary_group_id) . '_status';
		$event_form_options = get_post_meta( $event_id, 'mec_rsvp_forms_options', true );
		$primary_form_group_status = isset($event_form_options[$primary_group_id_status_key]) && 'enable' === $event_form_options[$primary_group_id_status_key]  ? true : false;
		$form_group_status = isset($event_form_options[$group_id_status_key]) && 'enable' === $event_form_options[$group_id_status_key]  ? true : false;

		//primary
		$event_fields = [];
		if ( $primary_form_group_status && isset($event_form_options[$primary_group_id]) && is_array( $event_form_options[$primary_group_id] ) ) {

			$event_fields = $event_form_options[$primary_group_id];
		}elseif( $form_group_status && isset($event_form_options[$group_id]) && is_array( $event_form_options[$group_id] ) ){

			$event_fields = $event_form_options[$group_id];
		}

		if(!empty($event_fields)){

			$fields = $event_fields;
		}

		if ( $translated_event_id && $event_id !== $translated_event_id ) {
			$translated_fields = get_post_meta( $translated_event_id, 'mec_reg_fields', true );
			if ( !is_array( $translated_fields ) ) {
				$translated_fields = array();
			}

			foreach ( $translated_fields as $field_id => $translated_field ) {
				if ( !isset( $fields[ $field_id ] ) ) {
					continue;
				}
				if ( isset( $translated_field['label'] ) && trim( $translated_field['label'] ) ) {
					$fields[ $field_id ]['label'] = $translated_field['label'];
				}
				if ( isset( $translated_field['options'] ) && is_array( $translated_field['options'] ) ) {
					$fields[ $field_id ]['options'] = $translated_field['options'];
				}
			}
		}

		return $fields;
	}

	/**
	 * @hooked
	 */
	public function display_rsvp_form( $event_id ) {

		$is_popup = isset( $_REQUEST['action'] ) && 'mec_load_single_page' === $_REQUEST['action'];
		if( !$event_id ){

			$event_id = $is_popup ? absint( $_REQUEST['id'] ) : get_the_ID();
		}

		if('mec-events' !== get_post_type( $event_id )){

			return;
		}

		if(!is_single() && !$is_popup ){

			return;
		}

		echo do_shortcode( '[mec-rsvp event-id="'.$event_id.'"]' );

		if( $is_popup ){

			echo '<script>mec_rsvp_module_init(jQuery);</script>';
		}
	}

	/**
	 * Init
	 *
	 * @return void
	 */
	public function init(): void {

		$this->load_plugin_textdomain();
	}

	/**
	 * Get MEC_Main
	 *
	 * @return \MEC_Main
	 */
	public static function get_main() {

		global $MEC_Main;
		if ( !is_a( $MEC_Main, \MEC_main::class ) ) {

			$MEC_Main = new \MEC_Main();
		}

		return $MEC_Main;
	}

	public function register_endpoints(){

		if(!defined('EP_MEC_EVENTS')) {
			define('EP_MEC_EVENTS', 555);
		}
		add_rewrite_endpoint('rsvp-verify', EP_MEC_EVENTS);
        add_rewrite_endpoint('rsvp-cancel', EP_MEC_EVENTS);

		add_rewrite_endpoint('rsvp-response', EP_MEC_EVENTS);
	}

	public function do_endpoints(){

		if(get_query_var('rsvp-verify')){

            $key = sanitize_text_field(get_query_var('rsvp-verify'));

            $db = \MEC\Base::get_main()->getDB();
            $rsvp_id = $db->select("SELECT `post_id` FROM `#__postmeta` WHERE `meta_key`='mec_verification_key' AND `meta_value`='$key'", 'loadResult');

            if(!$rsvp_id) return false;

			$rsvp = new RSVP($rsvp_id);
            $status = $rsvp->get_verification_status();
            if($status == '1'){

                $status_user = get_post_meta($rsvp_id, 'mec_verified_user', true);
                if(trim($status_user) == '') $status_user = 0;

                if(!$status_user){
                    // User Status
                    update_post_meta($rsvp_id, 'mec_verified_user', 1);

                    echo '<p class="mec-success">'.__('Your RSVP has been verified successfully!', 'mec-rsvp').'</p>';
                    return false;
                } else {
                    echo '<p class="mec-success">'.__('Your rsvp already verified!', 'mec-rsvp').'</p>';
                    return false;
                }
            }


            if($rsvp->set_verification_status('verified')){

				echo '<p class="mec-success">'.__('Your RSVP has been verified successfully!', 'mec-rsvp').'</p>';
			} else {

				echo '<p class="mec-error">'.__('Your RSVP cannot be verified!', 'mec-rsvp').'</p>';
			}
        } elseif(get_query_var('rsvp-cancel')) {
            $key = sanitize_text_field(get_query_var('rsvp-cancel'));

            $db = \MEC\Base::get_main()->getDB();
            $rsvp_id = $db->select("SELECT `post_id` FROM `#__postmeta` WHERE `meta_key`='mec_cancellation_key' AND `meta_value`='$key'", 'loadResult');

            if(!$rsvp_id) return false;

			$rsvp = new RSVP($rsvp_id);

            $status = $rsvp->get_verification_status();
            if($status == '-1'){

                $status_user = get_post_meta($rsvp_id, 'mec_canceled_user', true);
                if(trim($status_user) == '') {

					$status_user = 0;
				}

                if(!$status_user){
                    // User Status
                    update_post_meta($rsvp_id, 'mec_canceled_user', 1);

                    echo '<p class="mec-success">'.__('Your RSVP was successfully canceled.', 'mec-rsvp').'</p>';
                    return false;
                } else {
                    echo '<p class="mec-success">'.__('Your RSVP was already canceled!', 'mec-rsvp').'</p>';
                    return false;
                }
            }

            $timestamps = explode(':', $rsvp->get_event_times());
            $start = $timestamps[0];
            $end = $timestamps[1];

            $right_now = current_time('timestamp', 0);
            if($right_now >= $end){
                echo '<p class="mec-error">'.__('The event is already over!', 'mec-rsvp').'</p>';
                return false;
            }

            $settings = Settings::getInstance()->get_settings();

            $cancellation_period_from = isset($settings['rsvp_cancellation_period_from']) ? $settings['rsvp_cancellation_period_from'] : 0;
            $cancellation_period_to = isset($settings['rsvp_cancellation_period_time']) ? $settings['rsvp_cancellation_period_time'] : 0;
            $cancellation_period_p = isset($settings['rsvp_cancellation_period_p']) ? $settings['rsvp_cancellation_period_p'] : 'hour';
            $cancellation_period_type = isset($settings['rsvp_cancellation_period_type']) ? $settings['rsvp_cancellation_period_type'] : 'before';

            if($cancellation_period_from or $cancellation_period_to){
                if($cancellation_period_from){
                    if($cancellation_period_type == 'before'){
						$min_time = ($start - ($cancellation_period_from * ($cancellation_period_p == 'hour' ? 3600 : 86400)));
					}else{
						$min_time = ($start + ($cancellation_period_from * ($cancellation_period_p == 'hour' ? 3600 : 86400)));
					}

                    if($right_now < $min_time){
                        echo '<p class="mec-error">'.__("The cancelation window has not started yet.", 'mec-rsvp').'</p>';
                        return false;
                    }
                }

                if($cancellation_period_to){

                    if($cancellation_period_type == 'before'){
						$max_time = ($start - ($cancellation_period_to * ($cancellation_period_p == 'hour' ? 3600 : 86400)));
					} else {
						$max_time = ($start + ($cancellation_period_to * ($cancellation_period_p == 'hour' ? 3600 : 86400)));
					}

                    if($right_now > $max_time){

                        echo '<p class="mec-error">'.__("The cancelation window has passed.", 'mec-rsvp').'</p>';
                        return false;
                    }
                }
            }

            if($rsvp->set_verification_status('canceled')){

                echo '<p class="mec-success">'.__('Your rsvp successfully canceled.', 'mec-rsvp').'</p>';

                $cancel_page = (isset($settings['rsvp_cancel_page']) and trim($settings['rsvp_cancel_page'])) ? $settings['rsvp_cancel_page'] : NULL;
                $cancel_page_url = get_permalink($cancel_page);
                $cancel_page_time = (isset($settings['rsvp_cancel_page_time']) and trim($settings['rsvp_cancel_page_time']) != '') ? $settings['rsvp_cancel_page_time'] : 2500;

                if($cancel_page and $cancel_page_url) echo '<script>setTimeout(function(){window.location.replace("'.esc_js($cancel_page_url).'");}, '.$cancel_page_time.');</script>';
            }else{

				echo '<p class="mec-error">'.__('Your RSVP cannot be canceled.', 'mec-rsvp').'</p>';
			}
        } elseif(get_query_var('rsvp-response')){

			$args = sanitize_text_field(get_query_var('rsvp-response'));
			$args = explode('&&',$args);
			if(count($args) < 6 ){

				return;
			}
			$answer = isset($args[0]) ? sanitize_text_field( $args[0] ) : '';
			$event_id = isset($args[1]) ? absint( $args[1] ) : false;
			$event_start = isset($args[2]) ? sanitize_text_field( $args[2] ) : false;

			$email = isset($args[3]) && is_email( $args[3] ) ? sanitize_email( $args[3] ) : '';
			$name = isset($args[4]) && !empty($args[4]) ? sanitize_text_field( $args[4] ) : '';
			$nonce = isset($args[5]) && !empty($args[5]) ? sanitize_text_field( $args[5] ) : false;

			$allowed_answers = array(
				'yes',
				'no',
				'maybe',
			);

			$nonce_base = get_post_meta($event_id,'mec_rsvp_nonce_base',true);
			$action = $event_id.$event_start.$answer.$email;
			if(empty($nonce_base)){

				return;
			}

            $p_nonce = md5($nonce_base.$action);
			if(!$nonce || $p_nonce !== $nonce){

				return;
			}

			if(!in_array($answer,$allowed_answers,false)){

				return;
			}

			if(!$event_id){

				return;
			}

			if(!$event_start){

				return;
			}

			$rsvp = array(
				'event_id' => $event_id,
				'event_date' => $event_start,
				'answer' => $answer,
				'first_for_all' => 1,
				'attendees_count' => 1,
				'attendees' => array(
					array(
						'mec_email' => $email,
						'name' => $name,
					),
				),
				'creation_by_invite' => 'yes'
			);

			$rsvp_id = RSVPs::getInstance()->add_rsvp($rsvp);

			$html = '';
			if($rsvp_id){
				//display result
				$html = '<div id="mec-events-meta-group-booking-'.$rsvp_id.'" class="mec-events-meta-group mec-success">
							<div class="">
							'. __('Thank you for RSVP. please check your email.','mec-rsvp').'
							</div>
							<div id="mec_rsvp_message-'.$rsvp_id.'" class="mec-util-hidden" style="display: none;"></div>
						</div>';
			}

			echo $html;
		}
	}
}
