<?php


namespace MEC_RSVP\RSVP;

use MEC\Singleton;
use MEC\Events\Event;
use MEC\Forms\CustomForm;
use MEC\Settings\Settings;

class DisplayForm extends Singleton {

	/**
	 * @var array
	 */
	public $form_attributes = array();

	/**
	 * @var int
	 */
	private $event_id;

	/**
	 * @var \MEC\Events\Event
	 */
	private $event;

	/**
	 * @var bool
	 */
	public $enqueue;

	public function __construct() {

		$this->form_attributes = array(
				'input_class'   => '',
				'btn_box_class' => is_rtl() ? 'ltr' : 'rtl',
		);

		add_action( 'mec_rsvp_form_step_1',[ __CLASS__, 'google_recaptcha_html' ] );
	}

	public function rsvp_permitted_by_ip($event_id, $limit, $attendees_info){

        $count = isset($attendees_info['count']) ? intval($attendees_info['count']) : 0;

        $timestamp = isset($attendees_info['date']) ? $attendees_info['date'] : '';
        if(!is_numeric($timestamp)) $timestamp = strtotime($timestamp);

        $year = date('Y', $timestamp);
        $month = date('m', $timestamp);
        $day = date('d', $timestamp);
        $hour = date('H', $timestamp);
        $minutes = date('i', $timestamp);

        $attendee_ip = \MEC\Base::get_main()->get_client_ip();

        $args = array(
            'post_type' => 'mec-rsvp',
            'posts_per_page' => -1,
			'fields' => 'ids',
            'post_status' => array('publish', 'pending', 'draft', 'future', 'private'),
            'year'=>$year,
            'monthnum'=>$month,
            'day'=>$day,
            'hour'=>$hour,
            'minute'=>$minutes,
            'meta_query' => array(
                array(
                    'key' => 'mec_event_id',
                    'value' => $event_id,
                    'compare' => '=',
                ),
                array(
                    'key' => 'mec_verified',
                    'value' => '-1',
                    'compare' => '!=',
                ),
                array(
                    'key' => 'mec_confirmed',
                    'value' => '-1',
                    'compare' => '!=',
                ),
                array(
                    'key' => 'mec_user_ip',
                    'value' => $attendee_ip,
                    'compare' => '=',
                ),
            ),
        );

        $rsvps_count = 0;
        $permission = true;
        $mec_rsvps = get_posts($args);

        foreach($mec_rsvps as $rsvp_id){

			$rsvps_count += (int)get_post_meta($rsvp_id, 'mec_user_ip', true);
        }

        if(($rsvps_count + $count) > $limit) {
			$permission = false;
		}

        return array(
			'rsvp_count' => $rsvps_count,
			'permission' => $permission
		);
    }

	public function check_ip_restriction($event_id, $date, $attendees_count){

		// User Booking Limits
        list($limit, $unlimited) = RSVPs::getInstance()->get_user_rsvp_limit($event_id);

		$ip_restriction = (bool)Settings::getInstance()->get_settings('rsvp_ip_restriction');
		if($ip_restriction and !$unlimited){
			$permitted_by_ip_info = $this->rsvp_permitted_by_ip($event_id, $limit, array('date' => explode(':', $date)[0], 'count'=> $attendees_count));

			if($permitted_by_ip_info['permission'] === false){
				wp_send_json(array(
					'success'=> false,
					'error_message_html'=> '<div class="mec-error">'.__('Limited','mec-rsvp').'</div>',
				));

				die();
			}
		}
	}

	public static function google_recaptcha_html(){

		if(\MEC\base::get_main()->get_recaptcha_status('rsvp')){
			?>
				<div class="mec-form-row mec-google-recaptcha">
					<div id="mec-rsvp-g-recaptcha" class="g-recaptcha" data-sitekey="<?php echo Settings::getInstance()->get_settings('google_recaptcha_sitekey'); ?>"></div>
				</div>
			<?php
		}

	}

	public function get_html_attendees_form( $group_id, $event_id = null, $attendees_count = 1, $event_date = null, $email = null, $rsvp_id = null ) {

		if(empty($event_date)){

			wp_send_json(array(
				'success'=> false,
				'error_message_html'=> '<div class="mec-error">'. __('Please select a date first.','mec-rsvp') .'</div>',
			));

			die();
		}

		$primary_attendee_data = [];
		$other_attendees_data = [];

		if($rsvp_id){

			$rsvp = new RSVP($rsvp_id);
			$attendees = $rsvp->get_attendees_by_detail_for_each();
			$primary_attendee_data['reg'] = current($attendees);
			$primary_attendee_data['fixed'] = $rsvp->get_fixed_fields();

			$other_attendees_data = array_slice($attendees,1,null,true);
		}else{

			$primary_attendee_data['reg'] = array(
				'email' => $email,
			);
		}

		AjaxProcessHandler::getInstance()->validate_for_display_form( $event_id, $event_date, $email, $attendees_count, null, $rsvp_id );

		$event_id = !is_null($event_id) ? $event_id : $this->event_id;
		$title = __( 'Attendees', 'mec-rsvp' );

		$first_for_all = 1 == Settings::getInstance()->get_settings('rsvp_first_for_all');
		$first_for_all_value = !$first_for_all ? false : true;
		if($rsvp_id){

			$first_for_all_value = $rsvp->is_first_attendee_data_for_all();
		}

		$limit = EventRSVP::getInstance()->get_total_attendees_in_each_rsvp_limit( $event_id, $event_date);

		$registration = Settings::getInstance()->get_settings('rsvp_registration');
		$registration = empty($registration) || $registration == 1;

		$user_pass = Settings::getInstance()->get_settings('rsvp_userpass');
		$user_pass = $user_pass === 'manual' ? 'manual' : 'auto';

		//Check ip
		$this->check_ip_restriction($event_id,$event_date,$attendees_count);

		$attendees_count = (!$limit || $attendees_count <= $limit) ? $attendees_count : $limit;

		$lock_pre_filled = Settings::getInstance()->get_settings('rsvp_lock_prefilled');
		ob_start();
		if( $lock_pre_filled ): ?>
			<script>
				jQuery(document).ready(function($){
					var lock_pre_filled = <?php echo (int)$lock_pre_filled ?>;
					var fields = [];
					if( 1 === lock_pre_filled ){

						fields = $('.mec-events-meta-group [name$="[first_name]"], .mec-events-meta-group [name$="[last_name]"], .mec-events-meta-group [name$="[mec_email]"] ');
					}else if( 2 === lock_pre_filled ){

						fields = $('.mec-events-meta-group [name$="[0][first_name]"], .mec-events-meta-group [name$="[0][last_name]"], .mec-events-meta-group [name$="[0][mec_email]"]');
					}

					$.each(fields, function(i,v){

						var form = $(v).parents('form');
						if( $(v).val().length > 0 ){

							var name = $(v).attr('name');
							var value = $(v).val();

							if( $('input[type="hidden"][name="'+ name +'"]', form).length > 0 ){

								$('input[type="hidden"][name="'+ name +'"]', form).val( value );
							}else{

								$(form).append('<input type="hidden" name="'+ name +'" value="'+ value +'" />');
							}

							if( 'hidden' !== $(v).attr('type')){

								$(v).prop('disabled',true);
							}
						}
					});
				});
			</script>
		<?php endif; ?>
		<form class="mec_rsvp_form mec-events-meta-group" method="post" action="">
			<h4><?php echo $title ?></h4>
			<div class="mec-rsvp-message-box"></div>
			<input type="hidden" name="rsvp[event_date]" id="mec_rsvp_form_date" value="<?php echo $event_date; ?>"/>
			<input type="hidden" name="event_id" value="<?php echo $event_id; ?>">
			<input type="hidden" name="mec-action-2" value="mec-add-rsvp">

			<?php
			if($rsvp_id){
				echo '<input type="hidden" name="rsvp[rsvp_id]" value="'.$rsvp_id.'">';
				wp_nonce_field( 'mec-rsvp-form-edit-na'.$rsvp_id.$event_date, 'rsvp[rsvp_edit_nonce]' );
			}
			//primary attendee
			$form_group_id = 'rsvp_'.$group_id;
			echo CustomForm::getInstance()->display_fields( $form_group_id, $event_id, 0, $primary_attendee_data );

			if($attendees_count > 1){

				echo '<div class="mec-attendees-detail-inputs">';
					for ( $attendee_id = 1; $attendee_id < $attendees_count; $attendee_id++ ) {
						$attendee_data = isset($other_attendees_data[$attendee_id]) && is_array($other_attendees_data[$attendee_id]) ? $other_attendees_data[$attendee_id] : $primary_attendee_data;
						echo '<label class="col-md-12"><strong>'.esc_html__('Attendee','mec-rsvp').' '.$attendee_id.'</strong></label>';
						echo CustomForm::getInstance()->display_reg_fields( $form_group_id, $event_id, $attendee_id, null, $attendee_data );
						echo '<br>';
					}
				echo '</div>';
				?>
					<label class="mec-fill-attendees" <?php echo !$first_for_all ? 'style="display:none"' : ''; ?>>
						<input type="checkbox" name="rsvp[first_for_all]" value="1" <?php checked(true,$first_for_all_value); ?> class="mec_rsvp_first_for_all" id="mec_rsvp_first_for_all" />
						<?php _e("Fill other attendees' information like the first form.", 'mec-rsvp'); ?>
					</label>
				<?php
			}

			if($registration && $user_pass === 'manual' && !is_user_logged_in()): ?>
				<div class="mec-rsvp-username-password-wrapper">
					<h3><?php esc_html_e('Registration', 'mec-rsvp'); ?></h3>
					<div>
						<label for="mec_rsvp_form_username"><?php esc_html_e('Username', 'mec-rsvp'); ?></label>
						<input type="text" name="rsvp[username]" id="mec_rsvp_form_username">
					</div>
					<div>
						<label for="mec_rsvp_form_password"><?php esc_html_e('Password', 'mec-rsvp'); ?></label>
						<input type="password" name="rsvp[password]" id="mec_rsvp_form_password">
					</div>
				</div>
			<?php endif; ?>

			<div class="mec-btn-box <?php echo $this->form_attributes['btn_box_class']; ?>">
				<button class="button mec-move-to-step-1"><?php echo \MEC\Base::get_main()->svg('form/back-icon'); ?><?php esc_html_e( 'Back', 'mec-rsvp' ) ?></button>
				<button class="button mec-submit"><?php esc_html_e( 'Submit', 'mec-rsvp' ) ?></button>
			</div>
			<?php wp_nonce_field( 'mec_rsvp_step_2_nf'.$event_id.$event_date, '_mec_nonce_s2', false ); ?>
		</form>
		<?php
		return ob_get_clean();
	}

	/**
	 * Enqueue Scripts
	 * @used: Base
	 */
	public function enqueue() {

		wp_enqueue_style('mec-custom-form', MEC_CORE_URL.'src/Forms/custom-form.css');

		wp_enqueue_style( 'mec-rsvp-frontend', plugin_dir_url( __FILE__ ) . 'frontend.css' );
		wp_enqueue_script( 'mec-rsvp-frontend', plugin_dir_url( __FILE__ ) . 'frontend.js', array( 'jquery' ) );
		wp_localize_script( 'mec-rsvp-frontend', 'mec_rsvp', array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'error_form_data' => __('Please fill in all required fields','mec-rsvp'),
		) );
	}

	/**
	 * @param $event_id
	 */
	public function output( $event_id ) {

		$this->event_id = $event_id;
		$this->event    = new Event( $event_id );
		$repeating_status = $this->event->get_repeating_status();
		if($repeating_status){

			$next_occurrence = $this->event->get_occurrence_time( time() );
			$timestamp_start = isset($next_occurrence->tstart) ? $next_occurrence->tstart : 0;
		}else{

			$datetimes = $this->event->get_datetime();
			$timestamp_start = $datetimes['start']['timestamp'];
		}

		//expired
		if(!$timestamp_start || time() > $timestamp_start){

			return;
		}

		$interval_for_display_form = EventRSVP::getInstance()->get_show_rsvp_form_interval( $event_id );
		if($interval_for_display_form){

			$interval_for_display_form *= 60;
		}

		$show_from = $timestamp_start && $interval_for_display_form ? $timestamp_start - $interval_for_display_form : false;
		if(($show_from && time() < $show_from)){

			return;
		}

		$edit_form_html = '';
		if( !wp_doing_ajax() ){
			ob_start();
				$this->display_change_rsvp_box($event_id);
			$edit_form_html = ob_get_clean();
		}

		ob_start();
			$this->single_page($event_id);
		$primary_form_html = ob_get_clean();

		ob_start();
			$location = EventRSVP::getInstance()->get_rsvp_modification_form_location( $event_id );
			echo '<div>';

				if( 'bottom' === $location ){

					echo $primary_form_html . $edit_form_html;
				}else{

					echo $edit_form_html . $primary_form_html;
				}
			echo '</div>';

		return ob_get_clean();
	}

	/**
	 * @used step 1
	 */
	public function single_page($event_id) {

		?>
		<div id="mec-events-meta-group-rsvp-<?php echo $event_id ?>" class="mec-events-meta-group-booking mec-custom-form-box mec-rsvp-form-box">
			<div class="mec-rsvp-step-1-box" id="mec-rsvp-<?php echo $event_id ?>">
				<?php $this->display_step_1($event_id); ?>
			</div>
			<div class="mec-rsvp-step-2-box hidden"></div>
			<div class="mec-rsvp-result-box hidden"></div>
			<div class="mec-loading-box">
				<div class="mec-loading-1" style="display:none"></div>
			</div>
		</div>
		<?php
	}

	/**
	 * @used step 1 ajax
	 */
	public function get_html_step1_body_form( $event_id, $event_date ){

		ob_start();
		$max_attendees = EventRSVP::getInstance()->get_max_attendees_to_create_a_rsvp( $event_id, $event_date );
		$limit = 'unlimited' === $max_attendees ? '' : (int) $max_attendees;

		?>
		<div class="mec-form-row mec-event-rsvp-count-wrapper">
			<div class="mec-form-row mec-event-rsvp-count">
				<input id="rsvp_attendees_count" name="rsvp[attendees_count]" type="number" value="1" min="1" step="1" max="<?php echo $limit ?>">
			</div>
			<span class="mec-max-attendees-count">
				<?php
				printf(
					__('Maximum number of Attendees: %s'),
					$max_attendees
				);
				?>
			</span>

		</div>

		<div class="mec-form-row mec-field-wrapper">
			<?php
				$email = '';
				$lock_pre_filled = Settings::getInstance()->get_settings('rsvp_lock_prefilled');
				$email = is_user_logged_in() ? wp_get_current_user()->user_email : '';
			?>
			<span class="mec-field-icon mec-book-mec_email-field-icon"><?php echo \MEC\Base::get_main()->svg('form/email-icon'); ?> </span>
			<input id="rsvp_email" name="rsvp[email]" type="email" value="<?php echo esc_attr( $email ); ?>" placeholder="<?php esc_attr_e('ex: email@gmail.com','mec-rsvp'); ?>" required="required" <?php echo $lock_pre_filled && !empty( $email ) ? 'disabled' : ''; ?>>
			<?php if( $lock_pre_filled && !empty( $email ) ): ?>
				<input id="rsvp_email" name="rsvp[email]" type="hidden" value="<?php echo esc_attr( $email ); ?>" placeholder="<?php esc_attr_e('ex: email@gmail.com','mec-rsvp'); ?>" required="required">
			<?php endif; ?>
		</div>

		<?php $this->display_rsvp($event_id, $event_date); ?>
		<div class="clear"></div>

		<?php
		do_action('mec_rsvp_form_step_1', $event_id, $event_date);

		return ob_get_clean();
	}

	/**
	 * @used step 1
	 */
	public function display_step_1($event_id) {

		$title = __( 'RSVP', 'mec-rsvp' );
		?>
		<form class="mec_rsvp_form mec-events-meta-group">
			<h4><?php echo $title ?></h4>
			<div class="mec-rsvp-message-box"></div>
			<input type="hidden" name="rsvp_creation_by_invite" value="<?php echo get_query_var('rsvp-response') ? 1 : 0; ?>">
			<input type="hidden" name="event_id" value="<?php echo $event_id; ?>">
			<input type="hidden" name="rsvp_action" value="mec-rsvp-get-attendees-form">

			<?php $this->display_event_date($event_id); ?>

			<div class="mec-body-form"></div>

			<div class="mec-btn-box <?php echo $this->form_attributes['btn_box_class']; ?>">
				<button class="button mec-move-to-step-2"><?php esc_html_e( 'Next', 'mec-rsvp' ) ?>
					<svg xmlns="http://www.w3.org/2000/svg" width="13" height="10" viewBox="0 0 13 10"><path id="next-icon" d="M92.034,76.719l-.657.675,3.832,3.857H84v.937H95.208l-3.832,3.857.657.675,4.967-5Z" transform="translate(-84.001 -76.719)" fill="#07bbe9"></path></svg>
				</button>
			</div>
			<?php wp_nonce_field( 'mec_rsvp_step_1_nf', '_mec_nonce_s1' ); ?>
		</form>
		<?php
	}

	/**
	 * @used step 1
	 */
	public function display_event_date($event_id) {

		$date_format = Settings::getInstance()->get_settings( 'rsvp_date_format1' );
		$maximum_dates = Settings::getInstance()->get_settings( 'rsvp_maximum_dates' );

		$repeat_type = $this->event->get_repeating_type();
        if($repeat_type === 'custom_days') {

			$date_format .= ' '.get_option('time_format');
		}

		$start_occurrences = isset( $_GET['occurrence'] ) ? sanitize_text_field($_GET['occurrence']) : false;
		$start_occurrences = $start_occurrences ? strtotime( $start_occurrences ) : time();

		$classes     = $this->form_attributes['input_class'];

		$repeating_status = $this->event->get_repeating_status();
		if($repeating_status){

			echo '<div class="mec-select-date-dropdown-wrapper"><div class="mec-select-date-dropdown">';
			echo '<span class="mec-select-date-calendar-icon">' . \MEC\Base::get_main()->svg('form/calendar-icon') . '</span><select id="mec-rsvp-event-days-' . $event_id . '" name="rsvp[event_date]" class="mec-rsvp-event-days mec-custom-nice-select ' . $classes . '">';
			$occurrences = $this->event->get_occurrences_times( $start_occurrences, $maximum_dates );
			$interval_for_display_form = EventRSVP::getInstance()->get_show_rsvp_form_interval( $event_id );
			if($interval_for_display_form){

				$interval_for_display_form *= 60;
			}
			foreach ( $occurrences as $occurrence_id => $occurrence ) {

				$start['date'] = date_i18n( $date_format, $occurrence->tstart );
				$end['date']   = date_i18n( $date_format, $occurrence->tend );
				$text  = strip_tags(\MEC\Base::get_main()->date_label($start, $end, $date_format, ' - ', false));

				$disabled = false;
				$timestamp_start = isset($occurrence->tstart) ? $occurrence->tstart : 0;
				$show_from = $timestamp_start && $interval_for_display_form ? $timestamp_start - $interval_for_display_form : false;
				if($show_from && time() < $show_from){

					$disabled = true;
				}
				echo '<option value="' . $occurrence->tstart . '" '. ( $disabled ? 'disabled' : '' ) .'>' . $text . '</option>';
			}
			echo '</select>';
			echo '</div></div>';
		}else{

			$datetimes = $this->event->get_datetime();
			$timestamp_start = $datetimes['start']['timestamp'];
			echo '<input type="hidden" name="rsvp[event_date]" value="'.$timestamp_start.'" class="mec-rsvp-event-days" />';
		}
	}

	/**
	 * @used step 1
	 */
	public function display_rsvp($event_id, $event_date) {

		$reports = EventRSVP::getInstance()->get_rsvp_reports( $event_id, $event_date );
		$answers = array(
				'yes'   => __( 'Yes', 'mec-rsvp' ),
				'maybe' => __( 'Maybe', 'mec-rsvp' ),
				'no'    => __( 'No', 'mec-rsvp' ),
		);

		$tabs     = '';
		$links    = '';
		$selected = 'yes';
		foreach ( $answers as $answer_id => $answer_title ) {

			$group_id        = 'rsvp_' . $answer_id;
			$target_id       = 'mec-rsvp-answer-' . $group_id . '-details';
			$class           = ( $answer_id === $selected ) ? 'active' : '';
			$attendees_count = isset( $reports[ $answer_id ] ) && is_array( $reports[ $answer_id ] ) ? array_sum( $reports[ $answer_id ] ) : '0';
			$links           .= '<div class="' . $class . '">' .
								'<label for="mec-rsvp-' . $group_id . '">' .
								'<input ' . checked( 'active', $class, false ) . ' id="mec-rsvp-' . $group_id . '" type="radio" name="rsvp[answer]" data-target-id="' . $target_id . '" value="' . $answer_id . '" />' .
								'<span class="mec-answer-text">' . $answer_title . '</span>' .
								'<span class="mec-answer-attendees-count">'. $attendees_count.' '. __('Attendees','mec-rsvp') .'</span>'.
								'</label>' .

								'</div>';

			$tabs .= '<div id="' . $target_id . '" class="mec-attendees-info-box ' . $class . '">';
			$tabs .= $this->display_attendees( $event_id, $event_date, $answer_id, $reports );
			$tabs .= '</div>';
		}
		?>
		<div class="mec-rsvp-answer">
			<div class="mec-rsvp-answers-details-links">
				<?php echo $links ?>
			</div>
			<div class="mec-rsvp-answers-details-tabs">
				<?php echo $tabs ?>
			</div>
		</div>
		<?php
	}

	public function get_attendees_html($event_id, $event_date, $answer_id, $offset = 0){

		$can_display = EventRSVP::getInstance()->can_display_attendees_list($answer_id,$event_id);
		if(!$can_display){

			return;
		}

		$limit = 12;
		$attendees = EventRSVP::getInstance()->get_attendees(
			$event_id,
			$event_date,
			$answer_id,
			$limit + 1, // for check load more
			$offset
		);

		$total_attendees = count($attendees);
		$attendees = array_splice($attendees,0,$limit);

		$html = '<h5 class="mec-attendees-title">' . __( 'Attendees', 'mec-rsvp' ) . '</h5>';
		$html .= '<div class="mec-attendees">';
		if ( !empty($attendees) ) {
			foreach($attendees as $attendee){

				$attendees_count_html = '';
				if(isset($attendee['count']) && $attendee['count'] > 1){
					$attendees_count_html = '<div class="mec-attendees-count">'.$attendee['count'].'</div>';
				}
				$email = $attendee['email'];
				$name = $attendee['name'];
				$html .= '<div class="mec-attendee">'.
							$attendees_count_html.
							'<div class="mec-avatar">'.
								get_avatar($email) .
							'</div>' .
							'<span>'.$name.'</span>'.
						'</div>';
			}
		} else {
			$html .= '<p>' . __( 'We have not received an RSVP with this response yet.', 'mec-rsvp' ) . '</p>';
		}
		$html .= '</div>';
		$html .= $this->display_rsvp_report( $event_id, $answer_id, $event_date, $reports );

		if($limit < $total_attendees){

			$next_offset = $offset + $limit;
			$html .= '<div class="mec-load-more-wrap"><a href="#" class="mec-load-more-button mec-load-attendees" data-offset="'.$next_offset.'" data-answer="'.$answer_id.'" data-event_id="'.$event_id.'" data-event_date="'.$event_date.'">'.__('load more','mec-rsvp').'</a></div>';
		}

		return $html;
	}

	/**
	 * @used step 1
	 *
	 * @param string $answer_id
	 * @param array  $reports
	 *
	 * @return mixed
	 */
	public function display_attendees( $event_id, $event_date, $answer_id, $reports ) {

		$offset = isset( $_POST['offset'] ) ? absint($_POST['offset']) : 0;
		$html .= $this->get_attendees_html( $event_id, $event_date, $answer_id, $offset );

		return $html;
	}


	public function display_rsvp_report( $event_id, $answer_id, $event_date, $reports){

		if( 'yes' !== $answer_id ){

			return;
		}

		$can_display = EventRSVP::getInstance()->can_display_rsvp_report_in_rsvp_form($event_id);
		if(!$can_display){

			return;
		}

		$html = '';

		$reports = EventRSVP::getInstance()->get_rsvp_reports( $event_id, $event_date );
		$minimum_attendees_to_start_event = EventRSVP::getInstance()->get_minimum_attendees_to_start_event($event_id,$event_date);
		if($minimum_attendees_to_start_event){
			$total_attendees = isset($reports['yes']) && is_array($reports['yes']) ? array_sum($reports['yes'] ) : 0;
			$left_over = $minimum_attendees_to_start_event - $total_attendees;
			$html = '<div class="min-to-start">'.
					__('Minimum attendees to start the event','mec-rsvp').': <strong>'. $minimum_attendees_to_start_event. '</strong>' .
					(($left_over > 0) ? '. '. __('left over','mec-rsvp').': <strong>'.$left_over .'</strong>' : '').
				'</div>';
		}

		return $html;
	}

	public function display_change_rsvp_box($event_id){

		$can_display = EventRSVP::getInstance()->can_display_rsvp_modification($event_id);
		if(!$can_display){

			return;
		}

		$email = '';
		$user_id = get_current_user_id();
		if($user_id){

			$user = get_user_by('id', $user_id);
			$email = $user->user_email;
		}

		?>
			<div class="mec-rsvp-modification mec-custom-form-box">
				<h4 class="mec-box-title"><?php esc_html_e('Change my RSVP', 'mec-rsvp'); ?></h4>

				<div class="mec-rsvp-search-form">
					<div class="mec-rsvp-message-box"></div>
					<input type="hidden" name="action" value="mec-rsvp-search-form-by-email"/>
					<input type="hidden" name="event_id" value="<?php echo $event_id; ?>"/>
					<div class="mec-rsvp-search-form-field-wrapper">
						<span class="mec-field-icon mec-book-mec_email-field-icon"><?php echo \MEC\Base::get_main()->svg('form/email-icon'); ?> </span>
						<input type="email" name="mec-rsvp-search-form[email]" placeholder="<?php esc_html_e('Enter Your Email Address', 'mec-rsvp'); ?>" value="<?php echo $email; ?>" required/>
					</div>
					<?php
					wp_nonce_field( 'mec-rsvp-search-form-by-email', 'mec-rsvp-search-form-nonce');
					$can_modification_without_email_verification = EventRSVP::getInstance()->can_rsvp_modification_without_email_verification($event_id);
					if(!$can_modification_without_email_verification): ?>
						<div class="mec-email-confirmation hidden">
							<input type="text" name="mec-rsvp-verification-code" placeholder="<?php esc_html_e('Enter Email Verification Code', 'mec-rsvp'); ?>" />
						</div>
					<?php endif; ?>
						<button id="mec-rsvp-search-by-email" class="button"><?php esc_html_e('Find My RSVP', 'mec-rsvp'); ?></button>
					<div class="mec-loading-box">
						<div class="mec-loading-1" style="display:none"></div>
					</div>
				</div>
				<div class="mec-rsvps"></div>
				<div class="mec-edit-form-rsvp"></div>

			</div>
		<?php
	}

}
