<?php

namespace MEC_RSVP\RSVP;

use MEC\Singleton;
use MEC_RSVP\User;
use MEC\Forms\CustomForm;
use MEC\Settings\Settings;
use MEC\Notifications\SendEmail;

class RSVPNotification extends SendEmail{

    public $event_id;
    public $rsvp_id;
    public $rsvp;

    public function get_default_notification_settings(){

        return NotificationsSettings::getInstance()->get_default_settings($this->group_id) ;
    }

    public function get_rsvp_author($rsvp_id){

        $user = new User();
        return $user->rsvp($rsvp_id);
    }

    public function get_author($rsvp_id){

        return $this->get_rsvp_author($rsvp_id);
    }

    public function get_event_times(){

        return $this->rsvp->get_event_times();
    }

    public function prepare($rsvp_id){

        $this->rsvp_id = $rsvp_id;
        $this->rsvp = new RSVP($this->rsvp_id);
        $this->event_id = $this->rsvp->get_event_id();
    }

    public function get_rsvp_custom_text_for_answer( $answer_id, $default = '' ){

        $custom_text = $this->get_notification_settings( 'custom_text_'.$answer_id );

        return !is_null($custom_text) ? __($custom_text, 'mec-rsvp') : $default;
    }

    public function is_creation_by_invite(){

        $is_creation_by_invite = $this->rsvp->get_creation_type();

        if(!empty($is_creation_by_invite) && 'yes' === $is_creation_by_invite){

            return 'yes';
        }

        return 'no';
    }

    public function can_send_notification($rsvp_id,$mode = 'auto',$resend = false){

        $can = true;

        $sended = get_post_meta( $rsvp_id, 'mec_'.$this->group_id, true );
        if( !empty($sended) && !$resend ){

            $can = false;
        }

        if(!$this->get_enabled_status()){

            $can = false;
        }

        // $rsvp_author = $this->get_rsvp_author($rsvp_id);
        // if(!isset($rsvp_author->user_email)){

        //     $can = false;
        // }


        return apply_filters( 'mec_can_send_email_notification', $can, $this->group_id, $this );
    }

    public function get_cc_bcc_method(){

        $recipients_method = Settings::getInstance()->get_settings('booking_recipients_method');

        return trim($recipients_method) ? strtoupper($recipients_method) : 'BCC';
    }

    public function get_headers($recipients){

        $headers = array(
            'Content-Type: text/html; charset=UTF-8'
        );
        // Recipient Type
        $CCBCC = $this->get_cc_bcc_method();

        foreach($recipients as $recipient){
            // Skip if it's not a valid email
            if(trim($recipient) == '' or !filter_var($recipient, FILTER_VALIDATE_EMAIL)) continue;

            $headers[] = $CCBCC.': '.$recipient;
        }

        return $headers;
    }

    public function get_reg_fields(){

        $group_id = 'rsvp_'. $this->rsvp->get_answer();
        return CustomForm::getInstance()->get_reg_fields( $group_id, $this->event_id );
    }

    public function get_fixed_fields(){

        $group_id = 'rsvp_'. $this->rsvp->get_answer();
        return CustomForm::getInstance()->get_fixed_fields( $group_id, $this->event_id );
    }

    public function mec_sender_email_notification_filter(){

        // MEC Notification Sender Email
        add_filter('wp_mail_from_name', array($this, 'notification_sender_name'));
        add_filter('wp_mail_from', array($this, 'notification_sender_email'));
    }

    public function notification_sender_name($sender_name){

        $s_sender_name = Settings::getInstance()->get_settings('booking_sender_name');

        return !empty($s_sender_name) ? $s_sender_name : $sender_name;
    }

    public function notification_sender_email($sender_email){

        $s_sender_email = Settings::getInstance()->get_settings('booking_sender_email');

        return !empty($s_sender_email) ? $s_sender_email : $sender_email;
    }

    public function get_cancellation_link(){

        $key = get_post_meta($this->rsvp_id, 'mec_cancellation_key', true);

        return trim(get_permalink($this->event_id), '/').'/rsvp-cancel/'.$key.'/';
    }

    public function get_full_attendees_info(){

        $attendees_full_info = '';

        $attendees = $this->rsvp->get_attendees();

        $reg_fields = $this->get_reg_fields();

        $attachments = (isset($attendees['attachments']) and is_array($attendees['attachments'])) ? $attendees['attachments'] : array();
        $attachment_field = array();
        if(count($attachments)){
            foreach($reg_fields as $reg_field_id => $reg_field){
                if(!is_numeric($reg_field_id)) continue;
                if($reg_field['type'] !== 'file') continue;

                $attachment_field = $reg_field;
                break;
            }
        }

        foreach($attendees as $key=>$attendee){
            if($key === 'attachments') continue;

            $reg_form = isset($attendee['reg']) ? $attendee['reg'] : array();

            $attendees_full_info .= __('Name', 'mec-rsvp').': '.((isset($attendee['name']) and trim($attendee['name'])) ? $attendee['name'] : '---')."\r\n";
            $attendees_full_info .= __('Email', 'mec-rsvp').': '.((isset($attendee['email']) and trim($attendee['email'])) ? $attendee['email'] : '---')."\r\n";

            if(is_array($reg_form) and count($reg_form)){
                foreach($reg_form as $field_id=>$value){
                    // Placeholder Keys
                    if(in_array($field_id, [':i:',':fi:','_i_','_fi_',], true)){

                        continue;
                    }

                    $reg_fields = apply_filters( 'mec_rsvp_notification_reg_fields', $reg_fields, $field_id);

                    $type = $reg_fields[$field_id]['type'];

                    $label = isset($reg_fields[$field_id]) ? $reg_fields[$field_id]['label'] : '';
                    if(trim($label) == '') continue;

                    if($type == 'agreement'){
                        $label = sprintf(__($label, 'mec-rsvp'), '<a href="'.get_the_permalink($reg_fields[$field_id]['page']).'">'.get_the_title($reg_fields[$field_id]['page']).'</a>');
                        $attendees_full_info .= $label.': '.($value == '1' ? __('Yes', 'mec-rsvp') : __('No', 'mec-rsvp'))."\r\n";
                    } else {
                        $attendees_full_info .= __($label, 'mec-rsvp').': '.(is_string($value) ? $value : (is_array($value) ? implode(', ', $value) : '---'))."\r\n";
                    }
                }
            }

            $attendees_full_info .= "\r\n";
        }

        // Attachment
        if(count($attachments)){
            $attendees_full_info .= __($attachment_field['label'], 'mec-rsvp').': <a href="'.esc_url($attachments[0]['url']).'" target="_blank">'.esc_url($attachments[0]['url']).'</a>'."\r\n";
        }

        return $attendees_full_info;
    }

    public function render_rsvp_data(&$content,$rsvp_id,$timestamps){

        // Date & Time Format
        $date_format = get_option('date_format');
        $time_format = get_option('time_format');

        if(!trim($timestamps)) {

            $timestamps = $this->rsvp->get_event_times();
        }
        list($start_timestamp, $end_timestamp) = explode(':', $timestamps);

        // Occurrence Params
        $params = \MEC_feature_occurrences::param($this->event_id, $start_timestamp, '*');

        // RSVP Date
        if(trim($timestamps) and strpos($timestamps, ':') !== false){

            if(trim($start_timestamp) != trim($end_timestamp) ){

                $start_date = date($date_format, $start_timestamp);
                $end_date = !empty( $end_timestamp ) ? date($date_format, $end_timestamp) : '';
                if( !empty( $end_date ) ){

                    $rsvp_date = sprintf(
                        __('%s to %s', 'mec-rsvp'),
                        \MEC\Base::get_main()->date_i18n($date_format, $start_timestamp),
                        \MEC\Base::get_main()->date_i18n($date_format, $end_timestamp)
                    );
                }else{

                    $rsvp_date = \MEC\Base::get_main()->date_i18n($date_format, $start_timestamp);
                }
            } else{

                $rsvp_date = get_the_date($date_format, $rsvp_id);
            }
        }else{

            $rsvp_date = get_the_date($date_format, $rsvp_id);
        }

        $content = str_replace('%%rsvp_date%%', $rsvp_date, $content);

        // RSVP Time
        $event_start_time = \MEC\Base::get_main()->get_time($start_timestamp);
        $event_end_time = !empty( $end_timestamp ) ? \MEC\Base::get_main()->get_time($end_timestamp) : '';

        $allday = get_post_meta($this->event_id, 'mec_allday', true);
        $hide_time = get_post_meta($this->event_id, 'mec_hide_time', true);
        $hide_end_time = get_post_meta($this->event_id, 'mec_hide_end_time', true);
        $event_time = $allday ? \MEC\Base::get_main()->m('all_day', __('All Day' , 'mec-rsvp')) : (!$hide_end_time ? sprintf(__('%s to %s', 'mec-rsvp'), $event_start_time, $event_end_time) : $event_start_time);
        $datetime_format = $date_format.((!$allday and !$hide_time) ? ' '.$time_format : '');

        // Condition for check some parameter simple hide event time
        if(!$hide_time) {

            $content = str_replace('%%rsvp_time%%', $event_time, $content);
        }else{

            $content = str_replace('%%rsvp_time%%', '', $content);
        }

        // RSVP Date & Time
        if(trim($timestamps) and strpos($timestamps, ':') !== false){

            if(trim($start_timestamp) != trim($end_timestamp)){

                $start_date = date($date_format, $start_timestamp);
                $end_date = !empty( $end_timestamp ) ? date($date_format, $end_timestamp) : '';
                if( !empty( $end_date ) ){

                    $rsvp_datetime = sprintf(
                        __('%s to %s', 'mec-rsvp'),
                        \MEC\Base::get_main()->date_i18n($datetime_format, $start_timestamp),
                        \MEC\Base::get_main()->date_i18n($date_format.((!$allday and !$hide_time and !$hide_end_time) ? ' '.$time_format : ''), $end_timestamp)
                    );
                } else {

                    $rsvp_datetime = \MEC\Base::get_main()->date_i18n($datetime_format, $start_timestamp);
                }
            }
        } else {

            $rsvp_datetime = get_the_date($datetime_format, $rsvp_id);
        }

        $content = str_replace('%%rsvp_datetime%%', $rsvp_datetime, $content);

        // Order Time
        $order_time = $this->rsvp->get_create_datetime();
        $content = str_replace('%%rsvp_order_time%%', \MEC\Base::get_main()->date_i18n($date_format.' '.$time_format, strtotime($order_time)), $content);

        $cancellation_link = $this->get_cancellation_link();
        $content = str_replace('%%cancellation_link%%', $cancellation_link, $content);

        // Attendee Full Information
        if(strpos($content, '%%attendee_full_info%%') !== false or strpos($content, '%%attendees_full_info%%') !== false){
            $attendees_full_info = $this->get_full_attendees_info();

            $content = str_replace('%%attendee_full_info%%', $attendees_full_info, $content);
            $content = str_replace('%%attendees_full_info%%', $attendees_full_info, $content);
        }

        $content = str_replace('%%total_attendees%%', $this->rsvp->get_attendees_count(), $content);

        // RSVP IDs
        $content = str_replace('%%rsvp_id%%', $rsvp_id, $content);
        $content = str_replace('%%rsvp_response%%', $this->rsvp->get_answer(), $content);

        // RSVP Fixed Fields
        $fixed_fields = $this->get_fixed_fields();
        if(is_array($fixed_fields) and count($fixed_fields) and isset($transaction['fields']) and is_array($transaction['fields']) and count($transaction['fields'])){
            foreach($fixed_fields as $b => $fixed_field){

                if(!is_numeric($b)) {

                    continue;
                }

                $fixed_field_name = isset($fixed_field['label']) ? $fixed_field['label'] : '';
                $fixed_value = isset($transaction['fields'][$b]) ? $transaction['fields'][$b] : NULL;
                if(trim($fixed_value) === '') {

                    continue;
                }

                if(is_array($fixed_value)){

                    $fixed_value = implode(', ', $fixed_value);
                }

                $content = str_replace('%%rsvp_field_'.$b.'%%', trim(stripslashes($fixed_value), ', '), $content);
                $content = str_replace('%%rsvp_field_'.$b.'_with_name%%', trim((trim($fixed_field_name) ? stripslashes($fixed_field_name).': ' : '').trim(stripslashes($fixed_value), ', ')), $content);
            }
        }

        return $content;

    }

    public function render_content($content, $rsvp_id, $attendee = array(), $timestamps = NULL){

        $this->render_author($content,$rsvp_id,$attendee);

        // Site Data
        $this->render_site_data($content,$rsvp_id);


        // Event Data
        $this->render_event_data($content,$rsvp_id,$timestamps);

        // RSVP Data
        $this->render_rsvp_data($content,$rsvp_id,$timestamps);

        return apply_filters( 'mec_render_content_email', $content, $this );
    }

    public function get_attendees(){

        return $this->rsvp->get_attendees();
    }

    public function allowed_check_settings_for_attendees(){

        return array(
            'rsvp_cancellation_notification',
            'rsvp_rejection_notification',
            'rsvp_admin_notification',
        );
    }

    public function send($rsvp_id, $mode = 'auto', $resend = false){

        $mails = $this->get_mails_for_send( $rsvp_id, $mode, $resend );
        if(!is_array($mails)){

            return false;
        }

        // Set Email Type to HTML
        add_filter('wp_mail_content_type', array(\MEC\Base::get_main(), 'html_email_type'));

        foreach($mails as $mail_arg){

            $this->send_mail( $mail_arg );
        }

        // Remove the HTML Email filter
        remove_filter('wp_mail_content_type', array(\MEC\Base::get_main(), 'html_email_type'));

        update_post_meta( $rsvp_id, 'mec_'.$this->group_id, current_time('timestamp') );

        return true;
    }

    public function get_mails_for_send($rsvp_id, $mode = 'auto', $resend = false ){
        $this->prepare($rsvp_id);

        if(!$this->can_send_notification( $rsvp_id, $mode, $resend )){

            return false;
        }

        $subject = $this->get_subject();
        $content = $this->get_content();
        $recipients = $this->get_all_recipients_emails();
        $headers = $this->get_headers($recipients);
        $target_users = $this->get_target_users_or_emails();

        $this->mec_sender_email_notification_filter();
        // Do not send email twice!
        $done_emails = array();

        // Send the emails
        $mails = [];
        foreach($target_users as $attendee_or_email) {

            $attendee = is_array($attendee_or_email) ? $attendee_or_email : [];
            $to = isset($attendee['email']) ? $attendee['email'] : $attendee_or_email;

            if(!trim($to) or in_array($to, $done_emails) or !filter_var($to, FILTER_VALIDATE_EMAIL)) {

                continue;
            }

            $subject = $this->render_content($subject, $rsvp_id, $attendee);
            $message = $this->render_content($content, $rsvp_id, $attendee);

            // Remove remained placeholders
            $message = preg_replace('/%%.*%%/', '', $message);

            $message = $this->add_template($message);

            // Filter the email
            $mail_arg = array(
                'to'            => $to,
                'subject'       => $subject,
                'message'       => $message,
                'headers'       => $headers,
                'attachments'   => array(),
            );

            $mail_arg = apply_filters('mec_rsvp_before_send_email', $mail_arg, $rsvp_id, $this->group_id);

            $mails[] = $mail_arg;

            // For prevention of email repeat send
            $done_emails[] = $to;
        }

        return $mails;
    }

}
