<?php

namespace MEC_RSVP\StatusProvider;

use MEC\Singleton;
use MEC_RSVP\RSVP\RSVP;

class StatusProvider extends Singleton {

	public function init() {

		add_action( 'pre_get_posts', array( $this, 'filter_query' ) );
		add_filter( 'manage_mec_rsvp_posts_columns', array( $this, 'filter_columns' ) );
		add_filter( 'manage_edit-mec_rsvp_sortable_columns', array( $this, 'filter_sortable_columns' ) );
		add_action( 'manage_mec_rsvp_posts_custom_column', array( $this, 'filter_columns_content' ), 10, 2 );
		add_action( 'add_meta_boxes', array( $this, 'meta_boxes' ) );

		add_action( 'restrict_manage_posts', array( $this, 'add_filters' ) );

		// Admin List Table
		add_filter( 'bulk_actions-edit-mec_rsvp', array( $this, 'add_bulk_actions' ) );
		add_filter( 'handle_bulk_actions-edit-mec_rsvp', array( $this, 'do_bulk_actions' ), 10, 3 );

		add_action( 'save_post_mec_rsvp', array( $this, 'save_rsvp' ) );
		add_action( 'edit_post_mec_rsvp', array( $this, 'save_rsvp' ) );
	}

	public function meta_boxes() {

		add_meta_box(
			'mec-status-provider',
			__( 'Status & Invoice', 'mec-rsvp' ),
			'\MEC_RSVP\StatusProvider\Display::output',
			array( 'mec_rsvp' ),
			'side'
		);
	}

	/**
	 * @param array $columns
	 *
	 * @return array
	 */
	public function filter_columns( $columns ) {

		$columns['confirmation'] = __( 'Confirmation', 'mec-rsvp' );
		$columns['verification'] = __( 'Verification', 'mec-rsvp' );

		return $columns;
	}

	/**
	 * @param array $columns
	 *
	 * @return array
	 */
	public function filter_sortable_columns( $columns ) {

		$columns['confirmation'] = 'confirmation';
		$columns['verification'] = 'verification';

		return $columns;
	}

	/**
	 * Get RSVP
	 *
	 * @param int $rsvp_id
	 *
	 * @return MEC_RSVP\RSVP\RSVP
	 */
	public function get_rsvp( $rsvp_id ) {

		global $MEC_RSVP;
		if ( !is_a( $MEC_RSVP, RSVP::class ) || $rsvp_id !== $MEC_RSVP->get_id() ) {

			$MEC_RSVP = new RSVP( $rsvp_id );
		}

		return $MEC_RSVP;
	}

	/**
	 * Get MEC_Main
	 *
	 * @return \MEC_Main
	 */
	public function get_main() {

		return \MEC_RSVP\Base::get_main();
	}

	/**
	 * Display Content
	 *
	 * @param string $column_name
	 * @param int    $post_id
	 *
	 * @return void
	 */
	public function filter_columns_content( $column_name, $post_id ) {

		$rsvp = $this->get_rsvp( $post_id );
		switch ( $column_name ) {
			case 'confirmation':
				$confirmed = $rsvp->get_confirm_status();
				echo '<a href="' . $this->get_main()->add_qs_var( 'mec_confirmed', $confirmed ) . '">' . $this->get_main()->get_confirmation_label( $confirmed ) . '</a>';

				break;
			case 'verification':
				$verified = $rsvp->get_verification_status();
				echo '<a href="' . $this->get_main()->add_qs_var( 'mec_verified', $verified ) . '">' . $this->get_main()->get_verification_label( $verified ) . '</a>';

				break;
		}
	}

	/**
	 * @param \WP_Query $query
	 */
	public function filter_query( $query ) {

		if ( !is_admin() || !$query->is_main_query() || 'mec_rsvp' !== $query->get( 'post_type' ) ) {
			return;
		}

		$meta_query = array();
		$orderby    = $query->get( 'orderby' );

		switch ( $orderby ) {
			case 'confirmation':
				$query->set( 'meta_key', 'mec_confirmed' );
				$query->set( 'orderby', 'mec_confirmed' );
				break;
			case 'verification':
				$query->set( 'meta_key', 'mec_verified' );
				$query->set( 'orderby', 'mec_verified' );
				break;
		}

		// Filter by Confirmation
		if ( isset( $_REQUEST['mec_confirmed'] ) and trim( $_REQUEST['mec_confirmed'] ) != '' ) {
			$meta_query[] = array(
				'key'     => 'mec_confirmed',
				'value'   => sanitize_text_field( $_REQUEST['mec_confirmed'] ),
				'compare' => '=',
				'type'    => 'numeric',
			);
		}

		// Filter by Verification
		if ( isset( $_REQUEST['mec_verified'] ) and trim( $_REQUEST['mec_verified'] ) != '' ) {
			$meta_query[] = array(
				'key'     => 'mec_verified',
				'value'   => sanitize_text_field( $_REQUEST['mec_verified'] ),
				'compare' => '=',
				'type'    => 'numeric',
			);
		}

		if ( count( $meta_query ) ) {
			$query->set( 'meta_query', $meta_query );
		}

	}

	public function add_filters( $post_type ) {

		if ( 'mec_rsvp' !== $post_type ) {
			return;
		}

		$mec_confirmed = isset( $_REQUEST['mec_confirmed'] ) ? sanitize_text_field( $_REQUEST['mec_confirmed'] ) : '';

		echo '<select name="mec_confirmed">';
		echo '<option value="">' . __( 'Confirmation', 'mec-rsvp' ) . '</option>';
		echo '<option value="1" ' . ( $mec_confirmed == '1' ? 'selected="selected"' : '' ) . '>' . __( 'Confirmed', 'mec-rsvp' ) . '</option>';
		echo '<option value="0" ' . ( $mec_confirmed == '0' ? 'selected="selected"' : '' ) . '>' . __( 'Pending', 'mec-rsvp' ) . '</option>';
		echo '<option value="-1" ' . ( $mec_confirmed == '-1' ? 'selected="selected"' : '' ) . '>' . __( 'Rejected', 'mec-rsvp' ) . '</option>';
		echo '</select>';

		$mec_verified = isset( $_REQUEST['mec_verified'] ) ? sanitize_text_field( $_REQUEST['mec_verified'] ) : '';

		echo '<select name="mec_verified">';
		echo '<option value="">' . __( 'Verification', 'mec-rsvp' ) . '</option>';
		echo '<option value="1" ' . ( $mec_verified == '1' ? 'selected="selected"' : '' ) . '>' . __( 'Verified', 'mec-rsvp' ) . '</option>';
		echo '<option value="0" ' . ( $mec_verified == '0' ? 'selected="selected"' : '' ) . '>' . __( 'Waiting', 'mec-rsvp' ) . '</option>';
		echo '<option value="-1" ' . ( $mec_verified == '-1' ? 'selected="selected"' : '' ) . '>' . __( 'Canceled', 'mec-rsvp' ) . '</option>';
		echo '</select>';
	}

	public function add_bulk_actions( $bulk_actions ) {

		$bulk_actions['pending'] = __( 'Pending', 'mec-rsvp' );
		$bulk_actions['confirm'] = __( 'Confirm', 'mec-rsvp' );
		$bulk_actions['reject']  = __( 'Reject', 'mec-rsvp' );

		$bulk_actions['waiting']  = __( 'Waiting', 'mec-rsvp' );
		$bulk_actions['verified'] = __( 'Verified', 'mec-rsvp' );
		$bulk_actions['canceled'] = __( 'Canceled', 'mec-rsvp' );

		return $bulk_actions;
	}

	public function do_bulk_actions( $redirect, $action, $object_ids ) {

		switch ( $action ) {
			case 'confirm':
			case 'pending':
			case 'reject':
				foreach ( $object_ids as $post_id ) {
					$rsvp = new RSVP( $post_id );
					$rsvp->set_confirm_status( $action );
				}

				break;
			case 'verified':
			case 'waiting':
			case 'canceled':
				foreach ( $object_ids as $post_id ) {
					$rsvp = new RSVP( $post_id );
					$rsvp->set_verification_status( $action );
				}

				break;
		}

		return $redirect;
	}

	public function save_rsvp( $rsvp_id ) {

		if ( wp_is_post_revision( $rsvp_id ) ) {
			return;
		}

		if ( wp_doing_ajax() ) {
			return;
		}

		if ( !isset( $_POST['mec_status_nonce'] ) || !wp_verify_nonce( $_POST['mec_status_nonce'], 'mec_status_nf_' . date( 'd' ) ) ) {
			return;
		}

		$rsvp         = new RSVP( $rsvp_id );
		$confirmation = isset( $_POST['mec_confirmation'] ) ? $_POST['mec_confirmation'] : 0;
		$verification = isset( $_POST['mec_verification'] ) ? $_POST['mec_verification'] : 0;

		$rsvp->set_confirm_status( $confirmation );
		$rsvp->set_verification_status( $verification );

        if(! \MEC_RSVP\RSVP\EventRSVP::getInstance()->can_send_verify_email($rsvp->get_event_id(),'auto')){
            \MEC_RSVP\RSVP\RSVPNotifications::getInstance()->send_notification_verification($rsvp_id,'',$verification,'auto');
        }

	}

}
