<?php

namespace MEC_Invoice\Helper;

use MEC_Invoice\Helper;
use MEC_Invoice\Attendee;
use MEC_Invoice\Notifications;

// Don't load directly
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Mec Invoice Information.
 *
 * @author     Webnus <info@webnus.biz>
 * @package     Modern Events Calendar
 * @since     1.0.0
 **/
class InvoiceInformation {

    public static function get_invoice_hash( $invoice_id ) {

        return get_post_meta( $invoice_id, 'invoiceID', true );
    }

    public static function get_checkin_hash( $invoice_id ) {

        $doCheckinHash = get_post_meta( $invoice_id, 'CheckinHash', true );
        if ( !$doCheckinHash ) {

            $doCheckinHash = sha1(md5(microtime()));
            update_post_meta($invoice_id, 'CheckinHash', $doCheckinHash);
        }

        return $doCheckinHash;
    }

    public static function get_qr_code_link( $invoice_id, $Hash, $attendee = null ) {

        if( !is_null( $attendee ) ) {

            $email = $attendee['email'] ?? '';
            $row_id = $attendee['row_id'] ?? 0;
            $attendee_key = $attendee['attendee_key'] ?? $row_id;
            $place = $attendee_key + 1;

            return get_site_url(null, '?invoiceID=' . $invoice_id . '&makeQRCode=' . $Hash . '&attendee=' . $email) . '&place=' . $place . '&Hash=' . $Hash;
        }else{

        }
    }

    public static function get_pdf_link( $invoice_id, $Hash, $email = null ) {

        if( !is_null( $email ) ) {

            $email = is_array($email) ? $email['email'] : $email;
            return get_site_url(null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash . '&showPDF=true&attendee=' . $email);
        }else{

            return get_site_url(null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash . '&showPDF=true');
        }
    }

    public static function get_book_id( $invoice_id ) {

        return get_post_meta( $invoice_id, 'book_id', true );
    }

    public static function get_company_information( $invoice_id, &$InvoiceInformation ) {

        // Company Information
        $InvoiceInformation['CompanyURL'] = Helper::getOption('mec_invoice_company_url', '');
        $InvoiceInformation['CompanyURLV'] = str_replace(['http://', 'https://'], '', Helper::getOption('mec_invoice_company_url', ''));

        $InvoiceInformation['CompanyName'] = Helper::getOption('mec_invoice_company_name', '');
        $InvoiceInformation['CompanyAddress'] = Helper::getOption('mec_invoice_address', '');
        $InvoiceInformation['CompanyDescription'] = Helper::getOption('mec_invoice_description', '');
        $InvoiceInformation['CompanyPhone'] = Helper::getOption('mec_invoice_company_phone', '');
        $InvoiceInformation['CompanyVatNumber'] = Helper::getOption('mec_invoice_vat_number', '');
        $InvoiceInformation['CompanyEmail'] = Helper::getOption('mec_invoice_company_email', '');
        $InvoiceInformation['CompanyLogoUrl'] = Helper::getOption('mec_invoice_logo', '');
        $InvoiceInformation['CompanyLogoWidth'] = Helper::getOption('mec_invoice_logo_width', 50);
        $InvoiceInformation['CompanyInformationStyle'] = '';

        if(
            !trim($InvoiceInformation['CompanyName']) &&
            !trim($InvoiceInformation['CompanyDescription']) &&
            !trim($InvoiceInformation['CompanyLogoUrl']) &&
            !trim($InvoiceInformation['CompanyEmail']) &&
            !trim($InvoiceInformation['CompanyAddress']) &&
            !trim($InvoiceInformation['CompanyPhone']) &&
            !trim($InvoiceInformation['CompanyVatNumber']) &&
            !trim($InvoiceInformation['CompanyURL'])) {

            $InvoiceInformation['CompanyInformationStyle'] = 'display:none';
        }

        return $InvoiceInformation;
    }

    public static function get_event_information( $event_id, &$InvoiceInformation ) {

        $eventDescriptions = get_post_meta($event_id, 'i_event_descriptions', true);
        $InvoiceInformation['EventDescriptions'] = $eventDescriptions;

        # Event Data
        $organizer_id = get_post_meta($event_id, 'mec_organizer_id', true);
        $location_id = get_post_meta($event_id, 'mec_location_id', true);
        $organizer = get_term($organizer_id, 'mec_organizer');
        $organizerImage = get_term_meta($organizer_id, 'thumbnail', true) ? get_term_meta($organizer_id, 'thumbnail', true) : 'https://webnus.net/wp-content/uploads/2018/10/deep-emm-pi-3.png';
        $location = get_term($location_id, 'mec_location');
        $location_id = get_post_meta($event_id, 'mec_location_id', true);
        $location = get_term($location_id, 'mec_location');
        $locationName = '';
        if (isset($location->name)) {
            $locationName = $location->name;
        }

        $show_location_address = 'on' === Helper::getOption( 'show_location_address', 'off' ) ? true : false;
        $locationAddress = '';
        if ( isset($location->name) && $show_location_address ) {//TODO: get settings

            $locationAddress = get_term_meta( $location->term_id, 'address', true );
        }

        // Organizer
        $InvoiceInformation['OrganizerImage'] = $organizerImage;
        $InvoiceInformation['OrganizerStyle'] = isset($organizer->name) ? '' : 'display:none;';
        $InvoiceInformation['OrganizerName'] = isset($organizer->name) ? $organizer->name : '';
        $InvoiceInformation['OrganizerPhone'] = get_term_meta($organizer_id, 'tel', true) ? get_term_meta($organizer_id, 'tel', true) : '';
        $InvoiceInformation['OrganizerEmail'] = get_term_meta($organizer_id, 'email', true) ? get_term_meta($organizer_id, 'email', true) : '';
        $InvoiceInformation['OrganizerSite'] = get_term_meta($organizer_id, 'url', true) ? get_term_meta($organizer_id, 'url', true) : '';

        // Location
        $InvoiceInformation['EventLocation'] = $locationName;
        $InvoiceInformation['EventLocationStyle'] = '';

        // Location
        $InvoiceInformation['EventLocationAddress'] = $locationAddress;
        $InvoiceInformation['EventLocationAddressStyle'] = '';


        return $InvoiceInformation;
    }

    public static function get_buttons_information( $invoice_id, $attendee, &$InvoiceInformation ) {

        $Hash = static::get_invoice_hash( $invoice_id );
        $PDFLink = static::get_pdf_link( $invoice_id, $Hash, $attendee );

        $InvoiceInformation['PrintButtonStyles'] = '';
        $InvoiceInformation['PrintAndDownloadButtonStyles'] = 'display:none;';
        $InvoiceInformation['PDFLink'] = $PDFLink;
        $InvoiceInformation['DownloadBtnStyles'] = '';

        $html2pdfApiKey = Helper::getOption('mec_invoice_html2pdf', '');
        $selectpdfApiKey = Helper::getOption('mec_invoice_selectpdf', '');
        $hide_invoice_download_button_in_email = 'on' === Helper::getOption('hide_invoice_download_button_in_email', 'off') ? true : false;

        if( ( !$html2pdfApiKey && !$selectpdfApiKey ) || ( !isset( $_GET['makePreview']) && $hide_invoice_download_button_in_email ) ) {

            $InvoiceInformation['DownloadBtnStyles'] = 'display:none;';
            $InvoiceInformation['PDFLink'] = '';
        }

        if( !isset( $_GET['invoiceID'] ) ) {

            $InvoiceInformation['PrintButtonStyles'] = 'display:none;';
            $InvoiceInformation['PrintAndDownloadButtonStyles'] = '';
        }

        return $InvoiceInformation;
    }

    public static function get_invoice_description( $event_id = 0 ){

        $InvoiceDescription = Helper::getOption('mec_invoice_descriptions', '');////

        $InvoiceDescription = trim(html_entity_decode(stripslashes($InvoiceDescription)));
        return trim(strip_tags($InvoiceDescription)) ? wpautop( $InvoiceDescription ) : '';
    }

    public static function get_allowed_reg_fields( $event_id ){

        $allowed_fields = get_post_meta( $event_id, 'mec_invoice_attendees_custom_fields', true );
        $allowed_fields = is_array( $allowed_fields ) ? $allowed_fields : false;

        if( !$allowed_fields || in_array( 'global', $allowed_fields, false ) ){

            $global_allowed_fields = \MEC\Settings\Settings::getInstance()->get_settings('attendees_custom_fields');
            $global_allowed_fields = is_array( $global_allowed_fields ) ? $global_allowed_fields : array();

            $allowed_fields = $global_allowed_fields;
        }

        return $allowed_fields;
    }

    public static function get_attendee_detail_html( $attendee, $event_id ){

        $allowed_fields = self::get_allowed_reg_fields( $event_id );
        $reg_fields = \MEC\Base::get_main()->get_reg_fields( $event_id );
        $reg_fields = apply_filters( 'mec_bookign_reg_form', $reg_fields, $event_id, null );

        $html = '';
        if( !empty($reg_fields) ){

            if ( isset( $attendee['reg'] ) && is_array( $attendee['reg'] ) ) {

                foreach ( $attendee['reg'] as $field_id => $value ) {


                    $field_args = isset( $reg_fields[ $field_id ] ) ? $reg_fields[ $field_id ] : false;
                    if ( ! $field_args || ! isset( $field_args['label'] ) ) {

                        continue;
                    }

                    $label = isset( $field_args['label'] ) ? $field_args['label'] : '';
                    $type = isset( $field_args['type'] ) ? $field_args['type'] : '';



                    if( false  !== array_search( $field_id, $allowed_fields, false ) ){

                        $label = __($label, 'mec-invoice');
                        $value = (is_string($value) ? $value : (is_array($value) ? implode(', ', $value) : '---'));
                        if( 'agreement' === $type ){

                            $label = sprintf( $label, '<a href="'.get_the_permalink($reg_fields[$field_id]['page']).'">'.get_the_title($reg_fields[$field_id]['page']).'</a>');
                            $value =  $value == '1' ? __('Yes', 'mec-invoice') : __('No', 'mec-invoice');
                        }

                        $html .= '<div class="mec-row" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;"><strong>'.$label.':</strong> '.$value.'</div>';
                    }
                }
            }
        }

        return $html;
    }

    public static function get_attendees_html( $invoice_id, $event_id, $tickets, $is_multi_date = false, $event_time = '', $DisplayQrcode = '' ) {

        $Hash = static::get_invoice_hash( $invoice_id );

        $datetime_format = Helper::get_invoice_date_format();
        $html_attendees = array();

        $book_id = get_post_meta($invoice_id, 'book_id', true);
        $transaction_id = get_post_meta($book_id, 'mec_transaction_id', true);
        if(!$transaction_id) {
            $transaction_id = get_post_meta($book_id, 'transaction_id', true);
        }
        $transactionObject = new \MEC\Transactions\Transaction( $transaction_id );
        $attendees = $transactionObject->get_tickets();
        $main = new \MEC_main();
        $reg_fields = $main->get_reg_fields($event_id);

        foreach( $tickets as $key => $ticket ) {

            $ticket_id = $ticket['id'] ?? '';
            $email = $ticket['email'] ?? '';
            $ticket_date = explode( ':',$ticket['date'] );
            $start_timestamp = $ticket_date[0];
            $ticket_date_start = date_i18n( $datetime_format, $start_timestamp );

            $ticket_html = '';
            $attendee_key = isset( $ticket['attendee_key'] ) ? $ticket['attendee_key'] + 1 : $key+1;

            self::add_ticket_info( $ticket_id, $tickets, $key, $ticket, $ticket_date );

            $transaction['date'] = Helper::get_date_label( $ticket['date'], $event_id );
            if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {
                $ticket_html .= '
                <div class="mec-invoice-attendee" style="clear: both;">
                    <img class="attendee-profile" src="' . get_avatar_url( $email ) . '" />
                ';
            } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4' ) {
                $ticket_html .= '
                <div class="mec-invoice-attendee" style="clear: both; padding-bottom: 20px; padding-right: 10px; margin-bottom: 25px; border-bottom: 1px solid #d1d2d3; clear: both; overflow: hidden;">
                    <img class="attendee-profile" style="width: 40px; height: 40px; border-radius: 8px; margin-right: 12px; margin-bottom: 5px;" src="' . get_avatar_url( $email ) . '" />
                ';
            }
            if ( Attendee::doIHaveAccess() && Attendee::hasCheckedIn( $invoice_id, $email, $attendee_key, $start_timestamp) ) {
                if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {
                    $ticket_html .= '<span class ="attendee-checked-badge" style="font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; color: #fff; font-size: 11px; line-height: 15px; padding: 3px 8px; border-radius: 13px; background: #28bf15; margin-top: 2px; display: inline-block; letter-spacing: 0.5px;">Checked</span>';
                } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                          \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                          \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                          \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                          \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4' ) {
                    $ticket_html .= '<span class ="attendee-checked-badge" style="font-family: DM Sans, Roboto, sans-serif; color: #fff; font-size: 11px; line-height: 15px; padding: 2px 7px; border-radius: 13px; background: #28bf15; margin-top: 2px; display: inline-block; letter-spacing: 0.5px; position: absolute;">Checked</span>';
                }
            }

            $more_info = self::get_attendee_detail_html( $ticket, $event_id );

            $ticket_name = $is_multi_date && $ticket_date_start ?  $ticket['name'] . ' --- ' . $ticket_date_start  :  $ticket['name'];
            $qr_code_link = static::get_qr_code_link( $invoice_id, $Hash, $ticket );

            if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {
                $ticket_html .= '<img class="attendee-qr" alt="Attendee QRCode" src="' . $qr_code_link . '" style="'.$DisplayQrcode.'" />';
            } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                     \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                     \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                     \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                     \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4' ) {
                $ticket_html .= '<img class="attendee-qr" style="float: right; width: 62px; height: 62px; border-radius: 0; margin-left: 12px;'.$DisplayQrcode.'" alt="Attendee QRCode" src="' . $qr_code_link . '" />';
            }

            if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {
                $ticket_html .=
                    '<div class="attendee-box" >
                        <span class="mec-invoice-attendee-name" style="color: #000; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 700; line-height: 24px; display: block;">
                            ' .$ticket_name . '
                        </span>
                        <span class="mec-invoice-attendee-email" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;">
                            ' . $email . '
                        </span>';

                        if(is_null($event_time) || isset($ticket['TicketTime']) && $ticket['TicketTime'] !== $event_time){

                            $ticket_html .= '<br>
                                <span class="mec-invoice-ticket-time" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;">
                                    ' .'<strong>' . __( 'Ticket Time:', 'mec-invoice' ) . '</strong> ' . $ticket['TicketTime'] . '
                                </span>';
                        }
            } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4') {
                $ticket_html .=
                    '<div class="attendee-box" >
                        <span class="mec-invoice-attendee-name" style="color: #000; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 700; line-height: 18px; display: block;">
                            ' .$ticket_name . '
                        </span>
                        <span class="mec-invoice-attendee-email" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 12px; font-weight: 400; line-height: 18px;">
                            ' . $email . '
                        </span>';

                        if(is_null($event_time) || isset($ticket['TicketTime']) && $ticket['TicketTime'] !== $event_time){

                            $ticket_html .= '<br>
                                <span class="mec-invoice-ticket-time" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 12px; font-weight: 400; line-height: 18px;">
                                    ' .'<strong>' . __( 'Ticket Time:', 'mec-invoice' ) . '</strong> ' . $ticket['TicketTime'] . '
                                </span>';
                        }
            }
            $attendee_variations="";
            foreach ($attendees as $key => $attendee){
                $reg_form = isset($attendee['reg']) && is_array($attendee['reg']) ? $attendee['reg'] : array();
                if($email==$attendee['email']){
                    if (isset($attendee['variations']) and is_array($attendee['variations']) and count($attendee['variations'])) {
                        $ticket_variations = $main->ticket_variations($event_id, $attendee['id']);
                        foreach ($attendee['variations'] as $variation_id => $variation_count) {
                            if (!$variation_count or ($variation_count and $variation_count < 0)) continue;

                            $variation_title = (isset($ticket_variations[$variation_id]) and isset($ticket_variations[$variation_id]['title'])) ? $ticket_variations[$variation_id]['title'] : '';
                            if (!trim($variation_title)) continue;

                            $attendee_variations .= '<div class="mec-row" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;">
                            <span> ' . $variation_title . '</span>
                            <span>(' . $variation_count . ')</span>
                        </div>';
                        }
                    }
                    foreach ($reg_form as $field_id => $value) {
                        $label = isset($reg_fields[$field_id]) ? $reg_fields[$field_id]['label'] : '';
                        $type = isset($reg_fields[$field_id]) ? $reg_fields[$field_id]['type'] : '';
                        if ($type == 'agreement'){
                            $attendee_variations .= '<div class="mec-row" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;">'.
                                '<strong>'.sprintf(__($label, 'mec-invoice'), '<a href="' . get_the_permalink($reg_fields[$field_id]['page']) . '">' . get_the_title($reg_fields[$field_id]['page']) . '</a>').': </strong>'.
                                ' <span>'. ($value == '1' ? __('Yes', 'mec-invoice') : __('No', 'mec-invoice')).'</span>'.
                                '</div>';
                        } elseif($label != 'Send Email') {
                            $attendee_variations .= '<div class="mec-row" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 13px; font-weight: 400; line-height: 22px;">'.
                                '<strong>'. __($label, 'mec-invoice').': </strong>'.
                                '<span>'.(is_string($value) ? $value : (is_array($value) ? implode(', ', $value) : '---')).'</span>'.
                                '</div>';
                        }
                    }
                }
            }

            $ticket_html .= $more_info.
                '</div>
            </div>';

            if( !isset($html_attendees[ $ticket_date_start ]) ){

                $html_attendees[ $ticket_date_start ] = '';
            }
            $html_attendees[ $ticket_date_start ] .= $ticket_html;

        }

        return implode( '',$html_attendees );
    }

    public static function get_tickets_html( $invoice_id, $event_id, $tickets, $book_all_occurrences, $InvoiceInformation ) {

        $date_format = Helper::get_invoice_date_format();
        $datetime_format = Helper::get_invoice_date_format();
        $tickets_html = '';
        $book_id = static::get_book_id( $invoice_id );
        $Hash = static::get_invoice_hash( $invoice_id );

        $event_title = $InvoiceInformation['EventName'];
        foreach ( $tickets as $key => $ticket ) {

            $attendee_key = $ticket['attendee_key'] ?? $key+1;

            $ticket_id = $ticket['id'] ?? '';
            $email = $ticket['email'] ?? '';
            $ticket_date = explode( ':',$ticket['date'] );
            $start_timestamp = $ticket_date[0];
            $ticket_date_start = date_i18n( $datetime_format, $start_timestamp );

            self::add_ticket_info( $ticket['id'], $tickets, $key, $InvoiceInformation );
            // PDF Link
            if(!$InvoiceInformation['PDFLink']) {
                $PDFLink = get_site_url(null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash . '&showPDF=true&attendee=' . $ticket['email']);
                $InvoiceInformation['PDFLink'] = $PDFLink;
            }

            $totalPrice = Helper::TotalPrice( $invoice_id, $email, $attendee_key );
            $totalPrice = $totalPrice ? $totalPrice : 0;
            $InvoiceInformation['TotalPrice'] = Helper::render_price( $totalPrice, $event_id ) ;
            $InvoiceInformation['AttendeeQrCode'] = static::get_qr_code_link( $invoice_id, $Hash, $ticket );
            $InvoiceInformation['AttendeeName'] = isset($ticket['name']) ? $ticket['name'] : $email;
            $InvoiceInformation['TicketID'] = $key;
            $InvoiceInformation['AttendeeDetail'] = self::get_attendee_detail_html( $ticket, $event_id );
            $InvoiceInformation['EventName'] = $event_title . ( !empty( $ticket_name ) ? ' - ' . $ticket_name : '' );


            if( $ticket_date ){

                $ticket_start_timestamp = $ticket_date[0];
                if($book_all_occurrences){
                    $ticket_event_url = get_post_permalink($event_id);
                    $ticket_event_date = date_i18n($date_format, $ticket_start_timestamp);
                } else {
                    $ticket_start_date = date_i18n($datetime_format, $ticket_date[0]);
                    $ticket_end_date = date_i18n($datetime_format, $ticket_date[1]);
                    $ticket_event_date = ($ticket_end_date == $ticket_start_date) ? $ticket_start_date : $ticket_start_date . ' - ' . $ticket_end_date;
                    $ticket_event_url = get_post_permalink($event_id) . '?occurrence=' . str_replace('/','-',date_i18n('Y-m-d', $ticket_start_timestamp));
                }

                $InvoiceInformation['EventDate'] = $ticket_event_date;
                $InvoiceInformation['EventURL'] = $ticket_event_url;
            }

            $InvoiceInformation = apply_filters( 'mec_invoice_information', $InvoiceInformation, $ticket, $attendee_key, $ticket_start_timestamp, $invoice_id, $book_id );

            $tickets_html.= Notifications::get_ticket_html($InvoiceInformation,'tickets');

        }

        return $tickets_html;
    }

    public static function get_modern_tickets_html( $invoice_id, $event_id, $tickets, $book_all_occurrences, $InvoiceInformation ) {

        $date_format = Helper::get_invoice_date_format();
        $datetime_format = Helper::get_invoice_date_format();
        $tickets_html = '';
        $book_id = static::get_book_id( $invoice_id );
        $Hash = static::get_invoice_hash( $invoice_id );

        $event_title = $InvoiceInformation['EventName'];
        foreach ( $tickets as $key => $ticket ) {

            $attendee_key = $ticket['attendee_key'] ?? $key+1;

            $ticket_id = $ticket['id'] ?? '';
            $email = $ticket['email'] ?? '';
            $ticket_date = explode( ':',$ticket['date'] );
            $start_timestamp = $ticket_date[0];
            $ticket_date_start = date_i18n( $datetime_format, $start_timestamp );

            self::add_ticket_info( $ticket['id'], $tickets, $key, $InvoiceInformation );
            // PDF Link
            if(!$InvoiceInformation['PDFLink']) {
                $PDFLink = get_site_url(null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash . '&showPDF=true&attendee=' . $ticket['email']);
                $InvoiceInformation['PDFLink'] = $PDFLink;
            }

            $totalPrice = Helper::TotalPrice( $invoice_id, $email, $attendee_key );
            $totalPrice = $totalPrice ? $totalPrice : 0;
            $InvoiceInformation['TotalPrice'] = Helper::render_price( $totalPrice, $event_id ) ;
            $InvoiceInformation['AttendeeQrCode'] = static::get_qr_code_link( $invoice_id, $Hash, $ticket );
            $InvoiceInformation['AttendeeName'] = isset($ticket['name']) ? $ticket['name'] : $email;
            $InvoiceInformation['TicketID'] = $key;
            $InvoiceInformation['AttendeeDetail'] = self::get_attendee_detail_html( $ticket, $event_id );
            $InvoiceInformation['EventName'] = $event_title . ( !empty( $ticket_name ) ? ' - ' . $ticket_name : '' );


            if( $ticket_date ){

                $ticket_start_timestamp = $ticket_date[0];
                if($book_all_occurrences){
                    $ticket_event_url = get_post_permalink($event_id);
                    $ticket_event_date = date_i18n($date_format, $ticket_start_timestamp);
                } else {
                    $ticket_start_date = date_i18n($datetime_format, $ticket_date[0]);
                    $ticket_end_date = date_i18n($datetime_format, $ticket_date[1]);
                    $ticket_event_date = ($ticket_end_date == $ticket_start_date) ? $ticket_start_date : $ticket_start_date . ' - ' . $ticket_end_date;
                    $ticket_event_url = get_post_permalink($event_id) . '?occurrence=' . str_replace('/','-',date_i18n('Y-m-d', $ticket_start_timestamp));
                }

                $InvoiceInformation['EventDate'] = $ticket_event_date;
                $InvoiceInformation['EventURL'] = $ticket_event_url;
            }

            $InvoiceInformation = apply_filters( 'mec_invoice_information', $InvoiceInformation, $ticket, $attendee_key, $ticket_start_timestamp, $invoice_id, $book_id );

            if ($key % 2 == 1 ) {
                $InvoiceInformation['PageBreak'] = 'page-break-inside:avoid; page-break-after:always;';
            } else {
                $InvoiceInformation['PageBreak'] = '';
            }

            $tickets_html.= Notifications::get_ticket_html($InvoiceInformation,'modern-ticket');

        }

        return $tickets_html;
    }

    public static function get_cart_items_html( $invoice_id, $event_id, $tickets, $event_time = '' ) {

        $CartItems = '';

        foreach ( $tickets as $key => $ticket) {

            $attendee_key = $ticket['attendee_key'] ?? $key++;
            $ticket_name = $ticket['ticket_name'];
            $ticket_count = $ticket['count'];
            $total_amount = $ticket['tickets_amount'];
            $ticket_variations = $ticket['variations_details'];
            $ticket_price = $ticket['ticket_price'];

            if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {

                $CartItems .= '<tr style="border-bottom: 1px solid #e7e8e9;">
                    <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                        ' . $ticket_name . '
                    </td>
                    <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                        ' . Helper::render_price( $ticket_price, $event_id ) . '
                    </td>
                    <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;text-align:center">
                        ' . $ticket_count . '
                    </td>
                    <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 600; line-height: 24px; padding: 20px 2px;">
                        ' .  Helper::render_price( $total_amount, $event_id ) . '
                    </td>
                </tr>';

            } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                      \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4') {

                $CartItems .= '<tr style="border-bottom: 1px solid #d1d2d3;">
                    <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                        ' . $ticket_name . '
                    </td>
                    <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                        ' . Helper::render_price( $ticket_price, $event_id ) . '
                    </td>
                    <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;text-align: center;">
                        ' . $ticket_count . '
                    </td>
                    <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 600; line-height: 24px; padding: 20px 2px;">
                        ' .  Helper::render_price( $total_amount, $event_id ) . '
                    </td>
                </tr>';

            }

            foreach ( $ticket_variations as $tk => $ticket_variation ) {

                if ( $ticket_variation ) {
                    // if ( $first_for_all ) {
                    //     $v = $v *  $ticket_count;
                    // }

                    if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {

                        $CartItems .= '<tr style="border-bottom: 1px solid #e7e8e9;">
                            <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                            <img src="' . MECINVOICEASSETS . '/img/child.svg" class="child-variation" style="width:12px;margin-left:10px;">
                                ' . $ticket_variation['description'] . '
                            </td>
                            <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                                ' . Helper::render_price($ticket_variation['price'], $event_id) . '
                            </td>
                            <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;text-align:center">
                                ' . $ticket_variation['count'] . '
                            </td>
                            <td class="mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 600; line-height: 24px; padding: 20px 2px;">
                                ' . Helper::render_price($ticket_variation['amount'], $event_id ) . '
                            </td>
                        </tr>';

                    } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                              \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                              \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                              \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                              \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4' ) {

                        $CartItems .= '<tr style="border-bottom: 1px solid #e7e8e9;">
                            <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                            <img src="' . MECINVOICEASSETS . '/img/child.svg" class="child-variation" style="width:12px;margin-left:10px;">
                                ' . $ticket_variation['description'] . '
                            </td>
                            <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;">
                                ' . Helper::render_price($ticket_variation['price'], $event_id) . '
                            </td>
                            <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;text-align: center">
                                ' . $ticket_variation['count'] . '
                            </td>
                            <td class="mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 600; line-height: 24px; padding: 20px 2px;">
                                ' . Helper::render_price($ticket_variation['amount'], $event_id ) . '
                            </td>
                        </tr>';
                    }
                }
            }
        }

        return $CartItems;
    }

    public static function tax_detail_rows_html ( $invoice_id, $attendee = null ) {

        $rows = Helper::ِInvoiceFeeDetailRows( $invoice_id, $attendee );

        $html = '';

        if (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'full') {

            $attributes = 'class="test mec-invoice-item" style="color: #75787b; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;"';
            foreach( $rows as $description => $rendered_price ){

                $html .= '<tr style="border-bottom: 1px solid #e7e8e9;"><td '.$attributes.'>'. $description .'</td><td '.$attributes.'>'. $rendered_price .'</td></tr>';
            }

            return !empty( $html ) ? '<tr style="border-bottom: 1px solid #e7e8e9;"><td class="mec-invoice-item-tax" style="color: #000; font-family: -apple-system,BlinkMacSystemFont, \'Segoe UI\',Roboto, sans-serif; font-size: 14px; font-weight: 600; line-height: 24px; padding: 25px 2px 5px;{{TaxStyle}}">'. __('Tax\Fee','mec-invoice') .'</td></tr>' . $html : '';

        } elseif (\MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'standard' ||
                  \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
                  \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
                  \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
                  \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4') {

            $attributes = 'class="test mec-invoice-item" style="color: #141516; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 24px; padding: 20px 2px;"';
            foreach( $rows as $description => $rendered_price ){

                $html .= '<tr style="border-bottom: 1px solid #d1d2d3;"><td '.$attributes.'>'. $description .'</td><td '.$attributes.'>'. $rendered_price .'</td></tr>';
            }

            return !empty( $html ) ? '<tr style="border-bottom: 1px solid #d1d2d3;"><td class="mec-invoice-item-tax" style="color: #000; font-family: DM Sans, Roboto, sans-serif; font-size: 14px; font-weight: 500; line-height: 24px; padding: 25px 2px 5px; {{TaxStyle}}">'. __('Tax\Fee','mec-invoice') .'</td></tr>' . $html : '';
        }
    }

    public static function add_ticket_info($ticket_id, $tickets, $ticket_key, &$InvoiceInformation, $ticket_dates = null ){

        $time_format = get_option('time_format');

        $event_date_timestamp = isset( $InvoiceInformation['EventDate'] ) ? strtotime( $InvoiceInformation['EventDate'] ) : '';
        $ticket_info = isset( $tickets[ $ticket_key ] ) ? $tickets[ $ticket_key ] : false;
        $display_ticket_name = (bool) \MEC\Settings\Settings::getInstance()->get_settings('invoice_display_ticket_name');

        if(\MEC\Settings\Settings::getInstance()->get_settings('disable_ticket_times')){
            $display_ticket_time= false;
        }else{
            $display_ticket_time = (bool) \MEC\Settings\Settings::getInstance()->get_settings('invoice_display_ticket_time');
        }

        $ticket_times_status = $display_ticket_time ? true : false;

        if( !$ticket_info ){

            return $InvoiceInformation;
        }

        if($display_ticket_name){

            //ticket_name
            $InvoiceInformation['TicketName'] = isset($ticket_info['ticket_name']) && !empty($ticket_info['ticket_name'])  ? $ticket_info['ticket_name'] : '';
            $InvoiceInformation['TicketNameStyle'] = '';
        } else {
            $InvoiceInformation['TicketNameStyle'] = 'display:none;';
        }
        $start_time = $end_time = '';


        if( $ticket_times_status ){

            $ticket_timestamp_start = isset($ticket_info['ticket_timestamp_start']) ? $ticket_info['ticket_timestamp_start'] : $event_date_timestamp;
            $ticket_timestamp_end = isset($ticket_info['ticket_timestamp_end']) ? $ticket_info['ticket_timestamp_end'] : '';



            $InvoiceInformation['TicketDateTimeStartTimestamp'] = $ticket_timestamp_start;
            // $InvoiceInformation['TicketDateTimeEndTimestamp'] = $ticket_timestamp_end;

            $start_time = date_i18n( $time_format, $ticket_timestamp_start );
            $end_time = date_i18n( $time_format, $ticket_timestamp_end );
        }else{
            if( is_array( $ticket_dates ) ){

                $InvoiceInformation['TicketDateTimeStartTimestamp'] = $ticket_dates[0];
                // $InvoiceInformation['TicketDateTimeEndTimestamp'] = $ticket_dates[1];

                $start_time = date_i18n( $time_format, $ticket_dates[0] );
                $end_time = date_i18n( $time_format, $ticket_dates[1] );
            }
        }

        if( $display_ticket_time && !empty( $start_time ) ){

            $transactionObject = Helper::TransactionObject( $InvoiceInformation['InvoiceID'] );
            $date = $transactionObject->get_date();
            $event_date = explode( ':', $date );
            $timeFormat = Helper::getOption('tickets_time_format', 'h:i A');
            $timeFormat = $timeFormat ? $timeFormat : 'h:i A';

            $InvoiceInformation['TicketTimeStart'] = date_i18n( $timeFormat, $event_date[0] );
            $InvoiceInformation['TicketTimeEnd'] = $end_time;
            $InvoiceInformation['TicketTime'] = $start_time . ' - ' .$end_time;
        }

        return $InvoiceInformation;
    }

    public static function get_invoice_information( $invoice_id, $attendee_list = array(), $attendee = false ) {

        $attendee_email = is_array( $attendee_list ) ? $attendee_list['email'] : $attendee_list;

        $transaction_id = Helper::TransactionID( $invoice_id );
        $transactionObject = Helper::TransactionObject( $invoice_id );
        $filters = Helper::get_ticket_filters_for_transaction( $transactionObject, $invoice_id, $attendee );
        $tickets = $transactionObject->get_tickets_details( $filters );
        $total_tickets = count( $tickets );
        $is_multi_date = $transactionObject->get_total_dates( $filters ) > 1 ? true : false;
        $event_id = $transactionObject->get_event_id();
        $date = $transactionObject->get_date();

        $book_id = static::get_book_id( $invoice_id );
        $Hash = static::get_invoice_hash( $invoice_id );
        $doCheckinHash = static::get_checkin_hash( $invoice_id );
        $currencies = \MEC\Base::get_main()->get_currencies();

        $dateFormat = Helper::getOption('tickets_date_format', 'M j, Y');
        $dateFormat = $dateFormat ? $dateFormat : 'M j, Y';
        $timeFormat = Helper::getOption('tickets_time_format', 'h:i A');
        $timeFormat = $timeFormat ? $timeFormat : 'h:i A';

        $CustomStyle = Helper::getOption('mec_invoice_custom_styles', '');
        $CustomStyle = trim(html_entity_decode(stripslashes($CustomStyle)));
        $CustomStyle = trim(strip_tags($CustomStyle)) ? $CustomStyle : '';

        if($total_tickets > 3) {

			$CustomStyle .= '@media print {.mec-invoice-body-half2.mec-invoice-attendees {page-break-before: always;}}';
		}

        if (Helper::getOption('mec_invoice_show_invoice_description_in_pdf', 'off') != 'on') {

            $CustomStyle .= '@media print {.mec-invoice-footer {display: none;}}';
        }

        $FavIcon = get_site_icon_url();
        $FavIcon = trim(html_entity_decode(stripslashes($FavIcon)));
        $FavIcon = trim(strip_tags($FavIcon)) ? $FavIcon : '';

        $invoice = get_post( $invoice_id );
        $event = get_post( $event_id );

        $event_img = wp_get_attachment_url( get_post_thumbnail_id($event_id), 'thumbnail' );

        $render = \MEC::getInstance('app.libraries.render');
        $event->data = $render->data($event_id);

        // new event date
        $main = new \MEC_main();
        $date_format = Helper::get_invoice_date_format();
        $event_date_format = Helper::getOption('event_date_format', get_option('date_format'));
        $time_format = get_option('time_format');
        $event_date = explode( ':', $date );

        if (is_numeric($event_date[0]) and is_numeric($event_date[1])) {
            $start_datetime = date($date_format . ' ' . $time_format, $event_date[0]);
            $end_datetime = date($date_format . ' ' . $time_format, $event_date[1]);
            $start_timestamp = $event_date[0];
            $end_timestamp = $event_date[1];
        } else {
            $start_datetime = $event_date[0];
            $end_datetime = $event_date[1];
            $start_timestamp = strtotime( $event_date[0] );
            $end_timestamp = strtotime( $event_date[1] );
        }

        $event_time_date = ((isset($event_date[0]) and isset($event_date[1])) ? sprintf(__('%s to %s', 'mec-invoice'), $start_datetime, $end_datetime) : __('Unknown', 'mec-invoice'));

        if (isset($start_datetime) and !empty($start_datetime)) {
            $event_start_time = date_i18n( $time_format, $start_timestamp );
        }
        if (isset($end_datetime) and !empty($end_datetime)){
            $event_end_time = date_i18n( $time_format, $end_timestamp );
        }

        $event_start_date = date_i18n( $event_date_format, $start_timestamp );
        $event_end_date = date_i18n( $event_date_format, $end_timestamp );

        $esd = $start_timestamp;
        $evd = $end_timestamp;
        $ticket_start_timestamp = $start_timestamp;
        $ticket_end_timestamp = $start_timestamp;

        $event_booking_options = get_post_meta($event_id, 'mec_booking', true);
        if(!is_array($event_booking_options)) $event_booking_options = array();

        $book_all_occurrences = 0;
        if(isset($event_booking_options['bookings_all_occurrences'])) {

            $book_all_occurrences = (int) $event_booking_options['bookings_all_occurrences'];
        }

        $maximum_dates = Helper::getOption('booking_maximum_dates', false) ? Helper::getOption('booking_maximum_dates', false) : 6;
        $dd = explode( ':', $date );
        $ed = explode(' - ', $date );
        if($book_all_occurrences){
            $next_occurrences = $main->getRender()->dates($event_id, NULL, $maximum_dates, date('Y-m-d', strtotime('-1 day', strtotime($start_datetime))));
            $new_event_date = '';
            foreach($next_occurrences as $next_occurrence) $new_event_date .= $main->date_label($next_occurrence['start'], $next_occurrence['end'], $date_format, ' - ', false)."<br>";
            $eventURL = get_post_permalink($event_id);
        } else {
            $new_event_date = ($event_end_date == $event_start_date) ? $event_start_date : $event_start_date . ' - ' . $event_end_date;
            $eventURL = get_post_permalink($event_id) . '?occurrence=' . str_replace('/','-',date_i18n('Y-m-d', $esd));
        }


        $allday = isset($event->data->meta['mec_allday']) ? $event->data->meta['mec_allday'] : 0;
        if ($allday == '0' and isset($event->data->time) and trim($event->data->time['start'])) :
            $event_time = $event_start_time . ' - ' . $event_end_time; //TODO
        else:
            $event_time =  __('All Day', 'mec-invoice');
        endif;

        $total_price = $transactionObject->get_total( $filters );
        $payable = $transactionObject->get_payable( $filters );
        $tax = $transactionObject->get_fees( $filters );
        $show_tax_details = 'on' == Helper::getOption('show_invoice_tax_detail', 'off');
        $tax_detail_html = '';
        if( $show_tax_details ) {

            $tax_detail_html = static::tax_detail_rows_html( $invoice_id );
        }
        $discount = $transactionObject->get_discounts( $filters );

        $datetime_format = Helper::get_invoice_date_format();
//        $invoiceDate = date_i18n($datetime_format, $date);
        $invoiceDate = date_i18n($datetime_format, get_post_meta($invoice_id, 'date', true));

        $InvoiceInformation = array(

            // Links
            'InvoiceLink' => get_site_url( null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash ),

            'WebSiteUrl' => get_site_url(),
            'blog_name' => get_option( 'blogname' ),
            'CustomStyle' => $CustomStyle,
			'FavIcon' => $FavIcon,

            // Prices
            'TotalPrice' => Helper::render_price( $total_price, $event_id ),
            'Paid' => Helper::render_price( $payable, $event_id ),
            'Discount' => $discount ? '-' .  Helper::render_price($discount, $event_id) : '',
            'Currency'  => isset($currencies[Helper::getOption('currency', false)]) ? $currencies[Helper::getOption('currency', false)] : '',
            'Tax'  => $tax ? Helper::render_price( $tax, $event_id ) : '',
            'TaxDetail'  => $tax_detail_html,

            'attendeeParam' => '',
            'TicketSeatNumbers' => '-',

            // QRCode
            'QRCodeUrl' => get_site_url(null, '?invoiceID='. $invoice_id . '&makeQRCode='.$Hash . '&checkin='.$doCheckinHash),

            // Event Information
            'EventID'     => $event->ID,
            'EventName'     => $event->post_title,
            'EventUrl'      => $eventURL,
            'EventImg'      => $event_img,
            'EventImgStyle' => empty($event_img) ? 'display: none;' : '',
            'EventDate'     => $new_event_date,
            'EventTime'     => $event_time,
            'EventStartTime'     => $event_time,
            'EventStartDate'     => $new_event_date,


            'TicketName' => '',
            'TicketDate' => $new_event_date,
            'TicketTime' => $event_time,
            'TicketTimeStart' => date_i18n( $timeFormat, $start_timestamp ),
            'TicketTimeEnd' => date_i18n($timeFormat, $tickets[0]['ticket_timestamp_end']),


            // Invoice Information
            'InvoiceID' => $invoice_id,
            'InvoiceTitle' => __('Invoice', 'mec-invoice') . ' #' . Helper\Invoice::get_invoice_number( $invoice_id ),
            'InvoiceDescription' => static::get_invoice_description( $event_id ),
            'InvoiceDate' => $invoiceDate,
            'InvoiceDateD' => date_i18n($dateFormat, strtotime($ed[0])),
            'InvoiceDateT' => date_i18n($timeFormat, strtotime($event->data->time['start'])),
            'InvoiceNumber' => '#' . Helper\Invoice::get_invoice_number($invoice_id),
            'TransactionID' => $transaction_id,
            'InvoiceMoreInfo' => apply_filters( 'mec_invoice_more_info', '', $invoice->ID, $book_id ),

            // Attendees
            'AttendeesList' => '',
            'name' => $attendee_list['name']??'',

            // Cart Items
            'CartItems' => '',

            // Cart Tickets Items
            'Tickets' => '',

            'PageBreak' => '',

            'AssetsUrl' => MECINVOICEASSETS,

            'SeatActive' => is_plugin_active( 'mec-seat/mec-seat.php' ) ? '':'display:none;',

            'DisplayQrcode' => \MEC\Settings\Settings::getInstance()->get_settings('invoice_hide_ticket_qrcode') == 'on' ? 'display:none;':'',
        );

        $InvoiceInformation = static::get_company_information( $invoice_id, $InvoiceInformation );
        $InvoiceInformation = static::get_event_information( $event_id, $InvoiceInformation );
        $InvoiceInformation = static::get_buttons_information( $invoice_id, $attendee_email, $InvoiceInformation );

        if ( Helper::getOption('mec_invoice_rendering_mode', 'full') == 'ticket' ) {
            $html = static::get_tickets_html( $invoice_id, $event_id, $tickets, $book_all_occurrences, $InvoiceInformation );
            preg_match_all('/\[\[(.*?)\]\]/i', $html, $translatableMatches);
            if (is_array($translatableMatches)) {

                foreach ($translatableMatches[0] as $key => $text) {

                    $html = str_replace($text, __($translatableMatches[1][$key], 'mec-invoice'), $html);
                }
            }
            $InvoiceInformation['Tickets'] = $html;
        } else if ( Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-1' ||
            Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-2' ||
            Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-3' ||
            Helper::getOption('mec_invoice_rendering_mode', 'full') == 'modern-ticket-4' ) {
            $html = static::get_modern_tickets_html( $invoice_id, $event_id, $tickets, $book_all_occurrences, $InvoiceInformation );
            preg_match_all('/\[\[(.*?)\]\]/i', $html, $translatableMatches);
            if (is_array($translatableMatches)) {

                foreach ($translatableMatches[0] as $key => $text) {

                    $html = str_replace($text, __($translatableMatches[1][$key], 'mec-invoice'), $html);
                }
            }
            $InvoiceInformation['Tickets'] = $html;
            // Cart Items
            $InvoiceInformation['CartItems'] = static::get_cart_items_html( $invoice_id, $event_id, $tickets, $event_time );
        } else {

            // Attendees
            $InvoiceInformation['AttendeesList'] = static::get_attendees_html( $invoice_id, $event_id, $tickets, $is_multi_date, $event_time, $InvoiceInformation['DisplayQrcode'] );
            // Cart Items
            $InvoiceInformation['CartItems'] = static::get_cart_items_html( $invoice_id, $event_id, $tickets, $event_time );
        }

        if( !empty( $attendee_email ) ){

            $InvoiceInformation['attendeeParam'] = '&attendee='.$attendee_email;
        }

        return $InvoiceInformation;
    }
}
