<?php

namespace MEC_Invoice\Core\Manager;

use MEC_main;
use WP_REST_Response;
use MEC_Invoice\Attendee;
use MEC_Invoice\NotificationsSettings;

// Don't load directly
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 *   Controller.
 *
 * @author      Webnus <info@webnus.biz>
 * @package     Modern Events Calendar
 * @since       1.0.0
 */
class Controller
{

    /**
     *  Instance of this class.
     *
     * @since   1.0.0
     * @access  public
     * @var     MEC_Invoice
     */
    public static $instance;

    /**
     *  The directory of this file
     *
     * @access  public
     * @var     string
     */
    public static $dir;

    /**
     *  Provides access to a single instance of a module using the Singleton pattern.
     *
     * @return  object
     * @since   1.0.0
     */
    public static function instance()
    {

        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }


    public function __construct()
    {
        if (self::$instance === null) {
            self::$instance = $this;
        }

        self::settingUp();
        self::setHooks();
        self::init();
    }

    /**
     *  Set Hooks.
     *
     * @since   1.0.0
     */
    public static function setHooks()
    {
        add_action('mec_booking_completed', [self::$instance, 'addInvoiceManually'], 1000, 1);
        add_action('admin_enqueue_scripts', [self::$instance, 'enqueueAdminScripts']);
        add_action('wp_ajax_mec_invoice_change_ticket_status', [self::$instance, 'changeTicketStatus']);
        add_action('wp_ajax_mec_invoice_import_old_invoices', [self::$instance, 'importOldInvoices']);
        add_action('mec-settings-page-before-form-end', [self::$instance, 'renderSettingsPage'], 10, 1);
        add_filter('mec-settings-items-settings', [self::$instance, 'applySettingsPageMenu'], 10, 2);
        add_filter('mec_attendees_list_html', [self::$instance, 'applyCheckStatus'], 10, 5);
        add_filter('mec_attendees_list_header_html', [self::$instance, 'applyCheckStatusCol'], 10, 1);
        add_filter('mec_attendees_list_header_html', [self::$instance, 'addExportButton'], 10, 3);
        add_action('wp_ajax_export_attendees_as_csv', [self::$instance, 'exportAttendeesAsCsv'], 10);
        add_action('wp_loaded', [self::$instance, 'processExportAction']);
        add_action('mec_booking_verified', [self::$instance, 'verifyInvoice']);
        add_action('mec_booking_confirmed', [self::$instance, 'verifyInvoice']);
        add_action('mec_booking_pended', [self::$instance, 'verifyInvoice']);
        add_action('mec_booking_waiting', [self::$instance, 'verifyInvoice']);
        add_action('mec_booking_canceled', [self::$instance, 'cancelInvoice']);
        add_action('mec_booking_rejected', [self::$instance, 'cancelInvoice']);
        add_action('admin_menu', [self::$instance, 'exportMenu'], 100);
        add_action('admin_menu', [self::$instance, 'apiKeyAppMenu'], 101);
        add_action('wp_ajax_mec_update_translate_invoice_app', [self::$instance, 'updateTranslateApp']);
        add_action('rest_api_init', [self::$instance, 'routes_api_app']);
        add_action('delete_post', [self::$instance, 'deleteInvoice'], 10, 1);
        add_action('add_event_booking_sections_left_menu', [self::$instance, 'renderEventDescriptionMenu'], 10, 1);
        add_action('mec_metabox_booking', [self::$instance, 'renderEventDescriptionMetaBox'], 10);
        add_action('save_post', [self::$instance, 'save_event'], 10);
        add_action('mec-invoice-check-in', [self::$instance, 'sendCheckInStatus'], 10, 4);
        add_action('mec-invoice-check-out', [self::$instance, 'sendCheckOutStatus'], 10, 3);
        add_action('mec_display_report_page', [self::$instance, 'renderMECDisplayReportPageScripts'], 10);
        add_action('mec_booking_added', [self::$instance, 'sendNewBookingStatus'], 10);
        add_filter('script_loader_src', [self::$instance, 'DisableAutoSave'], 500, 2);

        add_action( 'mec_import_item',  array( __CLASS__, 'import_form' ), 10, 1 );
		add_filter( 'mec_import_item_action', array( __CLASS__, 'import_action' ), 10, 2 );

        NotificationsSettings::init();
    }

    /**
     * Disable AutoSave
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function DisableAutoSave($src, $handle)
    {
        global $typenow;
        if ('autosave' != $handle || $typenow != 'mec_invoice')
            return $src;
        return '';
    }

    /**
     * Send Pusher Signal for checked in attendee
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function sendCheckInStatus($invoiceID, $attendee, $place, $start_timestamp)
    {
        if (!\MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', '')) {
            return;
        }
        include realpath(MECINVOICEDIR . '/libraries/pusher/vendor/autoload.php');
        $options = array(
            'cluster' => \MEC_Invoice\Helper::getOption('mec_invoice_pusher_cluster', ''),
            'useTLS' => true
        );
        $pusher = new \Pusher\Pusher(
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_secret', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_app_id', ''),
            $options
        );

        $book_id = get_post_meta($invoiceID, 'book_id', true);
        $event_id = get_post_meta($invoiceID, 'event_id', true);
        $event_name = get_the_title($event_id);
        $feature_class = new \MEC_feature_mec();
        $event = $feature_class->db->select("SELECT * FROM `#__mec_events` WHERE `post_id` = {$event_id}", 'loadAssoc');
        if (!$event) {
            return;
        }
        $transaction_id = get_post_meta($book_id, 'mec_transaction_id', true);
        $transaction = get_option($transaction_id);
        $transaction['date'] = explode(':', $transaction['date']);
        $main = \MEC::getInstance('app.libraries.main');
        foreach ($transaction['date'] as $k => $d) {
            if (!$d) {
                continue;
            }
            $transaction['date'][$k] = $main->standardize_format(date('Y/m/d H:i:s', $d), \MEC_Invoice\Helper::getOption('invoice_date_format', 'Y/m/d'));
        }
        $dates = $transaction['date'];
        $past_start_date = isset($dates[0]) ? $dates[0] : '';
        $past_end_date = isset($dates[1]) ? $dates[1] : '';

        $occurrence = sprintf(__('%s to %s', 'mec-invoice'), $past_start_date, $past_end_date);
        $data['status'] = 'check-in';
        $data['attendee'] = $attendee;
        $data['event_name'] = $event_name;
        $data['event_id'] = $event_id;
        $data['occurrence'] = $occurrence;
        $data['start_date'] = $past_start_date;
        $pusher->trigger('Attendees', 'Check_in_Check_out', $data);
    }

    /**
     * Send Pusher Signal for checked in attendee
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function sendNewBookingStatus($book_id)
    {
        if (!\MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', '')) {
            return;
        }
        include realpath(MECINVOICEDIR . '/libraries/pusher/vendor/autoload.php');
        $options = array(
            'cluster' => \MEC_Invoice\Helper::getOption('mec_invoice_pusher_cluster', ''),
            'useTLS' => true
        );
        $pusher = new \Pusher\Pusher(
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_secret', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_app_id', ''),
            $options
        );

        $transaction_id = get_post_meta($book_id, 'mec_transaction_id', true);
        $transaction = get_option($transaction_id);
        $event_id = $transaction['event_id'];
        $event_name = get_the_title($event_id);
        $book_name = get_the_title($book_id);
        $feature_class = new \MEC_feature_mec();
        $event = $feature_class->db->select("SELECT * FROM `#__mec_events` WHERE `post_id` = {$event_id}", 'loadAssoc');
        if (!$event) {
            return;
        }
        $transaction['date'] = explode(':', $transaction['date']);
        $main = \MEC::getInstance('app.libraries.main');
        foreach ($transaction['date'] as $k => $d) {
            if (!$d) {
                continue;
            }
            $transaction['date'][$k] = $main->standardize_format(date('Y/m/d H:i:s', $d), \MEC_Invoice\Helper::getOption('invoice_date_format', 'Y/m/d'));
        }
        $dates = $transaction['date'];
        $past_start_date = isset($dates[0]) ? $dates[0] : '';
        $past_end_date = isset($dates[1]) ? $dates[1] : '';

        $occurrence = sprintf(__('%s to %s', 'mec-invoice'), $past_start_date, $past_end_date);
        $data['status'] = 'new-booking';
        $data['event_name'] = $event_name;
        $data['book_name'] = $book_name;
        $data['event_id'] = $event_id;
        $data['occurrence'] = $occurrence;
        $data['start_date'] = $past_start_date;
        $pusher->trigger('Attendees', 'Check_in_Check_out', $data);
    }

    /**
     * Send Pusher Signal for checked out attendee
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function sendCheckOutStatus($invoiceID, $attendee, $place)
    {
        if (!\MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', '')) {
            return;
        }
        include realpath(MECINVOICEDIR . '/libraries/pusher/vendor/autoload.php');
        $options = array(
            'cluster' => \MEC_Invoice\Helper::getOption('mec_invoice_pusher_cluster', ''),
            'useTLS' => true
        );
        $pusher = new \Pusher\Pusher(
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_secret', ''),
            \MEC_Invoice\Helper::getOption('mec_invoice_pusher_app_id', ''),
            $options
        );

        $book_id = get_post_meta($invoiceID, 'book_id', true);
        $event_id = get_post_meta($invoiceID, 'event_id', true);
        $event_name = get_the_title($event_id);
        $feature_class = new \MEC_feature_mec();
        $event = $feature_class->db->select("SELECT * FROM `#__mec_events` WHERE `post_id` = {$event_id}", 'loadAssoc');
        if (!$event) {
            return;
        }
        $transaction_id = get_post_meta($book_id, 'mec_transaction_id', true);
        $transaction = get_option($transaction_id);
        $transaction['date'] = explode(':', $transaction['date']);
        $main = \MEC::getInstance('app.libraries.main');
        foreach ($transaction['date'] as $k => $d) {
            if (!$d) {
                continue;
            }
            $transaction['date'][$k] = $main->standardize_format(date('Y/m/d H:i:s', $d), \MEC_Invoice\Helper::getOption('invoice_date_format', 'Y/m/d'));
        }
        $dates = $transaction['date'];
        $past_start_date = isset($dates[0]) ? $dates[0] : '';
        $past_end_date = isset($dates[1]) ? $dates[1] : '';

        $occurrence = sprintf(__('%s to %s', 'mec-invoice'), $past_start_date, $past_end_date);
        $data['status'] = 'check-out';
        $data['attendee'] = $attendee;
        $data['event_name'] = $event_name;
        $data['event_id'] = $event_id;
        $data['occurrence'] = $occurrence;
        $data['start_date'] = $past_start_date;
        $pusher->trigger('Attendees', 'Check_in_Check_out', $data);
    }

    /**
     * Send Pusher Signal for checked in attendee
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function renderMECDisplayReportPageScripts()
    {
        if (!\MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', '')) {
            return;
        }
        echo '<script src="https://js.pusher.com/5.1/pusher.min.js"></script>';
        ?>
        <script>
            Pusher.logToConsole = false;
            var pusher = new Pusher('<?php echo \MEC_Invoice\Helper::getOption('mec_invoice_pusher_key', '') ?>', {
                cluster: '<?php echo \MEC_Invoice\Helper::getOption('mec_invoice_pusher_cluster', '') ?>',
                forceTLS: true
            });

            var channel = pusher.subscribe('Attendees');
            channel.bind('Check_in_Check_out', function (data) {
                if (data.status == 'check-in') {
                    jQuery('#mec-pusher-messages').append('<div class="message-wrap check-in"><span class="check-in">Check-in</span><div class="message">' + data.attendee + ' - ' + '<b>' + data.event_name + '</b>' + ' (' + data.occurrence + ')' + '</div></div>')
                } else if (data.status == 'check-out') {
                    jQuery('#mec-pusher-messages').append('<div class="message-wrap check-out"><span class="check-out">Check-out</span><div class="message">' + data.attendee + ' - ' + '<b>' + data.event_name + '</b>' + ' (' + data.occurrence + ')' + '</div></div>')
                } else {
                    jQuery('#mec-pusher-messages').append('<div class="message-wrap new-booking"><span class="new-booking">New Booking</span><div class="message">' + data.book_name + ' - ' + '<b>' + data.event_name + '</b>' + ' (' + data.occurrence + ')' + '</div></div>')
                }
                if (jQuery('.mec-reports-selectbox-dates').val() == data.start_date && jQuery('.mec-reports-selectbox-event').val() == data.event_id) {
                    mec_event_attendees(data.event_id, data.start_date)
                }
            });

            jQuery(document).on('click', '#mec-pusher-messages .message-wrap', function () {
                jQuery(this).remove();
            })
        </script>
        <div id="mec-pusher-messages"></div>
        <?php
    }

    /**
     * codeMirror Enqueue Scripts
     *
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function codeMirrorEnqueueScripts()
    {
        $cm_settings['codeEditor'] = wp_enqueue_code_editor(array('type' => 'text/css'));
        wp_localize_script('jquery', 'cm_settings', $cm_settings);

        wp_enqueue_script('wp-theme-plugin-editor');
        wp_enqueue_style('wp-codemirror');
    }

    /**
     * Save event data
     *
     * @param int $post_id
     * @return void
     * @author Webnus <info@webnus.biz>
     */
    public function save_event($post_id)
    {

        // Check if our nonce is set.
        if (!isset($_POST['mec_event_nonce'])) {
            return;
        }

        // Verify that the nonce is valid.
        if (!wp_verify_nonce($_POST['mec_event_nonce'], 'mec_event_data')) {
            return;
        }
        // If this is an autosave, our form has not been submitted, so we don't want to do anything.
        if (defined('DOING_AUTOSAVE') and DOING_AUTOSAVE) {
            return;
        }

        $_mec = isset($_POST['mec-invoice']) ? $_POST['mec-invoice'] : array();
        if (!$_mec) {
            return;
        }

        if (isset($_mec['i_event_descriptions'])) {
            update_post_meta($post_id, 'i_event_descriptions', $_mec['i_event_descriptions']);
        }

        if (isset($_mec['invoice_attendees_custom_fields'])) {

            $_mec['invoice_attendees_custom_fields'] = !in_array('global', (array)$_mec['invoice_attendees_custom_fields']) ? $_mec['invoice_attendees_custom_fields'] : array('global');
            update_post_meta($post_id, 'mec_invoice_attendees_custom_fields', $_mec['invoice_attendees_custom_fields']);
        }
    }

    /**
     * Render Event Description MetaBox
     * @param object $post
     */
    public function renderEventDescriptionMetaBox($post)
    {
        ?>
        <script>
            jQuery(document).ready(function($){

                $.each($(".mec-select2"), function(i,v){

                if( $(v).attr('name').search(":i:") > 0 ){

                    return;
                }

                if( typeof $(v).data('select2-id') == 'undefined' ){

                    $(v).select2();
                }
                });
            });
        </script>
        <div id="mec-organizer-payments">
            <div id="mec-event-descriptions" class="mec-meta-box-fields mec-booking-tab-content">
                <h4 class="mec-title"><?php _e('Event Excerpt for Invoice', 'mec-invoice'); ?></h4>
                <span style="font-size: 11px;color: #888;margin: 0;margin-top: -10px;display:block;"><?php _e('Short descriptions about the event in the invoice. It is set to Modern Style as default.', 'mec-invoice'); ?></span>
                <div class="mec-form-row" id="mec_organizer_gateways_form_container">
                    <ul>
                        <?php
                        $content = get_post_meta($post->ID, 'i_event_descriptions', true);

                        wp_editor( $content, 'mec-invoice_event_descriptions', array(
                            'textarea_name' => 'mec-invoice[i_event_descriptions]',
                            'textarea_rows' => 4,
                        ));
                        ?>
                    </ul>
                </div>

                <div>
                    <label class="mec-col-3"
                           for="mec_settings_invoice_attendees_custom_fields"><?php esc_html_e('Custom Fields in Invoice', 'mec-invoice'); ?></label>
                    <div class="mec-col-9">
                        <select style="width:100%" name="mec-invoice[invoice_attendees_custom_fields][]" class="mec-select2"
                                multiple>
                            <?php
                            $fields = \MEC\Base::get_main()->get_reg_fields($post->ID);

                            if (is_array($fields)) {

                                $attendees_custom_fields = get_post_meta($post->ID, 'mec_invoice_attendees_custom_fields', true);
                                $attendees_custom_fields = is_array($attendees_custom_fields) ? $attendees_custom_fields : array('global');

                                $selected = in_array('global', $attendees_custom_fields);
                                echo '<option value="global" ' . selected(true, $selected, false) . '>' . esc_html__('Global', 'mec-invoice') . '</option>';
                                foreach ($fields as $field_id => $field_args) {

                                    $type = isset($field_args['type']) && !empty($field_args['type']) ? $field_args['type'] : false;
                                    if (!$type || in_array($type, array('name', 'mec_email'), false)) {
                                        continue;
                                    }

                                    $label = isset($field_args['label']) && !empty($field_args['label']) ? $field_args['label'] : false;
                                    if ($label==="I agree with %s") {
                                        continue;
                                    }

                                    $selected = in_array($field_id, $attendees_custom_fields);
                                    echo '<option value="' . $field_id . '" ' . selected(true, $selected, false) . '>' . $label . '</option>';
                                }
                            }
                            ?>
                        </select>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Event Description Menu
     *
     * @since     1.0.0
     */
    public function renderEventDescriptionMenu($post)
    {
        // $menuItems[__('Event Excerpt for Invoice', 'mec-invoice')] =  'mec-event-descriptions';
        echo '<a class="mec-add-booking-tabs-link" data-href="mec-event-descriptions" href="#">' . esc_html('Invoice', 'mec-invoice') . '</a>';
    }

    /**
     * Delete the invoice when a booking is deleted
     *
     * @since     1.0.0
     */
    public function deleteInvoice($postID)
    {

        if (get_post_type($postID) === 'mec-books') {
            $args = array(
                'post_type' => 'mec_invoice',
                'meta_query' => array(
                    array(
                        'key' => 'book_id',
                        'value' => $postID,
                        'compare' => '=',
                    )
                )
            );
            $invoice = current(get_posts($args));
            if ($invoice) {
                wp_delete_post($invoice->ID, true);
            }
        }
    }


    /**
     *  Admin Scripts
     *
     * @since   1.0.0
     */
    public static function enqueueAdminScripts()
    {
        wp_enqueue_style('mec-invoice', MECINVOICEASSETS . 'css/mec-invoice.css', '', '');
        wp_enqueue_script('mec-invoice', MECINVOICEASSETS . 'js/mec-invoice.js', ['jquery']);

        wp_enqueue_style('mec-admin-invoice', MECINVOICEASSETS . 'css/report.css', '', '');
        wp_enqueue_script('mec-admin-invoice', MECINVOICEASSETS . 'js/report.js', ['jquery']);
        wp_localize_script(
            'mec-admin-invoice',
            'MEC_Invoice',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'checkin_options_select' => esc_html__('All', 'mec-invoice'),
                'checkin_options' => array(
                    'unchecked' => esc_html__('UnChecked', 'mec-invoice'),
                    'checked' => esc_html__('Checked', 'mec-invoice'),
                ),
                'placeholder_attendee_email' => esc_html__('Email', 'mec-invoice'),
                'placeholder_attendee_name' => esc_html__('Name', 'mec-invoice'),
            )
        );

        if ('mec_invoice' == get_post_type()) {
            wp_dequeue_script('autosave');
            echo '<style>#minor-publishing-actions,#misc-publishing-actions{display:none !important;}</style>';
        }
    }


    /**
     *  Global Variables.
     *
     * @since   1.0.0
     */
    public static function settingUp()
    {
        self::$dir = MECINVOICEDIR . 'core' . DS . 'manager';
    }

    /**
     *  Verify Invoice
     *
     * @since     1.0.0
     */
    public function verifyInvoice($bookID)
    {
        $invoiceID = get_post_meta($bookID, 'invoiceID', true);
        $confirmation_status = get_post_meta($bookID, 'mec_confirmed', true);

        if ($invoiceID && -1 != $confirmation_status) {
            update_post_meta($invoiceID, 'status', 'open');
        }
    }

    /**
     *  Apply Checkin Data In Attendees List
     *
     * @since     1.0.0
     */
    public function applyCheckStatus($html, $attendee, $key, $book_id, $timestamp)
    {

        $args = array(
            'post_type' => 'mec_invoice',
            'meta_query' => array(
                array(
                    'key' => 'book_id',
                    'value' => $book_id,
                    'compare' => '=',
                ),
            )
        );
        $post = get_posts($args);

        if (isset($post[0]->ID)) {
            $InvoiceID = $post[0]->ID;
        } else {
            $InvoiceID = 0;
        }

        $datetime_format = get_option('date_format') . ' - ' . get_option('time_format');

        $has_checkedin = Attendee::hasCheckedIn($InvoiceID, $attendee['email'], $key, $timestamp);

        if ( !$has_checkedin ) {
            $html .= '<div class="w-col-xs-2 checkin_status checked-in-status status-' . $checked_id . '" data-id="' . $book_id . '" data-invoice-id="' . $InvoiceID . '" data-me="' . $attendee['email'] . '" data-place="' . $key . '" data-timestamp="' . $timestamp . '">'
                . '<span class="unchecked">' . __('Unchecked', 'mec-invoice') . '</span>'
                . '</div>';
            $checked_id = 'no';
            $checkedTime = '-';
        } else {
            $html .= '<div class="w-col-xs-2 checkin_status checked-in-status status-' . $checked_id . '" data-id="' . $book_id . '" data-invoice-id="' . $InvoiceID . '" data-me="' . $attendee['email'] . '" data-place="' . $key . '" data-timestamp="' . $timestamp . '">'
                . '<span class="checked">' . __('Checked', 'mec-invoice') . '</span>'
                . '</div>';
            $checked_timestamp = Attendee::get_checkedin_time( $InvoiceID, $attendee['email'], $key, $timestamp );
            $checked_id = 'yes';
            $checkedTime = date_i18n( $datetime_format, $checked_timestamp);
        }

//        if (get_post_meta($book_id, 'mec_verified', true) === '1') {
//
//        }

        $html = str_replace('w-col-xs-3 name', 'w-col-xs-2 name', $html);
        $html = str_replace('w-col-xs-3 ticket', 'w-col-xs-2 ticket', $html);
        return $html;
    }

    /**
     *  Apply Checkin Col In Attendees List
     *
     * @since     1.0.0
     */
    public function applyCheckStatusCol($html)
    {
        $html = str_replace('w-col-xs-3 name', 'w-col-xs-2 name', $html);
        $html = str_replace('w-col-xs-3 ticket', 'w-col-xs-2 ticket', $html);

        $html .= ' <div class="w-col-xs-2">
                    <span>' . __('Checkin Status', 'mec-invoice') . '</span>
                </div>';
        return $html;
    }

    /**
     *  Add Export Button Into Booking Report Attendees Table
     *
     * @since     1.0.0
     */
    public function addExportButton($html, $eventID, $occurrence)
    {
        $filter_checkin_status = isset( $_POST['checkin'] ) ? sanitize_text_field( $_POST['checkin'] ) : '';

        $html = '<div class="w-col-xs-2">
                    <span class="mec-invoice-export-attendees-as-contact" data-id="' . $eventID . '" data-occurrence="' . $occurrence . '" data-checkin-status="' . $filter_checkin_status . '">' . __('Export Attendees As CSV', 'mec-invoice') . '</span>
                </div>' . $html;
        return $html;
    }

    /**
     *  Export Attendees As CSV
     *
     * @since     1.0.0
     */
    public function exportAttendeesAsCsv()
    {
        $main = \MEC::getInstance('app.libraries.main');
        $db = \MEC::getInstance('app.libraries.db');
        $id = isset($_POST['eventID']) ? sanitize_text_field($_POST['eventID']) : 0;
        $occurrence = isset($_POST['occurrence']) ? sanitize_text_field($_POST['occurrence']) : NULL;
        $filter_checkin_status = isset( $_POST['checkin'] ) ? sanitize_text_field( $_POST['checkin'] ) : '';
        $email = isset( $_POST['email'] ) ? sanitize_text_field( $_POST['email'] ) : '';
        $name = isset( $_POST['name'] ) ? sanitize_text_field( $_POST['name'] ) : '';

        $book_attendee_keys = isset( $_POST['book_attendee_keys'] ) && is_array( $_POST['book_attendee_keys'] ) ? $_POST['book_attendee_keys'] : 'all';

        $verified = true;
        $bookings = \MEC\Base::get_main()->get_bookings($id, $occurrence, '-1', NULL, $verified);

        // Attendees
        $attendees = array();
        foreach ($bookings as $booking) {
            $atts = get_post_meta($booking->ID, 'mec_attendees', true);
            if (isset($atts['attachments'])) unset($atts['attachments']);

            $timestamps = get_post_meta( $booking->ID, 'mec_attention_time', true );
            $timestamps = explode( ':', $timestamps );
            $event_datetime = date( 'Y-m-d H:i', $timestamps[0] );
            foreach ($atts as $key => $value) {
                if (!is_numeric($key)) continue;

                $attendee_key = $key + 1;
                if( 'all' !== $book_attendee_keys && !in_array( "{$booking->ID}-{$attendee_key}", $book_attendee_keys ) ) {

                    unset( $atts[ $key ] );
                    continue;
                }

                $atts[$key]['book_id'] = $booking->ID;
                $atts[$key]['invoice_id'] = get_post_meta( $booking->ID, 'invoiceID', true );
                $atts[$key]['key'] = ($key + 1);
                $atts[$key]['event_datetime'] = $event_datetime;

            }

            $attendees = array_merge($attendees, $atts);
        }


        $export_data = [];
        $export_data[] = [
            'Invoice ID',
            'Event DateTime',
            'Attendee Key',
            'First Name',
            'Title',
            'Contact Picture',
            'E-mail Address',
            'Variations',
            'Checkin Status',
            'Checkin DateTime',
        ];
        foreach ($attendees as $k => $attendee) {

            $book_id = $attendee['book_id'];
            $event_id = get_post_meta($book_id, 'mec_event_id', true);
            $tickets = get_post_meta($event_id, 'mec_tickets', true);

            $attendee_id = $attendee['id'] ?? null;
            $attendee_key = $attendee['key'] ?? null;
            $invoice_id = $attendee['invoice_id'] ?? null;

            $variations = '';
            if (isset($attendee['variations']) and is_array($attendee['variations']) and count($attendee['variations'])) {

                $ticket_variations = \MEC\Base::get_main()->ticket_variations($id, $attendee['id']);

                foreach ($attendee['variations'] as $variation_id => $variation_count) {
                    if (!$variation_count or ($variation_count and $variation_count < 0)) continue;

                    $variation_title = (isset($ticket_variations[$variation_id]) and isset($ticket_variations[$variation_id]['title'])) ? $ticket_variations[$variation_id]['title'] : '';
                    if (!trim($variation_title)) continue;

                    $variations .= $variation_title . ' (' . $variation_count . ')' . PHP_EOL;
                }
            }

            $t_place = $attendee_key;
            $has_checkedin = $invoice_id && Attendee::hasCheckedIn( $invoice_id, $attendee['email'], $t_place, $occurrence ) ? true : false;
            $checkedin_timestamp = $has_checkedin  ? Attendee::get_checkedin_time( $invoice_id, $t_place, $occurrence ) : false;

            if(
                ( 'unchecked' === $filter_checkin_status && $has_checkedin )
                ||
                ( 'checked' === $filter_checkin_status && !$has_checkedin )
                ){
                continue;
            }

            if( $email && false === strpos( strtolower( $attendee['email'] ), strtolower( $email ) ) ){

                continue;
            }

            if( $name && false === strpos( strtolower( $attendee['name'] ), strtolower( $name ) ) ){

                continue;
            }

            $attendee_detail = [
                $invoice_id,
                $attendee['event_datetime'],
                $attendee_key,
                $attendee['name'],
                ((isset($attendee['id']) and isset($tickets[$attendee['id']]['name'])) ? $tickets[$attendee['id']]['name'] : __('Unknow Ticket', 'mec-invoice')),
                get_avatar_url($attendee['email']),
                $attendee['email'],
                $variations,
                $has_checkedin ? __('Checked', 'mec-invoice') : __('Unchecked', 'mec-invoice'),
                is_numeric( $checkedin_timestamp ) ? date_i18n( 'Y-m-d H:i', $checkedin_timestamp ) : '-',
            ];

            $export_data[] = $attendee_detail;
        }

        // download
        header("Content-Description: File Transfer");
        header("Content-Disposition: attachment; filename=Attendees.csv");
        header('Content-type: text/csv');
        print $this->arrayToCSV($export_data);
        exit();
    }

    private function arrayToCSV($inputArray)
    {
        $csvFieldRow = array();
        foreach ($inputArray as $CSBRow) {
            $csvFieldRow[] = $this->str_putcsv($CSBRow);
        }
        $csvData = implode("\r\n", $csvFieldRow);
        return $csvData;
    }

    private function str_putcsv($input, $delimiter = ',', $enclosure = '"')
    {
        $fp = fopen('php://temp', 'r+');
        fputcsv($fp, $input, $delimiter, $enclosure);
        rewind($fp);
        $data = fread($fp, 1048576);
        fclose($fp);
        return rtrim($data, "\r\n");
    }

    /**
     *  Cancel Invoice
     *
     * @since     1.0.0
     */
    public function cancelInvoice($bookID)
    {
        $invoiceID = get_post_meta($bookID, 'invoiceID', true);
        if ($invoiceID) {
            update_post_meta($invoiceID, 'status', 'closed');
        }
    }


    /**
     *  ADD API Key App Menu
     *
     * @since     1.0.0
     */
    public function apiKeyAppMenu()
    {
        add_submenu_page(
            'edit.php?post_type=mec_invoice',
            'API Key for App Mobile',
            'API Key for App',
            'manage_options',
            'api-key-app',
            [static::instance(), 'apiKeyApp']
        );

        add_submenu_page(
            'edit.php?post_type=mec_invoice',
            'Language for App',
            'Language for App',
            'manage_options',
            'lang-app',
            [static::instance(), 'languageApp']
        );
    }

    /**
     * API Key App Page
     *
     * @since     1.0.0
     */
    public function apiKeyApp()
    {
        if (!get_option("api-key-app")) {
            update_option('api-key-app', substr(md5(rand(0, 1000)), 0, 8));
        }
        echo '<div class="mec-invoice-export-as-xml-wrap">';
        echo '<div class="wns-be-container"><div id="wns-be-infobar"></div></div>';
        echo '<div>';
        echo '<p class="mec_dash_count">' . get_option("api-key-app") . '</p>';
        echo '<p>'.esc_html_e('API Key for App Mobile', 'mec-invoice').'</p>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * language App Page
     *
     * @since     1.0.0
     */
    public function languageApp()
    {
        $settings_translate = json_decode(get_option('mec_translate_invoice_app',null));
        $enable = $settings_translate->enable??0;
        $translate = $settings_translate->translate??null;
        ?>
        <div class="mec-invoice-container">
            <div>
                <form id="mec_invoice_language_app_form">
                    <div class="mec-form-row">
                        <label>
                            <input onchange="jQuery('#mec_invoice_language_app_container_toggle').toggle(); jQuery(this).val(jQuery(this).val()==0?1:0);" value="<?php echo $settings_translate->enable??0; ?>" type="checkbox" name="activeLanguageApp" <?php echo $enable==1 ?'checked="checked"':''; ?> /> <?php esc_html_e('Enable Language App', 'mec-invoice'); ?>
                        </label>
                    </div>
                    <div id="mec_invoice_language_app_container_toggle" class="<?php echo $enable==1 ?'':'mec-util-hidden'; ?>">
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Ticket Scanner</label>
                                <input class="mec-col-6" value="<?php echo $translate->TicketScanner??'Ticket Scanner'; ?>" type="text" name="TicketScanner" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Click here for help!</label>
                                <input class="mec-col-6" value="<?php echo $translate->Clickhereforhelp??'Click here for help!'; ?>" type="text" name="Clickhereforhelp" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">To proceed, please connect to your website.</label>
                                <input class="mec-col-6" value="<?php echo $translate->Toproceedpleaseconnect??'To proceed, please connect to your website.'; ?>" type="text" name="Toproceedpleaseconnect" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Connect Your Site</label>
                                <input class="mec-col-6" value="<?php echo $translate->ConnectYourSite??'Connect Your Site'; ?>" type="text" name="ConnectYourSite" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">WEBSITE</label>
                                <input class="mec-col-6" value="<?php echo $translate->WEBSITE??'WEBSITE'; ?>" type="text" name="WEBSITE" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">API KEY</label>
                                <input class="mec-col-6" value="<?php echo $translate->APIKEY??'API KEY'; ?>" type="text" name="APIKEY" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">App is connected.</label>
                                <input class="mec-col-6" value="<?php echo $translate->Appisconnected??'App is connected.'; ?>" type="text" name="Appisconnected" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Something went wrong!</label>
                                <input class="mec-col-6" value="<?php echo $translate->Somethingwentwrong??'Something went wrong!'; ?>" type="text" name="Somethingwentwrong" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Logout</label>
                                <input class="mec-col-6" value="<?php echo $translate->Logout??'Logout'; ?>" type="text" name="Logout" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Let's get started</label>
                                <input class="mec-col-6" value="<?php echo $translate->Letsgetstarted??'Let\'s get started'; ?>" type="text" name="Letsgetstarted" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Try Again</label>
                                <input class="mec-col-6" value="<?php echo $translate->TryAgain??'Try Again'; ?>" type="text" name="TryAgain" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">No Events Found!</label>
                                <input class="mec-col-6" value="<?php echo $translate->NoEventsFound??'No Events Found!'; ?>" type="text" name="NoEventsFound" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">SELECT DATE</label>
                                <input class="mec-col-6" value="<?php echo $translate->SELECTDATE??'SELECTDATE'; ?>" type="text" name="SELECTDATE" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">All Events</label>
                                <input class="mec-col-6" value="<?php echo $translate->AllEvents??'All Events'; ?>" type="text" name="AllEvents" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Today's Events</label>
                                <input class="mec-col-6" value="<?php echo $translate->TodaysEvents??'Today\'s Events'; ?>" type="text" name="TodaysEvents" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Attendess</label>
                                <input class="mec-col-6" value="<?php echo $translate->Attendess??'Attendess'; ?>" type="text" name="Attendess" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Settings</label>
                                <input class="mec-col-6" value="<?php echo $translate->Settings??'Settings'; ?>" type="text" name="Settings" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Scan</label>
                                <input class="mec-col-6" value="<?php echo $translate->Scan??'Scan'; ?>" type="text" name="Scan" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Lists</label>
                                <input class="mec-col-6" value="<?php echo $translate->Lists??'Lists'; ?>" type="text" name="Lists" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Scan Again</label>
                                <input class="mec-col-6" value="<?php echo $translate->ScanAgain??'Scan Again'; ?>" type="text" name="ScanAgain" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Access Denied!</label>
                                <input class="mec-col-6" value="<?php echo $translate->AccessDenied??'Access Denied!'; ?>" type="text" name="AccessDenied" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Ticket Already Checked-in!</label>
                                <input class="mec-col-6" value="<?php echo $translate->SuccessfullyUnchecked??'Ticket Already Checked-in!'; ?>" type="text" name="SuccessfullyUnchecked" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Already Checked!</label>
                                <input class="mec-col-6" value="<?php echo $translate->AlreadyChecked??'Already Checked!'; ?>" type="text" name="AlreadyChecked" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Successfully Checked!</label>
                                <input class="mec-col-6" value="<?php echo $translate->SuccessfullyChecked??'Successfully Checked!'; ?>" type="text" name="SuccessfullyChecked" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Vibration</label>
                                <input class="mec-col-6" value="<?php echo $translate->Vibration??'Vibration'; ?>" type="text" name="Vibration" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Sound</label>
                                <input class="mec-col-6" value="<?php echo $translate->Sound??'Sound'; ?>" type="text" name="Sound" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">System</label>
                                <input class="mec-col-6" value="<?php echo $translate->System??'System'; ?>" type="text" name="System" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Total Attendees</label>
                                <input class="mec-col-6" value="<?php echo $translate->TotalAttendees??'Total Attendees'; ?>" type="text" name="TotalAttendees" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Checked-in Attendees</label>
                                <input class="mec-col-6" value="<?php echo $translate->CheckedinAttendees??'Checked-in Attendees'; ?>" type="text" name="CheckedinAttendees" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">Check-in Status</label>
                                <input class="mec-col-6" value="<?php echo $translate->CheckinStatus??'Check-in Status'; ?>" type="text" name="CheckinStatus" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">All Attendees</label>
                                <input class="mec-col-6" value="<?php echo $translate->AllAttendees??'All Attendees'; ?>" type="text" name="AllAttendees" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">OK</label>
                                <input class="mec-col-6" value="<?php echo $translate->OK??'OK'; ?>" type="text" name="OK" />
                            </div>
                        </div>
                        <div class="mec-form-row">
                            <div class="mec-col-12">
                                <label class="mec-col-6" for="">CANCEL</label>
                                <input class="mec-col-6" value="<?php echo $translate->CANCEL??'CANCEL'; ?>" type="text" name="CANCEL" />
                            </div>
                        </div>
                    </div>
                    <div class="mec-form-row">
                        <div class="mec-col-12">
                            <a class="button-translate-save"><?php esc_html_e('Save Translate', 'mec-invoice'); ?></a>
                        </div>
                    </div>
                </form>
            </div>

            <div class="mec-wizard-loading"><div class="mec-loader"></div></div>
        </div>

        <style>
            .mec-invoice-container{
                display: block;
                position: relative;
                background-color: #fff;
                width: 80%;margin: auto;
                text-align: center;
                margin-top: 30px;
                padding: 25px;
                border: 1px solid #e3e5e7;
                border-radius: 15px;
            }
            .mec-wizard-loading {
                background: #ffffff54;
                width: 100%;
                height: 100%;
                position: absolute;
                left: 0px;
                bottom: 0;
                top: 0;
                border-radius: 22px;
                display: none;
                z-index: 99;
            }
            .mec-loader {
                top: calc(50% - 25px);
                left: calc(50% - 25px);
            }
            .button-translate-save{
                box-shadow: 0 0 0 4px #e4ffe6;
                text-shadow: none;
                margin-right: 5px;
                background: #64e385;
                color: #fff;
                border: none;
                cursor: pointer;
                padding: 1px 22px;
                display: inline-block;
                text-decoration: none;
                height: 36px;
                line-height: 36px;
                font-weight: 500;
                font-size: 14px;
                border-radius: 20px;
                letter-spacing: .2px;
                transition: .24s all ease;
            }
        </style>

        <script>
            jQuery(".button-translate-save").click(function()
            {
                let translate_app={};
                jQuery('#mec_invoice_language_app_container_toggle input').each(function( index ) {
                    translate_app[jQuery(this).attr('name')]=jQuery(this).val();
                });
                jQuery.ajax(
                    {
                        type: "POST",
                        url: ajaxurl,
                        data: "action=mec_update_translate_invoice_app&active="+jQuery("input[name='activeLanguageApp']").val()+"&translate_string="+JSON.stringify(translate_app),
                        dataType: 'json',
                        beforeSend: function () {
                            save_loading(true)
                        },
                        success: function(data)
                        {
                            save_loading(false)
                        },
                        error: function(jqXHR, textStatus, errorThrown)
                        {
                            console.log('error');
                        }
                    });
            });

            function save_loading($value)
            {
                if ($value) {
                    jQuery('.mec-wizard-loading').show()
                } else {
                    jQuery('.mec-wizard-loading').hide()
                }
            }
        </script>
   <?php
    }

    /**
     * Update Translate App
     *
     * @since     1.0.0
     */
    public function updateTranslateApp()
    {
        if( isset($_POST['active']) && $_POST['translate_string']){
            update_option('mec_translate_invoice_app',json_encode(array(
                'enable'=> $_POST['active'],
                'translate'=>json_decode(stripslashes($_POST['translate_string']))
            )));
        }
        echo json_encode([
            'status' => 1,
            'text' => "Updated Translate App.",
        ]);
        die();
    }

    /**
     * Route API App Mobile
     *
     * @since     1.0.0
     */
    function routes_api_app()
    {
        register_rest_route('invoices/v1', '/connect', array(
            'methods' => 'Post',
            'callback' => [static::instance(), 'connectApiApp'],
        ));

        register_rest_route('invoices/v1', '/translate', array(
            'methods' => 'Get',
            'callback' => [static::instance(), 'translateApiApp'],
        ));

        register_rest_route('invoices/v1', '/events', array(
            'methods' => 'Get',
            'callback' => [static::instance(), 'eventsApiApp'],
        ));

        register_rest_route('invoices/v1', '/attedees', array(
            'methods' => 'Get',
            'callback' => [static::instance(), 'attedeesApiApp'],
        ));

        register_rest_route('invoices/v1', '/attedees/check', array(
            'methods' => 'Post',
            'callback' => [static::instance(), 'checkAttedeesApiApp'],
        ));
    }

    /**
     * Connect API App
     *
     * @since     1.0.0
     */
    function connectApiApp($data)
    {
        if (isset($data['key']) && get_option("api-key-app") === $data['key']) {
            $data_response = array('status' => 200, 'msg' => "Successful", 'data' => null);
            return $this->responseApiApp($data_response, 200);
        } else {
            $data_response = array('status' => 404, 'msg' => "Invalid Key.", 'data' => null);
            return $this->responseApiApp($data_response, 404);
        }
    }


    /**
     * Translate API App
     *
     * @since     1.0.0
     */
    function translateApiApp($data)
    {
        if (isset($data['key']) && get_option("api-key-app") === $data['key']) {
            $data_response = array('status' => 200, 'msg' => "Successful", 'data' => get_option('mec_translate_invoice_app',null));
            return $this->responseApiApp($data_response, 200);
        } else {
            $data_response = array('status' => 404, 'msg' => "Invalid Key.", 'data' => null);
            return $this->responseApiApp($data_response, 404);
        }
    }

    /**
     * Connect API App
     *
     * @since     1.0.0
     */
    function eventsApiApp($data)
    {
        $db = \MEC::getInstance('app.libraries.db');
        $main = \MEC::getInstance('app.libraries.main');

        if (isset($data['key']) && get_option("api-key-app") === $data['key']) {

            if (isset($data['date'])) {

//                $list = array();
//                $i = 0;
//                $events = $db->select("SELECT `post_id` FROM `#__mec_events` WHERE `start` = '" . $data['date'] . "'");
//                foreach ($events as $key => $event) {
//                    $j = 0;
//                    $post_id = $event->post_id;
//                    $list[$i]['event_id'] = $post_id;
//                    $list[$i]['title'] = get_the_title($post_id);
//
//                    // Invoices and Attendees
//                    $invoices = get_posts(array(
//                        'post_type' => 'mec_invoice',
//                        'posts_per_page' => -1
//                    ));
//
//                    foreach ($invoices as $key_invoices => $invoice) {
//                        $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
//                        if ($post_id == $invoice_event_id) {
//                            $book_id = get_post_meta($invoice->ID, 'book_id', true);
//                            $atts = get_post_meta($book_id, 'mec_attendees', true);
//                            foreach ($atts as $key_att => $att) {
//                                $j++;
//                            }
//                        }
//                    }
//
//                    $list[$i]['total_attendees'] = $j;
//                    $i++;
//                }
//
//                $data_response = array('status' => 200, 'msg' => "Successful", 'data' => $list);
//                return $this->responseApiApp($data_response, 200);

                $events = get_posts(array(
                    'post_type' => 'mec-events',
                    'posts_per_page' => -1
                ));
                $list = array();
                $i = 0;
                $timestamp_date_select = strtotime($data['date']);
                foreach ($events as $key_event => $event) {
                    $j = 0;
                    $event_details = $db->select("SELECT * FROM `#__mec_events` WHERE `post_id`='$event->ID'", "loadObject");

                    if (isset($event_details->repeat) and $event_details->repeat == '0') {
                        if (strtotime($event_details->start) <= strtotime($data['date']) && strtotime($event_details->end) >= strtotime($data['date'])) {

                            $item_event['event_id'] = $event->ID;
                            $item_event['title'] = $event->post_title;
                            $item_event['repeat'] = 0;

                            //Invoices and Attendees
                            $invoices = get_posts(array(
                                'post_type' => 'mec_invoice',
                                'posts_per_page' => -1
                            ));

                            foreach ($invoices as $key_invoices => $invoice) {
                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                if ($event->ID == $invoice_event_id) {
                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                    foreach ($atts as $key_att => $att) {
                                        $j++;
                                    }
                                }
                            }

                            $item_event['total_attendees'] = $j;
                            $list[] = $item_event;
                            $i++;
                        }
                    } else {

                        if (strtotime($event_details->start) <= strtotime($data['date']) && strtotime($event_details->end) >= strtotime($data['date'])) {

                            $item_event['event_id'] = $event->ID;
                            $item_event['title'] = $event->post_title;
                            $item_event['repeat'] = 1;

                            //Invoices and Attendees
                            $invoices = get_posts(array(
                                'post_type' => 'mec_invoice',
                                'posts_per_page' => -1
                            ));

                            foreach ($invoices as $key_invoices => $invoice) {
                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                if ($event->ID == $invoice_event_id) {
                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                    foreach ($atts as $key_att => $att) {
                                        $j++;
                                    }
                                }
                            }

                            $item_event['total_attendees'] = $j;
                            $list[] = $item_event;
                            $i++;
                            continue 1;
                        } else {
                            $repeat_type = get_post_meta($event->ID, 'mec_repeat_type', true);
                            $today = NULL;
                            $start_date = get_post_meta($event->ID, 'mec_start_date', true);
                            $end_date = get_post_meta($event->ID, 'mec_end_date', true);
                            $finish_date = array('date' => get_post_meta($event->ID, 'mec_end_date', true), 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true));
                            $exceptional_days = (trim(get_post_meta($event->ID, 'mec_end_date', true))) ? explode(',', trim(get_post_meta($event->ID, 'mec_not_in_days', true), ', ')) : array();
                            $allday = get_post_meta($event->ID, 'mec_allday', true);
                            $hide_time = get_post_meta($event->ID, 'mec_hide_time', true);
                            $event_period = $main->date_diff($start_date, $end_date);
                            $event_period_days = $event_period ? $event_period->days : 0;
                            $dates = array();
                            $original_start_date = $today;

                            if (in_array($repeat_type, array('daily', 'weekly'))) {
                                $repeat_interval = get_post_meta($event->ID, 'mec_repeat_interval', true);

//                                $date_interval = $main->date_diff($start_date, $today);
//                                $passed_days = $date_interval ? $date_interval->days : 0;
//
//                                // Check if date interval is negative (It means the event didn't start yet)
//                                if ($date_interval and $date_interval->invert == 1) $remained_days_to_next_repeat = $passed_days;
//                                else $remained_days_to_next_repeat = $repeat_interval - fmod($passed_days, $repeat_interval);

                                $loop = true;
                                $start_date = date('Y-m-d', strtotime('+' . $repeat_interval . ' Days', strtotime($start_date)));

                                if ($timestamp_date_select >= strtotime($start_date)) {
                                    if ($start_date == $data['date']) {

                                        $item_event['event_id'] = $event->ID;
                                        $item_event['title'] = $event->post_title;
                                        $item_event['repeat'] = 1;

                                        //Invoices and Attendees
                                        $invoices = get_posts(array(
                                            'post_type' => 'mec_invoice',
                                            'posts_per_page' => -1
                                        ));

                                        foreach ($invoices as $key_invoices => $invoice) {
                                            $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                            if ($event->ID == $invoice_event_id) {
                                                $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                foreach ($atts as $key_att => $att) {
                                                    $j++;
                                                }
                                            }
                                        }

                                        $item_event['total_attendees'] = $j;
                                        $list[] = $item_event;
                                        $i++;
                                        $loop = false;
                                    }
                                } else {
                                    $loop = false;
                                }

                                while ($loop) {
                                    $start_date = date('Y-m-d', strtotime('+' . $repeat_interval . ' Days', strtotime($start_date)));

                                    // Event finished
                                    // if ($main->is_past($finish_date['date'], $start_date)) break;

                                    if (!in_array($start_date, $exceptional_days)) {

                                        $dates[] = $this->add_timestamps(array(
                                            'start' => array('date' => $start_date, 'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true)),
                                            'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true)),
                                            'allday' => $allday,
                                            'hide_time' => $hide_time,
                                            'past' => 0
                                        ));

                                        if ($timestamp_date_select >= strtotime($start_date)) {
                                            if ($start_date == $data['date']) {

                                                $item_event['event_id'] = $event->ID;
                                                $item_event['title'] = $event->post_title;
                                                $item_event['repeat'] = 1;

                                                //Invoices and Attendees
                                                $invoices = get_posts(array(
                                                    'post_type' => 'mec_invoice',
                                                    'posts_per_page' => -1
                                                ));

                                                foreach ($invoices as $key_invoices => $invoice) {
                                                    $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                    if ($event->ID == $invoice_event_id) {
                                                        $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                        $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                        foreach ($atts as $key_att => $att) {
                                                            $j++;
                                                        }
                                                    }
                                                }

                                                $item_event['total_attendees'] = $j;
                                                $list[] = $item_event;
                                                $i++;
                                                break;
                                            }
                                        } else {
                                            $loop = false;
                                        }
                                    }
                                }

                            } else if (in_array($repeat_type, array('weekday', 'weekend', 'certain_weekdays'))) {
                                $date_interval = $main->date_diff($start_date, $today);
                                $passed_days = $date_interval ? $date_interval->days : 0;

                                $today = $start_date;
                                // Check if date interval is negative (It means the event didn't start yet)
                                if ($date_interval and $date_interval->invert == 1) $today = date('Y-m-d', strtotime('+' . $passed_days . ' Days', strtotime($today)));

                                $event_days = explode(',', trim($event_details->weekdays, ', '));

                                $today_id = date('N', strtotime($today));
                                $loop = true;
                                while ($loop) {
                                    //  if ($main->is_past($finish_date['date'], $today)) break;

                                    if (!in_array($today_id, $event_days)) {
                                        $today = date('Y-m-d', strtotime('+1 Days', strtotime($today)));
                                        $today_id = date('N', strtotime($today));
                                        continue;
                                    }

                                    $start_date = $today;
                                    if ($timestamp_date_select >= strtotime($start_date)) {
                                        if ($start_date == $data['date']) {

                                            $item_event['event_id'] = $event->ID;
                                            $item_event['title'] = $event->post_title;
                                            $item_event['repeat'] = 1;

                                            //Invoices and Attendees
                                            $invoices = get_posts(array(
                                                'post_type' => 'mec_invoice',
                                                'posts_per_page' => -1
                                            ));

                                            foreach ($invoices as $key_invoices => $invoice) {
                                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                if ($event->ID == $invoice_event_id) {
                                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                    foreach ($atts as $key_att => $att) {
                                                        $j++;
                                                    }
                                                }
                                            }

                                            $item_event['total_attendees'] = $j;
                                            $list[] = $item_event;
                                            $i++;
                                            break;
                                        }
                                    } else {
                                        $loop = false;
                                    }

                                    if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                                        'start' => array('date' => $start_date, 'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true)),
                                        'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true)),
                                        'allday' => $allday,
                                        'hide_time' => $hide_time,
                                        'past' => 0
                                    ));

                                    $today = date('Y-m-d', strtotime('+1 Days', strtotime($today)));
                                    $today_id = date('N', strtotime($today));
                                }
                            } else if ($repeat_type == 'monthly') {
                                $repeat_interval = max(1, get_post_meta($event->ID, 'mec_repeat_interval', true));

                                // Start from Event Start Date
                                $original_start_date = $start_date;

                                $event_days = explode(',', trim($event_details->day, ', '));
                                $event_start_day = $event_days[0];

                                $diff = $main->date_diff($start_date, $end_date);
                                $event_period_days = $diff->days;

                                $q = $repeat_interval;
                                $t = strtotime($original_start_date . ' + ' . $q . ' months');

                                $loop = true;
                                while ($loop) {

                                    //   if (!$t) break;

                                    $today = date('Y-m-d', $t);
                                    // if ($main->is_past($finish_date['date'], $today)) break;

                                    $year = date('Y', $t);
                                    $month = date('m', $t);
                                    $day = $event_start_day;
                                    $hour = get_post_meta($event->ID, 'mec_end_time_hour', true);
                                    $minutes = get_post_meta($event->ID, 'mec_end_time_minutes', true);
                                    $ampm = get_post_meta($event->ID, 'mec_end_time_ampm', true);

                                    // Fix for 31st, 30th, 29th of some months
                                    while (!checkdate($month, $day, $year)) $day--;

                                    $start_date = $year . '-' . $month . '-' . $day;
                                    $end_time = $hour . ':' . $minutes . ' ' . $ampm;

                                    // Wrong Date & Time
                                    //  if (!strtotime($start_date . ' ' . $end_time)) break;

//                                    if (strtotime($start_date . ' ' . $end_time) < strtotime($original_start_date)) {
//                                        $q += $repeat_interval;
//                                        $t = strtotime($original_start_date. ' + ' . $q . ' months');
//                                        continue;
//                                    }

                                    if (!in_array($start_date, $exceptional_days)) {

                                        if ($timestamp_date_select >= strtotime($start_date)) {
                                            if ($start_date == $data['date']) {

                                                $item_event['event_id'] = $event->ID;
                                                $item_event['title'] = $event->post_title;
                                                $item_event['repeat'] = 1;

                                                //Invoices and Attendees
                                                $invoices = get_posts(array(
                                                    'post_type' => 'mec_invoice',
                                                    'posts_per_page' => -1
                                                ));

                                                foreach ($invoices as $key_invoices => $invoice) {
                                                    $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                    if ($event->ID == $invoice_event_id) {
                                                        $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                        $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                        foreach ($atts as $key_att => $att) {
                                                            $j++;
                                                        }
                                                    }
                                                }

                                                $item_event['total_attendees'] = $j;
                                                $list[] = $item_event;
                                                $i++;
                                                break;
                                            }
                                        } else {
                                            $loop = false;
                                        }

                                        $dates[] = $this->add_timestamps(array(
                                            'start' => array('date' => $start_date, 'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true)),
                                            'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true)),
                                            'allday' => $allday,
                                            'hide_time' => $hide_time,
                                            'past' => 0
                                        ));

                                    }

                                    $q += $repeat_interval;
                                    $t = strtotime($original_start_date . ' + ' . $q . ' months');
                                }

                            } else if ($repeat_type == 'yearly') {
                                // Start from Event Start Date
                                if (strtotime($start_date) > strtotime($original_start_date)) $original_start_date = $start_date;

                                $event_days = explode(',', trim($event_details->day, ', '));
                                $event_months = explode(',', trim($event_details->month, ', '));

                                $event_start_day = $event_days[0];
                                $event_period_days = $main->date_diff($start_date, $end_date)->days;

                                $event_start_year = date('Y', strtotime($original_start_date));
                                $event_start_month = date('n', strtotime($original_start_date));

                                $q = 0;

                                $loop = true;
                                while ($loop) {
                                    $today = date('Y-m-d', strtotime($event_start_year . '-' . $event_start_month . '-' . $event_start_day));
                                    //if ($main->is_past($finish_date['date'], $today)) break;

                                    $year = date('Y', strtotime($today));
                                    $month = date('m', strtotime($today));

                                    if (!in_array($month, $event_months)) {
                                        if ($event_start_month == '12') {
                                            $event_start_month = 1;
                                            $event_start_year += 1;
                                        } else $event_start_month += 1;

                                        $q++;
                                        continue;
                                    }

                                    $day = $event_start_day;

                                    // Fix for 31st, 30th, 29th of some months
                                    while (!checkdate($month, $day, $year)) $day--;

                                    $event_date = $year . '-' . $month . '-' . $day;
                                    if (strtotime($event_date) >= strtotime($original_start_date)) {
                                        $start_date = $event_date;

                                        if ($timestamp_date_select >= strtotime($start_date)) {
                                            if ($start_date == $data['date']) {

                                                $item_event['event_id'] = $event->ID;
                                                $item_event['title'] = $event->post_title;
                                                $item_event['repeat'] = 1;

                                                //Invoices and Attendees
                                                $invoices = get_posts(array(
                                                    'post_type' => 'mec_invoice',
                                                    'posts_per_page' => -1
                                                ));

                                                foreach ($invoices as $key_invoices => $invoice) {
                                                    $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                    if ($event->ID == $invoice_event_id) {
                                                        $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                        $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                        foreach ($atts as $key_att => $att) {
                                                            $j++;
                                                        }
                                                    }
                                                }

                                                $item_event['total_attendees'] = $j;
                                                $list[] = $item_event;
                                                $i++;
                                                break;
                                            }
                                        } else {
                                            $loop = false;
                                        }

                                        if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                                            'start' => array('date' => $start_date, 'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true)),
                                            'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true)),
                                            'allday' => $allday,
                                            'hide_time' => $hide_time,
                                            'past' => 0
                                        ));

                                    }

                                    if ($event_start_month == '12') {
                                        $event_start_month = 1;
                                        $event_start_year += 1;
                                    } else $event_start_month += 1;

                                    $q++;
                                }

                            } else if ($repeat_type == 'custom_days') {
                                $custom_days = explode(',', $event_details->days);

                                // Add current time if we're checking today's events
                                if ($today == current_time('Y-m-d')) $today .= ' ' . current_time('H:i:s');

                                $loop = true;
                                if ((strtotime($event_details->start) + get_post_meta($event->ID, 'mec_start_day_seconds', true)) >= strtotime($today) and !in_array($event_details->start, $exceptional_days)) {
                                    $dates[] = $this->add_timestamps(array(
                                        'start' => array('date' => $event_details->start, 'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true)),
                                        'end' => array('date' => $event_details->end, 'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true), 'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true), 'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true)),
                                        'allday' => $allday,
                                        'hide_time' => $hide_time,
                                        'past' => 0
                                    ));

                                    if ($timestamp_date_select >= strtotime($event_details->start)) {
                                        if ($event_details->start == $data['date']) {

                                            $item_event['event_id'] = $event->ID;
                                            $item_event['title'] = $event->post_title;
                                            $item_event['repeat'] = 1;

                                            //Invoices and Attendees
                                            $invoices = get_posts(array(
                                                'post_type' => 'mec_invoice',
                                                'posts_per_page' => -1
                                            ));

                                            foreach ($invoices as $key_invoices => $invoice) {
                                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                if ($event->ID == $invoice_event_id) {
                                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                    foreach ($atts as $key_att => $att) {
                                                        $j++;
                                                    }
                                                }
                                            }

                                            $item_event['total_attendees'] = $j;
                                            $list[] = $item_event;
                                            $i++;
                                            continue 1;
                                        }
                                    } else {
                                        $loop = false;
                                    }
                                }

                                foreach ($custom_days as $custom_day) {
                                    // Found maximum dates
                                    if (!$loop) break;

                                    $cday = explode(':', $custom_day);

                                    $c_start = $cday[0];
                                    if (isset($cday[2])) $c_start .= ' ' . str_replace('-', ' ', substr_replace($cday[2], ':', strpos($cday[2], '-'), 1));

                                    // Date is past
                                    if (strtotime($c_start) < strtotime($today)) continue;

                                    $cday_start_hour = get_post_meta($event->ID, 'mec_start_time_hour', true);
                                    $cday_start_minutes = get_post_meta($event->ID, 'mec_start_time_minutes', true);
                                    $cday_start_ampm = get_post_meta($event->ID, 'mec_start_time_ampm', true);

                                    $cday_end_hour = get_post_meta($event->ID, 'mec_end_time_hour', true);
                                    $cday_end_minutes = get_post_meta($event->ID, 'mec_end_time_minutes', true);
                                    $cday_end_ampm = get_post_meta($event->ID, 'mec_end_time_ampm', true);

                                    if (isset($cday[2]) and isset($cday[3])) {
                                        $cday_start_ex = explode('-', $cday[2]);
                                        $cday_start_hour = $cday_start_ex[0];
                                        $cday_start_minutes = $cday_start_ex[1];
                                        $cday_start_ampm = $cday_start_ex[2];

                                        $cday_end_ex = explode('-', $cday[3]);
                                        $cday_end_hour = $cday_end_ex[0];
                                        $cday_end_minutes = $cday_end_ex[1];
                                        $cday_end_ampm = $cday_end_ex[2];
                                    }

                                    if (!in_array($cday[0], $exceptional_days)) $dates[] = $this->add_timestamps(array(
                                        'start' => array('date' => $cday[0], 'hour' => $cday_start_hour, 'minutes' => $cday_start_minutes, 'ampm' => $cday_start_ampm),
                                        'end' => array('date' => $cday[1], 'hour' => $cday_end_hour, 'minutes' => $cday_end_minutes, 'ampm' => $cday_end_ampm),
                                        'allday' => $allday,
                                        'hide_time' => $hide_time,
                                        'past' => 0
                                    ));

                                    if ($timestamp_date_select >= strtotime($cday[0])) {
                                        if ($cday[0] == $data['date']) {

                                            $item_event['event_id'] = $event->ID;
                                            $item_event['title'] = $event->post_title;
                                            $item_event['repeat'] = 1;

                                            //Invoices and Attendees
                                            $invoices = get_posts(array(
                                                'post_type' => 'mec_invoice',
                                                'posts_per_page' => -1
                                            ));

                                            foreach ($invoices as $key_invoices => $invoice) {
                                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                if ($event->ID == $invoice_event_id) {
                                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                    foreach ($atts as $key_att => $att) {
                                                        $j++;
                                                    }
                                                }
                                            }

                                            $item_event['total_attendees'] = $j;
                                            $list[] = $item_event;
                                            $i++;
                                            continue 1;
                                        }
                                    } else {
                                        $loop = false;
                                    }
                                }

                                // No future date found so the event is passed
                                if (!count($dates)) {
                                    $dates[] = $this->add_timestamps(array(
                                        'start' => $start_date,
                                        'end' => $finish_date,
                                        'allday' => $allday,
                                        'hide_time' => $hide_time,
                                        'past' => 0
                                    ));
                                }

                            } else if ($repeat_type == 'advanced') {
                                // Start from Event Start Date
                                if (strtotime($start_date) > strtotime($today)) $today = $start_date;

                                // Get user specifed days of month for repeat
                                $advanced_days = get_post_meta($event->ID, 'mec_advanced_days', true);

                                // Generate dates for event
                                $event_info = array('start' => $start_date, 'end' => $end_date, 'allday' => $allday, 'hide_time' => $hide_time, 'finish_date' => $finish_date['date'], 'exceptional_days' => $exceptional_days, 'mec_repeat_end' => get_post_meta($event->ID, 'mec_repeat_end', true), 'occurrences' => get_post_meta($event->ID, 'mec_repeat_end_at_occurrences', true));
                                $referer_date = $today;
                                $mode = 'render';

                                //   $dates = $this->generate_advanced_days($advanced_days, $event_info, $maximum, $today);

                                if (!count($advanced_days)) return array();
                                if (!trim($referer_date)) $referer_date = date('Y-m-d', current_time('timestamp', 0));

                                $levels = array('first', 'second', 'third', 'fourth', 'last');
                                $year = date('Y', strtotime($event_info['start']));
                                $dates = array();

                                // Set last month for include current month results
                                $month = date('m', strtotime('first day of last month', strtotime($event_info['start'])));

                                if ($month == '12') $year = $year - 1;

                                $q = 0;
                                $loop = true;
                                // Event info
                                $exceptional_days = array_key_exists('exceptional_days', $event_info) ? $event_info['exceptional_days'] : array();
                                $start_date = $event_info['start'];
                                $end_date = $event_info['end'];
                                $allday = array_key_exists('allday', $event_info) ? $event_info['allday'] : 0;
                                $hide_time = array_key_exists('hide_time', $event_info) ? $event_info['hide_time'] : 0;
                                $finish_date = array_key_exists('finish_date', $event_info) ? $event_info['finish_date'] : '0000-00-00';
                                $event_period = $main->date_diff($start_date, $end_date);
                                $event_period_days = $event_period ? $event_period->days : 0;
                                $mec_repeat_end = array_key_exists('mec_repeat_end', $event_info) ? $event_info['mec_repeat_end'] : '';
                                $occurrences = array_key_exists('occurrences', $event_info) ? $event_info['occurrences'] : 0;

                                // Include default start date to results
                                if (!$main->is_past($start_date, $referer_date) and !in_array($start_date, $exceptional_days)) {
                                    $dates[] = $this->add_timestamps(array(
                                        'start' => $start_date,
                                        'end' => $end_date,
                                        'allday' => $allday,
                                        'hide_time' => $hide_time,
                                        'past' => 0,
                                    ));

                                    if ($timestamp_date_select >= strtotime($start_date)) {
                                        if ($start_date == $data['date']) {

                                            $item_event['event_id'] = $event->ID;
                                            $item_event['title'] = $event->post_title;
                                            $item_event['repeat'] = 1;

                                            //Invoices and Attendees
                                            $invoices = get_posts(array(
                                                'post_type' => 'mec_invoice',
                                                'posts_per_page' => -1
                                            ));

                                            foreach ($invoices as $key_invoices => $invoice) {
                                                $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                if ($event->ID == $invoice_event_id) {
                                                    $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                    foreach ($atts as $key_att => $att) {
                                                        $j++;
                                                    }
                                                }
                                            }

                                            $item_event['total_attendees'] = $j;
                                            $list[] = $item_event;
                                            $i++;
                                            continue 1;
                                        }
                                    } else {
                                        $loop = false;
                                    }
                                    $q++;
                                }

                                while ($loop) {
                                    $start = NULL;

                                    foreach ($advanced_days as $day) {
                                        if (!$loop) break;

                                        // Explode $day value for example (Sun.1) to Sun and 1
                                        $d = explode('.', $day);

                                        // Set indexes for {$levels} index if number day is Last(Sun.l) then indexes set 4th {$levels} index
                                        $index = intval($d[1]) ? (intval($d[1]) - 1) : 4;

                                        // Generate date
                                        $date = date('Y-m-t', strtotime("{$year}-{$month}-01"));

                                        // Generate start date for example "first Sun of next month"
                                        $start = date('Y-m-d', strtotime("{$levels[$index]} {$d[0]} of next month", strtotime($date)));
                                        $end = date('Y-m-d', strtotime("+{$event_period_days} Days", strtotime($start)));

                                        // When ends repeat date set
                                        //if ($mode == 'render' and $main->is_past($finish_date, $start)) continue;

                                        // Jump to next level if start date is past
                                        if ($main->is_past($start, $referer_date) or in_array($start, $exceptional_days)) continue;

                                        // Add dates
                                        $dates[] = $this->add_timestamps(array(
                                            'start' => array(
                                                'date' => $start,
                                                'hour' => get_post_meta($event->ID, 'mec_start_time_hour', true),
                                                'minutes' => get_post_meta($event->ID, 'mec_start_time_minutes', true),
                                                'ampm' => get_post_meta($event->ID, 'mec_start_time_ampm', true),
                                            ),
                                            'end' => array(
                                                'date' => $end,
                                                'hour' => get_post_meta($event->ID, 'mec_end_time_hour', true),
                                                'minutes' => get_post_meta($event->ID, 'mec_end_time_minutes', true),
                                                'ampm' => get_post_meta($event->ID, 'mec_end_time_ampm', true),
                                            ),
                                            'allday' => $allday,
                                            'hide_time' => $hide_time,
                                            'past' => 0,
                                        ));

                                        if ($timestamp_date_select >= strtotime($start)) {
                                            if ($start == $data['date']) {

                                                $item_event['event_id'] = $event->ID;
                                                $item_event['title'] = $event->post_title;
                                                $item_event['repeat'] = 1;

                                                //Invoices and Attendees
                                                $invoices = get_posts(array(
                                                    'post_type' => 'mec_invoice',
                                                    'posts_per_page' => -1
                                                ));

                                                foreach ($invoices as $key_invoices => $invoice) {
                                                    $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                                                    if ($event->ID == $invoice_event_id) {
                                                        $book_id = get_post_meta($invoice->ID, 'book_id', true);
                                                        $atts = get_post_meta($book_id, 'mec_attendees', true);
                                                        foreach ($atts as $key_att => $att) {
                                                            $j++;
                                                        }
                                                    }
                                                }

                                                $item_event['total_attendees'] = $j;
                                                $list[] = $item_event;
                                                $i++;
                                                break;
                                            }
                                        } else {
                                            $loop = false;
                                        }

                                        $q++;
                                    }

                                    // When ends repeat date set
                                    //  if ($mode == 'render' and $main->is_past($finish_date, $start)) break;

                                    // Change month and years for next resualts
                                    if (intval($month) == 12) {
                                        $year = intval($year) + 1;
                                        $month = '00';
                                    }

                                    $month = sprintf("%02d", intval($month) + 1);
                                }

                                if (($mode == 'render') and (trim($mec_repeat_end) == 'occurrences') and (count($dates) > $occurrences)) {
                                    $max = strtotime(reset($dates)['start']['date']);
                                    $pos = 0;

                                    for ($c = 1; $c < count($dates); $c++) {
                                        if (strtotime($dates[$c]['start']['date']) > $max) {
                                            $max = strtotime($dates[$c]['start']['date']);
                                            $pos = $c;
                                        }
                                    }

                                    unset($dates[$pos]);
                                }

                            }
                        }
                    }
                }

                $listResult = array();
                foreach ($list as $key_event => $event) {
                    $listResult[] = $event;
                }

                $data_response = array('status' => 200, 'msg' => "Successful Get List Events.", 'data' => $listResult);
                return $this->responseApiApp($data_response, 200);
            } else {
                $args = array(
                    'post_type' => 'mec-events',
                    'posts_per_page' => -1
                );

                $posts = get_posts($args);
                $list = array();
                $i = 0;

                foreach ($posts as $key => $data) {
                    if($main->get_original_event($data->ID)!=$data->ID){
                        continue;
                    }
                    $event_details = $db->select("SELECT * FROM `#__mec_events` WHERE `post_id`='$data->ID'", "loadObject");
                    $j = 0;
                    $post_id = $data->ID;
                    $list[$i]['event_id'] = $post_id;
                    $list[$i]['title'] = $data->post_title;

                    if (isset($event_details->repeat) and $event_details->repeat == '0') {
                        $list[$i]['repeat'] = 0;
                    } else {
                        $list[$i]['repeat'] = 1;
                    }

                    // Invoices and Attendees
                    $invoices = get_posts(array(
                        'post_type' => 'mec_invoice',
                        'posts_per_page' => -1
                    ));

                    foreach ($invoices as $key_invoices => $invoice) {
                        $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                        if ($post_id == $invoice_event_id) {
                            $book_id = get_post_meta($invoice->ID, 'book_id', true);
                            $atts = get_post_meta($book_id, 'mec_attendees', true);
                            foreach ($atts as $key_att => $att) {
                                $j++;
                            }
                        }
                    }

                    $list[$i]['total_attendees'] = $j;
                    $i++;
                }

                $data_response = array('status' => 200, 'msg' => "Successful Get List Events.", 'data' => $list);
                return $this->responseApiApp($data_response, 200);
            }
        } else {
            $data_response = array('status' => 404, 'msg' => "Invalid Key.", 'data' => null);
            return $this->responseApiApp($data_response, 404);
        }

    }

    public function add_timestamps($date)
    {
        $start = (isset($date['start']) and is_array($date['start'])) ? $date['start'] : array();
        $end = (isset($date['end']) and is_array($date['end'])) ? $date['end'] : array();

        if (!count($start) or !count($end)) return $date;

        $s_hour = $start['hour'];
        if (strtoupper($start['ampm']) == 'AM' and $s_hour == '0') $s_hour = 12;

        $e_hour = $end['hour'];
        if (strtoupper($end['ampm']) == 'AM' and $e_hour == '0') $e_hour = 12;

        $allday = (isset($date['allday']) ? $date['allday'] : 0);

        // All Day Event
        if ($allday) {
            $s_hour = 12;
            $start['minutes'] = 1;
            $start['ampm'] = 'AM';

            $e_hour = 11;
            $end['minutes'] = 59;
            $end['ampm'] = 'PM';
        }

        $start_time = $start['date'] . ' ' . sprintf("%02d", $s_hour) . ':' . sprintf("%02d", $start['minutes']) . ' ' . $start['ampm'];
        $end_time = $end['date'] . ' ' . sprintf("%02d", $e_hour) . ':' . sprintf("%02d", $end['minutes']) . ' ' . $end['ampm'];

        $start['timestamp'] = strtotime($start_time);
        $end['timestamp'] = strtotime($end_time);

        $hide_time = (isset($date['hide_time']) ? $date['hide_time'] : 0);
        $past = (isset($date['past']) ? $date['past'] : 0);

        return array(
            'start' => $start,
            'end' => $end,
            'allday' => $allday,
            'hide_time' => $hide_time,
            'past' => $past,
        );
    }

    /**
     * Attedees API App
     *
     * @since     1.0.0
     */
    function attedeesApiApp($data)
    {
        if (isset($data['key']) && get_option("api-key-app") === $data['key']) {

            if (isset($data['event_id'])) {

                $list = array();
                $i = 0;
                $total_checkin = 0;
                $total_attendees = 0;
                $post_id = $data['event_id'];


                // Invoices and Attendees
                $invoices = get_posts(array(
                    'post_type' => 'mec_invoice',
                    'posts_per_page' => -1
                ));

                foreach ($invoices as $key_invoices => $invoice) {

                    $invoice_event_id = get_post_meta($invoice->ID, 'event_id', true);
                    if ($post_id == $invoice_event_id) {
//                        $total_checkin += intval(get_post_meta($invoice->ID, 'checkedInCount', true));

                        $book_id = get_post_meta($invoice->ID, 'book_id', true);
                        $atts = get_post_meta($book_id, 'mec_attendees', true);
                        $j = 1;

                        foreach ($atts as $key_att => $att) {

                            $user = get_user_by('email', $att['email']);
                            $list[$i]['user_id'] = intval($user->ID);
                            $list[$i]['event_id'] = intval($invoice_event_id);
                            $list[$i]['invoice_id'] = intval($invoice->ID);
                            $list[$i]['book_id'] = intval($book_id);
                            $list[$i]['transaction_id'] = get_post_meta($book_id, 'mec_transaction_id', true);
                            $list[$i]['name'] = $att['name'];
                            $list[$i]['email'] = $att['email'];
                            $list[$i]['avatar'] = get_avatar_url($att['email']);
                            $list[$i]['key_checkin'] = $j;

                            if (Attendee::hasCheckedIn($invoice->ID, $att['email'], $j)) {
                                $list[$i]['checkin_status'] = 1;
                                $total_checkin += 1;
                            } else {
                                $list[$i]['checkin_status'] = 0;
                            }

                            $i++;
                            $j++;
                        }
                    }
                }

                $total_attendees = $i;
                $data_response = array('status' => 200, 'msg' => "Successful Get List Attendees.", 'data' => array(
                    'list' => $list,
                    'total_attendees' => $total_attendees,
                    'total_checkin' => $total_checkin

                ));
                return $this->responseApiApp($data_response, 200);
            } else {
                $data_response = array('status' => 404, 'msg' => "Not Found Event.", 'data' => null);
                return $this->responseApiApp($data_response, 404);
            }
        } else {
            $data_response = array('status' => 404, 'msg' => "Invalid Key.", 'data' => null);
            return $this->responseApiApp($data_response, 404);
        }

    }

    /**
     * Checkin Attedees API App
     *
     * @since     1.0.0
     */
    function checkAttedeesApiApp($data)
    {

        if (isset($data['key']) && get_option("api-key-app") === $data['key']) {

            if (isset($data['book_id']) && isset($data['invoice_id'])) {
                $book_id = $data['book_id'];
                $invoiceId = $data['invoice_id'];
                if (isset($data['all_checked']) && $data['all_checked'] == "all") {
                    $verified = get_post_meta($book_id, 'mec_verified', true);

                    if ($verified !== '1') {
                        $data_response = array('status' => 404, 'status_scan' => -1, 'msg' => "Booking Is Not Verified!", 'data' => null);
                        return $this->responseApiApp($data_response, 404);
                    }

                    $atts = get_post_meta($book_id, 'mec_attendees', true);
                    $j = 1;
                    $checkin = 0;
                    $uncheck = 0;

                    foreach ($atts as $key_att => $att) {
                        if (!Attendee::hasCheckedIn($invoiceId, $att['email'], $j)) {
                            Attendee::doCheckIn($invoiceId, $att['email'], $j, null, true);
                            $checkin++;
                        } else {
                            Attendee::doCheckOut($invoiceId, $att['email'], $j, null, true);
                            $uncheck++;
                        }
                        $j++;
                    }

                    if ($checkin >= $uncheck) {
                        $text = __('All Checkin', 'mec-invoice');
                        $status = 1;
                        $status_scan = 1;
                    } else {
                        $text = __('All Uncheck', 'mec-invoice');
                        $status = 0;
                        $status_scan = 0;
                    }

                    $data_response = array('status' => 200, 'status_scan' => $status_scan, 'msg' => $text, 'data' => array(
                        'status' => $status,
                    ));
                    return $this->responseApiApp($data_response, 200);

                } else {
                    if (isset($data['email']) && isset($data['key_checkin'])) {
                        $email = $data['email'];
                        $key = $data['key_checkin'];
                        $verified = get_post_meta($book_id, 'mec_verified', true);

                        if ($verified !== '1') {
                            $data_response = array('status' => 404, 'status_scan' => -1, 'msg' => "Booking Is Not Verified!", 'data' => null);
                            return $this->responseApiApp($data_response, 404);
                        }

                        if (!Attendee::hasCheckedIn($invoiceId, $email, $key)) {
                            Attendee::doCheckIn($invoiceId, $email, $key, null, true);
                            $text = __('Checkin', 'mec-invoice');
                            $status = 1;
                            $status_scan = 1;
                        } else {
                            Attendee::doCheckOut($invoiceId, $email, $key, null, true);
                            $text = __('Uncheck', 'mec-invoice');
                            $status = 0;
                            $status_scan = 0;
                        }

                        $data_response = array('status' => 200, 'status_scan' => $status_scan, 'msg' => $text, 'data' => array(
                            'status' => $status,
                        ));
                        return $this->responseApiApp($data_response, 200);
                    } else {
                        $data_response = array('status' => 404, 'status_scan' => -1, 'msg' => "Not Found Event.", 'data' => null);
                        return $this->responseApiApp($data_response, 404);
                    }
                }

            } else {
                $data_response = array('status' => 404, 'status_scan' => -1, 'msg' => "Not Found Event.", 'data' => null);
                return $this->responseApiApp($data_response, 404);
            }
        } else {
            $data_response = array('status' => 404, 'status_scan' => -1, 'msg' => "Invalid Key.", 'data' => null);
            return $this->responseApiApp($data_response, 404);
        }
    }

    /**
     * Response API App
     *
     * @since     1.0.0
     */
    public
    function responseApiApp($data, $status)
    {
        $response = new WP_REST_Response($data);
        $response->set_status($status);
        return $response;
    }


    /**
     *  ADD Export Menu
     *
     * @since     1.0.0
     */
    public function exportMenu()
    {
        add_submenu_page(
            'edit.php?post_type=mec_invoice',
            'Export Invoice (XML)',
            'Export Invoice (XML)',
            'manage_options',
            'export-xml',
            [static::instance(), 'exportAsXML']
        );
    }

    /**
     * Export As XML Page
     *
     * @since     1.0.0
     */
    public function exportAsXML()
    {
        echo '<div class="mec-invoice-export-as-xml-wrap">';
        echo '<div class="wns-be-container"><div id="wns-be-infobar"></div></div>';
        echo '<form method="post" action="#">
            <input type="hidden" name="mec-invoice-option" value="export">
            ' . wp_nonce_field('mec-invoice-export-data') . '
            <input type="submit" class="primary mec-invoice-export-btn" value="Export Invoices">
            <span class="description">' . __('Export all the details of invoices in "XMl" format. To proceed, please use the WordPress importer.', 'mec-invoice') . '</span>
        </form>';
        echo '</div>';
    }

    /**
     *  Process Bulk Actions
     *
     * @since     1.0.0
     */
    public function processExportAction()
    {
        if (is_admin()) {
            if ($_POST) {

                if (isset($_POST['_wpnonce'])) {
                    if (!wp_verify_nonce($_POST['_wpnonce'], 'mec-invoice-export-data')) {
                        return;
                    }
                } else {
                    return;
                }
                if (isset($_POST['mec-invoice-option']) && $_POST['mec-invoice-option'] = 'export') {
                    require_once(ABSPATH . 'wp-admin/includes/export.php');
                    $args = array('content' => 'mec_invoice'); // custom post type cpt
                    export_wp($args);
                    exit();
                }
            }
        }
    }

    /**
     *  Change Ticket Status.
     *
     * @since   1.0.0
     */
    public static function changeTicketStatus()
    {
        if (!Attendee::doIHaveAccess()) {
            ob_start();
            header("Content-type:application/json");
            echo json_encode([
                'status' => 'error',
                'text' => 'You don`t have permission!',
                'checked_time' => ''
            ]);
            die();
        }

        $book_id = isset($_POST['book_id']) ? $_POST['book_id'] : '';
        $invoiceId = isset($_POST['invoice_id']) ? $_POST['invoice_id'] : '';
        $timestamp = isset($_POST['timestamp']) ? $_POST['timestamp'] : null;
        $verified = get_post_meta($book_id, 'mec_verified', true);
        if ($verified !== '1') {
            ob_start();
            header("Content-type:application/json");
            echo json_encode([
                'status' => 'error',
                'text' => 'Booking Is Not Verified!',
                'checked_time' => ''
            ]);
            die();
        }
        $me = isset($_POST['me']) ? $_POST['me'] : '';
        $place = isset($_POST['place']) ? $_POST['place'] : '';
        if (!$me || !$book_id || !$invoiceId) {
            return;
        }

        if (!Attendee::hasCheckedIn($invoiceId, $me, $place, $timestamp)) {
            Attendee::doCheckIn($invoiceId, $me, $place, $timestamp);
            $checkedTime = Attendee::get_checkedin_time($invoiceId, $place, $timestamp);
            $text = __('Uncheck', 'mec-invoice');
            $status = 'yes';
        } else {
            Attendee::doCheckOut($invoiceId, $me, $place, $timestamp);
            $checkedTime = '-';
            $text = __('Checkin', 'mec-invoice');
            $status = 'no';
        }

        ob_start();
        header("Content-type:application/json");
        echo json_encode([
            'status' => $status,
            'text' => $text,
            'checked_time' => date_i18n(get_option('date_format') . ' - ' . get_option('time_format'), $checkedTime)
        ]);
        die();
    }

    /**
     * Import Old Invoices (AJAX ACTION)
     *
     * @since     1.0.0
     */
    public function importOldInvoices()
    {
        if (!is_admin()) {
            return;
        }

        $args = array(
            'post_type' => 'mec-books',
            'post_status' => 'all',
            'numberposts' => -1
        );

        $latest_books = get_posts($args);
        $counter = 0;
        foreach ($latest_books as $book) {
            $invoiceId = get_post_meta($book->ID, 'invoiceID', true);
            $event_id = get_post_meta($book->ID, 'mec_event_id', true);

            if (FALSE === get_post_status($invoiceId)) {
                if (FALSE !== get_post_status($event_id)) {
                    $counter++;
                    static::add_invoice($book->ID, false);
                }
            }
        }
        echo __(sprintf('The %s invoice added successfully!', $counter), 'mec-invoice');
        die();
    }

    /**
     * Add Invoice Manually
     *
     * @since     1.0.0
     */
    public function addInvoiceManually($bookID)
    {
        if (!$bookID || get_post_meta($bookID, 'invoiceID', true)) {
            return;
        }

        global $mec_create_booking_from_waiting;
        if (!wp_doing_ajax() && is_admin() && !$mec_create_booking_from_waiting ) {
            global $wpdb;
            $wpdb->get_results("DELETE FROM " . $wpdb->prefix . "posts WHERE post_status = 'auto-draft' and post_type='mec_invoice'");
            $wpdb->get_results("DELETE FROM " . $wpdb->prefix . "posts WHERE post_title = 'Auto Draft' and post_type='mec_invoice'");

            $ID = static::add_invoice($bookID, false);
            global $MEC_Invoice_ID;
            $MEC_Invoice_ID = $ID;
            add_action('mec_booking_saved_and_process_completed', [$this, 'redirect_to_invoice_page'], 999);

        } else {
            static::add_invoice($bookID);
        }
        remove_action('mec_booking_completed', [$this, 'addInvoiceManually']);
    }

    public function redirect_to_invoice_page($post_id)
    {

        global $MEC_Invoice_ID;
        echo '<div style="position:fixed;left:0;top:0;width:100vw;height:100vh;background-color:#fefefe;color:#fff;">
            <h1 style="display: block;text-align: center;color: #bbb;">Invoice #' . $MEC_Invoice_ID . '</h1>
            <h1 style="display: block;text-align: center;color: #ebebeb;">Please Wait To Redirect!</h1>
            <a href="' . get_edit_post_link($MEC_Invoice_ID) . '" id="mec-editLink" style="display: block;text-align: center;color: #ebebeb;" >'. __('Manage Invoice', 'mec-invoice') .'</a>

            </div>';
        echo '<script>
            setTimeout(() => {
                document.getElementById("mec-editLink").click();
            },1000);
        </script>';
        die();
    }

    /**
     *  Add Invoice.
     *
     * @since   1.0.0
     */
    public static function add_invoice($bookID, $doAction = true, $attendee = false)
    {
        if (get_post_meta($bookID, 'invoiceID', true)) {
            return;
        }

        $main = \MEC::getInstance('app.libraries.main');
        $settings = $main->get_settings();
        if (!isset($settings['booking_status']) || !$settings['booking_status']) {
            return;
        }

        $price = get_post_meta($bookID, 'mec_price', true);
        $event_id = get_post_meta($bookID, 'mec_event_id', true);
        $confirmed = get_post_meta($bookID, 'mec_confirmed', true);
        $IID = sha1(microtime());
        $doCheckinHash = sha1(md5(microtime()));
        if (!$doAction) {
            $time = get_post_meta($bookID, 'mec_booking_time', true);
        } else {
            $time = date("Y-m-d H:i:s", current_time('timestamp'));
        }

        $ID = wp_insert_post([
            'post_title' => __('Invoice', 'mec-invoice') . ' #' . $bookID,
            'post_type' => 'mec_invoice',
            'post_status' => 'publish',
            'post_date' => $time,
            'meta_input' => array(
                'invoiceID' => $IID,
                'CheckinHash' => $doCheckinHash,
                'book_id' => $bookID,
                'transaction_id' => get_post_meta($bookID, 'mec_transaction_id', true),
                'price' => $price,
                'date' => strtotime($time),
                'confirmed' => $confirmed,
                'status' => 'open',
                'event_id' => $event_id,
                'date_submit' => date('YmdHis', current_time('timestamp', 0)),
            ),
        ]);
        wp_update_post([
            'ID' => $ID,
            'post_title' => __('Invoice', 'mec-invoice') . ' #' . \MEC_Invoice\Helper\Invoice::create_invoice_number($ID),
        ]);

        update_post_meta($bookID, 'invoiceID', $ID);

        if ($doAction) {
            do_action('mec-invoice-new-invoice-added', $ID);
        }


        return $ID;
    }

    /**
     *  Apply Settings Page Menu.
     *
     * @since   1.0.0
     */
    public static function applySettingsPageMenu($items, $active_menu)
    {
        $items[__('Invoice Options', 'mec-invoice')] = 'invoice_options';
        return $items;
    }

    /**
     *  Render Invoice Settings In MEC Settings Page.
     *
     * @since   1.0.0
     */
    public static function renderSettingsPage($settings)
    {
        $main = new \MEC_main();
        ?>
        <style>
            .CodeMirror {
                border: 1px solid #ddd;
            }

            .CodeMirror-gutters {
                width: 35px;
            }

            .CodeMirror-sizer {
                margin-left: 35px !important;
            }

            .CodeMirror-gutter-wrapper {
                margin-left: -20px !important;
            }
        </style>
        <style>
            #mec-invoice-custom-css {
                height: 200px;
                margin-bottom: 5px;
                font-family: Consolas, Monaco, monospace;
                font-size: 13px;
                width: 100%;
                background: #f9f9f9;
                outline: 0;
                line-height: 1.5;
            }
        </style>
        <div id="invoice_options" class="mec-options-fields">
            <h4 class="mec-form-subtitle"><?php _e('Invoice Options', 'mec-invoice'); ?></h4>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_invoice_title_pattern"><?php _e('Invoice Title', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_invoice_title_pattern"
                           name="mec[settings][mec_invoice_title_pattern]"
                           value="<?php echo((isset($settings['mec_invoice_title_pattern']) and trim($settings['mec_invoice_title_pattern']) != '') ? stripslashes(htmlentities($settings['mec_invoice_title_pattern'])) : ''); ?>"/>
                </div>
                <div class="mec-col-12">
                    <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                        <?php echo __('You can use this patterns in title', 'mec-invoice'); ?>:
                        <?php
                        $title_patterns = apply_filters('mec_invoice_title_patterns_for_display_settings', array(
                            'invoice_id' => '',
                            'invoice_number' => '',
                            'event_title' => '',
                        ));

                        foreach ($title_patterns as $pattern_key => $description) {

                            echo '<strong style="position: relative;display: block;margin-top: 5px;">{' . $pattern_key . '} ' . $description . '</strong>';
                        }
                        ?>
                        <?php _e('Example:', 'mec-invoice') ?>
                        <br/>
                        <?php echo sprintf(__('Your Invoice #%s in %s', 'mec-invoice'), '{invoice_number}', '{event_title}'); ?>
                    </p>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Company Information', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_company_url"><?php _e('Company URL', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_company_url" name="mec[settings][mec_invoice_company_url]"
                           value="<?php echo((isset($settings['mec_invoice_company_url']) and trim($settings['mec_invoice_company_url']) != '') ? stripslashes(htmlentities($settings['mec_invoice_company_url'])) : ''); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_company_name"><?php _e('Company Name', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_company_name" name="mec[settings][mec_invoice_company_name]"
                           value="<?php echo((isset($settings['mec_invoice_company_name']) and trim($settings['mec_invoice_company_name']) != '') ? stripslashes(htmlentities($settings['mec_invoice_company_name'])) : ''); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_company_email"><?php _e('Company Email', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_company_email" name="mec[settings][mec_invoice_company_email]"
                           value="<?php echo((isset($settings['mec_invoice_company_email']) and trim($settings['mec_invoice_company_email']) != '') ? stripslashes(htmlentities($settings['mec_invoice_company_email'])) : ''); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_company_email"><?php _e('Company Phone', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_company_phone" name="mec[settings][mec_invoice_company_phone]"
                           value="<?php echo((isset($settings['mec_invoice_company_phone']) and trim($settings['mec_invoice_company_phone']) != '') ? stripslashes(htmlentities($settings['mec_invoice_company_phone'])) : ''); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_settings_vat_number"><?php _e('Vat Number', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_vat_number" name="mec[settings][mec_invoice_vat_number]"
                           value="<?php echo((isset($settings['mec_invoice_vat_number']) and trim($settings['mec_invoice_vat_number']) != '') ? stripslashes(htmlentities($settings['mec_invoice_vat_number'])) : ''); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_description"><?php _e('Description', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <textarea id="mec_settings_description" name="mec[settings][mec_invoice_description]"
                              rows="4"><?php echo((isset($settings['mec_invoice_description']) and trim($settings['mec_invoice_description']) != '') ? stripslashes(htmlentities($settings['mec_invoice_description'])) : ''); ?></textarea>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_settings_address"><?php _e('Address', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <textarea id="mec_settings_address" name="mec[settings][mec_invoice_address]"
                              rows="4"><?php echo((isset($settings['mec_invoice_address']) and trim($settings['mec_invoice_address']) != '') ? stripslashes(htmlentities($settings['mec_invoice_address'])) : ''); ?></textarea>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_settings_logo"><?php _e('Logo', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_logo" name="mec[settings][mec_invoice_logo]"
                           value="<?php echo((isset($settings['mec_invoice_logo']) and trim($settings['mec_invoice_logo']) != '') ? stripslashes(htmlentities($settings['mec_invoice_logo'])) : ''); ?>"/>
                    <button class="select-company-logo button">Select</button>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_logo_width"><?php esc_html_e('Logo Width', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_logo_width" name="mec[settings][mec_invoice_logo_width]"
                           value="<?php echo(isset($settings['mec_invoice_logo_width']) ? (int)$settings['mec_invoice_logo_width'] : ''); ?>"
                           placeholder="<?php esc_attr_e('50', 'mec-invoice') ?>"/>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Invoice Number Generation Settings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_invoice_number"><?php _e('Invoice Number Start Point', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="number" id="mec_settings_invoice_number"
                           name="mec[settings][mec_invoice_number_start_point]"
                           value="<?php echo((isset($settings['mec_invoice_number_start_point']) and trim($settings['mec_invoice_number_start_point']) != '') ? stripslashes(htmlentities($settings['mec_invoice_number_start_point'])) : '0'); ?>"/>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_invoice_number_length"><?php _e('Invoice Number Length', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="number" id="mec_settings_invoice_number_length"
                           name="mec[settings][mec_invoice_number_length]"
                           value="<?php echo((isset($settings['mec_invoice_number_length']) and trim($settings['mec_invoice_number_length']) != '') ? stripslashes(htmlentities($settings['mec_invoice_number_length'])) : '0'); ?>"/>
                </div>
                <div class="mec-col-12">
                    <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                        <?php echo __('Examples', 'mec-invoice'); ?>:
                        <strong style="position: relative;display: block;margin-top: 5px;">"0"
                            <span class="mec-invoice-quote">12345</span>
                        </strong>
                        <strong style="position: relative;display: block;margin-top: 5px;">"6"
                            <span class="mec-invoice-quote">012345</span>
                        </strong>
                        <strong style="position: relative;display: block;margin-top: 5px;">"10"
                            <span class="mec-invoice-quote">0000012345</span>
                        </strong>
                    </p>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_invoice_number_pattern"><?php _e('Invoice Number Pattern', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_invoice_number_pattern"
                           name="mec[settings][mec_invoice_number_pattern]"
                           value="<?php echo((isset($settings['mec_invoice_number_pattern']) and trim($settings['mec_invoice_number_pattern']) != '') ? stripslashes(htmlentities($settings['mec_invoice_number_pattern'])) : '{number}'); ?>"/>
                </div>
                <div class="mec-col-12">
                    <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                        <?php echo __('Examples', 'mec-invoice'); ?>:
                        <strong style="position: relative;display: block;margin-top: 5px;"> {number}
                            <span class="mec-invoice-quote"> 12345</span>
                        </strong>
                        <strong style="position: relative;display: block;margin-top: 5px;">Prefix-{number}
                            <span class="mec-invoice-quote">Prefix-12345</span>
                        </strong>
                        <strong style="position: relative;display: block;margin-top: 5px;">Prefix-{number}-Suffix
                            <span class="mec-invoice-quote">Prefix-12345-Suffix</span>
                        </strong>
                    </p>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('PDF Generation Settings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_html2pdf"><?php _e('html2pdf API Key', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_html2pdf" name="mec[settings][mec_invoice_html2pdf]"
                           value="<?php echo((isset($settings['mec_invoice_html2pdf']) and trim($settings['mec_invoice_html2pdf']) != '') ? stripslashes(htmlentities($settings['mec_invoice_html2pdf'])) : ''); ?>"/>
                </div>
                <div class="mec-col-12">
                    <?php if (!isset($settings['mec_invoice_html2pdf']) || !trim($settings['mec_invoice_html2pdf'])) : ?>
                        <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                            Enter your API Key to download the PDF file. Use the following link to get a new API key: <a
                                    href="https://dash.html2pdf.app/registration"
                                    target="_blank">https://html2pdf.app</a></p>
                    <?php endif; ?>
                </div>
            </div>

            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_selectpdf"><?php _e('Selectpdf API Key', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_selectpdf" name="mec[settings][mec_invoice_selectpdf]"
                           value="<?php echo((isset($settings['mec_invoice_selectpdf']) and trim($settings['mec_invoice_selectpdf']) != '') ? stripslashes(htmlentities($settings['mec_invoice_selectpdf'])) : ''); ?>"/>
                </div>
                <div class="mec-col-12">
                    <?php if (!isset($settings['mec_invoice_selectpdf']) || !trim($settings['mec_invoice_selectpdf'])) : ?>
                        <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                            Enter your API Key to create the PDF file. Use the following link to purchase a new API key:
                            <a href="https://selectpdf.com/pricing/" target="_blank">SelectPdf Online HTML to PDF
                                API</a></p>
                    <?php endif; ?>
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_settings_selectpdf"><?php _e('Selectpdf Margins', 'mec-invoice'); ?>
                    <span class="mec-tooltip">
                        <div class="box">
                            <h5 class="title"><?php _e('PDF Margins', 'mec-invoice') ?></h5>
                            <div class="content">
                                <p><?php _e('Default margin is: 5 5 5 5 . Please don\'t use "px" or "%" or "em" on box field. ', 'mec-invoice') ?></p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </label>
                <div class="mec-col-2" style="margin:0">
                    <input style="width: 70px; text-align: center;padding: 0;margin: 0;" type="text" placeholder="Top"
                           id="mec_settings_selectpdf" name="mec[settings][mec_invoice_selectpdf_margin_top]"
                           value="<?php echo((isset($settings['mec_invoice_selectpdf_margin_top']) and trim($settings['mec_invoice_selectpdf_margin_top']) != '') ? stripslashes(htmlentities($settings['mec_invoice_selectpdf_margin_top'])) : ''); ?>"/>
                </div>
                <div class="mec-col-2" style="margin:0">
                    <input style="width: 70px; text-align: center;padding: 0;margin: 0;" type="text" placeholder="Right"
                           id="mec_settings_selectpdf" name="mec[settings][mec_invoice_selectpdf_margin_right]"
                           value="<?php echo((isset($settings['mec_invoice_selectpdf_margin_right']) and trim($settings['mec_invoice_selectpdf_margin_right']) != '') ? stripslashes(htmlentities($settings['mec_invoice_selectpdf_margin_right'])) : ''); ?>"/>
                </div>
                <div class="mec-col-2" style="margin:0">
                    <input style="width: 70px; text-align: center;padding: 0;margin: 0;" type="text"
                           placeholder="Bottom" id="mec_settings_selectpdf"
                           name="mec[settings][mec_invoice_selectpdf_margin_bottom]"
                           value="<?php echo((isset($settings['mec_invoice_selectpdf_margin_bottom']) and trim($settings['mec_invoice_selectpdf_margin_bottom']) != '') ? stripslashes(htmlentities($settings['mec_invoice_selectpdf_margin_bottom'])) : ''); ?>"/>
                </div>
                <div class="mec-col-2" style="margin:0">
                    <input style="width: 70px; text-align: center;padding: 0;margin: 0;" type="text" placeholder="Left"
                           id="mec_settings_selectpdf" name="mec[settings][mec_invoice_selectpdf_margin_left]"
                           value="<?php echo((isset($settings['mec_invoice_selectpdf_margin_left']) and trim($settings['mec_invoice_selectpdf_margin_left']) != '') ? stripslashes(htmlentities($settings['mec_invoice_selectpdf_margin_left'])) : ''); ?>"/>
                </div>
            </div>

            <div class="mec-form-row" id="show_invoice_description_in_pdf">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_show_invoice_description_in_pdf]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_show_invoice_description_in_pdf'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_show_invoice_description_in_pdf]" <?php echo((isset($settings['mec_invoice_show_invoice_description_in_pdf']) and trim($settings['mec_invoice_show_invoice_description_in_pdf']) != '' and $settings['mec_invoice_show_invoice_description_in_pdf'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Display Invoice Description in Print and PDF Mode', 'mec-invoice'); ?>
                </label>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Live Attendees Settings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_use]" value="0"/>
                    <input type="checkbox" id="mec_invoice_use_checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_use'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_use]" <?php echo((isset($settings['mec_invoice_use']) and trim($settings['mec_invoice_use']) != '' and $settings['mec_invoice_use'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Use Pusher API For Live Actions', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="mec_invoice_use_form">
                <div class="group-box">
                    <label class="mec-col-3"
                           for="mec_settings_pusher_app_id"><?php _e('Pusher.com API', 'mec-invoice'); ?></label>
                    <div class="mec-col-9">
                        <input type="text" id="mec_settings_pusher_app_id" placeholder="app_id"
                               name="mec[settings][mec_invoice_pusher_app_id]"
                               value="<?php echo((isset($settings['mec_invoice_pusher_app_id']) and trim($settings['mec_invoice_pusher_app_id']) != '') ? stripslashes(htmlentities($settings['mec_invoice_pusher_app_id'])) : ''); ?>"/>
                        <input type="text" id="mec_settings_pusher_key" placeholder="key"
                               name="mec[settings][mec_invoice_pusher_key]"
                               value="<?php echo((isset($settings['mec_invoice_pusher_key']) and trim($settings['mec_invoice_pusher_key']) != '') ? stripslashes(htmlentities($settings['mec_invoice_pusher_key'])) : ''); ?>"/>
                        <input type="text" id="mec_settings_pusher_secret" placeholder="secret"
                               name="mec[settings][mec_invoice_pusher_secret]"
                               value="<?php echo((isset($settings['mec_invoice_pusher_secret']) and trim($settings['mec_invoice_pusher_secret']) != '') ? stripslashes(htmlentities($settings['mec_invoice_pusher_secret'])) : ''); ?>"/>
                        <input type="text" id="mec_settings_pusher_cluster" placeholder="cluster"
                               name="mec[settings][mec_invoice_pusher_cluster]"
                               value="<?php echo((isset($settings['mec_invoice_pusher_cluster']) and trim($settings['mec_invoice_pusher_cluster']) != '') ? stripslashes(htmlentities($settings['mec_invoice_pusher_cluster'])) : ''); ?>"/>
                        <?php if (!isset($settings['mec_invoice_pusher_app_id']) || !trim($settings['mec_invoice_pusher_app_id'])) : ?>
                            <p style="color: #8a6d3b; background-color: #fcf8e3; padding: 15px; margin-bottom: 20px; border: 1px solid #faebcc; border-radius: 4px; font-size: 14px;">
                                Enter your API Data to make the report page live. Use the following link to create a new
                                channel and get a new API key: <a href="https://pusher.com" target="_blank">https://pusher.com</a>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Check-in & Notification Settings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_notification_status]" value="0"/>
                    <input type="checkbox" id="mec_invoice_notification_status_checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_notification_status'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_notification_status]" <?php echo((isset($settings['mec_invoice_notification_status']) and trim($settings['mec_invoice_notification_status']) != '' and $settings['mec_invoice_notification_status'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Disable Invoice Email Notifications', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="mec_invoice_early_checkin">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_early_checkin]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_early_checkin'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_early_checkin]" <?php echo((isset($settings['mec_invoice_early_checkin']) and trim($settings['mec_invoice_early_checkin']) != '' and $settings['mec_invoice_early_checkin'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Enable Check-in Using QR Code only On Event Opening Days', 'mec-invoice'); ?>
                    <p style="color:red;margin:5px 30px;"><?php _e('When the multiple-date booking feature is enabled, this option is activated by default', 'mec-invoice'); ?></p>
                </label>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3" for="mec_invoice_early_checkin_minutes"><?php _e('Checkin X minutes earlier than the start of the event', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input id="mec_invoice_early_checkin_minutes" type="number" placeholder="0"
                           value="<?php echo $settings['mec_invoice_early_checkin_minutes'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_early_checkin_minutes]" />
                </div>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_for_each_tickets]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_for_each_tickets'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_for_each_tickets]" <?php echo((isset($settings['mec_invoice_for_each_tickets']) and trim($settings['mec_invoice_for_each_tickets']) != '' and $settings['mec_invoice_for_each_tickets'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Send Ticket & Invoice for each Attendees', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_display_invoice_number_in_checkin]" value="0"/>
                    <input type="checkbox" id="mec_invoice_display_invoice_number_in_checkin" value="on"
                           value="<?php echo $settings['mec_invoice_display_invoice_number_in_checkin'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_display_invoice_number_in_checkin]" <?php echo((isset($settings['mec_invoice_display_invoice_number_in_checkin']) and trim($settings['mec_invoice_display_invoice_number_in_checkin']) != '' and $settings['mec_invoice_display_invoice_number_in_checkin'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Display Invoice Number in Check-in page', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_invoice_attendees_custom_fields"><?php esc_html_e('Custom Fields in Invoice', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <select name="mec[settings][attendees_custom_fields][]" id="invoice_attendees_custom_fields" class="mec-select2" multiple>
                        <?php
                        $fields = $main->get_reg_fields(0);

                        if (is_array($fields)) {

                            $attendees_custom_fields = isset($settings['attendees_custom_fields']) && is_array($settings['attendees_custom_fields']) ? $settings['attendees_custom_fields'] : array();
                            foreach ($fields as $field_id => $field_args) {

                                $type = isset($field_args['type']) && !empty($field_args['type']) ? $field_args['type'] : false;
                                if (!$type || in_array($type, array('name', 'mec_email'), false)) {
                                    continue;
                                }

                                $label = isset($field_args['label']) && !empty($field_args['label']) ? $field_args['label'] : false;
                                if ($label==="I agree with %s") {
                                    continue;
                                }

                                $selected = in_array($field_id, $attendees_custom_fields);
                                echo '<option value="' . $field_id . '" ' . selected(true, $selected, false) . '>' . $label . '</option>';
                            }
                        }
                        ?>
                    </select>
                </div>
            </div>

            <div class="mec-form-row mec_booking_notifications">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_booking_notifications]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['mec_booking_notifications'] ?? ''; ?>"
                           name="mec[settings][mec_booking_notifications]" <?php echo((isset($settings['mec_booking_notifications']) and trim($settings['mec_booking_notifications']) != '' and $settings['mec_booking_notifications'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Disable MEC Booking Notifications', 'mec-invoice'); ?>
                    <span class="mec-tooltip">
                        <div class="box">
                            <h5 class="title"><?php _e('MEC Booking Notifications', 'mec-invoice') ?></h5>
                            <div class="content">
                                <p>
                                    <?php echo __("By enabling this option, the pre-defined booking confirmation sent to the customer by MEC is disabled. Add-on's email is sent in both disable/enable states.", 'mec-invoice'); ?>
                                </p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </label>
            </div>

            <div class="mec-form-row">
                <label class="mec-col-3">
                    <?php _e('Checkin Roles', 'mec-invoice'); ?>
                    <span class="mec-tooltip">
                        <div class="box">
                            <h5 class="title"><?php _e('Checkin And Uncheck Roles', 'mec-invoice') ?></h5>
                            <div class="content">
                                <p><?php _e('What is the user access for Checkin and Uncheck?', 'mec-invoice') ?></p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </label>
                <div class="mec-col-9">
                    <div class="mec-box">
                        <?php foreach (get_editable_roles() as $role_name => $role_info) :
                            if ($role_name == 'administrator') {
                                $settings['mec_invoice_checkin_roles'][$role_name] = 'on';
                            }
                            ?>
                            <label>
                                <input type="hidden"
                                       name="mec[settings][mec_invoice_checkin_roles][<?php echo $role_name ?>]"
                                       value="0"/>
                                <input type="checkbox" value="on"
                                       value="<?php echo $settings['mec_invoice_checkin_roles'][$role_name] ?? ''; ?>"
                                       name="mec[settings][mec_invoice_checkin_roles][<?php echo $role_name ?>]" <?php echo((isset($settings['mec_invoice_checkin_roles'][$role_name]) and trim($settings['mec_invoice_checkin_roles'][$role_name]) != '' and $settings['mec_invoice_checkin_roles'][$role_name] == 'on') ? 'checked="checked"' : ''); ?> /><?php _e($role_name, 'mec-invoice'); ?>
                            </label>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Invoice Rendering Settings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_rendering_mode"><?php _e('Invoice Type', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <select id="mec_settings_rendering_mode" name="mec[settings][mec_invoice_rendering_mode]">
                        <option value="ticket" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'ticket' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Classic Ticket', 'mec-invoice'); ?></option>
                        <option value="full" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'full' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Classic Invoice', 'mec-invoice'); ?></option>
                        <option value="standard" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'standard' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Standard', 'mec-invoice'); ?></option>
                        <option value="modern-ticket-1" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'modern-ticket-1' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Modern Ticket Type 1', 'mec-invoice'); ?></option>
                        <option value="modern-ticket-2" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'modern-ticket-2' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Modern Ticket Type 2', 'mec-invoice'); ?></option>
                        <option value="modern-ticket-3" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'modern-ticket-3' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Modern Ticket Type 3', 'mec-invoice'); ?></option>
                        <option value="modern-ticket-4" <?php if (isset($settings['mec_invoice_rendering_mode']) and 'modern-ticket-4' == $settings['mec_invoice_rendering_mode']) echo 'selected="selected"'; ?>><?php _e('Modern Ticket Type 4', 'mec-invoice'); ?></option>
                    </select>
                </div>
            </div>
            <div class="mec-form-row" id="mec_invoice_display_ticket_name">
                <label class="mec-col-12">
                    <?php $checked = (isset($settings['invoice_display_ticket_name']) and trim($settings['invoice_display_ticket_name']) != '' and $settings['invoice_display_ticket_name'] == 'on') ? 'checked="checked"' : ''; ?>
                    <input type="hidden" name="mec[settings][invoice_display_ticket_name]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['invoice_display_ticket_name'] ?? ''; ?>"
                           name="mec[settings][invoice_display_ticket_name]" <?php echo $checked; ?> />
                        <?php _e('Display Ticket Name(Type) in Tickets', 'mec-invoice'); ?>
                </label>
            </div>
            <?php
            if(isset($settings['disable_ticket_times']) and trim($settings['disable_ticket_times']) != '' and $settings['disable_ticket_times'] == '1'){
                $checked = '';
                $hide = 'w-hide';
                $val_display_ticket_time=0;
            }else{
                $checked = (isset($settings['invoice_display_ticket_time']) and trim($settings['invoice_display_ticket_time']) != '' and $settings['invoice_display_ticket_time'] == 'on') ? 'checked="checked"' : '';
                $hide='';
                $val_display_ticket_time=$settings['invoice_display_ticket_time'] ?? '';
            }
            $val_hide_ticket_qrcode = $settings['invoice_hide_ticket_qrcode'] ?? '';
            $checked_qrcode = (isset($settings['invoice_hide_ticket_qrcode']) and trim($settings['invoice_hide_ticket_qrcode']) != '' and $settings['invoice_hide_ticket_qrcode'] == 'on') ? 'checked="checked"' : '';
            ?>
            <div class="mec-form-row <?php echo $hide; ?>" id="mec_invoice_display_ticket_time">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][invoice_display_ticket_time]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $val_display_ticket_time; ?>"
                           name="mec[settings][invoice_display_ticket_time]" <?php echo $checked; ?> />
                    <?php _e('Display Ticket Time', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="show_invoice_tax_detail">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][show_invoice_tax_detail]" value="0"/>
                    <input type="checkbox" value="on" value="<?php echo $settings['show_invoice_tax_detail'] ?? ''; ?>"
                           name="mec[settings][show_invoice_tax_detail]" <?php echo((isset($settings['show_invoice_tax_detail']) and trim($settings['show_invoice_tax_detail']) != '' and $settings['show_invoice_tax_detail'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Show Tax Detail on invoice types', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="show_location_address">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][show_location_address]" value="0"/>
                    <input type="checkbox" value="on" value="<?php echo $settings['show_location_address'] ?? ''; ?>"
                           name="mec[settings][show_location_address]" <?php echo((isset($settings['show_location_address']) and trim($settings['show_location_address']) != '' and $settings['show_location_address'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Show Location Address', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="show_invoice_time">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][mec_invoice_show_time_in_modern_type]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $settings['mec_invoice_show_time_in_modern_type'] ?? ''; ?>"
                           name="mec[settings][mec_invoice_show_time_in_modern_type]" <?php echo((isset($settings['mec_invoice_show_time_in_modern_type']) and trim($settings['mec_invoice_show_time_in_modern_type']) != '' and $settings['mec_invoice_show_time_in_modern_type'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Show Invoice Time in Invoice Styles', 'mec-invoice'); ?>
                    <span class="mec-tooltip">
                        <div class="box">
                            <h5 class="title"><?php _e('Show invoice time in invoice types', 'mec-invoice') ?></h5>
                            <div class="content">
                                <p>
                                    <?php echo __("Display invoice time (minuets and hours) in invoice styles", 'mec-invoice'); ?>
                                </p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </label>
            </div>
            <div class="mec-form-row" id="hide_invoice_download_button_in_email">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][hide_invoice_download_button_in_email]" value="0"/>
                    <input type="checkbox" value="on" value="<?php echo $settings['hide_invoice_download_button_in_email'] ?? ''; ?>"
                           name="mec[settings][hide_invoice_download_button_in_email]" <?php echo((isset($settings['hide_invoice_download_button_in_email']) and trim($settings['hide_invoice_download_button_in_email']) != '' and $settings['hide_invoice_download_button_in_email'] == 'on') ? 'checked="checked"' : ''); ?> />
                    <?php _e('Hide The Invoice Download Button in The Email', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="mec_invoice_hide_ticket_qrcode">
                <label class="mec-col-12">
                    <input type="hidden" name="mec[settings][invoice_hide_ticket_qrcode]" value="0"/>
                    <input type="checkbox" value="on"
                           value="<?php echo $val_hide_ticket_qrcode; ?>"
                           name="mec[settings][invoice_hide_ticket_qrcode]" <?php echo $checked_qrcode; ?> />
                    <?php _e('Hide QrCode', 'mec-invoice'); ?>
                </label>
            </div>
            <div class="mec-form-row" id="mec_settings_tickets_date_format">
                <label class="mec-col-3"
                       for="mec_settings_tickets_date_format"><?php _e('Date Format (Tickets)', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_tickets_date_format" placeholder="M j, Y"
                           name="mec[settings][tickets_date_format]"
                           value="<?php echo((isset($settings['tickets_date_format']) and trim($settings['tickets_date_format']) != '') ? stripslashes(htmlentities($settings['tickets_date_format'])) : ''); ?>"/>
                </div>
                <p class="description"><?php echo __('Exam: M j, Y'); ?></p>
            </div>
            <div class="mec-form-row" id="mec_settings_tickets_time_format">
                <label class="mec-col-3"
                       for="mec_settings_tickets_time_format"><?php _e('Time Format (Tickets)', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_tickets_time_format" placeholder="h:i A"
                           name="mec[settings][tickets_time_format]"
                           value="<?php echo((isset($settings['tickets_time_format']) and trim($settings['tickets_time_format']) != '') ? stripslashes(htmlentities($settings['tickets_time_format'])) : ''); ?>"/>
                </div>
                <p class="description"><?php echo __('Exam: h:i A'); ?></p>
            </div>
            <div class="mec-form-row" id="mec_settings_invoice_date_format">
                <label class="mec-col-3"
                       for="mec_settings_invoice_date_format"><?php _e('Default Date Format', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_invoice_date_format" placeholder="M j, Y"
                           name="mec[settings][invoice_date_format]"
                           value="<?php echo((isset($settings['invoice_date_format']) and trim($settings['invoice_date_format']) != '') ? stripslashes(htmlentities($settings['invoice_date_format'])) : ''); ?>"/>
                </div>
                <p class="description"><?php echo __('Exam: M j, Y'); ?></p>
            </div>
            <div class="mec-form-row" id="mec_settings_event_date_format">
                <label class="mec-col-3"
                       for="mec_settings_event_date_format"><?php _e('Event Date Format', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <input type="text" id="mec_settings_event_date_format" placeholder="M j, Y"
                           name="mec[settings][event_date_format]"
                           value="<?php echo((isset($settings['event_date_format']) and trim($settings['event_date_format']) != '') ? stripslashes(htmlentities($settings['event_date_format'])) : ''); ?>"/>
                </div>
                <p class="description"><?php echo __('Exam: M j, Y'); ?></p>
            </div>

            <div class="mec-form-row" id="mec_invoice_descriptions_wrap">
                <label class="mec-col-12" for="mec_invoice_descriptions">
                    <?php _e('Invoice Descriptions', 'mec-invoice'); ?>
                    <span class="mec-tooltip">
                        <div class="box">
                            <h5 class="title"><?php _e('Invoice Descriptions', 'mec-invoice') ?></h5>
                            <div class="content">
                                <p><?php _e('This description is displayed in every invoice. For example, Be at the location at least one hour before the event begins.', 'mec-invoice') ?></p>
                            </div>
                        </div>
                        <i title="" class="dashicons-before dashicons-editor-help"></i>
                    </span>
                </label>
                <div class="clear"></div>
                <?php
                $content = isset($settings['mec_invoice_descriptions']) ? html_entity_decode(stripslashes($settings['mec_invoice_descriptions'])) : '';
                $_settings = [
                    'textarea_name' => 'mec[settings][mec_invoice_descriptions]',
                    'editor_height' => 300
                ];
                \wp_editor($content, 'mec_invoice_descriptions', $_settings);
                ?>
            </div>

            <h5 class="mec-form-subtitle"><?php _e('Custom CSS', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-12" for="mec_invoice_custom_styles">
                    <?php _e('Invoice Custom Styles', 'mec-invoice'); ?>
                </label>
                <div class="clear"></div>
                <div class="mec-col-12" style="margin-top: 10px;">
                    <?php $content = isset($settings['mec_invoice_custom_styles']) ? html_entity_decode(stripslashes($settings['mec_invoice_custom_styles'])) : '';
                    echo '<textarea id="mec-invoice-custom-css" name="mec[settings][mec_invoice_custom_styles]">' . esc_textarea($content) . '</textarea>'; ?>
                </div>
            </div>
            <h5 class="mec-form-subtitle"><?php _e('Generate Invoice for Old Bookings', 'mec-invoice'); ?></h5>
            <div class="mec-form-row">
                <label class="mec-col-3"
                       for="mec_settings_import"><?php _e('Import Invoices from Bookings', 'mec-invoice'); ?></label>
                <div class="mec-col-9">
                    <a href="#"
                       class="dpr-btn dpr-import-btn mec-invoice-import-button"><?php _e('Import', 'mec-invoice'); ?></a>
                </div>
            </div>
        </div>

        <script>
            jQuery(document).ready(function () {
                if (jQuery('#mec_invoice_use_checkbox').prop('checked') === true) {
                    jQuery('#mec_invoice_use_form').show();
                } else {
                    jQuery('#mec_invoice_use_form').hide();
                }

                jQuery('#mec_invoice_use_checkbox').on('input', function () {
                    if (jQuery(this).prop('checked') === true) {
                        jQuery('#mec_invoice_use_form').show();
                    } else {
                        jQuery('#mec_invoice_use_form').hide();
                    }
                });

                // if (jQuery('#mec_settings_rendering_mode').val() == 'full') {
                //     jQuery('#mec_settings_tickets_date_format').css('display', 'none');
                //     jQuery('#mec_settings_tickets_time_format').css('display', 'none');
                //     jQuery('#show_invoice_description_in_pdf').css('display', 'block');
                //     jQuery('#mec_invoice_descriptions_wrap').css('display', 'block');
                //     jQuery('.mec_booking_notifications').css('display', 'block');
                //     jQuery('#show_invoice_time').css('display', 'block');
                //     jQuery('#show_invoice_tax_detail').css('display', 'block');
                // } else {
                //     jQuery('#mec_settings_tickets_date_format').css('display', 'block');
                //     jQuery('#mec_settings_tickets_time_format').css('display', 'block');
                //     jQuery('#show_invoice_description_in_pdf').css('display', 'none');
                //     jQuery('#mec_invoice_descriptions_wrap').css('display', 'none');
                //     jQuery('.mec_booking_notifications').css('display', 'none');
                //     jQuery('#show_invoice_time').css('display', 'none');
                //     jQuery('#show_invoice_tax_detail').css('display', 'none');
                // }
                // jQuery('#mec_settings_rendering_mode').on('change', function () {
                //     if (jQuery(this).val() == 'full') {
                //         jQuery('#mec_invoice_display_ticket_name').css('display', 'none');
                //         jQuery('#mec_settings_tickets_date_format').css('display', 'none');
                //         jQuery('#mec_settings_tickets_time_format').css('display', 'none');
                //         jQuery('#show_invoice_description_in_pdf').css('display', 'block');
                //         jQuery('#mec_invoice_descriptions_wrap').css('display', 'block');
                //         jQuery('.mec_booking_notifications').css('display', 'block');
                //         jQuery('#show_invoice_time').css('display', 'block');
                //         jQuery('#show_invoice_tax_detail').css('display', 'block');
                //     } else {
                //         jQuery('#mec_invoice_display_ticket_name').css('display', 'block');
                //         jQuery('#mec_settings_tickets_date_format').css('display', 'block');
                //         jQuery('#mec_settings_tickets_time_format').css('display', 'block');
                //         jQuery('#show_invoice_description_in_pdf').css('display', 'none');
                //         jQuery('#mec_invoice_descriptions_wrap').css('display', 'none');
                //         jQuery('.mec_booking_notifications').css('display', 'none');
                //         jQuery('#show_invoice_time').css('display', 'none');
                //         jQuery('#show_invoice_tax_detail').css('display', 'none');
                //     }
                // }).trigger('change');

                jQuery('.mec-invoice-import-button').on('click', function (e) {
                    var $this = jQuery(this);
                    $this.css('opacity', '0.5');
                    $this.css('pointer-events', 'none');

                    jQuery.ajax({
                        type: "post",
                        url: ajaxurl,
                        data: {
                            action: 'mec_invoice_import_old_invoices'
                        },
                        success: function (response) {
                            alert(response);
                            $this.css('opacity', '1');
                            $this.css('pointer-events', 'auto');
                        }
                    });

                    e.preventDefault();
                    return false;
                })
            })

            jQuery(document).ready(function ($) {
                var companyLogoSelector;
                $('.select-company-logo').click(function (e) {
                    e.preventDefault();
                    if (companyLogoSelector) {
                        companyLogoSelector.open();
                        return;
                    }

                    companyLogoSelector = wp.media.frames.file_frame = wp.media({
                        title: '<?php echo __('Select Logo', 'mec-invoice') ?>',
                        button: {
                            text: 'Select'
                        },
                        multiple: false
                    });

                    companyLogoSelector.on('select', function () {
                        var attachments = companyLogoSelector.state().get('selection').map(
                            function (attachment) {
                                attachment.toJSON();
                                return attachment;
                            }
                        );
                        var i;

                        jQuery('#mec_settings_logo').val(attachments[0].attributes.url);
                    });
                    companyLogoSelector.open();
                });
            });
            jQuery("#mec_settings_form").on('submit', function (event) {
                event.preventDefault();
                jQuery('.CodeMirror-code').click;
                jQuery("#mec_invoice_descriptions-html").click();
                jQuery("#mec_invoice_descriptions-tmce").click();
            });
            // jQuery(document).ready(function($) {
            //     wp.codeEditor.initialize($('#mec-invoice-custom-css'), cm_settings);
            //     setTimeout(() => {
            //         jQuery('.CodeMirror-code').trigger('click');
            //     }, 1000);
            // })
            jQuery(document).ready(function ($) {
                $('.mec-select2').select2();
            });
        </script>
        <?php
    }

    public static function import_form( $class ){

        $main = \MEC::getInstance('app.libraries.main');
        $url = admin_url('admin.php?page=MEC-report');

		?>
         <br><h3><?php esc_html_e('Import Attendees Checkin/Checkout CSV File', 'mec-invoice');?></h3>

        <form id="mec_import_csv_checkin_form" action="<?php echo $main->get_full_url(); ?>" method="POST" enctype="multipart/form-data">
            <div class="mec-form-row">
                <p><?php echo sprintf(__("You can export checkin/checkout list by %s using the checkin menu in the source website. You need a CSV export and then you're able to simply import it using this form in to your target website.", 'mec-invoice'), '<strong><a href="'. $url .'" target="_blank">' . __('Booking Report', 'mec-invoice') . '</a></strong>'); ?></p>
            </div>
            <div class="mec-form-row">
                <input type="file" name="feed" id="feed" title="<?php esc_attr_e('CSV File', 'mec-invoice');?>">
                <input type="hidden" name="mec-ix-action" value="import-start-checkin">
                <?php wp_nonce_field('mec_import_start_upload'); ?>
                <button class="button button-primary mec-button-primary mec-btn-2"><?php _e('Upload & Import', 'mec-invoice');?></button>
            </div>
        </form>
        <?php if (!empty($class->response['invoice-message'])): ?>
        <div class="mec-ix-import-started">
            <?php if ($class->response['success'] == 0): ?>
                <div class="mec-error"><?php echo $class->response['invoice-message']; ?></div>
            <?php else: ?>
                <div class="mec-success"><?php echo $class->response['invoice-message']; ?></div>
            <?php endif;?>
        </div>
        <?php endif;?>
        <?php
    }


    public static function import_action( $response, $action ) {

		if ("{$action}" !== 'import-start-checkin') {
			return $response;
		}

		$feed_file = $_FILES['feed'];
		$main = \MEC::getInstance('app.libraries.main');

		// File is not uploaded
		if (!isset($feed_file['name']) or (isset($feed_file['name']) and trim($feed_file['name']) == '')) {
			return array('success' => 0, 'invoice-message' => __('Please upload a CSV file.', 'mec-invoice'));
		}

		// File Type is not valid
		if (!isset($feed_file['type']) or (isset($feed_file['type']) and !in_array(strtolower($feed_file['type']), array('text/csv', 'application/vnd.ms-excel')))) {
			return array('success' => 0, 'invoice-message' => __('The file type should be CSV.', 'mec-invoice'));
		}

		// Upload the File
		$upload_dir = wp_upload_dir();

		$target_path = $upload_dir['basedir'] . '/' . basename($feed_file['name']);
		$uploaded = move_uploaded_file($feed_file['tmp_name'], $target_path);

		// Error on Upload
		if (!$uploaded) {
			return array('success' => 0, 'invoice-message' => __("An error occurred during the file upload! Please check permissions!", 'mec-invoice'));
		}

        $headers = array();
        $attendees = array();

		if (($h = fopen($target_path, 'r')) !== false) {

			while (($data = fgetcsv($h, 1000, ",")) !== false) {

                if( empty($headers) ){

                    foreach( $data as $k => $header_title ){

                        $headers[ $header_title ] = $k;
                    }

                    continue;
                }

                $invoice_id = $data[ $headers['Invoice ID'] ] ?? false;
                $attendee_key = $data[ $headers['Attendee Key'] ] ?? false;
                $book_datetime = $data[ $headers['Event DateTime'] ] ?? false;

                if ( !is_numeric( $invoice_id ) || !is_numeric( $attendee_key ) || empty( $book_datetime ) ) {

					continue;
				}

                $email = $data[ $headers['E-mail Address'] ] ?? false;
                $checkin_status = $data[ $headers[ 'Checkin Status' ] ] ?? false;
                $checkin_datetime = $data[ $headers[ 'Checkin DateTime' ] ] ?? false;

                $book_timestamp = strtotime( $book_datetime );
                $checkin_timestamp = $checkin_datetime ? strtotime( $checkin_datetime ) : false;

                $checkin_status = $checkin_status === __( 'Checked', 'mec-invoice' ) ? true : false;

                $attendee = array(
                    'invoice_id' => $invoice_id,
                    'book_datetime' => $book_datetime,
                    'attendee_key' => $attendee_key,
                    'email' => $email,
                    'checkin_datetime' => $checkin_datetime,
                    'checkin_status' => $checkin_status
                );

                if( $checkin_status ){

                    Attendee::doCheckIn( $invoice_id, $attendee, $attendee_key, $book_timestamp );

                    if( $checkin_timestamp ){

                        update_post_meta( $invoice_id, 'checkedInTime-' . $book_timestamp . '-' . $attendee_key, $checkin_timestamp );
                    }
                }else{

                    Attendee::doCheckOut( $invoice_id, $attendee, $attendee_key, $book_timestamp );
                }

                $attendees[] = $attendee;

			}

			fclose($h);
		}

        if( empty( $attendees ) ){

            return array('success' => 0, 'invoice-message' => __('Import encountered an error.', 'mec-invoice'));
        }

		// Delete File
		unlink($target_path);

		return array('success' => 1, 'invoice-message' => __('The Attendees Checkin/Checkout list was imported successfully!', 'mec-invoice'));
	}

    /**
     *  Init
     *
     * @since     1.0.0
     */
    public function init()
    {
        if (!class_exists('\MEC_Invoice\Autoloader')) {
            return;
        }
    }
} //Controller
