<?php

namespace MEC_Invoice;

use dawood\phpChrome\Chrome;
use MEC_Invoice\Helper\InvoiceInformation;

// Don't load directly
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
*  MEC Invoice Notifications.
*
*  @author      Webnus <info@webnus.biz>
*  @package     Modern Events Calendar
*  @since       1.0.0
*/
class Notifications
{
    /**
    * Instance of this class.
    *
    * @since   1.0.0
    * @access  public
    * @var     MEC_Invoice
    */
    public static $instance;

    /**
    * The directory of this file
    *
    * @access  public
    * @var     string
    */
    public static $dir;

    /**
    * The instance of MEC Main
    *
    * @access  public
    * @var     string
    */
    public static $main;

    /**
    * Provides access to a single instance of a module using the Singleton pattern.
    *
    * @since   1.0.0
    * @return  object
    */
    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }


    public function __construct()
    {
        if (self::$instance === null) {
            self::$instance = $this;
        }
        self::settingUp();
        self::setHooks();
    }

    /**
    * Set Hooks.
    *
    * @since   1.0.0
    */
    public static function setHooks()
    {
        add_action('mec-invoice-new-invoice-added', [__CLASS__, 'sendInvoiceNotification'], 10, 10);
        add_action('mec_booking_confirmed', [__CLASS__, 'sendInvoiceNotificationWhenBookIsConfirmed'], 10, 10);
        add_filter('mec_booking_notification', [self::$instance, 'disableMECBookingNotifications'], 10, 10);
        add_filter('mec_booking_confirmation', [self::$instance, 'disableMECBookingNotifications'], 10, 10);
        add_filter('mec_booking_invoice_url', [self::$instance, 'setInvoiceUrl'], 10, 2);
    }


    /**
    * Global Variables.
    *
    * @since   1.0.0
    */
    public static function settingUp()
    {
        self::$dir  = MECINVOICEDIR . 'core' . DS . 'notifications';
    }

    /**
    * Set Invoice Url in Books page
    *
    * @since   1.0.0
    */
    public static function setInvoiceUrl($url, $transaction_id)
    {
        $book = \MEC::getInstance('app.libraries.book');
        $booking = $book->get_bookings_by_transaction_id($transaction_id);
        if($booking) {
            $invoiceID = get_post_meta($booking[0]->ID, 'invoiceID',true);

            if($invoiceID){

                $url = self::get_invoice_url( $invoiceID );
            }
        }
        return $url;
    }

    /**
     * Return invoice url
     *
     * @param int $invoice_id
     * @return string
     */
    public static function get_invoice_url( $invoice_id ){

        $Hash = get_post_meta($invoice_id, 'invoiceID', true);
        return get_site_url(null, '?invoiceID=' . $invoice_id . '&makePreview=' . $Hash);
    }

    /**
    * Disable MEC Booking Notifications
    *
    * @since   1.0.0
    */
    public static function disableMECBookingNotifications($status)
    {
        if (\MEC_Invoice\Helper::getOption('mec_booking_notifications', false) == 'on') {
            return false;
        }

        return $status;
    }

    /**
    * Send Notification Too Customer When Booking is Confirmed
    *
    * @since     1.0.0
    */
    public static function sendInvoiceNotificationWhenBookIsConfirmed($book_id, $mode) {
        if(\MEC_Invoice\Helper::getOption('mec_invoice_notification_status', 'off') == 'on') {
            return;
        }

        $invoiceID = get_post_meta($book_id, 'invoiceID',true);
        if(!$invoiceID) {
            return;
        }

        $manual = 'manually' === $mode ? true : false;
        self::sendInvoiceNotification($invoiceID, $manual);
    }

    /**
    * Send Notification Too Customer
    *
    * @since     1.0.0
    */
    public static function sendInvoiceNotification ($invoiceID, $manual = false, $forced_send_to = false) {

        if(\MEC_Invoice\Helper::getOption('mec_invoice_notification_status', 'off') == 'on') {

            return;
        }

        if( 'mec_booking_confirmed' === current_action() ){

            remove_action('mec_booking_confirmed', array(__CLASS__,'sendInvoiceNotificationWhenBookIsConfirmed'), 1);
        }

        $sended = 'yes' === get_post_meta($invoiceID, 'mec_sended_invoice_by_email',true);
        if($sended && !$manual){
            return;
        }
        $book_id = get_post_meta($invoiceID, 'book_id', true);
        $transaction_id = get_post_meta($book_id, 'mec_transaction_id', true);
        $transaction = get_option($transaction_id);

        if(!get_post_meta($book_id, 'mec_confirmed', true)) {
            return;
        }
        $transaction['first_for_all'] = isset($transaction['first_for_all']) ? $transaction['first_for_all'] : \MEC_Invoice\Helper::getOption('booking_first_for_all', 0);

        $primary_email = Base::get_invoice_author_email( $invoiceID );
        $send_invoice_to_each_attendees = \MEC_Invoice\Helper::getOption('mec_invoice_for_each_tickets', false) == 'on' && $transaction['first_for_all'] != '1' ? true : false;
        switch( $forced_send_to ){
            case 'author':

                $send_invoice_to_each_attendees = false;

                break;
            case 'first_attendee':

                $first_attendee = isset( $transaction['tickets'] ) && is_array( $transaction['tickets'] ) ? current( $transaction['tickets'] ) : array();
                $primary_email = isset( $first_attendee['email'] ) ? $first_attendee['email'] : $primary_email;
                $send_invoice_to_each_attendees = false;

                break;
            case 'attendees':

                $send_invoice_to_each_attendees = true;

                break;
        }

        $Email = \MEC_Invoice\Email::instance();
        $from = \MEC_Invoice\Helper::getOption('booking_sender_email', get_option('admin_email'));
        $headers = [
            'MIME-Version' => 'MIME-Version: 1.0',
            'Content-type' => 'text/html; charset=UTF-8',
            'From' => \MEC_Invoice\Helper::getOption('booking_sender_name', get_bloginfo('name')) . ' <' . $from . '>',
            'Reply-To' => $from,
            'X-Mailer' => 'PHP/' . phpversion(),
        ];

        if ( $send_invoice_to_each_attendees ) {
            $InvoiceContent = static::getInvoiceContent( $invoiceID, 'email' , true, $forced_send_to );
            $content = $InvoiceContent->content;
            $emails = $InvoiceContent->emails;
            $post_author = $InvoiceContent->post_author;

            foreach ($emails as $email) {
                $Email->to = $email;
                $Email->headers = $headers;
                $Email->html_content = $content[$email];
                $Email->subject = self::$instance->get_invoice_title($invoiceID);
                $Email->send();
            }
        } else {

            $InvoiceContent = static::getInvoiceContent( $invoiceID, 'email' );
            $content = $InvoiceContent->content;
            $post_author = $InvoiceContent->post_author;

            $Email->headers = $headers;
            $Email->to = $primary_email;

          	if(empty($Email->to)){

                $Email->to = isset($transaction['tickets'][0]['email']) && !empty($transaction['tickets'][0]['email']) ? $transaction['tickets'][0]['email'] : '';
            }

            $Email->html_content = $content;
            $Email->subject = self::$instance->get_invoice_title($invoiceID);
            $Email->send();
        }

        update_post_meta($invoiceID, 'mec_sended_invoice_by_email','yes');
    }

    public function get_invoice_title($invoice_id,$event_id = false){

        if(!$event_id){

            $event_id = get_post_meta($invoice_id, 'event_id', true);
        }

        $event_title = get_the_title($event_id);
        $invoice_title_pattern = \MEC_Invoice\Helper::getOption('mec_invoice_title_pattern', '');
        if(!empty($invoice_title_pattern)){

            $invoice_title = $invoice_title_pattern;
        }else{

            $invoice_title = sprintf(__('Your Invoice #%s in %s', 'mec-invoice'),'{invoice_number}','{event_title}');
        }

        $patterns = array(
            'invoice_number' => get_post_meta($invoice_id,'invoice_number',true),
            'invoice_id' => $invoice_id,
            'event_title' => $event_title,
        );

        foreach($patterns as $pattern => $text){

            $invoice_title = str_replace('{'.$pattern.'}',$text,$invoice_title);
        }

        $invoice_title = apply_filters('mec_invoice_notification_title',$invoice_title,$invoice_title_pattern,$invoice_id,$event_id);

        return html_entity_decode(stripslashes($invoice_title));
    }

    public static function get_ticket_html($InvoiceInformation,$template = 'tickets'){

        $event_id = $InvoiceInformation['EventID'] ?? 0;
        $content = \MEC_Invoice\Email::instance()->getTemplate($template, $event_id);

        foreach ($InvoiceInformation as $name => $value) {
            if(empty(trim($value))) {
                $content = str_replace('{{' . $name . 'Style}}', 'display:none;', $content);
            } else {
                $content = str_replace('{{' . $name . 'Style}}', '', $content);
            }

            $content = str_replace('{{' . $name . '}}', $value, $content);
        }

        return $content;
    }

    /**
    * Return Invoice DateTime Format
    *
    * @since     1.3.0
    */
    public static function get_datetime_format(){

        return Helper::get_invoice_date_format();
    }

    public static function get_attendees( $book_id ){

        $attendees = get_post_meta($book_id, 'mec_attendees', true);

        return is_array( $attendees ) ? $attendees : array();
    }

    /**
    * Return Custom Email Template
    *
    * @since     1.3.0
    */
    public static function get_custom_email_template( $event_id ) {

        $event_notifications_settings = get_post_meta( $event_id, 'mec_notifications', true );
        $e_status = $event_notifications_settings['invoice_notification']['status'] ?? false;

        if( "1" == $e_status ){

            return $event_notifications_settings['invoice_notification'];
        }

        $notifications_settings = \MEC\Settings\Settings::getInstance()->get_options('notifications');
        $g_status = $notifications_settings['invoice_notification']['status'] ?? false;
        if( 1 == $g_status ){

            return $notifications_settings['invoice_notification'];
        }

        return false;
    }

    /**
    * Return Custom Email Template
    *
    * @since     1.3.0
    */
    public static function get_custom_email_template_content( $event_id ) {

        $custom_template = static::get_custom_email_template( $event_id );
        if( 1 == $custom_template['status'] || "1" == $custom_template['status']){

            return $custom_template['content'] ?? '';
        }

        return false;
    }

    /**
    * Get Invoice Content
    *
    * @since     1.0.0
    */
    public static function getInvoiceContent ($invoiceID, $template = 'invoice',$forEachOne = false, $forced_send_to = false ) {

        # Some Data
        $book_id = get_post_meta($invoiceID, 'book_id', true);
        if(!$book_id) {
            echo \MEC_Invoice\Helper::getTemplate('invoice-is-corrupted');
            die();
        }

        $transactionObject = Helper::TransactionObject( $invoiceID );
        $event_id = $transactionObject->get_event_id();
        $emails = $transactionObject->get_emails();

        if ( !$event_id ) {
            return false;
        }

        $custom_template_email = static::get_custom_email_template( $event_id );
        if( 'email' == $template && $custom_template_email ) {

            $template = 'custom-invoice';
        }elseif( 'email' == $template && !$custom_template_email ){

            $template = 'invoice';
        }

        $rendering_mode = \MEC_Invoice\Helper::getOption('mec_invoice_rendering_mode', 'full');
        if ($rendering_mode == 'ticket') {
            $template = $template == 'invoice' ? 'invoice-ticket' : $template;
        } elseif ($rendering_mode == 'standard') {
            $template = $template == 'invoice' ? 'standard-invoice' : $template;
        } elseif ($rendering_mode == 'modern-ticket-1') {
            $template = $template == 'invoice' ? 'modern-ticket-1' : $template;
        } elseif ($rendering_mode == 'modern-ticket-2') {
            $template = $template == 'invoice' ? 'modern-ticket-2' : $template;
        } elseif ($rendering_mode == 'modern-ticket-3') {
            $template = $template == 'invoice' ? 'modern-ticket-3' : $template;
        } elseif ($rendering_mode == 'modern-ticket-4') {
            $template = $template == 'invoice' ? 'modern-ticket-4' : $template;
        }

        # Invoice - Event
        $event = get_post($event_id);
        if(!$event) {
            wp_send_json_error( __('Event is missed!', 'mec-invoice') );
        }

        $post_author = get_post_field('post_author', $book_id);
        $attendee = false;
        if ($attendee = get_post_meta($invoiceID, 'attendee', true) && isset($_GET['attendee'])) {
            $_GET['attendee'] = $attendee;
        }

        $transaction['first_for_all'] = !$forced_send_to ? ( $transaction['first_for_all'] ?? false ) : false;

        // $transaction['date'] = \MEC_Invoice\Helper::get_date_label($transaction['date'], $event_id);

        $attendees = get_post_meta($book_id, 'mec_attendees', true);

        if($forEachOne) {
            $contents = [];
            foreach ($attendees as $at) {

                $InvoiceInformation = InvoiceInformation::get_invoice_information( $invoiceID, $at ,true);
                $contents[ $at['email'] ] = self::get_ticket_html( $InvoiceInformation,$template );
            }

            return (object) [
                'emails' => $emails,
                'content' => $contents,
                'post_author' => $post_author
            ];
        } else {
            $InvoiceInformation = InvoiceInformation::get_invoice_information( $invoiceID ,$attendees[0],true);

            $Email = \MEC_Invoice\Email::instance();
            $content = $Email->getTemplate( $template, $event_id );
            foreach ($InvoiceInformation as $name => $value) {
                if (empty(trim($value))) {
                    $content = str_replace('{{' . $name . 'Style}}', 'display:none;', $content);
                } else {
                    $content = str_replace('{{' . $name . 'Style}}', '', $content);
                }
                $content = str_replace('{{' . $name . '}}', $value, $content);
            }

            return (object) [
                'emails' => $emails,
                'content' => $content,
                'post_author' => $post_author
            ];
        }

    }

    /**
    *Apply Currency
    *
    *@since     1.0.0
    */
    public static function applyCurrency ($price) {
        if(!static::$main) {
            static::$main = \MEC::getInstance('app.libraries.main');
        }
        return static::$main->render_price($price);
    }



} // Notification
Notifications::instance();
