<?php

/**
 * Plugin Name: MEC Ticket and Invoice
 * Plugin URI: http://webnus.net/modern-events-calendar/
 * Description: You can easily sell your events on your website using the Ticket & Invoice add-on. this add-on allows you to manage your attendees' check-in, send out tickets and invoices in email format. Using this add-on, you can design your own invoice & tickets. The QR codes on the tickets enable you to check-in attendees using the QR code scanner device. Using this tool, you manage your payments on your own website. No need to refer your customers to any other third party websites.
 * Author: Webnus
 * Version: 3.4.0
 * Text Domain: mec-invoice
 * Domain Path: /languages
 * Author URI: http://webnus.net
 **/

namespace MEC_Invoice;

use MEC_Invoice\Helper\Invoice;

// Don't load directly
if (!defined('ABSPATH')) {
	header('Status: 403 Forbidden');
	header('HTTP/1.1 403 Forbidden');
	exit;
}
/**
 **   Base.
 **
 **   @author     Webnus <info@webnus.biz>
 **   @package     Modern Events Calendar
 **   @since     1.0.0
 */
class Base
{

	/**
	 **  Instance of this class.
	 **
	 **  @since   1.0.0
	 **  @access  public
	 **  @var     MEC_Invoice
	 */
	public static $instance;

	/**
	 **  Provides access to a single instance of a module using the Singleton pattern.
	 **
	 **  @since   1.0.0
	 **  @return  object
	 */
	public static function instance()
	{
		if (self::$instance === null) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function __construct()
	{
		if (defined('MECINVOICEVERSION')) {
			return;
		}
		self::settingUp();
		self::preLoad();
		self::setHooks($this);

		do_action('MEC_Invoice_init');
	}

	public function load_languages()
	{
		$locale = apply_filters('plugin_locale', get_locale(), 'mec-invoice');

		// WordPress language directory /wp-content/languages/mec-en_US.mo
		$language_filepath = MECINVOICEDIR . 'languages' . DIRECTORY_SEPARATOR . 'mec-invoice-' . $locale . '.mo';
		// If language file exists on WordPress language directory use it
		if (file_exists($language_filepath)) {
			load_textdomain('mec-invoice', $language_filepath);
		} else {
			load_plugin_textdomain('mec-invoice', false, dirname(plugin_basename(__FILE__)) . DIRECTORY_SEPARATOR . 'languages' . DIRECTORY_SEPARATOR);
		}
	}

	/**
	 **  Global Variables.
	 **
	 **  @since   1.0.0
	 */
	public static function settingUp()
	{
		define('MECINVOICEVERSION', '3.4.0');
		define('MECINVOICEDIR', plugin_dir_path(__FILE__));
		define('MECINVOICEURL', plugin_dir_url(__FILE__));
		define('MECINVOICEASSETS', MECINVOICEURL . 'assets/');
		define('MECINVOICENAME', 'Ticket and Invoice');
		define('MECINVOICESLUG', 'mec-invoice');
		define('MECINVOICEOPTIONS', 'mec_invoice_options');
		define('MECINVOICETEXTDOMAIN', 'mec-invoice');
		define('MECINVOICEMAINFILEPATH', __FILE__);
		define('MECINVOICEPABSPATH', dirname(__FILE__));
		if (!defined('DS')) {
			define('DS', DIRECTORY_SEPARATOR);
		}
	}

	/**
	 **  Set Hooks
	 **
	 **  @since     1.0.0
	 */
	public static function setHooks($This)
	{
		add_action('init', [$This, 'load_languages']);
		add_action('woocommerce_thankyou', array($This, 'display_invoice_info'), 10, 1);
		add_action('wp_head', array(__CLASS__, 'print_inline_style_and_scripts'));
		add_action('mec_save_options', array(__CLASS__, 'rest_invoice_number'));
		register_activation_hook(__FILE__, array(__CLASS__, 'install'));
		self::install();

		add_action('admin_notices', array(__CLASS__, 'upgrade_notice'));
		add_action('wp_ajax_mec-upgrade-invoice-prices-in-db', array(__CLASS__, 'upgrade_invoice_prices_in_db_by_ajax'));
	}

	public static function print_inline_style_and_scripts()
	{

?>
		<style>
			.mec-success .mec-invoice-download {
				background-color: #2CA84C;
				color: white;
				padding: 8px 25px;
				border-radius: 5px;
				line-height: 20px;
				display: block;
				text-align: center;
				margin: 20px 0 0 0;
			}
		</style>
	<?php
	}

	public function display_invoice_info($order_id)
	{

		$type = get_post_meta($order_id, 'mec_order_type', true);
		if ('mec_ticket' !== $type) {

			return;
		}

		$book_ids = get_posts(array(
			'post_type' => 'mec-books',
			'posts_per_page' => -1,
			'meta_query' => array(
				'' => array(
					'key' => 'mec_order_id',
					'value' => $order_id,
					'compare' => '=',
				),
			),
			'fields' => 'ids',
			'meta_key' => 'invoiceID',
		));

		$invoices = array();
		foreach ($book_ids as $book_id) {

			$invoice_id = get_post_meta($book_id, 'invoiceID', true);
			$invoices[$invoice_id] = array(
				'url' => Notifications::get_invoice_url($invoice_id),
				'number' => get_post_meta($invoice_id, 'invoice_number', true),
			);
		}

		echo '<h2>' . esc_html__('Invoice', 'mec-invoice') . '</h2>';
		echo '<table class="woocommerce-table woocommerce-table--order-details shop_table order_details">';
		foreach ($invoices as $invoice_id => $invoice_data) {

			echo '<tr><td><a href="' . $invoice_data['url'] . '">' . __('Download Invoice', 'mec-invoice') . ' #' . $invoice_data['number'] . '</a></td></tr>';
		}
		echo '</table>';
	}

	/**
	 **  MEC Version Admin Notice
	 **
	 **  @since     1.0.0
	 */
	public function MECVersionAdminNotice($type = false)
	{
		$screen = get_current_screen();
		if (isset($screen->parent_file) && 'plugins.php' === $screen->parent_file && 'update' === $screen->id) {
			return;
		}

		$plugin = 'modern-events-calendar/mec.php';

		if (!current_user_can('install_plugins')) {
			return;
		}

		$install_url = wp_nonce_url(self_admin_url('update.php?action=install-plugin&plugin=modern-events-calendar'), 'install-plugin_' . $plugin);
		$message     = '<p>' . __('MEC Invoice is not working because you need to install latest version of Modern Events Calendar plugin', 'mec-invoice') . '</p>';
		$message    .= esc_html__('Minimum version required') . ': <b> 4.2.3 </b>';
		$message    .= '<p>' . sprintf('<a href="%s" class="button-primary">%s</a>', $install_url, __('Update Modern Events Calendar Now', 'mec-invoice')) . '</p>';
	?>
		<div class="notice notice-error is-dismissible">
			<p><?php echo $message; ?></p>
		</div>
	<?php

	}


	/**
	 * Plugin Requirements Check
	 *
	 * @since 1.0.0
	 */
	public static function checkPlugins()
	{
		if (!function_exists('is_plugin_active')) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		if (!self::$instance) {
			self::$instance = static::instance();
		}

		if (!is_plugin_active('modern-events-calendar/mec.php') && !class_exists('\MEC')) {
			add_action('admin_notices', [self::$instance, 'MECNotice']);
			return false;
		} else {
			if (!defined('MEC_VERSION')) {
				$plugin_data = get_plugin_data(realpath(WP_PLUGIN_DIR . '/modern-events-calendar/mec.php'));
				$version     = str_replace('.', '', $plugin_data['Version']);
			} else {
				$version = str_replace('.', '', MEC_VERSION);
			}
			if ($version <= 421) {
				add_action('admin_notices', [self::$instance, 'MECVersionAdminNotice'], 'version');
				return false;
			}
		}

		return true;
	}

	/**
	 ** Send Admin Notice (MEC)
	 **
	 ** @since 1.0.0
	 **/
	public function MECNotice($type = false)
	{
		$screen = get_current_screen();
		if (isset($screen->parent_file) && 'plugins.php' === $screen->parent_file && 'update' === $screen->id) {
			return;
		}

		$plugin = 'modern-events-calendar/mec.php';
		if ($this->is_mec_installed()) {
			if (!current_user_can('activate_plugins')) {
				return;
			}
			$activation_url = wp_nonce_url('plugins.php?action=activate&amp;plugin=' . $plugin . '&amp;plugin_status=all&amp;paged=1&amp;s', 'activate-plugin_' . $plugin);
			$message        = '<p>' . __('MEC Invoice is not working because you need to activate the Modern Events Calendar plugin.', 'mec-invoice') . '</p>';
			$message       .= '<p>' . sprintf('<a href="%s" class="button-primary">%s</a>', $activation_url, __('Activate Modern Events Calendar Now', 'mec-invoice')) . '</p>';
		} else {
			if (!current_user_can('install_plugins')) {
				return;
			}
			$install_url = 'https://webnus.net/modern-events-calendar/';
			$message     = '<p>' . __('MEC Invoice is not working because you need to install the Modern Events Calendar plugin', 'mec-invoice') . '</p>';
			$message    .= '<p>' . sprintf('<a href="%s" class="button-primary">%s</a>', $install_url, __('Install Modern Events Calendar Now', 'mec-invoice')) . '</p>';
		}
	?>
		<div class="notice notice-error is-dismissible">
			<p><?php echo $message; ?></p>
		</div>
		<?php

	}

	/**
	 * Is MEC installed?
	 *
	 * @since     1.0.0
	 */
	public function is_mec_installed()
	{
		$file_path         = 'modern-events-calendar/mec.php';
		$installed_plugins = get_plugins();
		return isset($installed_plugins[$file_path]);
	}

	/**
	 **  PreLoad
	 **
	 **  @since     1.0.0
	 */
	public static function preLoad()
	{
		if (static::checkPlugins()) {
			include_once MECINVOICEDIR . DS . 'core' . DS . 'autoloader.php';
		}
	}

	/**
	 * Return invoice author email
	 *
	 * @param int $invoice_id
	 *
	 * @return string
	 */
	public static function get_invoice_author_email($invoice_id)
	{

		$book_id = get_post_meta($invoice_id, 'book_id', true);

		if (!$book_id) {

			return '';
		}

		$mec_user = new \MEC_user();
		$user = $mec_user->booking($book_id);

		return isset($user->user_email) ? $user->user_email : '';
	}

	public static function install()
	{

		$version = get_option('mec_invoice_version', '1.0.0');
		if (version_compare(MECINVOICEVERSION, $version, '>')) {

			$role = get_role('administrator');
			if ($role) {

				$role->add_cap('mec_invoice_cap', true);
				$role->add_cap('mec_invoice_cap_others', true);
			}
		}

		$upgrade_short_links = get_option('mec_invoice_upgrade_short_links', 'no');
		if ('yes' !== $upgrade_short_links) {

			Helper::upgrade_short_links_db();

			update_option('mec_invoice_upgrade_short_links', 'yes');
		}

		update_option('mec_invoice_version', MECINVOICEVERSION);
	}

	public static function rest_invoice_number($settings)
	{

		$saved_settings = \MEC\Settings\Settings::getInstance()->get_settings();

		$saved_start_point = $saved_settings['mec_invoice_number_start_point'] ?? 0;
		$start_point = $settings['settings']['mec_invoice_number_start_point'] ?? 0;

		if ($saved_start_point != $start_point) {

			Invoice::reset_invoice_number();
		}
	}

	public static function upgrade_notice($type = false)
	{

		if (!is_plugin_active('modern-events-calendar/mec.php') && !class_exists('\MEC')) {
			return;
		}

		$booking_module_status = (bool)\MEC\Settings\Settings::getInstance()->get_settings('booking_status');
		$upgrade_db = get_option('mec_upgrade_invoice_prices_in_db', 'no');
		if ('yes' !== $upgrade_db && $booking_module_status) {

			if (!current_user_can('activate_plugins')) {
				return;
			}

			$upgrade_url = admin_url('?mec_upgrade_db=true');
			$message        = '<p>'
				. __('Your invoice database needs updating. To do that, click the button below and wait until the operation is over. Do not refresh the page until the end.', 'mec-invoice')
				. '<br><b>' . __('Note: if you have many invoices, the operation might take longer, please be patient.', 'mec-invoice') . '</b>'
				. '</p>';
			$message       .= '<p>' . sprintf('<a href="%s" class="button-primary mec-upgrade-invoice-prices-in-db">%s</a>', $upgrade_url, __('Upgrade Database Now', 'mec-invoice')) . '</p>';

		?>
			<script>
				jQuery(document).ready(function($) {
					$('.mec-upgrade-invoice-prices-in-db').on('click', function(e) {
						e.preventDefault();

						var $btn = $(this);
						$btn.html("<?php echo __('Updating Database...', 'mec-invoice'); ?>");
						$.post(
							"<?php echo admin_url('admin-ajax.php'); ?>", {
								action: 'mec-upgrade-invoice-prices-in-db',
								nonce: "<?php echo wp_create_nonce('mec-upgrade-invoice-prices-in-db') ?>",
							},
							function(r) {

								if (false == r.done) {

									$('.mec-upgrade-invoice-prices-in-db').trigger('click');
								} else {

									$btn.html("<?php echo __('Database has been upgraded.', 'mec-invoice'); ?>");
								}
							}
						)
					});
				});
			</script>
			<div class="notice notice-error is-dismissible">
				<p><?php echo $message; ?></p>
			</div>
<?php
		}
	}

	public static function upgrade_invoice_prices_in_db_by_ajax()
	{

		if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'mec-upgrade-invoice-prices-in-db')) {

			return;
		}

		$upgrade_db = get_option('mec_upgrade_invoice_prices_in_db', 'no');
		if ('yes' !== $upgrade_db) {

			if (!Helper::upgrade_invoices_total_price()) {

				update_option('mec_upgrade_invoice_prices_in_db', 'yes');
			}

			wp_send_json(array(
				'done' => false,
			));
		} else {

			wp_send_json(array(
				'done' => true,
			));
		}
	}
} //Base

Base::instance();
