<?php

namespace MEC_Waiting_List\Core\Admin;

use MEC_Waiting_List\Core\User;

// don't load directly.
if (!defined('ABSPATH')) {
	header('Status: 403 Forbidden');
	header('HTTP/1.1 403 Forbidden');
	exit;
}

/**
 * Admin backend wp class.
 *
 * @author      author
 * @package     package
 * @since       1.0.0
 */
class MecAdmin {

	/**
	 * Instance of this class.
	 *
	 * @since   1.0.0
	 * @access  public
	 * @var     MEC_Waiting_List
	 */
	public static $instance;

	/**
	 * The directory of the file.
	 *
	 * @access  public
	 * @var     string
	 */
	public static $dir;

	/**
	 * The Args
	 *
	 * @access  public
	 * @var     array
	 */
	public static $args;

	public static $PT;

	/**
	 * Provides access to a single instance of a module using the singleton pattern.
	 *
	 * @since   1.0.0
	 * @return  object
	 */
	public static function instance() {
		if (self::$instance === null) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function __construct() {

		self::$PT = 'mec-waiting';
		self::init($this);
	}

	private static function init($class) {

		add_action( 'admin_enqueue_scripts', array($class,'load_assets'), 10, 1 );
		add_action( 'wp_enqueue_scripts', array($class,'fronend_assets'), 10, 1 );
        add_action( 'wp_enqueue_scripts', array( __CLASS__, 'register_style_script' ), 10, 1 );

		add_filter('mec-settings-item-booking', array($class, 'booking_menu_item'), 10, 2);
		add_action('mec_reg_menu_start', array($class, 'booking_menu_content'), 10, 2);

		add_filter('mec_save_options_final', array($class, 'save_options'), 10, 1);

		add_action('add_event_booking_sections_left_menu', array($class, 'add_waiting_list_left_menu'), 10, 1);
		add_action('mec_events_meta_box_regform_end', array($class, 'add_waiting_list_left_menu_content'), 10, 1);
		add_action('mec_after_publish_admin_event', array($class, 'save_event_waiting_fields'), 10, 2);

		add_filter('mec-settings-item-notifications', array($class, 'notification_menu_waiting'), 10, 2);
		add_action('mec_notification_menu_start', array($class, 'notification_menu_content'), 10, 2);
		add_action('mec_notification_menu_js', array($class, 'notification_menu_js'), 10, 1);

		add_action('mec_import_item', array($class, 'import_form'), 10, 1);
		add_filter('mec_import_item_action', array($class, 'import_action'), 10, 2);

		add_action( 'custom_field_ticket', array($class,'waiting_ticket_checkbox'), 10, 2 );
		add_action( 'custom_field_dynamic_ticket', array($class,'waiting_ticket_checkbox_dynamic'), 10, 1);

		add_filter( 'mec_booking_notification_reg_fields', array($class,'notification_reg_fields'), 10, 2 );

		add_action('current_screen', array($class, 'waiting_badge'));

	}


	public static function notification_reg_fields($reg_fields,$field_id){

		if(!isset($reg_fields[$field_id])){
			$reg_fields[$field_id] = array('type'=>null,'label'=>null);
		}

		return $reg_fields;
	}

	/**
     * Show Booking Badge.
     * @param object $screen
     * @return void
     */
    public function waiting_badge($screen)
    {
        $user_id = get_current_user_id();
        $user_last_view_date = get_user_meta($user_id, 'user_last_view_date', true);
        $count = 0;

        if(!trim($user_last_view_date))
        {
            update_user_meta($user_id, 'user_last_view_date', date('YmdHis', current_time('timestamp', 0)));
            return;
        }

        $args = array(
            'post_type' => self::$PT,
            'post_status' => 'any',
            'meta_query' => array(
                array(
                    'key' => 'mec_waiting_time',
                    'value' => $user_last_view_date,
                    'compare' => '>=',
                ),
            ),
        );

        $query = new \WP_Query($args);
        if($query->have_posts())
        {
            while($query->have_posts())
            {
                $query->the_post();
                $count += 1;
            }
        }

        if($count != 0)
        {
            if(isset($screen->id) and $screen->id == 'edit-mec-waiting')
            {
                update_user_meta($user_id, 'user_last_view_date', date('YmdHis', current_time('timestamp', 0)));
                return;
            }

            // Append waiting Badge To waiting Menu.
            global $menu;

            $badge = ' <span class="update-plugins count-%%count%%"><span class="plugin-count">%%count%%</span></span>';
            $menu_item = wp_list_filter($menu, array(2 =>'edit.php?post_type='.self::$PT));
            if(is_array($menu_item) and count($menu_item))
            {
                $menu[key($menu_item)][0] .= str_replace('%%count%%', esc_attr($count), $badge);
            }
        }
    }


	public static function waiting_ticket_checkbox($ticket, $key){

		$enable = isset($ticket['waiting_disable'])?$ticket['waiting_disable']==1:false;
		?>
		<div class="mec-form-row">
            <label class="mec-col-2" for="mec_tickets_waiting_disable_<?php echo $key; ?>" id="mec_bookings_waiting_disable_label<?php echo $key; ?>">
            	 <input type="hidden" name="mec[tickets][<?php echo $key; ?>][waiting_disable]" value="0"/>
                <input id="mec_tickets_waiting_disable_<?php echo $key; ?>"
                type="checkbox"
                value="1"
                name="mec[tickets][<?php echo $key; ?>][waiting_disable]"
                <?php if($enable) echo 'checked="checked"' ?> >
                <?php _e( 'Disable Waiting List', 'mec-waiting-list' ); ?>
            </label>
        </div>

		<?php
	}

	public static function waiting_ticket_checkbox_dynamic(){
		?>
		<div class="mec-form-row">
            <label class="mec-col-2" for="mec_tickets_waiting_disable_:i:"
                   id="mec_bookings_waiting_disable_label_::i:">
                <input type="hidden" name="mec[tickets][:i:][waiting_disable]" value="0"/>
                <input id="mec_tickets_waiting_disable_:i:" type="checkbox" value="1"
                       name="mec[tickets][:i:][waiting_disable]"/>
                <?php _e('Disable Waiting List', 'mec-waiting-list'); ?>
            </label>
        </div>
		<?php
	}

	public static function import_form($class) {
		$main = \MEC::getInstance('app.libraries.main');
		?>
         <br><h3><?php _e('Import Waiting CSV File', 'mec-waiting-list');?></h3>

        <form id="mec_import_csv_waiting_form" action="<?php echo $main->get_full_url(); ?>" method="POST" enctype="multipart/form-data">
            <div class="mec-form-row">
                <p><?php echo sprintf(__("You can export waiting list by %s using the waiting menu in the source website. You need a CSV export and then you're able to simply import it using this form in to your target website.", 'mec-waiting-list'), '<strong>' . __('Modern Events Calendar', 'mec-waiting-list') . '</strong>'); ?></p>
                <p style="color: red;"><?php echo __("Please note that you should create (or import) events and tickets before importing the waiting list, otherwise bookings won't import due to lack of data.", 'mec-waiting-list'); ?></p>
            </div>
            <div class="mec-form-row">
                <input type="file" name="feed" id="feed" title="<?php esc_attr_e('CSV File', 'mec-waiting-list');?>">
                <input type="hidden" name="mec-ix-action" value="import-start-waiting">
                <?php wp_nonce_field('mec_import_start_upload'); ?>
                <button class="button button-primary mec-button-primary mec-btn-2"><?php _e('Upload & Import', 'mec-waiting-list');?></button>
            </div>
        </form>
        <?php if (!empty($class->response['waiting-message'])): ?>
        <div class="mec-ix-import-started">
            <?php if ($class->response['success'] == 0): ?>
                <div class="mec-error"><?php echo $class->response['waiting-message']; ?></div>
            <?php else: ?>
                <div class="mec-success"><?php echo $class->response['waiting-message']; ?></div>
            <?php endif;?>
        </div>
        <?php endif;?>
        <?php
}

	public static function import_action( $response, $action ) {

		if ("{$action}" !== 'import-start-waiting') {
			return $response;
		}

		$feed_file = $_FILES['feed'];
		$main = \MEC::getInstance('app.libraries.main');

		// File is not uploaded
		if (!isset($feed_file['name']) or (isset($feed_file['name']) and trim($feed_file['name']) == '')) {
			return array('success' => 0, 'waiting-message' => __('Please upload a CSV file.', 'mec-waiting-list'));
		}

		// File Type is not valid
		if (!isset($feed_file['type']) or (isset($feed_file['type']) and !in_array(strtolower($feed_file['type']), array('text/csv', 'application/vnd.ms-excel')))) {
			return array('success' => 0, 'waiting-message' => __('The file type should be CSV.', 'mec-waiting-list'));
		}

		// Upload the File
		$upload_dir = wp_upload_dir();

		$target_path = $upload_dir['basedir'] . '/' . basename($feed_file['name']);
		$uploaded = move_uploaded_file($feed_file['tmp_name'], $target_path);

		// Error on Upload
		if (!$uploaded) {
			return array('success' => 0, 'waiting-message' => __("An error occurred during the file upload! Please check permissions!", 'mec-waiting-list'));
		}

		if (($h = fopen($target_path, 'r')) !== false) {
			// MEC Libraries
			$gateway = new \MEC_gateway();
			$book = \MEC::getInstance('app.libraries.book');

			$bookings = array();
			while (($data = fgetcsv($h, 1000, ",")) !== false) {
				$waiting_id = $data[0];
				if (!is_numeric($waiting_id)) {
					continue;
				}

				$event_title = $data[1];
				$event_id = post_exists($event_title, '', '', $main->get_main_post_type());

				// Event not Found
				if (!$event_id) {
					continue;
				}

				$tickets = get_post_meta($event_id, 'mec_tickets', true);
				if (!is_array($tickets)) {
					$tickets = array();
				}

				$ticket_id = 0;
				$ticket_name = $data[5];

				foreach ($tickets as $tid => $ticket) {
					if (strtolower($ticket['name']) == strtolower($ticket_name)) {
						$ticket_id = $tid;
						break;
					}
				}

				// Ticket ID not found!
				if (!$ticket_id) {
					continue;
				}

				$start_datetime = $data[2];
				$end_datetime = $data[3];
				$name = $data[8];
				$email = $data[9];

				$verified_label = "{$data[10]}";
				if ($verified_label == __('Verified', 'mec-waiting-list')) {
					$verified = 1;
				} elseif ($verified_label == __('Canceled', 'mec-waiting-list')) {
					$verified = -1;
				} else {
					$verified = 0;
				}

				$confirmed_label = "{$data[11]}";
				if ($confirmed_label == __('Confirmed', 'mec-waiting-list')) {
					$confirmed = 1;
				} elseif ($confirmed_label == __('Rejected', 'mec-waiting-list')) {
					$confirmed = -1;
				} else {
					$confirmed = 0;
				}

				$ticket_variations = explode(',', $data[10]);
				$variations = $main->ticket_variations($event_id);

				$v = array();
				foreach ($variations as $vid => $variation) {
					foreach ($ticket_variations as $ticket_variation) {
						$variation_ex = explode(':', $ticket_variation);
						if (!isset($variation_ex[1])) {
							continue;
						}

						$variation_name = $variation_ex[0];
						$variation_count = trim($variation_ex[1], '() ');

						if (strtolower($variation['title']) == strtolower($variation_name)) {
							$v[$vid] = $variation_count;
						}
					}
				}

				if (!isset($bookings[$waiting_id])) {
					$bookings[$waiting_id] = array('tickets' => array());
				}

				$bookings[$waiting_id]['tickets'][] = array(
					'email' => $email,
					'name' => $name,
					'variations' => $v,
					'id' => $ticket_id,
					'count' => 1,
				);

				$bookings[$waiting_id]['ticket_id'] = $ticket_id;

				if (!isset($bookings[$waiting_id]['date'])) {
					$bookings[$waiting_id]['date'] = strtotime($start_datetime) . ':' . strtotime($end_datetime);
				}

				if (!isset($bookings[$waiting_id]['event_id'])) {
					$bookings[$waiting_id]['event_id'] = $event_id;
				}

				if (!isset($bookings[$waiting_id]['confirmed'])) {
					$bookings[$waiting_id]['confirmed'] = $confirmed;
				}

				if (!isset($bookings[$waiting_id]['verified'])) {
					$bookings[$waiting_id]['verified'] = $verified;
				}

			}

			fclose($h);

			foreach ($bookings as $waiting_id => $transaction) {
				$event_id = $transaction['event_id'];
				$tickets = $transaction['tickets'];

				$event_tickets = get_post_meta($event_id, 'mec_tickets', true);
				if (!is_array($event_tickets)) {
					$event_tickets = array();
				}

				$raw_tickets = array();
				$raw_variations = array();

				foreach ($tickets as $ticket) {
					if (!isset($raw_tickets[$ticket['id']])) {
						$raw_tickets[$ticket['id']] = 1;
					} else {
						$raw_tickets[$ticket['id']] += 1;
					}

					if (isset($ticket['variations']) and is_array($ticket['variations']) and count($ticket['variations'])) {
						foreach ($ticket['variations'] as $variation_id => $variation_count) {
							if (!trim($variation_count)) {
								continue;
							}

							if (!isset($raw_variations[$variation_id])) {
								$raw_variations[$variation_id] = $variation_count;
							} else {
								$raw_variations[$variation_id] += $variation_count;
							}

						}
					}
				}

				// Calculate price of bookings
				$price_details = $book->get_price_details($raw_tickets, $event_id, $event_tickets, $raw_variations);

				$transaction['price_details'] = $price_details;
				$transaction['total'] = $price_details['total'];
				$transaction['discount'] = 0;
				$transaction['price'] = $price_details['total'];
				$transaction['coupon'] = NULL;

				$attendees = isset($transaction['tickets']) ? $transaction['tickets'] : array();

				$attention_date = isset($transaction['date']) ? $transaction['date'] : '';
				$attention_times = explode(':', $attention_date);
				$date = date('Y-m-d H:i:s', trim($attention_times[0]));

				$main_attendee = isset($attendees[0]) ? $attendees[0] : array();
				$name = isset($main_attendee['name']) ? $main_attendee['name'] : '';

				$ticket_ids = '';
				$attendees_info = array();

				foreach ($attendees as $i => $attendee) {
					if (!is_numeric($i)) {
						continue;
					}

					$ticket_ids .= $attendee['id'] . ',';
					if (!array_key_exists($attendee['email'], $attendees_info)) {
						$attendees_info[$attendee['email']] = array('count' => $attendee['count']);
					} else {
						$attendees_info[$attendee['email']]['count'] = ($attendees_info[$attendee['email']]['count'] + $attendee['count']);
					}

				}

				$ticket_ids = ',' . trim($ticket_ids, ', ') . ',';
                $user_id = $gateway->register_user(array_merge($main_attendee, array(
                    'event_id' => $event_id,
                    'source' => 'waiting'
                )));
				$book_subject = $name . ' - ' . get_userdata($user_id)->user_email;

				$values = array(
					'post_author' => $user_id,
					'post_type' => 'mec-waiting',
					'post_title' => $book_subject,
					'post_date' => $date,
					'post_status' => 'publish',
				);

				$waite_id = wp_insert_post($values);
                User::instance()->assign( $waite_id, $user_id );
				update_post_meta($waite_id, 'mec_gateway', 'MEC_gateway');
				update_post_meta($waite_id, 'mec_gateway_label', $gateway->label());
				update_post_meta($waite_id, 'mec_confirmed', $transaction['confirmed']);
				update_post_meta($waite_id, 'mec_verified', $transaction['verified']);
				update_post_meta($waite_id, 'mec_date', $attention_date);
				update_post_meta($waite_id, 'mec_price', $transaction['total']);
                update_post_meta($waite_id, 'attendees_info', $attendees_info);
                update_post_meta($waite_id, 'mec_attendees', $attendees);
				update_post_meta($waite_id, 'mec_event_id', $event_id);
				update_post_meta($waite_id, 'mec_ticket_id', $transaction['ticket_id']);
				update_post_meta($waite_id, 'mec_waiting_time', $date);
				wp_publish_post($waite_id);
			}
		}

		// Delete File
		unlink($target_path);

		return array('success' => 1, 'waiting-message' => __('The Waiting list was imported successfully!', 'mec-waiting-list'));
	}

    public static function register_style_script(){

        wp_register_style('waiting-list-backend-css', MECWAITINGLISTASSETS . '/css/backend.css', array(), MECWAITINGLISTVERSION);
		wp_register_script('waiting-list-backend-js', MECWAITINGLISTASSETS . '/js/backend.js', array('jquery'), MECWAITINGLISTVERSION, true);
    }

	public static function load_assets() {

        static::register_style_script();

		wp_enqueue_style('waiting-list-backend-css');
		wp_enqueue_script('waiting-list-backend-js');

	}

	public static function fronend_assets() {

		wp_register_style('waiting-list-frontend-styles', MECWAITINGLISTASSETS . '/frontend/css/waiting-list-styles.css', array(), MECWAITINGLISTVERSION);
		wp_enqueue_style('waiting-list-frontend-styles');

	}

	public static function booking_menu_item($arr, $menu) {

		$arr[__('Waiting List', 'mec-waiting-list')] = 'waiting_list';
		return $arr;

	}

	public static function notification_menu_waiting($arr, $menu) {
		$arr[__('Waiting Verification', 'mec-waiting-list')] = 'waiting_verification';
		$arr[__('Waiting Confirmation', 'mec-waiting-list')] = 'waiting_confirmation';

		return $arr;
	}

	public static function notification_menu_content($main, $notifications) {
		$enable_verification = isset($notifications['waiting_verification'])&& isset($notifications['waiting_verification']['enable']) && $notifications['waiting_verification']['enable']=='1'?true:false;
		$enable_confirmation = isset($notifications['waiting_confirmation'])&& isset($notifications['waiting_confirmation']['enable']) && $notifications['waiting_confirmation']['enable']=='1'?true:false;
		?>
        <div id="waiting_verification" class="mec-options-fields">

            <h4 class="mec-form-subtitle"><?php _e('Waiting Verification', 'mec-waiting-list');?></h4>

            <div class="mec-form-row">
                <label>
                    <input type="hidden" name="mec[notifications][waiting_verification][enable]" value="0" />
                    <input onchange="jQuery('#mec_notification_waiting_verification_toggle').toggle();" value="1" type="checkbox" name="mec[notifications][waiting_verification][enable]"
                     <?php if($enable_verification) echo 'checked="checked"'; ?> />
                     <?php _e('Enable Waiting Verification', 'mec'); ?>
                </label>
            </div>

            <div <?php if(!$enable_verification) echo 'style="display:none;"'; ?> id="mec_notification_waiting_verification_toggle" >

            <p class="description"><?php _e('Sends an email to the attendee to verifying their waiting/email.', 'mec-waiting-list');?></p>

            <div class="mec-form-row">
                <label for="mec_notifications_waiting_verification_subject"><?php _e('Email Subject', 'mec-waiting-list');?></label>
                <input type="text" name="mec[notifications][waiting_verification][subject]" id="mec_notifications_email_verification_subject" value="<?php echo (isset($notifications['waiting_verification']['subject']) ? stripslashes($notifications['waiting_verification']['subject']) : ''); ?>" />
            </div>

            <!-- Start Receiver Users -->
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_verification_receiver_users"><?php _e('Receiver Users', 'mec-waiting-list');?></label>
                <?php
$users = isset($notifications['waiting_verification']['receiver_users']) ? $notifications['waiting_verification']['receiver_users'] : array();
		echo $main->get_users_dropdown($users, 'waiting_verification');
		?>
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Receiver Users', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Select users to send a copy of the email to!', 'mec-waiting-list');?></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
            <!-- End Receiver Users -->

            <!-- Start Receiver Roles -->
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_verification_receiver_roles"><?php _e('Receiver Roles', 'mec-waiting-list');?></label>
                <?php
$roles = isset($notifications['waiting_verification']['receiver_roles']) ? $notifications['waiting_verification']['receiver_roles'] : array();
		echo $main->get_roles_dropdown($roles, 'waiting_verification');
		?>
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Receiver Roles', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Select a specific user role.', 'mec-waiting-list');?></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
            <!-- End Receiver Roles -->

            <div class="mec-form-row">
                <label for="mec_notifications_waiting_verification_recipients"><?php _e('Custom Recipients', 'mec-waiting-list');?></label>
                <input type="text" name="mec[notifications][waiting_verification][recipients]" id="mec_notifications_email_verification_recipients" value="<?php echo (isset($notifications['waiting_verification']['recipients']) ? $notifications['waiting_verification']['recipients'] : ''); ?>" />
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Custom Recipients', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Insert comma separated emails for multiple recipients.', 'mec-waiting-list');?><a href="https://webnus.net/dox/modern-events-calendar/notifications/" target="_blank"><?php _e('Read More', 'mec-waiting-list');?></a></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_verification_content"><?php _e('Email Content', 'mec-waiting-list');?></label>
                <?php wp_editor((isset($notifications['waiting_verification']) ? stripslashes($notifications['waiting_verification']['content']) : ''), 'mec_notifications_waiting_verification_content', array('textarea_name' => 'mec[notifications][waiting_verification][content]'));?>
            </div>
            <p class="description"><?php _e('You can use the following placeholders', 'mec-waiting-list');?></p>
            <ul>
                <li><span>%%first_name%%</span>: <?php _e('Attendee\'s first name', 'mec-waiting-list');?></li>
                <li><span>%%last_name%%</span>: <?php _e('Attendee\'s last name', 'mec-waiting-list');?></li>
                <li><span>%%user_email%%</span>: <?php _e('Attendee\'s email address', 'mec-waiting-list');?></li>
                <li><span>%%book_date%%</span>: <?php _e('Booked date for the event', 'mec-waiting-list');?></li>
                <li><span>%%book_time%%</span>: <?php _e('Booked time for the event', 'mec-waiting-list');?></li>
                <li><span>%%book_price%%</span>: <?php _e('Booking Price', 'mec-waiting-list');?></li>
                <li><span>%%book_order_time%%</span>: <?php _e('Date and time of booking', 'mec-waiting-list');?></li>
                <li><span>%%blog_name%%</span>: <?php _e('Your website title', 'mec-waiting-list');?></li>
                <li><span>%%blog_url%%</span>: <?php _e('Your website URL', 'mec-waiting-list');?></li>
                <li><span>%%blog_description%%</span>: <?php _e('Your website description', 'mec-waiting-list');?></li>
                <li><span>%%event_title%%</span>: <?php _e('Event title', 'mec-waiting-list');?></li>
                <li><span>%%event_link%%</span>: <?php _e('Event link', 'mec-waiting-list');?></li>
                <li><span>%%event_start_date%%</span>: <?php _e('Event Start Date', 'mec-waiting-list');?></li>
                <li><span>%%event_end_date%%</span>: <?php _e('Event End Date', 'mec-waiting-list');?></li>
                <li><span>%%event_speaker_name%%</span>: <?php _e('Speaker name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_name%%</span>: <?php _e('Organizer name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_tel%%</span>: <?php _e('Organizer tel for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_email%%</span>: <?php _e('Organizer email for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_location_name%%</span>: <?php _e('Location name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_location_address%%</span>: <?php _e('Location address for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_other_locations_name%%</span>: <?php _e('Additional location\'s name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_other_locations_address%%</span>: <?php _e('Additional location\'s address for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_featured_image%%</span>: <?php _e('Featured image of the booked event', 'mec-waiting-list');?></li>
                <li><span>%%attendees_full_info%%</span>: <?php _e('Full attendee info such as booking form data, name, email etc.', 'mec-waiting-list');?></li>
                <li><span>%%booking_id%%</span>: <?php _e('Booking ID', 'mec-waiting-list');?></li>
                <li><span>%%booking_transaction_id%%</span>: <?php _e('Transaction ID of Booking', 'mec-waiting-list');?></li>
                <li><span>%%verification_link%%</span>: <?php _e('Email/Booking verification link.', 'mec-waiting-list');?></li>
                <li><span>%%total_attendees%%</span>: <?php _e('Total attendees of current booking', 'mec-waiting-list');?></li>
                <li><span>%%amount_tickets%%</span>: <?php _e('Number of booked tickets (total attendees of all bookings)', 'mec-waiting-list');?></li>
                <li><span>%%ticket_name%%</span>: <?php _e('Ticket name', 'mec-waiting-list');?></li>
                <li><span>%%ticket_time%%</span>: <?php _e('Ticket time', 'mec-waiting-list');?></li>
                <li><span>%%ticket_name_time%%</span>: <?php _e('Ticket name & time', 'mec-waiting-list');?></li>
                <li><span>%%payment_gateway%%</span>: <?php _e('Payment Gateway', 'mec-waiting-list');?></li>
                <li><span>%%ics_link%%</span>: <?php _e('Download ICS file', 'mec-waiting-list');?></li>
                <li><span>%%google_calendar_link%%</span>: <?php _e('Add to Google Calendar', 'mec-waiting-list');?></li>
            </ul>
        	</div>
    	</div>


        <div id="waiting_confirmation" class="mec-options-fields">

            <h4 class="mec-form-subtitle"><?php _e('Waiting Confirmation', 'mec-waiting-list');?></h4>

            <div class="mec-form-row">
                <label>
                    <input type="hidden" name="mec[notifications][waiting_confirmation][enable]" value="0" />
                    <input onchange="jQuery('#mec_notification_waiting_confirmation_toggle').toggle();" value="1" type="checkbox" name="mec[notifications][waiting_confirmation][enable]"
                     <?php if($enable_confirmation) echo 'checked="checked"'; ?> />
                     <?php _e('Enable Waiting Confirmation', 'mec'); ?>
                </label>
            </div>

            <div <?php if(!$enable_confirmation) echo 'style="display:none;"'; ?> id="mec_notification_waiting_confirmation_toggle" >


            <p class="description"><?php _e('It sends to attendee an email to confirm their waiting/email.', 'mec-waiting-list');?></p>
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_confirmation_subject"><?php _e('Email Subject', 'mec-waiting-list');?></label>
                <input type="text" name="mec[notifications][waiting_confirmation][subject]" id="mec_notifications_email_confirmation_subject" value="<?php echo (isset($notifications['waiting_confirmation']['subject']) ? stripslashes($notifications['waiting_confirmation']['subject']) : ''); ?>" />
            </div>

            <!-- Start Receiver Users -->
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_confirmation_receiver_users"><?php _e('Receiver Users', 'mec-waiting-list');?></label>
                <?php
$users = isset($notifications['waiting_confirmation']['receiver_users']) ? $notifications['waiting_confirmation']['receiver_users'] : array();
		echo $main->get_users_dropdown($users, 'waiting_confirmation');
		?>
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Receiver Users', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Select users to send a copy of the email to!', 'mec-waiting-list');?></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
            <!-- End Receiver Users -->

            <!-- Start Receiver Roles -->
            <div class="mec-form-row">
                <label for="mec_notifications_waiting_confirmation_receiver_roles"><?php _e('Receiver Roles', 'mec-waiting-list');?></label>
                <?php
$roles = isset($notifications['waiting_confirmation']['receiver_roles']) ? $notifications['waiting_confirmation']['receiver_roles'] : array();
		echo $main->get_roles_dropdown($roles, 'waiting_confirmation');
		?>
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Receiver Roles', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Select a specific user role.', 'mec-waiting-list');?></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
            <!-- End Receiver Roles -->

            <div class="mec-form-row">
                <label for="mec_notifications_waiting_confirmation_recipients"><?php _e('Custom Recipients', 'mec-waiting-list');?></label>
                <input type="text" name="mec[notifications][waiting_confirmation][recipients]" id="mec_notifications_email_verification_recipients" value="<?php echo (isset($notifications['waiting_confirmation']['recipients']) ? $notifications['waiting_confirmation']['recipients'] : ''); ?>" />
                <span class="mec-tooltip">
                    <div class="box top">
                        <h5 class="title"><?php _e('Custom Recipients', 'mec-waiting-list');?></h5>
                        <div class="content"><p><?php esc_attr_e('Insert comma separated emails for multiple recipients.', 'mec-waiting-list');?><a href="https://webnus.net/dox/modern-events-calendar/notifications/" target="_blank"><?php _e('Read More', 'mec-waiting-list');?></a></p></div>
                    </div>
                    <i title="" class="dashicons-before dashicons-editor-help"></i>
                </span>
            </div>
           	<div class="mec-form-row">
                <label for="mec_notifications_waiting_confirmation_content"><?php _e('Email Content', 'mec-waiting-list');?></label>
                <?php wp_editor((isset($notifications['waiting_confirmation']) ? stripslashes($notifications['waiting_confirmation']['content']) : ''), 'mec_notifications_waiting_confirmation_content', array('textarea_name' => 'mec[notifications][waiting_confirmation][content]'));?>
            </div>
            <p class="description"><?php _e('You can use the following placeholders', 'mec-waiting-list');?></p>
            <ul>
                <li><span>%%first_name%%</span>: <?php _e('Attendee\'s first name', 'mec-waiting-list');?></li>
                <li><span>%%last_name%%</span>: <?php _e('Attendee\'s last name', 'mec-waiting-list');?></li>
                <li><span>%%user_email%%</span>: <?php _e('Attendee\'s email address', 'mec-waiting-list');?></li>
                <li><span>%%waiting_cancellation_link%%</span>: <?php _e('Waiting Cancel link', 'mec-waiting-list');?></li>
                <li><span>%%book_date%%</span>: <?php _e('Booked date for the event', 'mec-waiting-list');?></li>
                <li><span>%%book_time%%</span>: <?php _e('Booked time for the event', 'mec-waiting-list');?></li>
                <li><span>%%book_price%%</span>: <?php _e('Booking Price', 'mec-waiting-list');?></li>
                <li><span>%%book_order_time%%</span>: <?php _e('Date and time of booking', 'mec-waiting-list');?></li>
                <li><span>%%blog_name%%</span>: <?php _e('Your website title', 'mec-waiting-list');?></li>
                <li><span>%%blog_url%%</span>: <?php _e('Your website URL', 'mec-waiting-list');?></li>
                <li><span>%%blog_description%%</span>: <?php _e('Your website description', 'mec-waiting-list');?></li>
                <li><span>%%event_title%%</span>: <?php _e('Event title', 'mec-waiting-list');?></li>
                <li><span>%%event_link%%</span>: <?php _e('Event link', 'mec-waiting-list');?></li>
                <li><span>%%event_start_date%%</span>: <?php _e('Event Start Date', 'mec-waiting-list');?></li>
                <li><span>%%event_end_date%%</span>: <?php _e('Event End Date', 'mec-waiting-list');?></li>
                <li><span>%%event_speaker_name%%</span>: <?php _e('Speaker name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_name%%</span>: <?php _e('Organizer name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_tel%%</span>: <?php _e('Organizer tel for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_organizer_email%%</span>: <?php _e('Organizer email for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_location_name%%</span>: <?php _e('Location name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_location_address%%</span>: <?php _e('Location address for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_other_locations_name%%</span>: <?php _e('Additional location\'s name for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_other_locations_address%%</span>: <?php _e('Additional location\'s address for the booked event', 'mec-waiting-list');?></li>
                <li><span>%%event_featured_image%%</span>: <?php _e('Featured image of the booked event', 'mec-waiting-list');?></li>
                <li><span>%%attendees_full_info%%</span>: <?php _e('Full attendee info such as booking form data, name, email etc.', 'mec-waiting-list');?></li>
                <li><span>%%booking_id%%</span>: <?php _e('Booking ID', 'mec-waiting-list');?></li>
                <li><span>%%booking_transaction_id%%</span>: <?php _e('Transaction ID of Booking', 'mec-waiting-list');?></li>
                <li><span>%%verification_link%%</span>: <?php _e('Email/Booking verification link.', 'mec-waiting-list');?></li>
                <li><span>%%total_attendees%%</span>: <?php _e('Total attendees of current booking', 'mec-waiting-list');?></li>
                <li><span>%%amount_tickets%%</span>: <?php _e('Number of booked tickets (total attendees of all bookings)', 'mec-waiting-list');?></li>
                <li><span>%%ticket_name%%</span>: <?php _e('Ticket name', 'mec-waiting-list');?></li>
                <li><span>%%ticket_time%%</span>: <?php _e('Ticket time', 'mec-waiting-list');?></li>
                <li><span>%%ticket_name_time%%</span>: <?php _e('Ticket name & time', 'mec-waiting-list');?></li>
                <li><span>%%payment_gateway%%</span>: <?php _e('Payment Gateway', 'mec-waiting-list');?></li>
                <li><span>%%ics_link%%</span>: <?php _e('Download ICS file', 'mec-waiting-list');?></li>
                <li><span>%%google_calendar_link%%</span>: <?php _e('Add to Google Calendar', 'mec-waiting-list');?></li>
            </ul>
        </div>
        </div>

        <?php
}

	public static function notification_menu_js() {
		?>
        jQuery("#mec_notifications_waiting_verification_content-html").click();
        jQuery("#mec_notifications_waiting_verification_content-tmce").click();
        jQuery("#mec_notifications_waiting_confirmation_content-html").click();
        jQuery("#mec_notifications_waiting_confirmation_content-tmce").click();
        <?php
}

	public static function booking_menu_content($main, $setting) {

		$options = $main->get_options();
		$waiting_fields = isset($options['waiting_fields']) ? $options['waiting_fields'] : array();
        $settings = isset($options['settings']) && is_array($options['settings']) ? $options['settings'] : $options;
		$notif_page = isset($settings['waiting_notif_page'])?$settings['waiting_notif_page']:null;

		$mec_email = false;
		$mec_name = false;

		if (!is_array($waiting_fields)) {
			$waiting_fields = array();
		}

		foreach ($waiting_fields as $field) {
			if (isset($field['type'])) {
				if ($field['type'] == 'name') {
					$mec_name = true;
				}

				if ($field['type'] == 'mec_email') {
					$mec_email = true;
				}

			} else {
				break;
			}

		}

		if (!$mec_name) {
			array_unshift(
				$waiting_fields,
				array(
					'mandatory' => '0',
					'type' => 'name',
					'label' => esc_html__('Name', 'mec-waiting-list'),
				)
			);
		}

		if (!$mec_email) {
			array_unshift(
				$waiting_fields,
				array(
					'mandatory' => '0',
					'type' => 'mec_email',
					'label' => esc_html__('Email', 'mec-waiting-list'),
				)
			);
		}
		?>

        <div id="waiting_list" class="mec-options-fields">
        <h4 class="mec-form-subtitle"><?php _e('Waiting List Setting', 'mec-waiting-list');?></h4>
<div class="mec-form-row">
    <label class="mec-col-3" for="mec_settings_waiting_limit"><?php echo __('Limit', 'mec-waiting-list') ?></label>
    <div class="mec-col-9">
        <input type="number" id="mec_settings_waiting_limit" name="mec[settings][waiting_limit]" value="<?php echo ((isset($settings['waiting_limit']) and trim($settings['waiting_limit']) != '') ? $settings['waiting_limit'] : ''); ?>" placeholder="Default: 12" class="">
        <span class="mec-tooltip">
            <div class="box left">
                <h5 class="title"><?php echo __('Waiting List Limit', 'mec-waiting-list') ?></h5>
                <div class="content"><p><?php echo __('Total number of waitings that a user can book. It is useful if you\'re providing a maximum amount for user registrations in the waiting list. Leave it empty for unlimited waitings. Default: 12', 'mec-waiting-list') ?></p></div>
            </div>
            <i title="" class="dashicons-before dashicons-editor-help"></i>
        </span>
    </div>
</div>
<div class="mec-form-row">

    <div class="mec-form-row">
        <div class="mec-col-12">
            <label for="waiting_verified">
                <input type="hidden" name="mec[settings][waiting_verified]" value="0">
                <input type="checkbox" name="mec[settings][waiting_verified]" id="waiting_verified" <?php echo ((!isset($settings['waiting_verified']) or (isset($settings['waiting_verified']) and $settings['waiting_verified'] == '1')) ? 'checked="checked"' : ''); ?> value="1">
                <?php _e('Auto verification for waiting list', 'mec-waiting-list');?>
            </label>
            <span class="mec-tooltip">
                <div class="box top">
                    <h5 class="title"><?php _e('Auto verification for waiting list', 'mec-waiting-list');?></h5>
                    <div class="content"><p><?php esc_attr_e("User's submitted email is the default to verify or send verify link to.", 'mec-waiting-list');?><a href="https://webnus.net/dox/modern-events-calendar/booking/" target="_blank"><?php _e('Read More', 'mec-waiting-list');?></a></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
        </div>
    </div>

    <div class="mec-form-row">
        <div class="mec-col-12">
            <label for="waiting_confirmation">
                <input type="hidden" name="mec[settings][waiting_confirmation]" value="0">
                <input type="checkbox" name="mec[settings][waiting_confirmation]" id="waiting_confirmation" <?php echo ((!isset($settings['waiting_confirmation']) or (isset($settings['waiting_confirmation']) and $settings['waiting_confirmation'] == '1')) ? 'checked="checked"' : ''); ?> value="1">
				<?php _e('Auto Confirmation for waiting list', 'mec-waiting-list');?>
            </label>
            <span class="mec-tooltip">
                <div class="box top">
                    <h5 class="title"><?php _e('Waiting Auto Confirmation', 'mec-waiting-list');?></h5>
                    <div class="content"><p><?php esc_attr_e("User submitted waiting, auto confirm without admin approval.", 'mec-waiting-list');?><a href="https://webnus.net/dox/modern-events-calendar/booking/" target="_blank"><?php _e('Read More', 'mec-waiting-list');?></a></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
        </div>
    </div>

    <div class="mec-form-row">
        <div class="mec-col-12">
            <label for="mec_settings_waiting_auto_confirm_send_email">
                <input type="hidden" name="mec[settings][waiting_auto_confirm_send_email]" value="0" />
                <input type="checkbox" name="mec[settings][waiting_auto_confirm_send_email]" id="mec_settings_waiting_auto_confirm_send_email" <?php echo ((isset($settings['waiting_auto_confirm_send_email']) and $settings['waiting_auto_confirm_send_email'] == '1') ? 'checked="checked"' : ''); ?> value="1" />
                <?php esc_html_e('Send confirmation email in auto confirmation mode', 'mec'); ?>
            </label>
        </div>
    </div>

    <div class="mec-form-row">
        <div class="mec-col-12">
            <label for="waiting_first_for_all">
                <input type="hidden" name="mec[settings][waiting_first_for_all]" value="0">
                <input type="checkbox" name="mec[settings][waiting_first_for_all]" id="waiting_first_for_all" <?php echo ((!isset($settings['waiting_first_for_all']) or (isset($settings['waiting_first_for_all']) and $settings['waiting_first_for_all'] == '1')) ? 'checked="checked"' : ''); ?> value="1">
                <?php echo __('Enable Express Attendees Form', 'mec-waiting-list') ?>
            </label>
            <span class="mec-tooltip">
                <div class="box top">
                    <h5 class="title"><?php _e('Attendees Form', 'mec-waiting-list');?></h5>
                    <div class="content"><p><?php esc_attr_e("Apply the info from the first attendee to all purchased waitings by that user. Uncheck if you want every waiting to have its own attendee’s info.", 'mec-waiting-list');?><a href="https://webnus.net/dox/modern-events-calendar/booking/" target="_blank"><?php _e('Read More', 'mec-waiting-list');?></a></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
        </div>
    </div>

    <div class="mec-form-row">
        <label class="mec-col-3" for="mec_settings_waiting_notif_page"><?php echo __('Verify & Cancel Page', 'mec-waiting-list') ?></label>
        <div class="mec-col-9">
            <select id="mec_settings_waiting_notif_page" name="mec[settings][waiting_notif_page]">
            	<?php foreach (get_pages() as $k => $v):?>
                <option value="<?php echo $v->ID; ?>" <?php if($notif_page==$v->ID) echo 'selected="selected"'; ?>>
                	<?php echo $v->post_title; ?>
                </option>

                <?php endforeach; ?>
            </select>
            <span class="mec-tooltip">
                <div class="box left">
                    <h5 class="title"><?php echo __('Verify & Cancel Page', 'mec-waiting-list') ?></h5>
                    <div class="content"><p><?php echo __('Send Verify and Cancel Page link to attendees. After creating a page, add [waiting_notif_page] shortcode, and then select the page.', 'mec-waiting-list') ?></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
        </div>
    </div>
    <h5 class="title"><?php esc_html_e('User Registration', 'mec'); $roles = wp_roles()->roles;
?></h5>
        <div class="mec-form-row">
            <label class="mec-col-3" for="mec_settings_waiting_registration"><?php esc_html_e('Registration', 'mec'); ?></label>
            <div class="mec-col-9">
            <select id="mec_settings_waiting_registration" name="mec[settings][waiting_registration]">
                <option value="1" <?php selected(get_option('mec_options')['settings']['waiting_registration'], '1'); ?>><?php esc_html_e('Enabled', 'mec'); ?></option>
                <option value="0" <?php selected(get_option('mec_options')['settings']['waiting_registration'], '0'); ?>><?php esc_html_e('Disabled', 'mec'); ?></option>
            </select>

                <span class="mec-tooltip">
                <div class="box left">
                    <h5 class="title"><?php esc_html_e('Registration', 'mec'); ?></h5>
                    <div class="content"><p><?php esc_attr_e("By enabaling this option MEC will create a WordPress User for the main attendee. It's recommended to keep it enabled.", 'mec'); ?></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
            </div>
        </div>
        <div id="mec_settings_waiting_registration_wrapper" class="<?php echo !isset($settings['waiting_registration']) || $settings['waiting_registration'] ? "" : "w-hidden"; ?>">
            <div id="mec_settings_waiting_registration_userpass_wrapper" class="mec-form-row <?php echo !isset($settings['waiting_registration']) || $settings['waiting_registration'] == '1' ? "" : "w-hidden"; ?>">
                <label class="mec-col-3" for="mec_settings_waiting_userpass"><?php esc_html_e('Username & Password', 'mec'); ?></label>
                <div class="mec-col-9">
                    <select id="mec_settings_waiting_userpass" name="mec[settings][waiting_userpass]">
                        <option value="auto" <?php echo ((isset($settings['waiting_userpass']) and trim($settings['waiting_userpass']) == 'auto') ? 'selected="selected"' : ''); ?>><?php echo esc_html__('Auto', 'mec'); ?></option>
                        <option value="manual" <?php echo ((isset($settings['waiting_userpass']) and trim($settings['waiting_userpass']) == 'manual') ? 'selected="selected"' : ''); ?>><?php echo esc_html__('Manual', 'mec'); ?></option>
                    </select>
                    <span class="mec-tooltip">
                <div class="box left">
                    <h5 class="title"><?php esc_html_e('Username & Password', 'mec'); ?></h5>
                    <div class="content"><p><?php esc_attr_e("If you set it to the manual option, users can insert a username and password during the waiting for registration; otherwise, MEC will use their email and an auto-generated password.", 'mec'); ?></p></div>
                </div>
                <i title="" class="dashicons-before dashicons-editor-help"></i>
            </span>
                </div>
            </div>
        </div>

		<h5 class="title"><?php _e('Waiting List Form', 'mec-waiting-list');?></h5>
        <div class="mec-booking-per-attendee-fields">
            <div class="mec-container">

                <div class="mec-form-row" id="mec_waiting_form_container">
                    <?php /** Don't remove this hidden field **/?>
                    <input type="hidden" name="mec[waiting_fields]" value="" />

                    <ul id="mec_waiting_form_fields" class="mec-form-row">
                        <?php
			$i = 0;
		foreach ($waiting_fields as $key => $waite_field) {
			if (!is_numeric($key)) {
				continue;
			}

			$i = max($i, $key);

			if ($waite_field['type'] == 'text') {
				echo $main->field_text($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'name') {
				echo $main->field_name($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'mec_email') {
				echo $main->field_mec_email($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'email') {
				echo $main->field_email($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'date') {
				echo $main->field_date($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'file') {
				echo $main->field_file($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'tel') {
				echo $main->field_tel($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'textarea') {
				echo $main->field_textarea($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'p') {
				echo $main->field_p($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'checkbox') {
				echo $main->field_checkbox($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'radio') {
				echo $main->field_radio($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'select') {
				echo $main->field_select($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'agreement') {
				echo $main->field_agreement($key, $waite_field, 'waiting');
			}

		}
		?>
                    </ul>
                    <div id="mec_waiting_form_field_types">
                        <button type="button" class="button red" data-type="name"><?php _e('MEC Name', 'mec-waiting-list');?></button>
                        <button type="button" class="button red" data-type="mec_email"><?php _e('MEC Email', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="text"><?php _e('Text', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="email"><?php _e('Email', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="date"><?php _e('Date', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="tel"><?php _e('Tel', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="file"><?php _e('File', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="textarea"><?php _e('Textarea', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="checkbox"><?php _e('Checkboxes', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="radio"><?php _e('Radio Buttons', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="select"><?php _e('Dropdown', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="agreement"><?php _e('Agreement', 'mec-waiting-list');?></button>
                        <button type="button" class="button" data-type="p"><?php _e('Paragraph', 'mec-waiting-list');?></button>
                    </div>
                    <?php do_action('mec_waiting_fields_form_end');?>
                </div>
                <?php do_action('after_mec_waiting_fields_form');?>
            </div>
            <input type="hidden" id="mec_new_waiting_field_key" value="<?php echo $i + 1; ?>" />
            <div class="mec-util-hidden">
                <div id="mec_waiting_field_text">
                    <?php echo $main->field_text(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_email">
                    <?php echo $main->field_email(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_mec_email">
                    <?php echo $main->field_mec_email(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_name">
                    <?php echo $main->field_name(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_tel">
                    <?php echo $main->field_tel(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_date">
                    <?php echo $main->field_date(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_file">
                    <?php echo $main->field_file(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_textarea">
                    <?php echo $main->field_textarea(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_checkbox">
                    <?php echo $main->field_checkbox(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_radio">
                    <?php echo $main->field_radio(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_select">
                    <?php echo $main->field_select(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_agreement">
                    <?php echo $main->field_agreement(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_p">
                    <?php echo $main->field_p(':i:', array(), 'waiting'); ?>
                </div>
                <div id="mec_waiting_field_option">
                    <?php echo $main->field_option(':fi:', ':i:', array(), 'waiting'); ?>
                </div>
            </div>
        </div>
</div>
    </div>

        <?php
}

	public static function save_options($final) {

		$mec = isset( $_POST['mec'] ) && is_array( $_POST['mec'] ) ? $_POST['mec'] : array();

		$current = get_option('mec_options', array());

		$filtered = array();
		foreach ($mec as $key => $value) {
			$filtered[$key] = (is_array($value) ? $value : array());
		}


		// Bellow conditional block codes is used for sortable booking form items.
		if (isset($filtered['waiting_fields'])) {
			if (is_array($filtered['waiting_fields'])) {
				$filtered_waiting_fields_count = count($filtered['waiting_fields']);
				if ($filtered_waiting_fields_count) {
					$filtered_waiting_fields_slice_first = array_slice($filtered['waiting_fields'], 0, $filtered_waiting_fields_count - 2);
					$filtered_waiting_fields_slice_last = array_slice($filtered['waiting_fields'], ($filtered_waiting_fields_count - 2), $filtered_waiting_fields_count, true);
					$filtered['waiting_fields'] = array_merge($filtered_waiting_fields_slice_first, $filtered_waiting_fields_slice_last);
				}
			} else {
				$filtered['waiting_fields'] = array();
			}
		}

		if (isset($current['waiting_fields']) and isset($filtered['waiting_fields'])) {
			$current['waiting_fields'] = array();
			$current['waiting_fields'] = $filtered['waiting_fields'];
		}else if(!isset($current['waiting_fields'])){
			$current['waiting_fields'] = $filtered['waiting_fields'];
		}

		$final['waiting_fields'] = $current['waiting_fields'];


		return $final;
	}

	public static function add_waiting_list_left_menu() {
		?>
        <a class="mec-add-booking-tabs-link" data-href="mec-waiting-fields" href="#"><?php echo esc_html__('Waiting List Form', 'mec-waiting-list'); ?></a>
        <?php
}

	public static function add_waiting_list_left_menu_content($post_id) {

		$main = \MEC::getInstance('app.libraries.main');
		$post = get_post($post_id);
        if( !is_a( $post, '\WP_Post' ) ){
            return;
        }

		$global_inheritance = get_post_meta($post->ID, 'mec_waiting_fields_global_inheritance', true);
		if (trim($global_inheritance) == '') {
			$global_inheritance = 1;
		}

		$waiting_fields = get_post_meta($post->ID, 'mec_waiting_fields', true);
		$global_waiting_fields = array();

		$options = $main->get_options();
		$global_waiting_fields = isset($options['waiting_fields']) ? $options['waiting_fields'] : array();

		if ((is_array($waiting_fields) and !count($waiting_fields)) or (!is_array($waiting_fields) and trim($waiting_fields) == '')) {
			$waiting_fields = $global_waiting_fields;
		}

		if (!is_array($waiting_fields)) {
			$waiting_fields = array();
		}

		$mec_email = false;
		$mec_name = false;
		foreach ($waiting_fields as $field) {
			if (isset($field['type'])) {
				if ($field['type'] == 'mec_email') {
					$mec_email = true;
				}

				if ($field['type'] == 'name') {
					$mec_name = true;
				}

			} else {
				break;
			}
		}

		if (!$mec_name) {
			$waiting_fields[] = array(
				'mandatory' => '0',
				'type' => 'name',
				'label' => esc_html__('Name', 'mec-waiting-list'),
			);
		}

		if (!$mec_email) {
			$waiting_fields[] = array(
				'mandatory' => '0',
				'type' => 'mec_email',
				'label' => esc_html__('Email', 'mec-waiting-list'),
			);
		}

		?>
        <div class="mec-meta-box-fields mec-booking-tab-content" id="mec-waiting-fields">
            <h4 class="mec-meta-box-header"><?php _e('Waiting List Form', 'mec-waiting-list');?></h4>
            <div id="mec_meta_box_waiting_fields_form">
                <div class="mec-form-row">
                    <label class="label-checkbox waiting_fields_global_inheritance_label">
                        <input type="hidden" name="mec[waiting_fields_global_inheritance]" value="0"/>
                        <input onchange="jQuery('#mec_waitingform_container_toggle').toggle();" value="1" type="checkbox"
                               name="mec[waiting_fields_global_inheritance]"
                            <?php
if ($global_inheritance) {
			echo 'checked="checked"';
		}
		?>
                        /> <?php _e('Inherit from global options', 'mec-waiting-list');?>
                    </label>
                </div>
                <?php do_action('mec_meta_box_waiting_fields_form', $post->ID);?>
                <div id="mec_waitingform_container_toggle" class="
                <?php
if ($global_inheritance) {
			echo 'mec-util-hidden';
		}
		?>">

                    <div class="mec-booking-per-attendee-fields">
                        <h5 class="mec-form-subtitle"><?php _e('Per Attendee Fields', 'mec-waiting-list');?></h5>
                        <?php /** Don't remove this hidden field **/?>
                        <input type="hidden" name="mec[waiting_fields]" value=""/>

                        <ul id="mec_waiting_form_fields" class="mec-form-row">
                            <?php
$i = 0;
		foreach ($waiting_fields as $key => $waite_field) {
			if (!is_numeric($key)) {
				continue;
			}

			$i = max($i, $key);

			if ($waite_field['type'] == 'text') {
				echo $main->field_text($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'mec_email') {
				echo $main->field_mec_email($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'name') {
				echo $main->field_name($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'email') {
				echo $main->field_email($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'date') {
				echo $main->field_date($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'file') {
				echo $main->field_file($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'tel') {
				echo $main->field_tel($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'textarea') {
				echo $main->field_textarea($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'p') {
				echo $main->field_p($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'checkbox') {
				echo $main->field_checkbox($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'radio') {
				echo $main->field_radio($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'select') {
				echo $main->field_select($key, $waite_field, 'waiting');
			} elseif ($waite_field['type'] == 'agreement') {
				echo $main->field_agreement($key, $waite_field, 'waiting');
			}

		}
		?>
                        </ul>
                        <div id="mec_waiting_form_field_types">
                            <button type="button" class="button red" data-type="name"><?php _e('MEC Name', 'mec-waiting-list');?></button>
                            <button type="button" class="button red" data-type="mec_email"><?php _e('MEC Email', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="text"><?php _e('Text', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="email"><?php _e('Email', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="date"><?php _e('Date', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="tel"><?php _e('Tel', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="file"><?php _e('File', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="textarea"><?php _e('Textarea', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="checkbox"><?php _e('Checkboxes', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="radio"><?php _e('Radio Buttons', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="select"><?php _e('Dropdown', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="agreement"><?php _e('Agreement', 'mec-waiting-list');?></button>
                            <button type="button" class="button" data-type="p"><?php _e('Paragraph', 'mec-waiting-list');?></button>
                        </div>
                        <input type="hidden" id="mec_new_waiting_field_key" value="<?php echo $i + 1; ?>"/>
                        <div class="mec-util-hidden">
                            <div id="mec_waiting_field_text">
                                <?php echo $main->field_text(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_email">
                                <?php echo $main->field_email(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_mec_email">
                                <?php echo $main->field_mec_email(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_name">
                                <?php echo $main->field_name(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_tel">
                                <?php echo $main->field_tel(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_date">
                                <?php echo $main->field_date(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_file">
                                <?php echo $main->field_file(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_textarea">
                                <?php echo $main->field_textarea(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_checkbox">
                                <?php echo $main->field_checkbox(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_radio">
                                <?php echo $main->field_radio(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_select">
                                <?php echo $main->field_select(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_agreement">
                                <?php echo $main->field_agreement(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_p">
                                <?php echo $main->field_p(':i:', array(), 'waiting'); ?>
                            </div>
                            <div id="mec_waiting_field_option">
                                <?php echo $main->field_option(':fi:', ':i:', array(), 'waiting'); ?>
                            </div>
                        </div>
                    </div>


                </div>
            </div>
        </div>
        <?php
}

	public static function save_event_waiting_fields($post_id, $mec_update) {

		$_mec = isset($_POST['mec']) ? $_POST['mec'] : array();

		// Waiting Fields options
		$waiting_fields_global_inheritance = isset($_mec['waiting_fields_global_inheritance']) ? $_mec['waiting_fields_global_inheritance'] : 1;
		update_post_meta($post_id, 'mec_waiting_fields_global_inheritance', $waiting_fields_global_inheritance);

		$waiting_fields = isset($_mec['waiting_fields']) ? $_mec['waiting_fields'] : array();
		if ($waiting_fields_global_inheritance) {
			$waiting_fields = array();
		}

		update_post_meta($post_id, 'mec_waiting_fields', $waiting_fields);

	}

} //Mec_Admin