<?php

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Box_Shadow;

/** no direct access */
defined('MECEXEC') or die();

/**
 * Webnus MEC elementor addon shortcode class
 *
 * @author Webnus <info@webnus.net>
 */
class MecShortCodeDesignerACF extends Widget_Base
{

	/**
	 * Retrieve MEC widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mec-acf';
	}

	/**
	 * Retrieve MEC widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __('MEC ACF', 'mec-shortcode-designer');
	}

	/**
	 * Retrieve MEC widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-post-excerpt';
	}


	/**
	 * Set widget category.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget category.
	 */
	public function get_categories()
	{
		return ['mec_shortcode_designer'];
	}

	/**
	 * Register MEC widget controls.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		if (!is_plugin_active('advanced-custom-fields/acf.php')) {
			return;
		}

		$this->start_controls_section(
			'content_section',
			[
				'label' => __('Typography', 'mec-shortcode-designer'),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'field_label',
			[
				'label' => __('Field Label', 'mec-shortcode-designer'),
				'type' => Controls_Manager::TEXT,
				'default' => '',
			]
		);

		$field_groups = acf_get_field_groups();
		$groups = array();
		foreach ($field_groups as $group) {
			$groups[$group['key']] = $group['title'];
		}

		$this->add_control(
			'field_group',
			[
				'label' => __('Field Groups', 'mec-shortcode-designer'),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => $groups,
			]
		);


		foreach ($field_groups as $group) {
			$group_id = $group['key'];
			$fields = acf_get_fields($group);

			if (! $fields) continue;

			$group_fields = array();
			foreach ($fields as $field) {
				$group_fields[$field['key']] = $field['label'];
			}

			$this->add_control(
				'field_key_' . $group_id,
				[
					'label' => __('Field Label', 'mec-shortcode-designer'),
					'type' => Controls_Manager::SELECT,
					'options' => $group_fields,
					'default' => key($group_fields),
					'condition' => [
						'field_group' => $group_id,
					],
				]
			);
		}


		$this->end_controls_section();

		// typography
		$this->start_controls_section(
			'styling_section',
			[
				'label' => __('Typography', 'mec-shortcode-designer'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'		=> 'title_typography',
				'label'		=> __('Typography', 'mec-shortcode-designer'),
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector'	=> '{{WRAPPER}} .mec-event-acf',
			]
		);
		$this->add_control(
			'title_align',
			[
				'label'		=> __('Alignment', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::CHOOSE,
				'options'	=> [
					'left' => [
						'title' => __('Left', 'mec-shortcode-designer'),
						'icon'	=> 'mec-fa-align-left',
					],
					'center' => [
						'title' => __('Center', 'mec-shortcode-designer'),
						'icon'	=> 'mec-fa-align-center',
					],
					'right' => [
						'title' => __('Right', 'mec-shortcode-designer'),
						'icon'	=> 'mec-fa-align-right',
					],
				],
				'default' => 'left',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf' => 'text-align: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'display',
			[
				'label'		=> __('Display', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::SELECT,
				'default' 	=> 'block',
				'options' 	=> [
					'inherit'		=> __('inherit', 'mec-shortcode-designer'),
					'inline'		=> __('inline', 'mec-shortcode-designer'),
					'inline-block'	=> __('inline-block', 'mec-shortcode-designer'),
					'block'			=> __('block', 'mec-shortcode-designer'),
					'none'			=> __('none', 'mec-shortcode-designer'),
				],
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf' => 'display: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
		// color
		$this->start_controls_section(
			'title_color_style',
			[
				'label' => __('Colors', 'mec-shortcode-designer'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'label_color',
			[
				'label'		=> __('Label Color', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> '#000',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf .mec-acf-label' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'label_color_hover',
			[
				'label'		=> __('Label Hover', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> '#000',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf .mec-acf-label:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'value_color',
			[
				'label'		=> __('Value Color', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> '#000',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf .mec-acf-value' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'value_color_hover',
			[
				'label'		=> __('Value Hover', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> '#000',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf .mec-acf-value:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
		// background
		$this->start_controls_section(
			'title_background_style',
			[
				'label' => __('Background', 'mec-shortcode-designer'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'title_background',
			[
				'label'		=> __('Background', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> 'rgba(255,255,255,0)',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf' => 'background: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'title_background_hover',
			[
				'label'		=> __('Hover', 'mec-shortcode-designer'),
				'type'		=> Controls_Manager::COLOR,
				'global' => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
				'default'	=> 'rgba(255,255,255,0)',
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf:hover' => 'background: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
		// Spacing
		$this->start_controls_section(
			'title_spacing_style',
			[
				'label' => __('Spacing', 'mec-shortcode-designer'),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'label_margin',
			[
				'label'			=> __('Label Margin', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'default'		=> [
					'top'		=> '0',
					'right'		=> '0',
					'bottom'	=> '0',
					'left'		=> '0',
					'isLinked' => true,
				],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf .mec-acf-label' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};display: inline-block;',
				],
			]
		);
		$this->add_control(
			'label_padding',
			[
				'label'			=> __('Label Padding', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf .mec-acf-label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};display: inline-block;',
				],
			]
		);

		$this->add_control(
			'value_margin',
			[
				'label'			=> __('Value Margin', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'default'		=> [
					'top'		=> '0',
					'right'		=> '0',
					'bottom'	=> '0',
					'left'		=> '0',
					'isLinked' => true,
				],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf .mec-acf-value' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};display: inline-block;',
				],
			]
		);
		$this->add_control(
			'value_padding',
			[
				'label'			=> __('Value Padding', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf .mec-acf-value' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};display: inline-block;',
				],
			]
		);

		$this->add_control(
			'title_margin',
			[
				'label'			=> __('Margin', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'default'		=> [
					'top'		=> '0',
					'right'		=> '0',
					'bottom'	=> '0',
					'left'		=> '0',
					'isLinked' => true,
				],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label'			=> __('Padding', 'mec-shortcode-designer'),
				'type'			=> Controls_Manager::DIMENSIONS,
				'size_units'	=> ['px', '%', 'em'],
				'selectors'		=> [
					'{{WRAPPER}} .mec-event-acf' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'		=> 'border',
				'label'		=> __('Border', 'mec-shortcode-designer'),
				'selector'	=> '{{WRAPPER}} .mec-event-acf',
			]
		);
		$this->add_control(
			'date_border_radius', //param_name
			[
				'label' 		=> __('Border Radius', 'mec-shortcode-designer'),
				'type' 			=> Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors' => [
					'{{WRAPPER}} .mec-event-acf' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'box_shadow',
				'label' => __('Box Shadow', 'mec-shortcode-designer'),
				'selector' => '{{WRAPPER}} .mec-event-acf',
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render MEC widget output on the frontend.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		if (!is_plugin_active('advanced-custom-fields/acf.php')) {
			echo __('Enabling this widget requires installing and activating the ACF plugin.', 'mec-shortcode-designer');
			return;
		}
		$settings = $this->get_settings();
		if (get_post_type() == 'mec_designer') {
			$event = get_posts('post_type=mec-events&numberposts=1');
			$event_id = $event ? $event[0]->ID : 0;
		} else {
			$event_id = get_the_ID();
		}
		$field_label = $settings['field_label'] ?? '';
		$group_id = $settings['field_group'] ?? '';
		$field_key = $settings['field_key_' . $group_id] ?? '';
		$field_object = get_field_object($field_key, $event_id);
		$meta_key = $field_object['name'] ?? $field_label;
		$field_value = $field_object['value'] ?? '';
		$field_type = $field_object['type'] ?? 'text';
		if (empty($field_value)) {
			global $wpdb;
			$meta_row = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT meta_value FROM {$wpdb->postmeta} 
					 WHERE meta_key = %s AND meta_value != '' 
					 ORDER BY meta_id DESC LIMIT 1",
					$meta_key
				),
				ARRAY_A
			);
			if (!empty($meta_row['meta_value'])) {
				$field_value = maybe_unserialize($meta_row['meta_value']);
			}
		}
		if ($field_type === 'url') {
			$url = trim($field_value);
			if (!empty($url) && !preg_match('#^https?://#i', $url)) {
				$url = 'https://' . $url;
			}
			if (filter_var($url, FILTER_VALIDATE_URL)) {
				$field_value = '<a href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($url) . '</a>';
			} else {
				$field_value = esc_html($field_value);
			}
		} elseif ($field_type === 'image') {
			if (is_array($field_value) && isset($field_value['url'])) {
				$field_value = '<img src="' . esc_url($field_value['url']) . '" alt="' . esc_attr($field_value['alt'] ?? $field_label) . '" />';
			} elseif (is_numeric($field_value)) {
				$image_url = wp_get_attachment_image_url($field_value, 'full');
				if ($image_url) {
					$field_value = '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($field_label) . '" />';
				} else {
					$field_value = '';
				}
			} else {
				$field_value = esc_html($field_value);
			}
		} elseif (is_array($field_value)) {
			$field_value = implode(', ', array_map('esc_html', $field_value));
		} else {
			$field_value = esc_html($field_value);
		}

?>
		<div class="mec-shortcode-designer">
			<div class="mec-event-acf">
				<span class="mec-acf-label"><?php echo esc_html($field_label); ?></span>
				<span class="mec-acf-value"><?php echo $field_value; ?></span>
			</div>
		</div>
<?php
	}
}
