<?php

namespace MEC_Social_Poster\Accounts;

use MEC\Singleton;
use MEC_Social_Poster\Settings\Settings;

/**
 * @since 1.0.0
 */
class Accounts extends Singleton {

	/**
	 * Return accounts
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function _get_accounts(){

		return Settings::getInstance()->get_settings( 'accounts', array() );
	}

	/**
	 * Return accounts
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function _update_accounts( $accounts ){

		return Settings::getInstance()->update_settings( 'accounts', $accounts );
	}

	/**
	 * Check Conditions
	 *
	 * @param array $account
	 * @param array $query
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function filter_account( $account, $query ){

		$taxonomies = array(
			'mec_category',
			'mec_organizer',
			'mec_speaker',
			'mec_location',
		);

		$compare = true;
		foreach( $taxonomies as $tax ){

			$tax_query = isset( $query[ $tax ] ) && is_array( $query[ $tax ] ) ? $query[ $tax ] : array();
			if( !$tax_query || empty( $tax_query ) ){

				continue;
			}
			$account_terms = isset( $account[ $tax ] ) && is_array( $account[ $tax ] ) ? $account[ $tax ] : array();
			$r = array_intersect( $tax_query, $account_terms );

			$compare = $compare && count( $r );
		}

		return $compare;
	}

	/**
	 * Return accounts list
	 *
	 * @param array $query
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_accounts( $query ){

		$accounts = $this->_get_accounts();
		$account_type = $query['account_type'] ?? '';
		$account_status = $query['account_status'] ?? '';
		$check_auth = $query['check_auth'] ?? false;

		foreach( $accounts as $id => $account ){

			if( empty( $id ) ){

				continue;
			}

			$type = $account['account_type'] ?? false;
			$status = $account['active'] ?? false;

			if( !empty( $account_type ) && ( !$type || $account_type !== $type ) ){

				unset( $accounts[ $id ] );
				continue;
			}

			if( !empty( $account_status ) && (bool)$status !== (bool)$account_status ){

				unset( $accounts[ $id ] );
				continue;
			}

			if( $check_auth && isset( $account['need_auth'] ) && $account['need_auth'] ){

				$auth_status = Settings::getInstance()->get_auth_status( $account['account_type'], $account['id'] );
				if( !$auth_status ){

					unset( $accounts[ $id ] );
					continue;
				}
			}

			$validate = $this->filter_account( $account, $query );
			if( !$validate ){

				unset( $accounts[ $id ] );
				continue;
			}

			$accounts[ $id ]['id'] = $id;
		}

		return $accounts;
	}

	/**
	 * Return account data
	 *
	 * @param string $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_account_data( $account_id ){

		$accounts = $this->_get_accounts();

		return isset( $accounts[ $account_id ] ) && is_array( $accounts[ $account_id ] ) ? $accounts[ $account_id ] : array();
	}

	/**
	 * Return account data
	 *
	 * @param string $account_id
	 * @param array $data
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function update_account_data( $account_id, $data ){

		$accounts = $this->_get_accounts();

		$accounts[ $account_id ] = $data;

		$this->_update_accounts( $accounts );
	}

	/**
	 * Delete account
	 *
	 * @param string $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function delete_account( $account_id ){

		$accounts = $this->_get_accounts();
		foreach ($accounts as $k => $account) {

			if ( (int)$account['id'] == (int)$account_id) {

				unset($accounts[$k]);

				$this->_update_accounts( $accounts );
				return true;
			}
		}

		return false;
	}
}
