<?php
/**
 * Class MEC_Social_Poster Core
 *
 * @since           1.0.0
 */

namespace MEC_Social_Poster;

use MEC_Social_Poster\History\History;

/**
 * Core Class in plugin
 */
final class Base {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public static $version = '1.0.0';

	/**
	 * Session instance
	 *
	 * @since 1.0.0
	 *
	 * @var bool
	 */
	protected static $instance;

	/**
	 * @since 1.0.0
	 *
	 * Constructor
	 */
	public function __construct() {

		$this->define();
		$this->includes();
		$this->init_hooks();
		$this->admin();
		$this->enqueue_scripts();
	}

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @return self()
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Set Constants
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function define() {

		define( 'MEC_SOCIAL_POSTER_PD', plugin_dir_path( MEC_SOCIAL_POSTER_FILE ) );
		define( 'MEC_SOCIAL_POSTER_PDI', plugin_dir_path( MEC_SOCIAL_POSTER_FILE ) . 'src/' );
		define( 'MEC_SOCIAL_POSTER_PU_JS', plugins_url( 'assets/js/', MEC_SOCIAL_POSTER_FILE ) );
		define( 'MEC_SOCIAL_POSTER_PU_CSS', plugins_url( 'assets/css/', MEC_SOCIAL_POSTER_FILE ) );
		define( 'MEC_SOCIAL_POSTER_PU_IMG', plugins_url( 'assets/img/', MEC_SOCIAL_POSTER_FILE ) );
		define( 'MEC_SOCIAL_POSTER_PU_FONTS', plugins_url( 'assets/fonts/', MEC_SOCIAL_POSTER_FILE ) );
		define( 'MEC_SOCIAL_POSTER_TEMPLATES', plugin_dir_path( MEC_SOCIAL_POSTER_FILE ) . 'templates/' );

		define( 'MEC_SOCIAL_POSTER_VERSION', self::$version );

	}

	/**
	 * Include Files
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function includes() {

		require_once MEC_SOCIAL_POSTER_PDI . 'functions.php';
	}

	/**
	 * Include Files If is Admin
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function admin() {

		if ( !is_admin() ) {

			return;
		}

		Admin::getInstance()->init();
	}

	/**
	 * @since 1.0.0
	 *
	 * Load Plugin Text Domain
	 */
	public function load_plugin_textdomain() {

		load_plugin_textdomain(
			MECSPTEXTDOMAIN,
			false,
			dirname( plugin_basename( MEC_SOCIAL_POSTER_FILE ) ) . '/languages'
		);
	}

	/**
	 * Register actions enqueue scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue' ), 99 );
	}

	/**
	 * Enqueue style and scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function enqueue(){

		$vars = array(
			'ajaxurl' => admin_url('admin-ajax.php'),
			'title' => array(
				'needauthentication' => __('Needs authentication', MECSPTEXTDOMAIN),
				'authenticated' => __('Authentication successful', MECSPTEXTDOMAIN),
                'no_account_selected' => __('No accounts selected', MECSPTEXTDOMAIN),
			),
		);

		wp_register_script( 'mec-social-poster-repost-script', MEC_SOCIAL_POSTER_PU_JS.'/repost.js', null, '1.0.0' );
        wp_localize_script('mec-social-poster-repost-script', 'MEC_SP_VARS', $vars);
	}

	/**
	 * Add Hooks - Actions and Filters
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init_hooks() {

		add_action( 'init', array( $this, 'init' ) );
		add_action( 'init', array( $this, 'register_taxonomies' ) );

		History::getInstance()->init();
		$this->schedule_init();
		SocialController::getInstance()->init();
		EventSocialFESMetaBox::getInstance()->init();

		register_deactivation_hook( MEC_SOCIAL_POSTER_FILE, array( $this, 'remove_schedules' ) );

		do_action( 'mec_social_poster_init' );
	}

	/**
	 * Init
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init() {

		$this->load_plugin_textdomain();
	}

	/**
	 * Get MEC_Main
	 *
	 * @since 1.0.0
	 *
	 * @return \MEC_Main
	 */
	public static function get_main() {

		global $MEC_Main;
		if ( !is_a( $MEC_Main, \MEC_main::class ) ) {

			$MEC_Main = new \MEC_Main();
		}

		return $MEC_Main;
	}

	/**
	 * Remove Schedules
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function remove_schedules(){

		wp_clear_scheduled_hook( 'mec_social_poster_cron_hook_publish_on_socials' );
	}

	/**
	 * Schedule init
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function schedule_init(){

		add_filter('cron_schedules', function ($schedules) {

			$schedules['10m'] = array(
				'interval' => 600,
				'display' => esc_html__('Every 10 Minutes')
			);

			return $schedules;
		});

		// wp_clear_scheduled_hook( 'mec_social_poster_cron_hook_publish_on_socials' );
		if (!wp_next_scheduled('mec_social_poster_cron_hook_publish_on_socials')) {

			wp_schedule_event(time(), '10m', 'mec_social_poster_cron_hook_publish_on_socials');
		}

		add_action( 'mec_social_poster_cron_hook_publish_on_socials', array( $this, 'cron_hook_publish_on_socials' ) );
	}

	/**
	 * Run social poster
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function cron_hook_publish_on_socials(){

		SocialController::getInstance()->publish_on_socials();
	}

	/**
	 * Register Taxonomies
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function register_taxonomies(){

		$plural_label = __( 'Social Tags', MECSPTEXTDOMAIN );
		$singular_label = __( 'Social Tag', MECSPTEXTDOMAIN );
		register_taxonomy(
			'mec_social_tag',
			'mec-events',
			array(
				'label'=> $plural_label,
				'labels'=>array(
					'name'=> $plural_label,
					'singular_name'=> $singular_label,
					'all_items'=>sprintf(__('All %s', MECSPTEXTDOMAIN), $plural_label),
					'edit_item'=>sprintf(__('Edit %s', MECSPTEXTDOMAIN), $singular_label),
					'view_item'=>sprintf(__('View %s', MECSPTEXTDOMAIN), $singular_label),
					'update_item'=>sprintf(__('Update %s', MECSPTEXTDOMAIN), $singular_label),
					'add_new_item'=>sprintf(__('Add New %s', MECSPTEXTDOMAIN), $singular_label),
					'new_item_name'=>sprintf(__('New %s Name', MECSPTEXTDOMAIN), $singular_label),
					'popular_items'=>sprintf(__('Popular %s', MECSPTEXTDOMAIN), $plural_label),
					'search_items'=>sprintf(__('Search %s', MECSPTEXTDOMAIN), $plural_label),
					'back_to_items'=>sprintf(__('← Back to %s', MECSPTEXTDOMAIN), $plural_label),
					'not_found'=>sprintf(__('no %s found.', MECSPTEXTDOMAIN), strtolower($plural_label)),
				),
				'rewrite'=>array('slug'=>'events-social-tag'),
				'public'=>false,
				'show_ui'=>true,
				'hierarchical'=>false,
			)
		);

		register_taxonomy_for_object_type('mec_social_tag', 'mec-events');
	}
}
