<?php

namespace MEC_Social_Poster;

use MEC\Singleton;
use MEC\Events\Event;
use MEC_Social_Poster\Accounts\Accounts;
use MEC_Social_Poster\Settings\Settings;

/**
 * @since 1.0.0
 */
class EventSocial extends Singleton {

	/**
	 * Return Event
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return \MEC\Events\Event
	 */
	public function get_event( $event_id ){

		return new Event( $event_id );
	}

	/**
	 * Return social post title
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_title( $event_id ){

		return get_the_title( $event_id );
	}

	/**
	 * Return social post content
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_message( $event_id ){

		add_filter('excerpt_more', array( __CLASS__, 'remove_more_link' ) );
		$excerpt = get_the_excerpt( $event_id );
		remove_filter('excerpt_more', array( __CLASS__, 'remove_more_link' ) );

		return $excerpt;
	}

	/**
	 * Return event link
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_event_link( $event_id ){

		return get_the_permalink( $event_id );
	}


	/**
	 * Return thumbnail url
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_thumbnail_url( $event_id ){

		return get_the_post_thumbnail_url(
			$event_id,
			array(
				500,
				500
			)
		);
	}

	/**
	 * Return hashtags
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_hashtags( $event_id ){

		$hashtags = wp_get_post_terms( $event_id, 'mec_social_tag' );
		foreach( $hashtags as $k => $hashtag ){

			$hashtags[ $k ] = str_replace( ' ', '_', $hashtag->name );
		}

		return $hashtags;
	}

	/**
	 * Return
	 *
	 * @param int $event_id
	 * @param string $taxonomy
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_terms_ids( $event_id, $taxonomy ){

		return wp_get_post_terms( $event_id, $taxonomy );
	}

	/**
	 * Return start datetime
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_start_datetime( $event_id ){

		$datetime = $this->get_event( $event_id )->get_datetime();

		return date( 'Y-m-d\TH:i:s', $datetime['start']['timestamp'] ?? '' );
	}

	/**
	 * Return duration datetime
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_duration( $event_id ){

		$datetime = $this->get_event( $event_id )->get_datetime();

		$diff = $datetime['end']['timestamp'] - $datetime['start']['timestamp'];
		$days = floor( $diff / 86400 );
		$hours = floor( ( $diff % 86400 ) / 3600 );
		$minutes = floor( ( $diff % 3600 ) / 60 );

		return "{$days} days {$hours} hours {$minutes} minutes";
	}

	/**
	 * Return repeat data
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return array|false
	 */
	public function get_repeat_data( $event_id ){

		$data = get_post_meta( $event_id, 'mec_date', true );

		return is_array( $data ) && isset( $data['repeat']['status'] ) && $data['repeat']['status'] ? $data : false ;
	}

	/**
	 * Return social post id
	 *
	 * @param int $event_id
	 * @param string $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return string|false
	 */
	public function get_social_post_id( $event_id, $account_id ){

		$social_post_id = get_post_meta( $event_id, "mec_social_{$account_id}_post_id", true );

		return !empty( $social_post_id ) ? $social_post_id : false;
	}

	/**
	 * Update social post id
	 *
	 * @param int $event_id
	 * @param string $account_id
	 * @param string $social_post_id
	 *
	 * @since 1.0.0
	 *
	 * @return string|false
	 */
	public function update_social_post_id( $event_id, $account_id, $social_post_id ){

		return update_post_meta( $event_id, "mec_social_{$account_id}_post_id", $social_post_id );
	}

	/**
	 * Return location detail
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_location( $event_id ){

		$location_id = get_post_meta( $event_id, 'mec_location_id', true );

		return array(
			'latitude' => get_term_meta( $location_id, 'latitude', true ),
			'longitude' => get_term_meta( $location_id, 'longitude', true ),
		);
	}

	/**
	 * Return event options
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_event_options( $event_id ){

		$options = get_post_meta( $event_id, 'mec_social_poster', true );

		return is_array( $options ) ? $options : array();
	}

	/**
	 * Return publish status
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function get_publish_status( $event_id ){

		return (bool)get_post_meta( $event_id, 'mec_social_poster_status', true );
	}

	/**
	 * Return global schedules
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_global_schedules(){

		$options = Settings::getInstance()->get_settings( 'general', array() );

		return isset( $options['schedules'] ) ? $options['schedules'] : array();
	}

	/**
	 * Return schedules
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_schedules( $event_id ){

		$options = $this->get_event_options( $event_id );
		$global_accounts_status = isset( $options['global_accounts_status'] ) ? (bool)$options['global_accounts_status'] : true;

		$global_options = Settings::getInstance()->get_settings( 'general', array() );
		$global_schedules = isset( $global_options['schedules'] ) ? $global_options['schedules'] : array();

		$schedules = array();
		if( !$global_accounts_status ){

			$event_schedules = isset( $options['schedules'] ) ? $options['schedules'] : array();
			foreach( $event_schedules as $schedule_id => $schedule ){

				if( 'global' === $schedule['publish_on_type'] ){

					if( isset( $global_schedule[ $schedule_id ] ) ){

						$schedule[ 'publish_on_type' ] = $global_schedule[ $schedule_id ]['publish_on_type'] ?? 'now';
						$schedule[ 'publish_on' ] = $global_schedule[ $schedule_id ]['publish_on'] ?? array();
					}else{

						$schedule[ 'publish_on_type' ] = 'now'; //TODO: get from account type settings
						$schedule[ 'publish_on' ] = array();
					}
				}

				$schedules[ $schedule_id ] = $schedule;
			}
		}else{

			$global_accounts = isset( $global_options['accounts'] ) && is_array( $global_options['accounts'] ) ? $global_options['accounts'] : array();
			foreach( $global_accounts as $schedule_id ){

				if( isset( $global_schedules[ $schedule_id ] ) ){

					$schedules[ $schedule_id ] = $global_schedules[ $schedule_id ];
				}
			}
		}

		return $schedules;
	}

	/**
	 * Can publish event to social account
	 *
	 * @param int $event_id
	 * @param int $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return boolean|\WP_Error
	 */
	public function can_send_event_to_social_account( $event_id, $account_id ){

		$account_settings = Accounts::getInstance()->get_account_data( $account_id );

		$taxonomies = array(
			'mec_category',
			'mec_organizer',
			'mec_speaker',
			'mec_location',
		);

		foreach( $taxonomies as $tax ){

			if( !taxonomy_exists( $tax ) ){

				continue;
			}

			$account_terms = isset( $account_settings[ $tax ] ) && is_array( $account_settings[ $tax ] ) ? $account_settings[ $tax ] : array();
			$event_terms = wp_get_object_terms( $event_id, $tax, array( 'fields' => 'ids' ) );
			$intersect = array_intersect( $account_terms, $event_terms );
			if( !empty( $account_terms ) && 0 == count( $intersect ) ){

				$taxonomy = get_taxonomy( $tax );
				return new \WP_Error(
					$taxonomy->label,
					sprintf(
						__( 'The event does not have the required %1s', MECSPTEXTDOMAIN ),
						$taxonomy->label
					)
				);
			}
		}

		return true;
	}

	/**
	 * Send event to social account without check
	 *
	 * @param int $event_id
	 * @param int $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function send_event_to_account( $event_id, $account_id, $save_social_post_id = true ){

		if( !$event_id ){

			return false;
		}

		$account_settings = Accounts::getInstance()->get_account_data( $account_id );
		$account_type = $account_settings['account_type'];

		$sended = false;
		switch( $account_type ){
			case 'facebook':
				$data = array(
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
				);

				$sended = \MEC_Social_Poster\APIs\Facebook\Post::getInstance()->publish_post( $account_id, $data );

				break;
			case 'linkedin':
				$data = array(
					'title' => $this->get_title( $event_id ),
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
				);

				$sended = \MEC_Social_Poster\APIs\Linkedin\Post::getInstance()->publish_post( $account_id, $data );

				break;

			case 'meetup':

				$location = $this->get_location( $event_id );

				$data = array(
					'title' => $this->get_title( $event_id ),
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
					'start_datetime' => $this->get_start_datetime( $event_id ),
					'duration' => $this->get_duration( $event_id ),
					'group_id' => $account_settings['meetup_account_id'] ?? '',
					'latitude' => $location['latitude'],
					'longitude' => $location['longitude'],
					'repeat_data' => $this->get_repeat_data( $event_id ),
				);

				$sended = \MEC_Social_Poster\APIs\Meetup\Post::getInstance()->publish_post( $account_id, $data );

				break;
			case 'twitter':

				$data = array(
					'title' => $this->get_title( $event_id ),
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
				);

				$sended = \MEC_Social_Poster\APIs\Twitter\Post::getInstance()->publish_post( $account_id, $data );

				break;
			case 'instagram':

				$data = array(
					'title' => $this->get_title( $event_id ),
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
				);

				$sended = \MEC_Social_Poster\APIs\Instagram\Post::getInstance()->publish_post( $account_id, $data );

				break;
			default:

				$data = array(
					'title' => $this->get_title( $event_id ),
					'message' => $this->get_message( $event_id ),
					'link' => $this->get_event_link( $event_id ),
					'thumbnail' => $this->get_thumbnail_url( $event_id ),
					'hashtags' => $this->get_hashtags( $event_id ),
				);

				$sended = apply_filters( 'mec_social_poster_send_event_to_account', null, $event_id, $account_id, $account_type, $data );

				break;
		}

		do_action( 'mec_social_poster_published_on_social', $event_id, $account_id, $data, $sended );

		if( $save_social_post_id && $sended && !is_wp_error( $sended ) ){

			$this->update_social_post_id( $event_id, $account_id, $sended );
		}

		return $sended;
	}

	/**
	 * Remove more link
	 *
	 * @param string $more
	 *
	 * @since 1.0.1
	 *
	 * @return string
	 */
	public static function remove_more_link( $more ){

		return '';
	}
}
