<?php

namespace MEC_Social_Poster\History;

use MEC\Singleton;
use MEC_Social_Poster\Accounts\Accounts;

/**
 * @since 1.0.0
 */
class History extends Singleton {

	/**
	 * Init
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init() {

		add_action( 'init', array( __CLASS__, 'register_post_type' ) );
	}

	/**
	 * Register post type
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function register_post_type() {

		$labels = array(
			'name'               => esc_html__( 'MEC Histories', MECSPTEXTDOMAIN ),
			'all_items'          => esc_html__( 'Histories', MECSPTEXTDOMAIN ),
			'singular_name'      => esc_html__( 'History', MECSPTEXTDOMAIN ),
			'add_new'            => esc_html__( 'Add New', MECSPTEXTDOMAIN ),
			'add_new_item'       => esc_html__( 'Add New History', MECSPTEXTDOMAIN ),
			'edit_item'          => esc_html__( 'Edit History', MECSPTEXTDOMAIN ),
			'new_item'           => esc_html__( 'New History', MECSPTEXTDOMAIN ),
			'view_item'          => esc_html__( 'View History', MECSPTEXTDOMAIN ),
			'search_items'       => esc_html__( 'Search History', MECSPTEXTDOMAIN ),
			'not_found'          => esc_html__( 'No History found', MECSPTEXTDOMAIN ),
			'not_found_in_trash' => esc_html__( 'No History found in Trash', MECSPTEXTDOMAIN ),
		);

		$args = array(
			'public'              => false,
			'show_ui'             => false,
			'show_in_menu'        => false,
			'show_in_admin_bar'   => false,
			'has_archive'         => false,
			'exclude_from_search' => true,
			'publicly_queryable'  => false,
			'show_in_rest'        => false,
			'show_in_menu'        => false,
			'labels'              => $labels,
		);

		$args = apply_filters( 'mec_history_register_post_type_args', $args );

		register_post_type( 'mec-history', $args );
	}

	/**
	 * Add new history
	 *
	 * @param int $event_id
	 * @param int $account_id
	 * @param string $type
	 * @param array $history_data
	 *
	 * @since 1.0.0
	 *
	 * @return int|\WP_Error
	 */
	public function insert_log( $event_id, $account_id, $type, $history_data  ){

		$account_data = Accounts::getInstance()->get_account_data( $account_id );
		$account_type = $account_data['account_type'] ?? '';

		$data = $history_data['data'] ?? '';
		$data = maybe_serialize( $data );

		$args = array(
			'post_title' => "e:{$event_id}-a:{$account_id}-t:{$type}",
			'post_type' => 'mec-history',
			'post_status' => 'publish',
			'post_content' => $data,
			'meta_input' => array(
				'mec_event_id' => $event_id,
				'mec_account_id' => $account_id,
				'mec_account_type' => $account_type,
				'mec_history_type' => $type,
				'mec_result' => $history_data['result'] ?? '',
				'mec_result_object' => $history_data['result_object'] ?? '',
			),
		);

		return wp_insert_post( $args );
	}

	/**
	 * Return query args
	 *
	 * @param array $
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function parse_args( $query = array() ){

		$args = array(
			'post_type' => 'mec-history',
			'post_status' => 'publish',
		);

		$args = wp_parse_args( $query, $args );

		if( isset( $query['event_id'] ) && $query['event_id'] ){

			$args['meta_query']['event_id'] = array(
				'key' => 'mec_event_id',
				'value' => $query['event_id'],
				'compare' => '=',
			);
		}

		if( isset( $query['account_id'] ) && $query['account_id'] ){

			$args['meta_query']['account_id'] = array(
				'key' => 'mec_account_id',
				'value' => $query['account_id'],
				'compare' => '=',
			);
		}

		if( isset( $query['history_type'] ) && $query['history_type'] ){

			$args['meta_query']['history_type'] = array(
				'key' => 'mec_history_type',
				'value' => $query['history_type'],
				'compare' => '=',
			);
		}

		return $args;
	}

	/**
	 * Return logs
	 *
	 * @param array $query
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function get_logs( $query = array() ){

		$args = $this->parse_args( $query );

		$logs = get_posts( $args );
		foreach( $logs as $k => $log ){

			$history_id = $log->ID;
			$logs[ $k ] = array(
				'id' => $history_id,
				'title' => $log->post_title,
				'data' => maybe_unserialize( $log->post_content ),
				'event_id' => get_post_meta( $history_id, 'mec_event_id', true ),
				'account_id' => get_post_meta( $history_id, 'mec_account_id', true ),
				'history_type' => get_post_meta( $history_id, 'mec_history_type', true ),
				'result' => get_post_meta( $history_id, 'mec_result', true ),
				'result_object' => get_post_meta( $history_id, 'mec_result_object', true ),
				'creation_date' => $log->post_date,
			);
		}

		return $logs;
	}

	/**
	 * Delete log
	 *
	 * @param int $id
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function delete_log( $id ){

		if( 'mec-history' !== get_post_type( $id ) ){

			return;
		}

		wp_delete_post( $id );
	}

	/**
	 * Return Color
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_color(){

		return sprintf('#%06X', mt_rand(0, 0xFFFFFF));
	}

	/**
	 * Return chart js data config
	 *
	 * @param array $query_args
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_chart_data( $query_args = array(), $report_type = 'daily', $date_query_after_value = '', $date_query_before_value = '' ){

		$query_args = wp_parse_args(
			$query_args,
			array(
				'posts_per_page' => -1,
			)
		);

		switch( $report_type ){
			case 'monthly':
				$date_format = 'Y-m';
				break;
			case 'daily':
				$date_format = 'Y-m-d';
				break;
			case 'hourly':
				$date_format = 'Y-m-d H:00';
				break;
		}

		$accounts = Accounts::getInstance()->get_accounts( array() );

		$groups = array();
		$logs_by_accounts_group = array();

		$args = $this->parse_args( $query_args );
		$args['date_query'] = array(
			'column'  => 'post_date',
			'after'   => !empty( $date_query_after_value ) ? $date_query_after_value : '- 30 days',
			'before'  => !empty( $date_query_before_value ) ? $date_query_before_value : 'now',
		);
		$logs = get_posts( $args );

		foreach( $logs as $k => $log ){

			$log_id = $log->ID;
			$timestamp = strtotime( $log->post_date );
			$group_id = date_i18n( $date_format, $timestamp );
			$account_id = get_post_meta( $log_id, 'mec_account_id', true );

			$logs_by_accounts_group[ $account_id ][ $group_id ][ $log_id ] = $log_id;
			$groups[ $group_id ] = $group_id;
		}

		sort( $groups );

		$datasets = array();
		foreach( $logs_by_accounts_group as $account_id => $group_logs ){

			$account_title = $accounts[ $account_id ]['title'] ?? __( 'Undefined', MECSPTEXTDOMAIN );

			foreach( $groups as $group ){

				$data[ $group ] = isset( $group_logs[ $group ] ) && is_array( $group_logs[ $group ] ) ? count( $group_logs[ $group ] ) : 0;
			}

			$datasets[] = array(
				'label' => $account_title,
				'data' => array_values($data),
				'borderColor' =>  $this->get_color(),
				'backgroundColor' => 'rgba(0, 0, 0, 0.0)',
			);
		}


		$data = array(
			'labels' => $groups,
			'datasets' => $datasets,
		);

		return $data;
	}
}
