<?php

namespace MEC_Social_Poster\Settings;

use MEC\Singleton;
use MEC_Social_Poster\Accounts\Accounts;

/**
 * @since 1.0.0
 */
class Settings extends Singleton {

	/**
	 * Return saved settings
	 *
	 * @param string $key
	 *
	 * @since 1.0.0
	 *
	 * @return mixed
	 */
	public function get_settings( $key, $default = '' ){

		$option_key = "mec_social_poster_{$key}";

		return get_option( $option_key, $default );
	}

	/**
	 * Update settings
	 *
	 * @param string $key
	 * @param mixed $value
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function update_settings( $key, $value ){

		$option_key = "mec_social_poster_{$key}";

		update_option( $option_key, $value );

		return true;
	}

	/**
	 * Return auth url
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_auth_url( $account_type ){

		$admin_url = admin_url( 'admin.php' );
		$args = array(
			'page'=> "mec-social-poster",
			'cmd' => "{$account_type}_auth",
			'class' => $account_type
		);

		return add_query_arg( $args , $admin_url );
	}

	/**
	 * Return auth callback url
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_auth_callback_url( $account_type ){

		$admin_url = admin_url( 'admin.php' );
		$args = array(
			'page'=> "mec-social-poster",
			'tab' => "mec-{$account_type}",
			'cmd' => "{$account_type}_callback",
			'class' => $account_type,
		);

		// $admin_url = str_replace( 'http:', 'https:', $admin_url );//TODO: Remove

		return add_query_arg( $args , $admin_url );
	}

	/**
	 * Return auth settings
	 *
	 * @param string $accounts_type
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_auth_settings( $accounts_type ){

		return get_option("mec_sp_auth_{$accounts_type}", array());
	}

	/**
	 * Update auth settings
	 *
	 * @param string $accounts_type
	 * @param array $settings
	 *
	 * @since 1.0.0
	 *
	 * @return int|null
	 */
	public function update_auth_settings( $accounts_type, $settings ){

		return update_option("mec_sp_auth_{$accounts_type}", $settings);
	}

	/**
	 * Return auth status
	 *
	 * @param string $accounts_type
	 * @param int $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_auth_account_settings( $accounts_type, $account_id ){

		$auth_config = $this->get_auth_settings( $accounts_type );

		return isset($auth_config[ $account_id ]) ? $auth_config[ $account_id ] : array();
	}

	/**
	 * Update auth account settings
	 *
	 * @param string $accounts_type
	 * @param int $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function update_auth_account_settings( $accounts_type, $account_id, $settings ){

		$auth_config = $this->get_auth_settings( $accounts_type );

		$auth_config[ $account_id ] = $settings;

		return $this->update_auth_settings( $accounts_type, $auth_config );
	}

	/**
	 * Return auth status
	 *
	 * @param string $accounts_type
	 * @param int $account_id
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function get_auth_status( $accounts_type, $account_id ){

		$auth_config = $this->get_auth_settings( $accounts_type );

		$is_expired = isset($auth_config[ $account_id ]) && isset($auth_config[ $account_id ]['expire_at']) && $auth_config[ $account_id ]['expire_at'] < time() ? true: false;

		$auth_status = isset($auth_config[ $account_id ]) && isset($auth_config[ $account_id ]['status'])?$auth_config[ $account_id ]['status']:false;
		if( $is_expired == true ){

			$auth_status = false;
		}

		return $auth_status;
	}

	/**
	 * Return AccessToken
	 *
	 * @param string $account_type
	 * @param int $account_id
	 * @param string $token_type
	 *
	 * @since 1.0.0
	 *
	 * @return string|false
	 */
	public function get_access_token( $account_type, $account_id, $token_type = 'primary' ){

		$access_token = false;

		$auth_status = $this->get_auth_status( $account_type, $account_id );
		if( $auth_status ){

			$auth_config = $this->get_auth_account_settings( $account_type, $account_id );
			$account_settings = Accounts::getInstance()->get_account_data( $account_id );

			if( 'primary' === $token_type ){

				$key = 'access_token';
			}elseif( 'facebook' === $account_settings['account_type'] && 'group' === $account_settings['fb_account_type'] ){

				$key = 'access_token';
			}else{

				$key = 'account_access_token';
			}

			$access_token = isset( $auth_config[ $key ] ) ? $auth_config[ $key ] : false;
		}

		return $access_token;
	}

}