<?php

namespace MEC_Social_Poster;

use MEC\Singleton;
use MEC\Events\EventsQuery;
use MEC_Social_Poster\Accounts\Accounts;
use MEC_Social_Poster\History\Histories;
use MEC_Social_Poster\History\History;
use MEC\Libraries\FlushNotices;

/**
 * @since 1.0.0
 */
class SocialController extends Singleton {

	/**
	 * Init
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init(){

		add_action( 'save_post', array( __CLASS__, 'publish_events_handler' ), 99999 );
		add_action( 'wp_ajax_mec_social_poster_repost', array( $this, 'ajax_repost_handler' ) );

		add_action( 'mec_social_poster_published_on_social', array( __CLASS__, 'add_publish_on_history' ), 10, 4 );
	}

	/**
	 * Add publish on history
	 *
	 * @param int $event_id
	 * @param int $account_id
	 * @param array $data
	 * @param string|int|\WP_Error $sended
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function add_publish_on_history( $event_id, $account_id, $data, $sended ){

		$type = '';
		$history = array(
			'data' => $data,
			'result' => is_wp_error( $sended ) ? 0 : 1,
			'result_object' => $sended
		);

		return History::getInstance()->insert_log( $event_id, $account_id, $type, $history );
	}

	/**
	 * Ajax repost handler
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function ajax_repost_handler(){

		$data = array();

		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( $_POST['nonce'] ) : false;
		$action = 'mec_social_poster_nf_repost';

		if( !$nonce || !wp_verify_nonce( $nonce, $action ) ){
			return;
		}

		$event_id = isset( $_POST['event_id'] ) ? absint( $_POST['event_id'] ) : false;
		$accounts = isset( $_POST['accounts'] ) && is_array( $_POST['accounts'] ) ? $_POST['accounts'] : array();
		$disable_check = isset( $_POST['check'] ) ? (bool)$_POST['check'] : false;
		foreach( $accounts as $account_id ){

			$account_data = Accounts::getInstance()->get_account_data( $account_id );
			$account_title = $account_data['title'];

			$can_send = EventSocial::getInstance()->can_send_event_to_social_account( $event_id, $account_id );
			if( !$disable_check && ( is_wp_error($can_send) || !$can_send ) ){

				$data['error_messages'][ $account_id ] = sprintf(
					__( 'This event is not allowed to be posted on "%1s" %2s', MECSPTEXTDOMAIN ),
					$account_title,
					is_wp_error($can_send) ? '<br>' . $can_send->get_error_message() : ''
				);
				continue;
			}

			$sended = EventSocial::getInstance()->send_event_to_account( $event_id, $account_id );
			if( is_wp_error( $sended ) ){

				error_log( $sended->get_error_message() . "$event_id $account_id" );

				$data['error_messages'][ $account_id ] = sprintf(
					__( 'Publish on "%1s" encountered an error', MECSPTEXTDOMAIN ) . '<br>' . $sended->get_error_message(),
					$account_title
				);
			}elseif( $sended ){

				add_post_meta( $event_id, "mec_social_poster_{$account_id}_repost", $sended );
				$data['success_messages'][ $account_id ] = sprintf(
					__( 'Published on "%1s"', MECSPTEXTDOMAIN ),
					$account_title
				);
			}else{

				$data['error_messages'][ $account_id ] = sprintf(
					__( 'Publish on "%1s" encountered an error', MECSPTEXTDOMAIN ) . '<br>',
					$account_title
				);
			}
		}

		wp_send_json_success( $data );
	}

	/**
	 * Can auto send event to social
	 *
	 * @param int $event_id
	 * @param string $social
	 *
	 * @since 1.0.0
	 *
	 * @return boolean
	 */
	public static function can_auto_send_event_to_social_account( $event_id, $account_id ){

		$social_post_id = EventSocial::getInstance()->get_social_post_id( $event_id, $account_id );
		if( !empty( $social_post_id ) ){

			return false;
		}

		$can = EventSocial::getInstance()->can_send_event_to_social_account( $event_id, $account_id );
		if( is_wp_error( $can ) || !$can ){

			return $can;
		}

		return true;
	}

	/**
	 * Publish post handler
	 *
	 * @param int $event_id
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function publish_events_handler( $event_id ){

		if(!( isset($_POST['mec']['mec_social_poster_nonce']) || isset($_POST['mec_social_poster_nonce']))) return;
        if(
			!(
				( isset($_POST['mec']['mec_social_poster_nonce']) && wp_verify_nonce(sanitize_text_field($_POST['mec']['mec_social_poster_nonce']), 'mec_social_poster_data') )
				||
				( isset($_POST['mec_social_poster_nonce']) && wp_verify_nonce(sanitize_text_field($_POST['mec_social_poster_nonce']), 'mec_social_poster_data') )
			)
			) {
				return;
		}
        if(defined('DOING_AUTOSAVE') and DOING_AUTOSAVE) return;
		if( 'mec-events' !== get_post_type( $event_id ) ) return;

		$status = EventSocial::getInstance()->get_publish_status( $event_id );
		if( !$status ){

			return;
		}

		$schedule_accounts = EventSocial::getInstance()->get_schedules( $event_id );

		$saved_schedules_meta = get_post_meta( $event_id, 'mec_social_poster_schedule', false );

        $schedules = array();

		$added_timestamps = array();

		foreach( $schedule_accounts as $schedule_account ){

			$account_id = $schedule_account['account_id'];
			$publish_on_type = $schedule_account['publish_on_type'] ?? 'now';
			$publish_on = $schedule_account['publish_on'] ?? array();

			$can_send = self::can_auto_send_event_to_social_account( $event_id, $account_id );
			if( is_wp_error( $can_send ) || !$can_send ){

				if( is_wp_error( $can_send ) ){

					$account_data = Accounts::getInstance()->get_account_data( $account_id );

					$message = sprintf(
						__( 'MEC Social Auto Poster - "%1s" : %2s', MECSPTEXTDOMAIN ),
						$account_data['title'],
						$can_send->get_error_message()
					);

					FlushNotices::getInstance()->add_flash_notice( $message, 'error' );
				}
				continue;
			}

			$publish_timestamp = false;
			$current_timestamp = current_time( 'timestamp' );
			switch( $publish_on_type ){
				case 'now':

					$sended = EventSocial::getInstance()->send_event_to_account( $event_id, $account_id );

					break;
				case 'x_hours_after_publish':

					$value = $publish_on['value'] ?? 0;
					if( !$value ){

						continue 2;
					}

					$publish_timestamp = $current_timestamp + ( $value * 3600 );

					break;
				case 'x_days_after_publish':

					$value = $publish_on['value'] ?? 0;
					if( !$value ){

						continue 2;
					}

					$publish_timestamp = $current_timestamp + ( $value * 86400 );

					break;
				case 'custom':

					$publish_on_date = isset( $publish_on['date'] ) ? $publish_on['date'] : current_time( 'Y-m-d' );
					$publish_on_time = isset( $publish_on['time'] ) ? $publish_on['time'] : array();

					$hour = sprintf( '%02d', $publish_on_time['hour'] ?? '' );
					$minutes = sprintf( '%02d', $publish_on_time['minutes'] ?? '' );
					$ampm = $publish_on_time['ampm'];

					$datetime = "{$publish_on_date} {$hour}:{$minutes} {$ampm}";

					$publish_timestamp = strtotime( $datetime );

					break;

			}

			if( $publish_timestamp ){

				$schedules[ $account_id ] = array(
					'send_in_timestamp' => $publish_timestamp,
					'account_id' => $account_id,
				);
			}
		}



        update_post_meta( $event_id, 'mec_social_poster_schedules_settings', $schedules );

        $timestamps = array_column( $schedules, 'send_in_timestamp' );
        foreach( $timestamps as $timestamp ){

            $sk = array_search( $timestamp, $saved_schedules_meta, false );
            if( false === $sk ){

                $added_timestamps[ $timestamp ] = $timestamp;
                add_post_meta( $event_id, 'mec_social_poster_schedule', $timestamp );
            }
        }

        foreach( $saved_schedules_meta as $ssm_timestamp ){

			if( !in_array( $ssm_timestamp, $timestamps, false ) ){

				delete_post_meta( $event_id, 'mec_social_poster_schedule', $ssm_timestamp );
			}
        }

		return true;
	}

	/**
     * Return schedules
     *
     * @param int $timestamp
     * @param array $schedules
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function filter_schedules( $timestamp, $schedules ){

        foreach( $schedules as $k => $schedule ){

            if( !isset( $schedule['send_in_timestamp'] ) || $schedule['send_in_timestamp'] != $timestamp ){

                unset( $schedules[ $k ] );
            }
        }

        return $schedules;
    }

	/**
	 * Return events schedules for publish
	 *
	 * @param int|string $before_timestamp
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_events_schedules_for_publish_on_socials( $before_timestamp = 'now' ){

		$before_timestamp = 'now' == $before_timestamp ? current_time('timestamp') : $before_timestamp;
        $q_args = array(
            'meta_query' => array(
                array(
                    'key' => 'mec_social_poster_schedule',
                    'compare' => '>',
                    'value' => $before_timestamp,
                )
            ),
        );

        $schedules = array();

        $events_ids = EventsQuery::getInstance()->get_events_ids( $q_args );
        foreach( $events_ids as $event_id ){

            $saved_schedules_timestamp = get_post_meta( $event_id, 'mec_social_poster_schedule', false );
            $e_schedules = get_post_meta( $event_id, 'mec_social_poster_schedules_settings', true );
            $event_schedules = array();
            foreach( $saved_schedules_timestamp as $s_timestamp ){

                if( $s_timestamp > $before_timestamp ){
                    continue;
                }

                $event_schedules = array_merge(
                    $event_schedules,
                    $this->filter_schedules( $s_timestamp, $e_schedules )
                );
                $schedule['timestamp'] = $s_timestamp;
            }

            if( !empty( $event_schedules ) ){

                $schedules[ $event_id ] = $event_schedules;
            }
        }

        return $schedules;
	}

	/**
	 * Publish events on socials
	 *
	 * @since 1.0.0
	 *
	 * @return bool
	 */
	public function publish_on_socials(){

		$events_schedules = SocialController::getInstance()->get_events_schedules_for_publish_on_socials();
		foreach( $events_schedules as $event_id => $schedules ){

			$saved_schedules_meta = get_post_meta( $event_id, 'mec_social_poster_schedule', false );
			$saved_event_schedules = get_post_meta( $event_id, 'mec_social_poster_schedules_settings', true );

			foreach( $schedules as $schedule_args ){

				$send_in_timestamp = $schedule_args['send_in_timestamp'] ?? false;
				$account_id = $schedule_args['account_id'] ?? false;
				if( !$account_id ){

					continue;
				}

				$sended = EventSocial::getInstance()->send_event_to_account( $event_id, $account_id );
				if( is_wp_error( $sended ) ){

					error_log( $sended->get_error_message() . "$event_id $account_id" );
				}

				unset( $saved_event_schedules[ $account_id ] );
			}

			update_post_meta( $event_id, 'mec_social_poster_schedules_settings', $saved_event_schedules );

			//remove empty donned
			$timestamps = array_column( $saved_event_schedules, 'send_in_timestamp' );
			foreach( $saved_schedules_meta as $ssm_timestamp ){

				if( !in_array( $ssm_timestamp, $timestamps, false ) ){

					delete_post_meta( $event_id, 'mec_social_poster_schedule', $ssm_timestamp );
				}
			}
		}

		return true;
	}
}
