<?php
/**
 * Gutenberg blocks for MEC single builder.
 *
 * @package MEC_GSB
 */

namespace MEC_GSB\Core\Gutenberg;

use MEC;
use DateTime;
use MEC\Base;
use MEC\SingleBuilder\Widgets\EventSocialShare\EventSocialShare;
use MEC_kses;
use MEC_main;
use stdClass;
use WP_Query;
use MEC_feature_occurrences;


if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Class SingleBlocks.
 */
class SingleBlocks
{

    /**
     * Instance of this class.
     *
     * @since   1.0.0
     *
     * @var     SingleBlocks
     */
    public static $instance;

    /**
     * Provides access to a single instance of a module using the singleton pattern.
     *
     * @return  object
     * @since   1.0.0
     *
     */
    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     *
     * @since 1.0.0
     */
    private function __construct()
    {
        if (is_admin()) {
            return;
        }
        $this->hooks();
    }

    /**
     * Init hooks.
     *
     * @since 1.0.0
     */
    private function hooks()
    {
        add_action('init', array($this, 'register_single_blocks'));
    }

    /**
     * Register mec single blocks.
     *
     * @since 1.0.0
     */
    public function register_single_blocks()
    {

        register_block_type('mec-gsb/event-title', [
            'render_callback' => array($this, 'event_title_render'),
        ]);

        register_block_type('mec-gsb/event-featured-image', [
            'render_callback' => array($this, 'event_featured_image_render'),
        ]);

        register_block_type('mec-gsb/event-content', [
            'render_callback' => array($this, 'event_content_render'),
        ]);

        register_block_type('mec-gsb/event-tags', [
            'render_callback' => array($this, 'event_tags_render'),
        ]);

        register_block_type('mec-gsb/event-count-down', [
            'render_callback' => array($this, 'event_count_down_render'),
        ]);

        register_block_type('mec-gsb/event-local-time', [
            'render_callback' => array($this, 'event_local_time_render'),
        ]);

        register_block_type('mec-gsb/event-category', [
            'render_callback' => array($this, 'event_category_render'),
        ]);

        register_block_type('mec-gsb/event-date-module', [
            'render_callback' => array($this, 'event_date_module_render'),
        ]);

        register_block_type('mec-gsb/event-time-module', [
            'render_callback' => array($this, 'event_time_module_render'),
        ]);

        register_block_type('mec-gsb/event-cost', [
            'render_callback' => array($this, 'event_cost_render'),
        ]);

        register_block_type('mec-gsb/event-banner', [
            'render_callback' => array($this, 'event_banner_render'),
        ]);

        register_block_type('mec-gsb/event-labels', [
            'render_callback' => array($this, 'event_labels_render'),
        ]);

        register_block_type('mec-gsb/event-organizers', [
            'render_callback' => array($this, 'event_organizers_render'),
        ]);

        register_block_type('mec-gsb/event-speakers', [
            'render_callback' => array($this, 'event_speakers_render'),
        ]);

        register_block_type('mec-gsb/event-more-info', [
            'render_callback' => array($this, 'event_more_info_render'),
        ]);

        register_block_type('mec-gsb/event-social-share', [
            'render_callback' => array($this, 'event_social_share_render'),
        ]);

        register_block_type('mec-gsb/event-qr-code', [
            'render_callback' => array($this, 'event_qr_code_render'),
        ]);

        register_block_type('mec-gsb/event-next', [
            'render_callback' => array($this, 'event_next_render'),
        ]);

        register_block_type('mec-gsb/event-hourly-schedule', [
            'render_callback' => array($this, 'event_hourly_schedule_render'),
        ]);

        register_block_type('mec-gsb/event-attendees', [
            'render_callback' => array($this, 'event_attendees_render'),
        ]);

        register_block_type('mec-gsb/event-booking', [
            'render_callback' => array($this, 'event_booking_render'),
        ]);

        register_block_type('mec-gsb/event-breadcrumbs', [
            'render_callback' => array($this, 'event_breadcrumbs_render'),
        ]);

        register_block_type('mec-gsb/event-cancellation', [
            'render_callback' => array($this, 'event_cancellation_render'),
        ]);

        register_block_type('mec-gsb/event-data', [
            'render_callback' => array($this, 'event_data_render'),
        ]);

        register_block_type('mec-gsb/event-map', [
            'render_callback' => array($this, 'event_map_render'),
        ]);

        register_block_type('mec-gsb/event-next-previous', [
            'render_callback' => array($this, 'event_next_previous_render'),
        ]);

        register_block_type('mec-gsb/event-public-download', [
            'render_callback' => array($this, 'event_public_download_render'),
        ]);

        register_block_type('mec-gsb/event-register-button', [
            'render_callback' => array($this, 'event_register_button_render'),
        ]);

        register_block_type('mec-gsb/event-related', [
            'render_callback' => array($this, 'event_related_render'),
        ]);

        register_block_type('mec-gsb/event-weather', [
            'render_callback' => array($this, 'event_weather_render'),
        ]);

        register_block_type('mec-gsb/event-gallery', [
            'render_callback' => array($this, 'event_gallery'),
        ]);

        register_block_type('mec-gsb/event-export', [
            'render_callback' => array($this, 'event_export'),
        ]);

        register_block_type('mec-gsb/event-faq', [
            'render_callback' => array($this, 'event_faq'),
        ]);

        register_block_type('mec-gsb/event-locations', [
            'render_callback' => array($this, 'event_locations'),
        ]);

        register_block_type('mec-gsb/event-sponsors', [
            'render_callback' => array($this, 'event_sponsors'),
        ]);

        register_block_type('mec-gsb/event-trailer-url', [
            'render_callback' => array($this, 'event_trailer_url'),
        ]);

        register_block_type('mec-gsb/event-rsvp', [
            'render_callback' => array($this, 'event_rsvp'),
        ]);

    }

    /**
     * Title block render.
     *
     * @since 1.0.0
     */
    public function event_title_render($attr)
    {
        $content_render = '<div class="event-title-gsb"><h1 class="mec-single-title">' . get_the_title() . '</h1></div>';
        return $content_render;
    }

    /**
     * Event Featured Image block render.
     *
     * @since 1.0.0
     */
    public function event_featured_image_render()
    {
        $image = get_the_post_thumbnail(get_the_ID(), 'single-post-thumbnail');
        $content_render = '<div class="event-image-gsb">' . $image . '</div>';
        return $content_render;
    }

    /**
     * Event Content block render.
     *
     * @since 1.0.0
     */
    public function event_content_render()
    {
        $content = get_the_content('', false, get_the_ID());
        $content = wpautop($content);
        $content = do_shortcode($content);
        $html = '';
        if (empty($content)) {

            $html .= '';
        } else {

            $html .= '<div class="mec-single-event-description mec-events-content">'
                . $content .
                '</div>';
        }

        $content_render = '<div class="event-content-gsb">' . $html . '</p></div>';
        return $content_render;
    }

    /**
     * Event Tags block render.
     *
     * @since 1.0.0
     */
    public function event_tags_render()
    {
        $tags = get_the_tags(get_the_ID());

        $html = '';
        if (!$tags) {
            $html .= '';
        } else {
            $html = '<div class="mec-events-meta-group mec-events-meta-group-tags">';
            $html .= esc_html__('Tags: ', 'mec');
            if ($tags) {
                $html .= implode(', ', array_map(function ($tag) {
                        return '<a href="' . esc_url(get_tag_link($tag->term_id)) . '">' . esc_html($tag->name) . ' </a>';
                    }, $tags
                    )
                );
            }
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-tags-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event CountDown block render.
     *
     * @since 1.0.0
     */
    public function event_count_down_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();

        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $events_detail = $event;
        } else {

            $events_detail = $events[0];
        }

        $html = '';
        if (!isset($settings['countdown_status']) || !$settings['countdown_status']) {

            $html .= '';
        } else {

            $wrap_class = 'mec-wrap';

            $html .= '<div class="' . esc_attr($wrap_class) . ' mec-events-meta-group mec-events-meta-group-countdown">'
                . Base::get_main()->module('countdown.details', array('event' => array($events_detail))) .
                '</div>';
        }

        $content_render = '<div><div class="event-count-down-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event LocalTime block render.
     *
     * @since 1.0.0
     */
    public function event_local_time_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();

        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';
        if (!isset($settings['local_time_module_status']) || !$settings['local_time_module_status']) {
            $html .= '';
        } else {
            $html .= '<div class="mec-event-meta mec-local-time-details mec-frontbox">'
                . Base::get_main()->module('local-time.details', array('event' => $event_detail)) .
                '</div>';
        }

        $content_render = '<div><div class="event-local-time-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Category block render.
     *
     * @since 1.0.0
     */
    public function event_category_render()
    {
        $taxonomies = array('mec_label', 'mec_organizer', 'mec_location', 'mec_category', apply_filters('mec_taxonomy_tag', ''));
        $categories = array();
        $terms = wp_get_post_terms(get_the_ID(), $taxonomies, array('fields' => 'all'));
        foreach ($terms as $term) {
            // First Validation
            if (!isset($term->taxonomy)) continue;

            if ($term->taxonomy == 'mec_category') {
                $categories[$term->term_id] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'icon' => get_metadata('term', $term->term_id, 'mec_cat_icon', true),
                    'color' => get_metadata('term', $term->term_id, 'mec_cat_color', true),
                );
            }
        }

        $html = '';
        if (empty($categories)) {
            $html .= '';
        } elseif (!empty($categories)) {
            $html .= '<div class="mec-single-event-category mec-event-meta mec-frontbox">';
            $html .= '<i class="mec-sl-folder"></i>';
            $html .= '<dt>' . Base::get_main()->m('taxonomy_categories', esc_html__('Category', 'mec')) . '</dt>';
            foreach ($categories as $category) {
                $icon = get_metadata('term', $category['id'], 'mec_cat_icon', true);
                $icon = isset($icon) && $icon != '' ? '<i class="' . esc_attr($icon) . ' mec-color"></i>' : '<i class="mec-fa-angle-right"></i>';
                $html .= '<dd class="mec-events-event-categories">
						<a href="' . esc_attr(get_term_link($category['id'], 'mec_category')) . '" class="mec-color-hover" rel="tag">' . $icon . esc_html($category['name']) . '</a></dd>';
            }
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-category-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event DateModule block render.
     *
     * @since 1.0.0
     */
    public function event_date_module_render()
    {

        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        // $event_detail = $this->get_event_detail(get_the_ID());
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $occurrence = isset($_GET['occurrence']) ? sanitize_text_field($_GET['occurrence']) : '';
        $start_datetime = !trim($occurrence) && isset($event_detail->date['start']) ? $event_detail->date['start'] : array( 'date' => $occurrence );
        $occurrence_end_date = trim($occurrence) ? Base::get_main()->get_end_date_by_occurrence($event_detail->data->ID, $start_datetime['date']) : '';
        $end_datetime = isset($event_detail->date['end']) ? $event_detail->date['end'] : array( 'date' => $occurrence_end_date );

        $date_format = isset($settings['single_date_format1']) ? $settings['single_date_format1'] : get_option('date_format');
        $date_label = Base::get_main()->date_label($start_datetime, $end_datetime, $date_format);

        $html = '<div class="mec-event-meta">' .
            '<div class="mec-single-event-date">' .
            '<i class="mec-sl-calendar"></i>' .
            '<h3 class="mec-date">' . esc_html(translate('Date', 'mec')) . '</h3>' .
            '<dl>';

        $html .= '<dd><abbr class="mec-events-abbr">' . wp_kses($date_label, array('span' => array('class' => array(), 'itemprop' => array()))) . '</abbr></dd>';
        $html .= '</dl>';
        $html .= '</div></div>';

        $content_render = '<div><div class="event-date-module-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event TimeModule block render.
     *
     * @since 1.0.0
     */
    public function event_time_module_render()
    {
        $main = MEC::getInstance('app.libraries.main');

        $html = '<div class="mec-event-meta">';
        $html .= '<style>.mec-single-event-time h3{display:inline;}</style>';

        // Event Time
        $mec_date_start = get_post_meta(get_the_ID(), 'mec_start_date', true);
        $mec_hide_time = get_post_meta(get_the_ID(), 'mec_hide_time', true);
        $mec_comment = get_post_meta(get_the_ID(), 'mec_comment', true);
        $mec_allday = get_post_meta(get_the_ID(), 'mec_allday', true);

        if (isset($mec_date_start) and !empty($mec_date_start)) {
            if (isset($mec_hide_time) and $mec_hide_time == '0') {

                $time_comment = isset($mec_comment) ? $mec_comment : '';
                $allday = isset($mec_allday) ? $mec_allday : 0;
                $time = array();
                $meta = $main->get_post_meta(get_the_ID(), true);

                $hide_time = isset($meta['mec_hide_time']) ? $meta['mec_hide_time'] : 0;
                $hide_end_time = isset($meta['mec_hide_end_time']) ? $meta['mec_hide_end_time'] : 0;

                $start_timestamp = ((isset($meta['mec_start_day_seconds']) and isset($meta['mec_start_date'])) ? (strtotime($meta['mec_start_date']) + $meta['mec_start_day_seconds']) : (isset($meta['mec_start_date']) ? strtotime($meta['mec_start_date']) : 0));
                $end_timestamp = ((isset($meta['mec_end_day_seconds']) and isset($meta['mec_end_date'])) ? (strtotime($meta['mec_end_date']) + $meta['mec_end_day_seconds']) : (isset($meta['mec_end_date']) ? strtotime($meta['mec_end_date']) : 0));

                $start_time = $main->get_time($start_timestamp);
                $end_time = $main->get_time($end_timestamp);

                if ($hide_time) {
                    $time = array(
                        'start' => '',
                        'end' => '',
                        'start_raw' => $start_time,
                        'end_raw' => $end_time,
                        'start_timestamp' => $start_timestamp,
                        'end_timestamp' => $end_timestamp,
                    );
                } elseif ($allday) {
                    $time = array(
                        'start' => $main->m('all_day', esc_html__('All Day', 'mec')),
                        'end' => '',
                        'start_raw' => $start_time,
                        'end_raw' => $end_time,
                        'start_timestamp' => $start_timestamp,
                        'end_timestamp' => $end_timestamp,
                    );
                } else {
                    $time = array(
                        'start' => $start_time,
                        'end' => ($hide_end_time ? '' : $end_time),
                        'start_raw' => $start_time,
                        'end_raw' => $end_time,
                        'start_timestamp' => $start_timestamp,
                        'end_timestamp' => $end_timestamp,
                    );
                }

                $html .= '<div class="mec-single-event-time">';
                $html .= '<i class="mec-sl-clock"></i>';
                $html .= '<h3 class="mec-time">' . esc_html(translate('Time', 'mec')) . '</h3>';
                $html .= '<i class="mec-time-comment">' . (isset($time_comment) ? esc_html(translate($time_comment)) : '') . '</i>';
                $html .= '<dl>';
                if ($allday == '0' and isset($time) and trim($time['start'])) {
                    $html .= '<dd><abbr class="mec-events-abbr">' . esc_html(translate($time['start'])) . (trim($time['end']) ? esc_html(translate(' - ' . $time['end'])) : '') . '</abbr></dd>';
                } else {
                    $html .= '<dd><abbr class="mec-events-abbr">' . esc_html(\MEC\Base::get_main()->m('all_day', esc_html__('All Day', 'mec'))) . '</abbr></dd>';
                }
                $html .= '</dl></div>';
            }
        }
        $html .= '</div>';

        $content_render = '<div><div class="event-time-module-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Cost block render.
     *
     * @since 1.0.0
     */
    public function event_cost_render()
    {
        //$events_detail = $this->get_event_detail(get_the_ID());

        $html = '';
        $cost = get_post_meta(get_the_ID(), 'mec_cost', true);
        if (!(isset($cost) && $cost != '')) {
            $html .= '';
        } else {

            if ($cost != '') {
                $html .= '<div class="mec-event-meta">';
                $html .= '<div class="mec-event-cost">';
                $html .= '<i class="mec-sl-wallet"></i>';
                $html .= '<h3 class="mec-cost">' . Base::get_main()->m('cost', esc_html__('Cost', 'mec')) . '</h3>';
                $html .= '<dd class="mec-events-event-cost">' . (is_numeric($cost) ? Base::get_main()->render_price($cost) : $cost) . '</dd>';
                $html .= '</div></div>';
            }
        }

        $content_render = '<div><div class="event-cost-gsb">' . $html . '</div></div>';
        return $content_render;
    }


    /**
     * Event Banner block render.
     *
     * @since 1.1.3
     */
    public function event_banner_render()
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-banner', 0, []);
        $content_render = '<div><div class="event-banner-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Labels block render.
     *
     * @since 1.0.0
     */
    public function event_labels_render()
    {
        $labels = array();
        $taxonomies = array('mec_label', 'mec_organizer', 'mec_location', 'mec_category', apply_filters('mec_taxonomy_tag', ''));
        $terms = wp_get_post_terms(get_the_ID(), $taxonomies, array('fields' => 'all'));
        foreach ($terms as $term) {
            // First Validation
            if (!isset($term->taxonomy)) continue;

            if ($term->taxonomy == 'mec_label') {
                $labels[$term->term_id] = array('id' => $term->term_id, 'name' => $term->name, 'color' => get_metadata('term', $term->term_id, 'color', true), 'style' => get_metadata('term', $term->term_id, 'style', true));
            }
        }

        $html = '';
        if (empty($labels)) {
            $html .= '';
        } elseif (!empty($labels)) {

            $html .= '<div class="mec-event-meta">';
            $mec_items = count($labels);
            $mec_i = 0;
            $html .= '<div class="mec-single-event-label">';
            $html .= '<i class="mec-fa-bookmark-o"></i>';
            $html .= '<h3 class="mec-cost">' . Base::get_main()->m('taxonomy_labels', esc_html__('Labels', 'mec')) . '</h3>';
            foreach ($labels as $k => $label) {
                $seperator = (++$mec_i === $mec_items) ? '' : ',';
                $html .= '<dd style="color:' . esc_attr($label['color']) . '">' . esc_html(translate($label["name"] . $seperator)) . '</dd>';
            }
            $html .= '</div>';
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-labels-gsb">' . $html . '</div></div>';
        return $content_render;
    }


    /**
     * Event Organizers block render.
     *
     * @since 1.0.0
     */
    public function event_organizers_render()
    {
        $html = '';
        $html .= '<style>
            .single-mec_esb.mec-single-event .mec-organizer-tel a {
                display: block;
                padding-left: 41px;
                color: #8d8d8d;
            }
        </style>';

        $main = MEC::getInstance('app.libraries.main');
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $meta = $main->get_post_meta(get_the_ID(), true);

        $organizers = array();
        $taxonomies = array('mec_label', 'mec_organizer', 'mec_location', 'mec_category', apply_filters('mec_taxonomy_tag', ''));
        $terms = wp_get_post_terms(get_the_ID(), $taxonomies, array('fields' => 'all'));
        foreach ($terms as $term) {
            // First Validation
            if (!isset($term->taxonomy)) continue;

            if ($term->taxonomy == 'mec_organizer')
                $organizers[$term->term_id] = array('id' => $term->term_id, 'name' => $term->name, 'tel' => get_metadata('term', $term->term_id, 'tel', true), 'email' => get_metadata('term', $term->term_id, 'email', true), 'url' => get_metadata('term', $term->term_id, 'url', true), 'page_label' => get_metadata('term', $term->term_id, 'page_label', true), 'thumbnail' => get_metadata('term', $term->term_id, 'thumbnail', true));
        }

        $primary_organizer_id = isset($meta['mec_organizer_id']) ? $meta['mec_organizer_id'] : '';

        $html = '';
        if (empty($organizers) || !isset($organizers[$primary_organizer_id])) {
            $html .= '';
        } elseif (!empty($organizers) && isset($organizers[$primary_organizer_id]) && !empty($organizers[$primary_organizer_id])) {

            $organizer = $organizers[$primary_organizer_id];

            $html .= '<div class="mec-event-meta">';
            $html .= '<div class="mec-single-event-organizer">';
            if (isset($organizer['thumbnail']) and trim($organizer['thumbnail'])) {
                $html .= '<img class="mec-img-organizer" src="' . esc_url($organizer['thumbnail']) . '" alt="' . (isset($organizer['name']) ? esc_attr($organizer['name']) : '') . '">';
            }
            $html .= '<h3 class="mec-events-single-section-title">' . Base::get_main()->m('taxonomy_organizer', esc_html__('Organizer', 'mec')) . '</h3>';
            if (isset($organizer['thumbnail'])) {
                $html .= '<dd class="mec-organizer">';
                $html .= '<i class="mec-sl-home"></i>';
                $html .= '<h6>' . (isset($organizer['name']) ? esc_html(translate($organizer['name'])) : '') . '</h6>';
                $html .= '</dd>';
            }
            if (isset($organizer['tel']) && !empty($organizer['tel'])) {
                $html .= '<dd class="mec-organizer-tel">';
                $html .= '<i class="mec-sl-phone"></i>';
                $html .= '<h6>' . esc_html(translate('Phone', 'mec')) . '</h6>';
                $html .= '<a href="tel:' . esc_attr($organizer['tel']) . '">' . esc_html(translate($organizer['tel'])) . '</a>';
                $html .= '</dd>';
            }
            if (isset($organizer['email']) && !empty($organizer['email'])) {
                $html .= '<dd class="mec-organizer-email">';
                $html .= '<i class="mec-sl-envelope"></i>';
                $html .= '<h6>' . esc_html(translate('Email', 'mec')) . '</h6>';
                $html .= '<a href="mailto:' . esc_attr($organizer['email']) . '">' . esc_html(translate($organizer['email'])) . '</a>';
                $html .= '</dd>';
            }
            if (isset($organizer['url']) && !empty($organizer['url']) and $organizer['url'] != 'http://') {
                $html .= '<dd class="mec-organizer-url">';
                $html .= '<i class="mec-sl-sitemap"></i>';
                $html .= '<h6>' . esc_html(translate('Website', 'mec')) . '</h6>';
                $html .= '<span><a href="' . esc_url($organizer['url']) . '" class="mec-color-hover" target="_blank">' . esc_url($organizer['url']) . '</a></span>';
                $html .= '</dd>';
            }
            $organizer_description_setting = isset($settings['organizer_description']) ? $settings['organizer_description'] : '';
            $organizer_terms = get_the_terms($data, 'mec_organizer');
            if ($organizer_description_setting == '1') {
                foreach ($organizer_terms as $organizer_term) {
                    if ($organizer_term->term_id == $organizer['id']) {
                        if (isset($organizer_term->description) && !empty($organizer_term->description)) {
                            $html .= '<dd class="mec-organizer-description">';
                            $html .= '<p>' . esc_html(translate($organizer_term->description)) . '</p>';
                            $html .= '</dd>';
                        }
                    }
                }
            }
            $html .= '</div>';
            $html .= $this->show_other_organizers(get_the_ID(), $organizers); // Show Additional Organizers
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-organizers-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Speakers block render.
     *
     * @since 1.0.0
     */
    public function event_speakers_render()
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-speakers', 0, []);
        $content_render = '<div><div class="event-speakers-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event More Info block render.
     *
     * @since 1.0.0
     */
    public function event_more_info_render()
    {
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta(get_the_ID(), true);

        $html = '';
        if (!(isset($meta['mec_more_info']) && $meta['mec_more_info'] != '')) {
            $html .= '';
        } else {
            $html .= '<style>.mec-event-more-info h3{display:inline;}</style>';
            $html .= '<div class="mec-event-meta">';
            $html .= '<div class="mec-event-more-info">';
            $html .= '<i class="mec-sl-info"></i>';
            $html .= '<h3 class="mec-more-info-label">' . Base::get_main()->m('more_info_link', esc_html__('More Info', 'mec')) . '</h3>';
            $html .= '<dd class="mec-events-event-more-info"><a class="mec-more-info-button a mec-color-hover" target="' . (isset($meta['mec_more_info_target']) ? esc_attr($meta['mec_more_info_target']) : '_self') . '" href="' . esc_attr($meta['mec_more_info']) . '">' . ((isset($meta['mec_more_info_title']) && trim($meta['mec_more_info_title'])) ? esc_html($meta['mec_more_info_title']) : esc_html__('Read More', 'mec')) . '</a></dd>';
            $html .= '</div>';
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-more-info-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Social Share block render.
     *
     * @since 1.0.0
     */
    public function event_social_share_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';
        if (!isset($settings['social_network_status']) || !$settings['social_network_status']) {
            $html .= '';
        } else {
            $main = MEC::getInstance('app.libraries.main');
            $meta = $main->get_post_meta(get_the_ID(), true);
            $permalink = ((isset($meta['mec_read_more']) and filter_var($meta['mec_read_more'], FILTER_VALIDATE_URL)) ? $meta['mec_read_more'] : get_post_permalink(get_the_ID()));

            $url = isset($permalink) ? $permalink : '';
            if (trim($url) == '') {
                $content_render = '<div><div class="event-social-share-gsb">' . $html . '</div></div>';
                return $content_render;
            }

//            $socials = new EventSocialShare();
            $socials = Base::get_main()->get_social_networks();

            $html .= '<div class="mec-event-social mec-frontbox">';
            $html .= '<h3 class="mec-social-single mec-frontbox-title">' . esc_html(translate('Share this event', 'mec')) . '</h3>';
            $html .= '<div class="mec-event-sharing">';
            $html .= '<div class="mec-links-details">';
            $html .= '<ul>';
            $social_networks = isset($settings['sn']) && is_array($settings['sn']) ? $settings['sn'] : array();
            foreach ($socials as $key => $social) {
                if ($key != "email") {
                    $social_id = $social['id'];
                    $is_enabled = isset($social_networks[$social_id]) && !$social_networks[$social_id];
                    if ($is_enabled) {
                        continue;
                    }

                    if (is_callable($social['function'])) {
                        $html .= call_user_func($social['function'], $url, $event_detail);
                    }
                }
            }
            $html .= '</ul>';
            $html .= '</div>';
            $html .= '</div>';
            $html .= '</div>';
//            $html =  EventSocialShare::getInstance()->output( $event_id );
        }

        $content_render = '<div><div class="event-social-share-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Qr Code block render.
     *
     * @since 1.0.0
     */
    public function event_qr_code_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';
        if (!isset($settings['qrcode_module_status']) || !$settings['qrcode_module_status']) {
            $html .= '';
        } else {
            $html .= Base::get_main()->module('qrcode.details', array('event' => $event_detail));
        }

        $content_render = '<div><div class="event-qr-code-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Next block render.
     *
     * @since 1.0.0
     */
    public function event_next_render()
    {
        $html= \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-next-occurrences', 0, []);
        $content_render = '<div><div class="event-next-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Hourly Schedule block render.
     *
     * @since 1.0.0
     */
    public function event_hourly_schedule_render()
    {
        $html= \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-hourly-schedule', 0, []);
        $content_render = '<div><div class="event-hourly-schedule-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Attendees block render.
     *
     * @since 1.0.0
     */
    public function event_attendees_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $events_detail = $event;
        } else {

            $events_detail = $events[0];
        }

        $html = '';
        if (isset($settings['bp_status']) && $settings['bp_status']) {
            $html .= Base::get_main()->module('attendees-list.details', array('event' => $events_detail));
        }

        if (empty($html)) {
            $html .= '';
        }

        $content_render = '<div><div class="event-attendees-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Booking block render.
     *
     * @since 1.0.0
     */
    public function event_booking_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';

        if ((!isset($settings['booking_status']) || !$settings['booking_status'])) {
            $html .= '';
        } else {

            $html .= $this->get_display_booking_form(get_the_ID());

            if (\MEC\Base::get_main()->can_show_booking_module($event_detail) && isset($settings['single_booking_style']) && $settings['single_booking_style'] == 'modal') {

                $html .= '<style>
					.lity-container {
						max-width: 480px;
						width: 480px;
					}
				</style>';

                $html .= '<div class="mec-content-notification"><p><span>'
                    . __('It seems that you have set "Booking" to modal from Single Event MEC Settings. You need to know that for this mode to work you must add Register Button Widget to this page, then in the front-end by clicking the Register button in your events you can then see the modal mode of the Booking.', 'mec')
                    . '</span></p></div>';
            } elseif (empty($html)) {
                $html .= '<div class="mec-content-notification"><p>'
                    . '<span>' . esc_html__('Not Found Booking', 'mec-gsb') . '</span>'
                    . '</p></div>';
            } else {
                // $html .= '<script>jQuery("#mec-book-form-btn-step-1").prop("disabled",true);</script>';
            }
        }

        $content_render = '<div><div class="event-booking-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Breadcrumbs block render.
     *
     * @since 1.0.0
     */
    public function event_breadcrumbs_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $html = '';
        if ((!isset($settings['breadcrumbs']) || !$settings['breadcrumbs'])) {
            $html = '';
        } else {

            $html .= '<div class="mec-breadcrumbs">';
            $html .= $this->display_breadcrumb_widget(get_the_ID());
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-breadcrumbs-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Cancellation block render.
     *
     * @since 1.0.0
     */
    public function event_cancellation_render()
    {
        $display_reason = get_post_meta(get_the_ID(), 'mec_display_cancellation_reason_in_single_page', true);
        $html = '';
        if (!$display_reason) {
            $html = '';
        } else {

            $display_reason = get_post_meta(get_the_ID(), 'mec_display_cancellation_reason_in_single_page', true);
            if ($display_reason) {
                $html = $this->display_cancellation_reason(get_the_ID(), $display_reason);
            }
        }

        $content_render = '<div><div class="event-cancellation-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Data block render.
     *
     * @since 1.0.0
     */
    public function event_data_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta(get_the_ID(), true);

        $data = (isset($meta['mec_fields']) and is_array($meta['mec_fields'])) ? $meta['mec_fields'] : get_post_meta(get_the_ID(), 'mec_fields', true);


        $html = '';
        if (!empty($data) && (isset($settings['display_event_fields']) && $settings['display_event_fields'])) {
            $html = $this->display_data_fields(get_the_ID());
        }

        if (empty($html)) {

            $html = '';
        }


        $content_render = '<div><div class="event-data-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Map block render.
     *
     * @since 1.0.0
     */
    public function event_map_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta(get_the_ID(), true);

        $primary_location_id = isset($meta['mec_location_id']) ? $meta['mec_location_id'] : '';
        $have_location = $primary_location_id && $primary_location_id > 1 ? true : false;

        $html = '';
        if (isset($settings['google_maps_status']) && $settings['google_maps_status'] && $have_location) {

            $html = Base::get_main()->module('googlemap.details', array('event' => [$event_detail]));
            if ($html) {

                $html = '<div class="mec-events-meta-group mec-events-meta-group-gmap">'
                    . $html .
                    '</div>';
            }
        }

        if ((!isset($settings['google_maps_status']) || !$settings['google_maps_status'])) {

            $html = '';
        } elseif (empty($html)) {

            $html = '';
        }

        $content_render = '<div><div class="event-map-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Next/Previous block render.
     *
     * @since 1.0.0
     */
    public function event_next_previous_render()
    {

        $settings = \MEC\Settings\Settings::getInstance()->get_settings();

        $html = '';
        if ((!isset($settings['next_previous_events']) || !$settings['next_previous_events'])) {

            $html .= '';
        } else {
            $event = get_the_ID();
            if (!isset($settings['next_previous_events'])) {
                $html = '';
            } else {
                if (isset($settings['next_previous_events']) && $settings['next_previous_events'] != '1') {
                    $html = '';
                } else {
                    if (is_numeric($event)) {
                        $event_id = $event;
                    } else {
                        $content_render = '<div><div class="event-next-gsb">' . $html . '</div></div>';
                        return $content_render;
                    }

                    $main = MEC::getInstance('app.libraries.main');
                    $db = MEC::getInstance('app.libraries.db');
                    $p_exclude = array($event_id);
                    $n_exclude = array($event_id);

                    $pskip = (isset($_REQUEST['pskip']) and is_numeric($_REQUEST['pskip']) and $_REQUEST['pskip'] > 0) ? sanitize_text_field($_REQUEST['pskip']) : NULL;
                    if ($pskip) $p_exclude[] = $pskip;

                    $nskip = (isset($_REQUEST['nskip']) and is_numeric($_REQUEST['nskip']) and $_REQUEST['nskip'] > 0) ? sanitize_text_field($_REQUEST['nskip']) : NULL;
                    if ($nskip) $n_exclude[] = $nskip;

                    $start_timestamp = !empty($start_timestamp) ? $start_timestamp : strtotime('Yesterday');
                    $dates = $this->dates(get_the_ID(), NULL, 6, date('Y-m-d H:i', $start_timestamp));

                    $date = $dates[0] ?? '';

                    $timestamp = (isset($date['start']) and isset($date['start']['timestamp'])) ? $date['start']['timestamp'] : NULL;

                    $args = array(
                        'post_type' => $main->get_main_post_type(),
                        'posts_per_page' => -1,
                        'post_status' => 'publish',
                        'orderby' => 'ASC',
                        'tax_query' => array(),
                    );

                    if (isset($settings['next_previous_events_category']) && $settings['next_previous_events_category'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, 'mec_category', array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => 'mec_category',
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    if (isset($settings['next_previous_events_organizer']) && $settings['next_previous_events_organizer'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, 'mec_organizer', array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => 'mec_organizer',
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    if (isset($settings['next_previous_events_location']) && $settings['next_previous_events_location'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, 'mec_location', array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => 'mec_location',
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    if (isset($settings['next_previous_events_speaker']) && $settings['next_previous_events_speaker'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, 'mec_speaker', array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => 'mec_speaker',
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    if (isset($settings['next_previous_events_label']) && $settings['next_previous_events_label'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, 'mec_label', array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => 'mec_label',
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    if (isset($settings['next_previous_events_tag']) && $settings['next_previous_events_tag'] == 1) {
                        $post_terms = wp_get_object_terms($event_id, apply_filters('mec_taxonomy_tag', ''), array('fields' => 'slugs'));
                        $args['tax_query'][] = array(
                            'taxonomy' => apply_filters('mec_taxonomy_tag', ''),
                            'field' => 'slug',
                            'terms' => $post_terms
                        );
                    }

                    $args['tax_query']['relation'] = 'OR';

                    $p_args = array_merge($args, array('post__not_in' => $p_exclude));
                    $n_args = array_merge($args, array('post__not_in' => $n_exclude));

                    $p_args = apply_filters('mec_next_previous_query', $p_args, $event_id);
                    $n_args = apply_filters('mec_next_previous_query', $n_args, $event_id);

                    $p_IDs = array();
                    $n_IDs = array();

                    $query = new WP_Query($p_args);
                    if ($query->have_posts()) {
                        while ($query->have_posts()) {
                            $query->the_post();
                            $p_IDs[] = get_the_ID();
                        }
                    }

                    wp_reset_postdata();

                    if ($p_args === $n_args) $n_IDs = $p_IDs;
                    else {
                        $query = new WP_Query($n_args);
                        if ($query->have_posts()) {
                            while ($query->have_posts()) {
                                $query->the_post();
                                $n_IDs[] = get_the_ID();
                            }
                        }

                        wp_reset_postdata();
                    }

                    // No Event Found!
                    if (!count($p_IDs) and !count($n_IDs)) {
                        $content_render = '<div><div class="event-next-gsb">' . $html . '</div></div>';
                        return $content_render;
                    }

                    $p = $db->select("SELECT `post_id`, `tstart` FROM `#__mec_dates` WHERE `tstart`<='" . $timestamp . "' AND `post_id` IN (" . implode(',', $p_IDs) . ") ORDER BY `tstart` DESC LIMIT 1", 'loadAssoc');
                    $n = $db->select("SELECT `post_id`, `tstart` FROM `#__mec_dates` WHERE `tstart`>='" . $timestamp . "' AND `post_id` IN (" . implode(',', $n_IDs) . ") ORDER BY `tstart` ASC LIMIT 1", 'loadAssoc');

                    // No Event Found!
                    if (!isset($p['post_id']) and !isset($n['post_id'])) {
                        $content_render = '<div><div class="event-next-previous-gsb">' . $html . '</div></div>';
                        return $content_render;
                    }

                    $html .= '<ul class="mec-next-previous-events">';

                    if (is_array($p) and isset($p['post_id'])) {
                        $p_url = $main->get_event_date_permalink(get_permalink($p['post_id']), date('Y-m-d', $p['tstart']));
                        $p_url = $main->add_qs_var('pskip', $event_id, $p_url);

                        $html .= '<li class="mec-previous-event"><a class="mec-color mec-bg-color-hover mec-border-color" href="' . esc_url($p_url) . '"><i class="mec-fa-long-arrow-left"></i>' . esc_html__('PRV Event', 'mec') . '</a></li>';
                    }

                    if (is_array($n) and isset($n['post_id'])) {
                        $n_url = $main->get_event_date_permalink(get_permalink($n['post_id']), date('Y-m-d', $n['tstart']));
                        $n_url = $main->add_qs_var('nskip', $event_id, $n_url);

                        $html .= '<li class="mec-next-event"><a class="mec-color mec-bg-color-hover mec-border-color" href="' . esc_html($n_url) . '">' . esc_html__('NXT Event', 'mec') . '<i class="mec-fa-long-arrow-right"></i></a></li>';
                    }

                    $html .= '</ul>';

                }
            }

        }

        $content_render = '<div><div class="event-next-previous-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Public Download block render.
     *
     * @since 1.0.0
     */
    public function event_public_download_render()
    {
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $events_detail = $event;
        } else {

            $events_detail = $events[0];
        }

        $html = '';
        $single = new \MEC_skin_single();
        $html .= $single->display_public_download_module($events_detail);

        if (empty($html)) {

            $html .= '';

        }

        $content_render = '<div><div class="event-public-download-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Register Button block render.
     *
     * @since 1.0.0
     */
    public function event_register_button_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';
        if ((!isset($settings['single_booking_style']) || !$settings['single_booking_style'])) {
            $html .= '';
        } else {

            $single = new \MEC_skin_single();
            $main = MEC::getInstance('app.libraries.main');
            $meta = $main->get_post_meta(get_the_ID(), true);

            $mec_more_info = isset($meta['mec_more_info']) && trim($meta['mec_more_info']) && $meta['mec_more_info'] != 'http://';
            $mec_more_info_target = isset($meta['mec_more_info_target']) ? $meta['mec_more_info_target'] : '_self';
            if (isset($meta['mec_more_info_title']) && trim($meta['mec_more_info_title'])) {
                $button_text = trim($meta['mec_more_info_title']);
            } else {
                $button_text = Base::get_main()->m('register_button', esc_html__('REGISTER', 'mec'));
            }

            $classes = '';
            if (isset($settings['single_booking_style']) && $settings['single_booking_style'] != 'modal') {
                $classes = 'simple-booking';
            }
            $html .= '<!-- Register Booking Button -->';
            $html .= '<div class="mec-reg-btn mec-frontbox">';
            if (Base::get_main()->can_show_booking_module($event_detail)) {

                $data_lity = '';
                if (isset($settings['single_booking_style']) && $settings['single_booking_style'] == 'modal') {
                    $data_lity = 'data-lity';
                    $classes .= ' mec-booking-data-lity';
                }
                $html .= '<a class="mec-booking-button mec-bg-color ' . esc_attr($classes) . '" href="#mec-events-meta-group-booking-' . esc_attr($single->uniqueid) . '" ' . esc_attr($data_lity) . '>' . esc_html__($button_text, 'mec') . '</a>';
                $html .= '<script>
                                    // Fix modal booking in some themes
                                    if( \'undefined\' === typeof mec_booking_lity_init){

                                        function mec_booking_lity_init(e){

                                            e.preventDefault();
                                            var book_id =  jQuery(this).attr(\'href\');
                                            lity(book_id);

                                            return false;
                                        }
                                        jQuery( ".mec-booking-button.mec-booking-data-lity" ).on(\'click\',mec_booking_lity_init);
                                    }
                                </script>';
            } elseif ($mec_more_info) {
                $html .= '<a target="' . esc_attr($mec_more_info_target) . '" class="mec-booking-button mec-bg-color" href="' . esc_url($meta['mec_more_info']) . '">';
                $html .= esc_html__($button_text, 'mec');
                $html .= '</a>';
            }
            $html .= '</div>';
        }

        $content_render = '<div><div class="event-register-button-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Related block render.
     *
     * @since 1.0.0
     */
    public function event_related_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $html = '';
        if ((!isset($settings['related_events']) || !$settings['related_events'])) {
            $html .= '';
        } else {
            $html .= $this->display_related_posts_widget(get_the_ID());
        }
        $content_render = '<div><div class="event-related-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * Event Weather block render.
     *
     * @since 1.0.0
     */
    public function event_weather_render()
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $event_id = get_the_ID();
        $single = new \MEC_skin_single();
        $events = $single->get_event_mec($event_id);

        if (!isset($events[0])) {
            $event = new \MEC\Events\Event($event_id);
            //$events_detail= $event->get_detail();
            $event_detail = $event;
        } else {

            $event_detail = $events[0];
        }

        $html = '';

        if (isset($settings['weather_module_status']) && $settings['weather_module_status']) {
            $html .= Base::get_main()->module('weather.details', array('event' => $event_detail));
        }

        if (empty($html)) {

            $html .= '';
        }

        $content_render = '<div><div class="event-weather-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    /**
     * @param int|WP_Post $event
     *
     * @return Object
     */
    public function get_event_detail($event_id)
    {
        $event = new \MEC\Events\Event($event_id);
        return $event->get_detail();
    }

    /**
     * Renders and Returns event dats
     * @param int $event_id
     * @param object $event
     * @param int $maximum
     * @param string $today
     * @return object|array
     * @author Webnus <info@webnus.net>
     */
    public function dates($event_id, $event = NULL, $maximum = 6, $today = NULL)
    {
        $main = MEC::getInstance('app.libraries.main');
        $db = MEC::getInstance('app.libraries.db');
        if (!trim($today)) $today = date('Y-m-d');

        // Original Start Date
        $original_start_date = $today;
        $dates = array();

        // Get event data if it is NULL
        if (is_null($event)) {
            $event = new stdClass();
            $event->meta = $main->get_post_meta($event_id, true);
            $event->mec = $db->select("SELECT * FROM `#__mec_events` WHERE `post_id`='$event_id'", "loadObject");
        }

        $start_date = isset($event->meta['mec_date']['start']) ? $event->meta['mec_date']['start'] : array();
        $end_date = isset($event->meta['mec_date']['end']) ? $event->meta['mec_date']['end'] : array();

        // Return empty array if date is not valid
        if (!isset($start_date['date']) or (isset($start_date['date']) and !strtotime($start_date['date']))) return $dates;

        // Return empty array if mec data is not exists on mec_events table
        if (!isset($event->mec->end)) return $dates;

        $allday = isset($event->meta['mec_allday']) ? $event->meta['mec_allday'] : 0;
        $hide_time = isset($event->meta['mec_hide_time']) ? $event->meta['mec_hide_time'] : 0;

        $event_period = $main->date_diff($start_date['date'], $end_date['date']);
        $event_period_days = $event_period ? $event_period->days : 0;

        $finish_date = array('date' => $event->mec->end, 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']);
        $exceptional_days = (isset($event->mec->not_in_days) and trim($event->mec->not_in_days)) ? explode(',', trim($event->mec->not_in_days, ', ')) : array();

        // Event Passed
        $past = $main->is_past($finish_date['date'], $today);

        // Event is not passed for custom days
        if ($past and isset($event->meta['mec_repeat_type']) and $event->meta['mec_repeat_type'] == 'custom_days') $past = 0;

        // Normal event
        if (isset($event->mec->repeat) and $event->mec->repeat == '0') {
            $dates[] = $this->add_timestamps(array(
                'start' => $start_date,
                'end' => $end_date,
                'allday' => $allday,
                'hide_time' => $hide_time,
                'past' => $past
            ));
        } elseif ($past) {
            $dates[] = $this->add_timestamps(array(
                'start' => $start_date,
                'end' => $end_date,
                'allday' => $allday,
                'hide_time' => $hide_time,
                'past' => $past
            ));
        } elseif (!$past) {
            $repeat_type = $event->meta['mec_repeat_type'];

            if (in_array($repeat_type, array('daily', 'weekly'))) {
                $repeat_interval = $event->meta['mec_repeat_interval'];

                $date_interval = $main->date_diff($start_date['date'], $today);
                $passed_days = $date_interval ? $date_interval->days : 0;

                // Check if date interval is negative (It means the event didn't start yet)
                if ($date_interval and $date_interval->invert == 1) $remained_days_to_next_repeat = $passed_days;
                else $remained_days_to_next_repeat = $repeat_interval - fmod($passed_days, $repeat_interval);

                $start_date = date('Y-m-d', strtotime('+' . $remained_days_to_next_repeat . ' Days', strtotime($today)));
                if (!$main->is_past($finish_date['date'], $start_date) and !in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                    'start' => array('date' => $start_date, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                    'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                    'allday' => $allday,
                    'hide_time' => $hide_time,
                    'past' => 0
                ));

                for ($i = 2; $i <= $maximum; $i++) {
                    $start_date = date('Y-m-d', strtotime('+' . $repeat_interval . ' Days', strtotime($start_date)));

                    // Event finished
                    if ($main->is_past($finish_date['date'], $start_date)) break;

                    if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                        'start' => array('date' => $start_date, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                        'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => 0
                    ));
                }
            } elseif (in_array($repeat_type, array('weekday', 'weekend', 'certain_weekdays'))) {
                $date_interval = $main->date_diff($start_date['date'], $today);
                $passed_days = $date_interval ? $date_interval->days : 0;

                // Check if date interval is negative (It means the event didn't start yet)
                if ($date_interval and $date_interval->invert == 1) $today = date('Y-m-d', strtotime('+' . $passed_days . ' Days', strtotime($original_start_date)));

                $event_days = explode(',', trim($event->mec->weekdays, ', '));

                $today_id = date('N', strtotime($today));
                $found = 0;
                $i = 0;

                while ($found < $maximum) {
                    if ($main->is_past($finish_date['date'], $today)) break;

                    if (!in_array($today_id, $event_days)) {
                        $today = date('Y-m-d', strtotime('+1 Days', strtotime($today)));
                        $today_id = date('N', strtotime($today));

                        $i++;
                        continue;
                    }

                    $start_date = $today;
                    if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                        'start' => array('date' => $start_date, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                        'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => 0
                    ));

                    $today = date('Y-m-d', strtotime('+1 Days', strtotime($today)));
                    $today_id = date('N', strtotime($today));

                    $found++;
                    $i++;
                }
            } elseif ($repeat_type == 'monthly') {
                $repeat_interval = ((isset($event->meta) and isset($event->meta['mec_repeat_interval'])) ? max(1, $event->meta['mec_repeat_interval']) : 1);

                // Start from Event Start Date
                if (strtotime($start_date['date']) > strtotime($original_start_date)) $original_start_date = $start_date['date'];

                $event_days = explode(',', trim($event->mec->day, ', '));
                $event_start_day = $event_days[0];

                $diff = $main->date_diff($start_date['date'], $end_date['date']);
                $event_period_days = $diff->days;

                $found = 0;
                $i = 0;

                while ($found < $maximum) {
                    $t = strtotime('+' . $i . ' Months', strtotime($original_start_date));
                    if (!$t) break;

                    $today = date('Y-m-d', $t);
                    if ($main->is_past($finish_date['date'], $today)) break;

                    $year = date('Y', strtotime($today));
                    $month = date('m', strtotime($today));
                    $day = $event_start_day;
                    $hour = isset($event->meta['mec_date']['end']['hour']) ? sprintf('%02d', $event->meta['mec_date']['end']['hour']) : '06';
                    $minutes = isset($event->meta['mec_date']['end']['minutes']) ? sprintf('%02d', $event->meta['mec_date']['end']['minutes']) : '00';
                    $ampm = isset($event->meta['mec_date']['end']['ampm']) ? strtoupper($event->meta['mec_date']['end']['ampm']) : 'PM';

                    // Fix for 31st, 30th, 29th of some months
                    while (!checkdate($month, $day, $year)) $day--;

                    $start_date = $year . '-' . $month . '-' . $day;
                    $end_time = $hour . ':' . $minutes . ' ' . $ampm;

                    // Wrong Date & Time
                    if (!strtotime($start_date . ' ' . $end_time)) break;

                    if (strtotime($start_date . ' ' . $end_time) < strtotime($original_start_date)) {
                        $i += $repeat_interval;
                        continue;
                    }

                    if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                        'start' => array('date' => $start_date, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                        'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => 0
                    ));

                    $found++;
                    $i += $repeat_interval;
                }
            } elseif ($repeat_type == 'yearly') {
                // Start from Event Start Date
                if (strtotime($start_date['date']) > strtotime($original_start_date)) $original_start_date = $start_date['date'];

                $event_days = explode(',', trim($event->mec->day, ', '));
                $event_months = explode(',', trim($event->mec->month, ', '));

                $event_start_day = $event_days[0];
                $event_period_days = $main->date_diff($start_date['date'], $end_date['date'])->days;

                $event_start_year = date('Y', strtotime($original_start_date));
                $event_start_month = date('n', strtotime($original_start_date));

                $found = 0;
                $i = 0;

                while ($found < $maximum) {
                    $today = date('Y-m-d', strtotime($event_start_year . '-' . $event_start_month . '-' . $event_start_day));
                    if ($main->is_past($finish_date['date'], $today)) break;

                    $year = date('Y', strtotime($today));
                    $month = date('m', strtotime($today));

                    if (!in_array($month, $event_months)) {
                        if ($event_start_month == '12') {
                            $event_start_month = 1;
                            $event_start_year += 1;
                        } else $event_start_month += 1;

                        $i++;
                        continue;
                    }

                    $day = $event_start_day;

                    // Fix for 31st, 30th, 29th of some months
                    while (!checkdate($month, $day, $year)) $day--;

                    $event_date = $year . '-' . $month . '-' . $day;
                    if (strtotime($event_date) >= strtotime($original_start_date)) {
                        $start_date = $event_date;
                        if (!in_array($start_date, $exceptional_days)) $dates[] = $this->add_timestamps(array(
                            'start' => array('date' => $start_date, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                            'end' => array('date' => date('Y-m-d', strtotime('+' . $event_period_days . ' Days', strtotime($start_date))), 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                            'allday' => $allday,
                            'hide_time' => $hide_time,
                            'past' => 0
                        ));

                        $found++;
                    }

                    if ($event_start_month == '12') {
                        $event_start_month = 1;
                        $event_start_year += 1;
                    } else $event_start_month += 1;

                    $i++;
                }
            } elseif ($repeat_type == 'custom_days') {
                $custom_days = explode(',', $event->mec->days);

                // Add current time if we're checking today's events
                if ($today == current_time('Y-m-d')) $today .= ' ' . current_time('H:i:s');

                $found = 0;
                if ((strtotime($event->mec->start) + $event->meta['mec_start_day_seconds']) >= strtotime($today) and !in_array($event->mec->start, $exceptional_days)) {
                    $dates[] = $this->add_timestamps(array(
                        'start' => array('date' => $event->mec->start, 'hour' => $event->meta['mec_date']['start']['hour'], 'minutes' => $event->meta['mec_date']['start']['minutes'], 'ampm' => $event->meta['mec_date']['start']['ampm']),
                        'end' => array('date' => $event->mec->end, 'hour' => $event->meta['mec_date']['end']['hour'], 'minutes' => $event->meta['mec_date']['end']['minutes'], 'ampm' => $event->meta['mec_date']['end']['ampm']),
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => 0
                    ));

                    $found++;
                }

                foreach ($custom_days as $custom_day) {
                    // Found maximum dates
                    if ($found >= $maximum) break;

                    $cday = explode(':', $custom_day);

                    $c_start = $cday[0];
                    if (isset($cday[2])) $c_start .= ' ' . str_replace('-', ' ', substr_replace($cday[2], ':', strpos($cday[2], '-'), 1));

                    // Date is past
                    if (strtotime($c_start) < strtotime($today)) continue;

                    $cday_start_hour = $event->meta['mec_date']['start']['hour'];
                    $cday_start_minutes = $event->meta['mec_date']['start']['minutes'];
                    $cday_start_ampm = $event->meta['mec_date']['start']['ampm'];

                    $cday_end_hour = $event->meta['mec_date']['end']['hour'];
                    $cday_end_minutes = $event->meta['mec_date']['end']['minutes'];
                    $cday_end_ampm = $event->meta['mec_date']['end']['ampm'];

                    if (isset($cday[2]) and isset($cday[3])) {
                        $cday_start_ex = explode('-', $cday[2]);
                        $cday_start_hour = $cday_start_ex[0];
                        $cday_start_minutes = $cday_start_ex[1];
                        $cday_start_ampm = $cday_start_ex[2];

                        $cday_end_ex = explode('-', $cday[3]);
                        $cday_end_hour = $cday_end_ex[0];
                        $cday_end_minutes = $cday_end_ex[1];
                        $cday_end_ampm = $cday_end_ex[2];
                    }

                    if (!in_array($cday[0], $exceptional_days)) $dates[] = $this->add_timestamps(array(
                        'start' => array('date' => $cday[0], 'hour' => $cday_start_hour, 'minutes' => $cday_start_minutes, 'ampm' => $cday_start_ampm),
                        'end' => array('date' => $cday[1], 'hour' => $cday_end_hour, 'minutes' => $cday_end_minutes, 'ampm' => $cday_end_ampm),
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => 0
                    ));

                    $found++;
                }

                // No future date found so the event is passed
                if (!count($dates)) {
                    $dates[] = $this->add_timestamps(array(
                        'start' => $start_date,
                        'end' => $finish_date,
                        'allday' => $allday,
                        'hide_time' => $hide_time,
                        'past' => $past
                    ));
                }
            } elseif ($repeat_type == 'advanced') {
                // Start from Event Start Date
                if (strtotime($start_date['date']) > strtotime($today)) $today = $start_date['date'];

                // Get user specifed days of month for repeat
                $advanced_days = get_post_meta($event_id, 'mec_advanced_days', true);

                // Generate dates for event
                $event_info = array('start' => $start_date, 'end' => $end_date, 'allday' => $allday, 'hide_time' => $hide_time, 'finish_date' => $finish_date['date'], 'exceptional_days' => $exceptional_days, 'mec_repeat_end' => ((isset($event->meta['mec_repeat']) and isset($event->meta['mec_repeat']['end'])) ? $event->meta['mec_repeat']['end'] : ''), 'occurrences' => ((isset($event->meta['mec_repeat']) and isset($event->meta['mec_repeat']['end_at_occurrences'])) ? $event->meta['mec_repeat']['end_at_occurrences'] : ''));

                $dates = $this->generate_advanced_days($advanced_days, $event_info, $maximum, $today);
            }
        }

        return $dates;
    }

    public function add_timestamps($date)
    {
        $start = (isset($date['start']) and is_array($date['start'])) ? $date['start'] : array();
        $end = (isset($date['end']) and is_array($date['end'])) ? $date['end'] : array();

        if (!count($start) or !count($end)) return $date;

        $s_hour = $start['hour'];
        if (strtoupper($start['ampm']) == 'AM' and $s_hour == '0') $s_hour = 12;

        $e_hour = $end['hour'];
        if (strtoupper($end['ampm']) == 'AM' and $e_hour == '0') $e_hour = 12;

        $allday = (isset($date['allday']) ? $date['allday'] : 0);

        // All Day Event
        if ($allday) {
            $s_hour = 12;
            $start['minutes'] = 1;
            $start['ampm'] = 'AM';

            $e_hour = 11;
            $end['minutes'] = 59;
            $end['ampm'] = 'PM';
        }

        $start_time = $start['date'] . ' ' . sprintf("%02d", $s_hour) . ':' . sprintf("%02d", $start['minutes']) . ' ' . $start['ampm'];
        $end_time = $end['date'] . ' ' . sprintf("%02d", $e_hour) . ':' . sprintf("%02d", $end['minutes']) . ' ' . $end['ampm'];

        $start['timestamp'] = strtotime($start_time);
        $end['timestamp'] = strtotime($end_time);

        $hide_time = (isset($date['hide_time']) ? $date['hide_time'] : 0);
        $past = (isset($date['past']) ? $date['past'] : 0);

        return array(
            'start' => $start,
            'end' => $end,
            'allday' => $allday,
            'hide_time' => $hide_time,
            'past' => $past,
        );
    }

    /**
     *  Render advanced dates
     * @param array $advanced_days
     * @param array $event_info
     * @param int $maximum
     * @param string $referer_date
     * @param string $mode
     * @return array
     * @author Webnus <info@webnus.net>
     */
    public function generate_advanced_days($advanced_days = array(), $event_info = array(), $maximum = 6, $referer_date = NULL, $mode = 'render')
    {
        $main = MEC::getInstance('app.libraries.main');
        if (!count($advanced_days)) return array();
        if (!trim($referer_date)) $referer_date = date('Y-m-d', current_time('timestamp', 0));

        $levels = array('first', 'second', 'third', 'fourth', 'last');
        $year = date('Y', strtotime($event_info['start']['date']));
        $dates = array();

        // Set last month for include current month results
        $month = date('m', strtotime('first day of last month', strtotime($event_info['start']['date'])));

        if ($month == '12') $year = $year - 1;

        $maximum = intval($maximum);
        $i = 0;

        // Event info
        $exceptional_days = array_key_exists('exceptional_days', $event_info) ? $event_info['exceptional_days'] : array();
        $start_date = $event_info['start'];
        $end_date = $event_info['end'];
        $allday = array_key_exists('allday', $event_info) ? $event_info['allday'] : 0;
        $hide_time = array_key_exists('hide_time', $event_info) ? $event_info['hide_time'] : 0;
        $finish_date = array_key_exists('finish_date', $event_info) ? $event_info['finish_date'] : '0000-00-00';
        $event_period = $main->date_diff($start_date['date'], $end_date['date']);
        $event_period_days = $event_period ? $event_period->days : 0;
        $mec_repeat_end = array_key_exists('mec_repeat_end', $event_info) ? $event_info['mec_repeat_end'] : '';
        $occurrences = array_key_exists('occurrences', $event_info) ? $event_info['occurrences'] : 0;

        // Include default start date to results
        if (!$main->is_past($start_date['date'], $referer_date) and !in_array($start_date['date'], $exceptional_days)) {
            $dates[] = $this->add_timestamps(array(
                'start' => $start_date,
                'end' => $end_date,
                'allday' => $allday,
                'hide_time' => $hide_time,
                'past' => 0,
            ));

            if ($mode == 'render') $i++;
        }

        while ($i < $maximum) {
            $start = NULL;

            foreach ($advanced_days as $day) {
                if ($i >= $maximum) break;

                // Explode $day value for example (Sun.1) to Sun and 1
                $d = explode('.', $day);

                // Set indexes for {$levels} index if number day is Last(Sun.l) then indexes set 4th {$levels} index
                $index = intval($d[1]) ? (intval($d[1]) - 1) : 4;

                // Generate date
                $date = date('Y-m-t', strtotime("{$year}-{$month}-01"));

                // Generate start date for example "first Sun of next month"
                $start = date('Y-m-d', strtotime("{$levels[$index]} {$d[0]} of next month", strtotime($date)));
                $end = date('Y-m-d', strtotime("+{$event_period_days} Days", strtotime($start)));

                // When ends repeat date set
                if ($mode == 'render' and $main->is_past($finish_date, $start)) continue;

                // Jump to next level if start date is past
                if ($main->is_past($start, $referer_date) or in_array($start, $exceptional_days)) continue;

                // Add dates
                $dates[] = $this->add_timestamps(array(
                    'start' => array(
                        'date' => $start,
                        'hour' => $start_date['hour'],
                        'minutes' => $start_date['minutes'],
                        'ampm' => $start_date['ampm'],
                    ),
                    'end' => array(
                        'date' => $end,
                        'hour' => $end_date['hour'],
                        'minutes' => $end_date['minutes'],
                        'ampm' => $end_date['ampm'],
                    ),
                    'allday' => $allday,
                    'hide_time' => $hide_time,
                    'past' => 0,
                ));

                $i++;
            }

            // When ends repeat date set
            if ($mode == 'render' and $main->is_past($finish_date, $start)) break;

            // Change month and years for next resualts
            if (intval($month) == 12) {
                $year = intval($year) + 1;
                $month = '00';
            }

            $month = sprintf("%02d", intval($month) + 1);
        }

        if (($mode == 'render') and (trim($mec_repeat_end) == 'occurrences') and (count($dates) > $occurrences)) {
            $max = strtotime(reset($dates)['start']['date']);
            $pos = 0;

            for ($i = 1; $i < count($dates); $i++) {
                if (strtotime($dates[$i]['start']['date']) > $max) {
                    $max = strtotime($dates[$i]['start']['date']);
                    $pos = $i;
                }
            }

            unset($dates[$pos]);
        }

        // Remove Duplicates
        $uniques = array();
        $timestamps = array();

        foreach ($dates as $key => $date) {
            $start_timestamp = $date['start']['timestamp'];
            $end_timestamp = $date['end']['timestamp'];
            $timestamp_key = $start_timestamp . '-' . $end_timestamp;

            if (isset($timestamps[$timestamp_key])) continue;

            $timestamps[$timestamp_key] = true;
            $uniques[] = $date;
        }

        // Sort
        usort($uniques, array($this, 'sort_dates'));

        return $uniques;
    }

    public function sort_dates($a, $b)
    {
        $a_timestamp = $a['start']['timestamp'];
        $b_timestamp = $b['end']['timestamp'];

        if ($a_timestamp == $b_timestamp) return 0;
        return ($a_timestamp > $b_timestamp) ? +1 : -1;
    }

    public function show_other_organizers($event, $organizers_data)
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta($event, true);

        $additional_organizers_status = (!isset($settings['additional_organizers']) or (isset($settings['additional_organizers']) and $settings['additional_organizers'])) ? true : false;
        if (!$additional_organizers_status) return;

        $organizer_id = $main->get_master_organizer_id($event);

        $organizers = array();
        $html = '';
        if (isset($organizers_data) && !empty($organizers_data)) {
            foreach ($organizers_data as $o)
                if ($o['id'] != $organizer_id)
                    $organizers[$o['id']] = $o;

            if (!count($organizers)) return;

            $organizer_ids = get_post_meta((int)$event, 'mec_additional_organizer_ids', true);
            if (!is_array($organizer_ids)) $organizer_ids = array();
            $organizer_ids = array_unique($organizer_ids);
            $html .= '<div class="mec-single-event-additional-organizers">';
            $html .= ' <h3 class="mec-events-single-section-title">' . esc_html($main->m('other_organizers', esc_html__('Other Organizers', 'mec'))) . '</h3>';
            foreach ($organizer_ids as $o_id) {
                $o_id = apply_filters('wpml_object_id', $o_id, 'mec_organizer', true);
                if ($o_id == $organizer_id) continue;
                $organizer = (isset($organizers[$o_id]) ? $organizers[$o_id] : NULL);
                if (!$organizer) continue;

                $html .= '<div class="mec-single-event-additional-organizer">';
                if (isset($organizer['thumbnail']) and trim($organizer['thumbnail'])) {
                    if (class_exists('MEC_Fluent\Core\pluginBase\MecFluent') && (isset($settings['single_single_style']) and $settings['single_single_style'] == 'fluent')) {
                        $html .= '<img class="mec-img-organizer" src="' . esc_url(MEC_Fluent\Core\pluginBase\MecFluent::generateCustomThumbnailURL($organizer['thumbnail'], 83, 83, true)) . '" alt="' . (isset($organizer['name']) ? esc_attr($organizer['name']) : '') . '">';
                    } else {
                        $html .= '<img class="mec-img-organizer" src="' . esc_url($organizer['thumbnail']) . '" alt="' . (isset($organizer['name']) ? esc_attr($organizer['name']) : '') . '">';
                    }
                }
                $html .= '<dl>';
                if (isset($organizer['thumbnail'])) {
                    $html .= ' <dd class="mec-organizer">';
                    $html .= '<i class="mec-sl-home"></i>';
                    $html .= '<h6>' . (isset($organizer['name']) ? esc_html($organizer['name']) : '') . '</h6>';
                    $html .= '</dd>';
                }
                if (isset($organizer['tel']) && !empty($organizer['tel'])) {
                    $html .= '<dd class="mec-organizer-tel">';
                    $html .= '<i class="mec-sl-phone"></i>';
                    $html .= '<h6>' . esc_html_e('Phone', 'mec') . '</h6>';
                    $html .= '<a href="tel:' . esc_attr($organizer['tel']) . '">' . esc_html($organizer['tel']) . '</a>';
                    $html .= '</dd>';
                }
                if (isset($organizer['email']) && !empty($organizer['email'])) {
                    $html .= '<dd class="mec-organizer-email">';
                    $html .= '<i class="mec-sl-envelope"></i>';
                    $html .= '<h6>' . esc_html_e('Email', 'mec') . '</h6>';
                    $html .= '<a href="mailto:' . esc_attr($organizer['email']) . '">' . esc_html($organizer['email']) . '</a>';
                    $html .= '</dd>';
                }
                if (isset($organizer['url']) && !empty($organizer['url']) and $organizer['url'] != 'http://') {
                    $html .= '<dd class="mec-organizer-url">';
                    $html .= '<i class="mec-sl-sitemap"></i>';
                    $html .= '<h6>' . esc_html_e('Website', 'mec') . '</h6>';
                    $html .= '<span><a href="' . esc_url($organizer['url']) . '" class="mec-color-hover" target="_blank">' . esc_html($organizer['url']) . '</a></span>';
                    $html .= '</dd>';
                }
                $organizer_description_setting = isset($settings['addintional_organizers_description']) ? $settings['addintional_organizers_description'] : '';
                $organizer_terms = get_the_terms($event->data, 'mec_organizer');
                if ($organizer_description_setting == '1') {
                    foreach ($organizer_terms as $organizer_term) {
                        if ($organizer_term->term_id == $organizer['id']) {
                            if (isset($organizer_term->description) && !empty($organizer_term->description)) {
                                $html .= '<dd class="mec-organizer-description">';
                                $html .= '<p>' . esc_html($organizer_term->description) . '</p>';
                                $html .= '</dd>';
                            }
                        }
                    }
                }
                $html .= '</dl>';
                $html .= '</div>';
            }
            $html .= '</div>';
        }

        return $html;
    }

    public function get_display_booking_form($event_id)
    {

        ob_start();
        \MEC\Books\BookingForm::getInstance()->display_form($event_id);
        return ob_get_clean();
    }

    public function display_related_posts_widget($event)
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');

        $html = '';
        if (!isset($settings['related_events'])) return;
        if (isset($settings['related_events']) && $settings['related_events'] != '1') return;

        if (is_numeric($event)) $event_id = $event;
        else return;

        $limit = (isset($settings['related_events_limit']) and trim($settings['related_events_limit'])) ? $settings['related_events_limit'] : 30;

        $related_args = array(
            'post_type' => $main->get_main_post_type(),
            'posts_per_page' => max($limit, 20),
            'post_status' => 'publish',
            'post__not_in' => array($event_id),
            'tax_query' => array(),
            'meta_query' => array(
                'mec_start_date' => array(
                    'key' => 'mec_start_date',
                ),
                'mec_start_day_seconds' => array(
                    'key' => 'mec_start_day_seconds',
                ),
            ),
            'orderby' => array(
                'mec_start_date' => 'ASC',
                'mec_start_day_seconds' => 'ASC',
            ),
        );

        if (isset($settings['related_events_basedon_category']) && $settings['related_events_basedon_category'] == 1) {
            $post_terms = wp_get_object_terms($event_id, 'mec_category', array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => 'mec_category',
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        if (isset($settings['related_events_basedon_organizer']) && $settings['related_events_basedon_organizer'] == 1) {
            $post_terms = wp_get_object_terms($event_id, 'mec_organizer', array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => 'mec_organizer',
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        if (isset($settings['related_events_basedon_location']) && $settings['related_events_basedon_location'] == 1) {
            $post_terms = wp_get_object_terms($event_id, 'mec_location', array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => 'mec_location',
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        if (isset($settings['related_events_basedon_speaker']) && $settings['related_events_basedon_speaker'] == 1) {
            $post_terms = wp_get_object_terms($event_id, 'mec_speaker', array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => 'mec_speaker',
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        if (isset($settings['related_events_basedon_label']) && $settings['related_events_basedon_label'] == 1) {
            $post_terms = wp_get_object_terms($event_id, 'mec_label', array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => 'mec_label',
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        if (isset($settings['related_events_basedon_tag']) && $settings['related_events_basedon_tag'] == 1) {
            $post_terms = wp_get_object_terms($event_id, apply_filters('mec_taxonomy_tag', ''), array('fields' => 'slugs'));
            $related_args['tax_query'][] = array(
                'taxonomy' => apply_filters('mec_taxonomy_tag', ''),
                'field' => 'slug',
                'terms' => $post_terms
            );
        }

        // Display Expired Events
        $display_expired_events = (isset($settings['related_events_display_expireds']) && $settings['related_events_display_expireds']);

        $related_args['tax_query']['relation'] = 'OR';
        $related_args = apply_filters('mec_add_to_related_post_query', $related_args, $event_id);

        $now = current_time('timestamp');
        $printed = 0;

        $query = new WP_Query($related_args);
        if ($query->have_posts()) {
            $html .= '<div class="row mec-related-events-wrap">';
            $html .= '<h3 class="mec-rec-events-title">' . esc_html__('Related Events', 'mec') . '</h3>';
            $html .= '<div class="mec-related-events">';
            while ($query->have_posts()) {
                if ($printed >= min($limit, 4)) break;
                $query->the_post();
                // Event Repeat Type
                $repeat_type = get_post_meta(get_the_ID(), 'mec_repeat_type', true);

                $occurrence = date('Y-m-d');
                if (!in_array($repeat_type, array('certain_weekdays', 'custom_days', 'weekday', 'weekend', 'advanced'))) $occurrence = date('Y-m-d', strtotime($occurrence));
                $new_occurrence = date('Y-m-d', strtotime('-1 day', strtotime($occurrence)));
                if (in_array($repeat_type, array('monthly')) and date('m', strtotime($new_occurrence)) != date('m', strtotime($occurrence))) $new_occurrence = date('Y-m-d', strtotime($occurrence));

                $occurrence = $new_occurrence;

                $start_timestamp = !empty($start_timestamp) ? $start_timestamp : strtotime('Yesterday');
                $dates = $this->dates(get_the_ID(), NULL, 1, $occurrence);

                $d = (isset($dates[0]) ? $dates[0] : array());

                // Don't show Expired Events
                $timestamp = (isset($d['start']) and isset($d['start']['timestamp'])) ? $d['start']['timestamp'] : 0;
                if ($display_expired_events or $timestamp <= 0 or $timestamp > $now) {

                    $printed += 1;
                    $mec_date = (isset($d['start']) and isset($d['start']['date'])) ? $d['start']['date'] : get_post_meta(get_the_ID(), 'mec_start_date', true);
                    $date = $main->date_i18n(get_option('date_format'), strtotime($mec_date));

                    $event_link = $main->get_event_date_permalink(get_the_permalink(), $mec_date);

                    // Custom Link
                    $read_more = get_post_meta(get_the_ID(), 'mec_read_more', true);
                    $read_more_occ_url = MEC_feature_occurrences::param(get_the_ID(), $timestamp, 'read_more', $read_more);

                    if ($read_more_occ_url and filter_var($read_more_occ_url, FILTER_VALIDATE_URL)) $event_link = $read_more_occ_url;

                    $html .= '<article class="mec-related-event-post col-md-3 col-sm-3">';
                    $html .= '<figure>';
                    $html .= '<a href="' . esc_url($event_link) . '">';
                    if (get_the_post_thumbnail(get_the_ID(), 'thumblist')) $html .= get_the_post_thumbnail(get_the_ID(), 'thumblist');
                    else $html .= '<img src="' . esc_url($main->asset('img/no-image.png')) . '" />';
                    $html .= '</a>';
                    $html .= '</figure>';
                    $html .= '<div class="mec-related-event-content">';
                    $html .= '<span>' . esc_html($date) . '</span>';
                    $html .= '<h5>';
                    $html .= '<a class="mec-color-hover" href="' . esc_url($event_link) . '">' . get_the_title() . '</a>';
                    if ($display_expired_events && $timestamp && $timestamp < $now) {
                        $html .= '<span class="mec-holding-status mec-holding-status-expired">' . esc_html(translate('Expired!', 'mec')) . '</span>';
                    }
                    $html .= '</h5>';
                    $html .= '</div>';
                    $html .= '</article>';
                }
            }
            $html .= '</div>';
            $html .= '</div>';
        }

        wp_reset_postdata();

        return $html;
    }

    public function display_data_fields($event, $sidebar = false, $shortcode = false)
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta($event, true);

        $html = '';

        $display = isset($settings['display_event_fields']) ? (boolean)$settings['display_event_fields'] : true;
        if (!$display and !$sidebar and !$shortcode) return;

        $fields = $main->get_event_fields();
        if (!is_array($fields) or (is_array($fields) and !count($fields))) return;

        $start_timestamp = !empty($start_timestamp) ? $start_timestamp : strtotime('Yesterday');
        $dates = $this->dates($event, NULL, 6, date('Y-m-d H:i', $start_timestamp));
        $date = $dates[0] ?? array();
        // Start Timestamp
        $start_timestamp = (isset($date) and isset($date['start']) and isset($date['start']['timestamp'])) ? $date['start']['timestamp'] : NULL;

        $data = (isset($meta) and isset($meta['mec_fields']) and is_array($meta['mec_fields'])) ? $meta['mec_fields'] : get_post_meta($event, 'mec_fields', true);
        if ($start_timestamp) $data = MEC_feature_occurrences::param($event, $start_timestamp, 'fields', $data);

        if (!is_array($data) or (is_array($data) and !count($data))) return;

        foreach ($fields as $n => $item) {
            // n meaning number
            if (!is_numeric($n)) continue;

            $result = isset($data[$n]) ? $data[$n] : NULL;
            if ((!is_array($result) and trim($result) == '') or (is_array($result) and !count($result))) continue;

            $content = isset($item['type']) ? $item['type'] : 'text';
            if ($content === 'checkbox') {
                $cleaned = array();
                foreach ($result as $k => $v) {
                    if (trim($v) !== '') $cleaned[] = $v;
                }

                $value = $cleaned;
                if (!count($value)) {
                    $content = NULL;
                    continue;
                }
            }
        }

        if (isset($content) && $content != NULL && (isset($settings['display_event_fields_backend']) and $settings['display_event_fields_backend'] == 1) or !isset($settings['display_event_fields_backend'])) {
            $date_format = get_option('date_format');

            $html .= '<div class="mec-event-data-fields mec-frontbox ' . ($sidebar ? 'mec-data-fields-sidebar' : '') . ' ' . ($shortcode ? 'mec-data-fields-shortcode mec-util-hidden' : '') . '">';
            $html .= '<div class="mec-data-fields-tooltip">';
            $html .= '<div class="mec-data-fields-tooltip-box">';
            $html .= '<ul class="mec-event-data-field-items">';
            foreach ($fields as $f => $field) {
                if (!is_numeric($f)) continue;
                $value = isset($data[$f]) ? $data[$f] : NULL;
                if ((!is_array($value) and trim($value) == '') or (is_array($value) and !count($value))) continue;

                $type = isset($field['type']) ? $field['type'] : 'text';
                if ($type === 'checkbox') {
                    $cleaned = array();
                    foreach ($value as $k => $v) {
                        if (trim($v) !== '') $cleaned[] = $v;
                    }

                    $value = $cleaned;
                    if (!count($value)) continue;
                }

                $html .= '<li class="mec-event-data-field-item mec-field-item-' . esc_attr($type) . '">';
                if (isset($field['label'])) {
                    $html .= '<span class="mec-event-data-field-name">' . esc_html(translate(stripslashes($field['label']), 'mec')) . ': </span>';
                }

                if ($type === 'email') {
                    $html .= '<span class="mec-event-data-field-value"><a href="mailto:' . esc_attr($value) . '">' . esc_html($value) . '</a></span>';
                } elseif ($type === 'tel') {
                    $html .= '<span class="mec-event-data-field-value"><a href="tel:' . esc_attr($value) . '">' . esc_html($value) . '</a></span>';
                } elseif ($type === 'url') {
                    $html .= '<span class="mec-event-data-field-value"><a href="' . esc_url($value) . '" target="_blank">' . esc_html($value) . '</a></span>';
                } elseif ($type === 'date') {
                    $value = $main->to_standard_date($value);
                    $html .= '<span class="mec-event-data-field-value">' . esc_html($main->date_i18n($date_format, strtotime($value))) . '</span>';
                } elseif ($type === 'textarea') {
                    $html .= '<span class="mec-event-data-field-value">' . wpautop(stripslashes($value)) . '</span>';
                } else {
                    $html .= '<span class="mec-event-data-field-value">' . (is_array($value) ? esc_html(stripslashes(implode(', ', $value))) : esc_html(stripslashes($value))) . '</span>';
                }
                $html .= '</li>';
            }
            $html .= '</ul>';
            $html .= '</div>';
            $html .= '</div>';
            $html .= '</div>';

        }

        return $html;
    }

    public function display_cancellation_reason($event, $display_reason = false)
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');
        $meta = $main->get_post_meta($event, true);

        $event_id = $event;

        $allday = isset($meta['mec_allday']) ? $meta['mec_allday'] : 0;
        $hide_time = isset($meta['mec_hide_time']) ? $meta['mec_hide_time'] : 0;
        $hide_end_time = isset($meta['mec_hide_end_time']) ? $meta['mec_hide_end_time'] : 0;

        $start_timestamp = ((isset($meta['mec_start_day_seconds']) and isset($meta['mec_start_date'])) ? (strtotime($meta['mec_start_date']) + $meta['mec_start_day_seconds']) : (isset($meta['mec_start_date']) ? strtotime($meta['mec_start_date']) : 0));
        $end_timestamp = ((isset($meta['mec_end_day_seconds']) and isset($meta['mec_end_date'])) ? (strtotime($meta['mec_end_date']) + $meta['mec_end_day_seconds']) : (isset($meta['mec_end_date']) ? strtotime($meta['mec_end_date']) : 0));

        $start_time = $main->get_time($start_timestamp);
        $end_time = $main->get_time($end_timestamp);
        $time = array();
        if ($hide_time) {
            $time = array(
                'start' => '',
                'end' => '',
                'start_raw' => $start_time,
                'end_raw' => $end_time,
                'start_timestamp' => $start_timestamp,
                'end_timestamp' => $end_timestamp,
            );
        } elseif ($allday) {
            $time = array(
                'start' => $main->m('all_day', esc_html__('All Day', 'mec')),
                'end' => '',
                'start_raw' => $start_time,
                'end_raw' => $end_time,
                'start_timestamp' => $start_timestamp,
                'end_timestamp' => $end_timestamp,
            );
        } else {
            $time = array(
                'start' => $start_time,
                'end' => ($hide_end_time ? '' : $end_time),
                'start_raw' => $start_time,
                'end_raw' => $end_time,
                'start_timestamp' => $start_timestamp,
                'end_timestamp' => $end_timestamp,
            );
        }

        $timestamp = !empty($start_timestamp) ? $start_timestamp : strtotime('Yesterday');
        $dates = $this->dates($event, NULL, 6, date('Y-m-d H:i', $timestamp));
        $date = $dates[0] ?? array();


        $start_timestamp = (isset($time['start_timestamp']) ? $time['start_timestamp'] : (isset($time['start']['timestamp']) ? $time['start']['timestamp'] : strtotime($date['start']['date'])));

        // All Params
        $params = MEC_feature_occurrences::param($event_id, $start_timestamp, '*');

        $event_status = (isset($meta['mec_event_status']) and trim($meta['mec_event_status'])) ? $meta['mec_event_status'] : 'EventScheduled';
        $event_status = (isset($params['event_status']) and trim($params['event_status']) != '') ? $params['event_status'] : $event_status;

        $reason = get_post_meta($event_id, 'mec_cancelled_reason', true);
        $reason = (isset($params['cancelled_reason']) and trim($params['cancelled_reason']) != '') ? $params['cancelled_reason'] : $reason;

        $output = '';
        if (isset($event_status) and $event_status == 'EventCancelled' && $display_reason != false and isset($reason) and !empty($reason)) {
            $output = '<div class="mec-cancellation-reason"><span>' . MEC_kses::element($reason) . '</span></div>';
        }

        return $output;
    }

    public function display_breadcrumb_widget($page_id)
    {
        $settings = \MEC\Settings\Settings::getInstance()->get_settings();
        $main = MEC::getInstance('app.libraries.main');
        $html = '';
        /**
         * Home Page
         */
        $homeURL = esc_url(home_url('/'));
        $html .= '<div class="mec-address"><a href="' . esc_url($homeURL) . '"> ' . esc_html__('Home', 'mec') . ' </a> <i class="mec-color mec-sl-arrow-right"></i> ';

        $archive_title = $main->get_archive_title();
        $archive_link = $main->get_archive_url();

        $referer_url = wp_get_referer();
        if (trim($referer_url)) {
            $referer_page_id = url_to_postid($referer_url);
            if ($referer_page_id and strpos(get_post_field('post_content', $referer_page_id), '[MEC') !== false) {
                $archive_link = $referer_url;
                $archive_title = get_the_title($referer_page_id);
            }
        }

        /**
         * Archive Page
         */
        if ($archive_link) $html .= '<a href="' . esc_url($archive_link) . '">' . esc_html($archive_title) . '</a> <i class="mec-color mec-sl-arrow-right"></i> ';

        /**
         * Categories Page
         */
        if (!isset($settings['breadcrumbs_category']) or (isset($settings['breadcrumbs_category']) and $settings['breadcrumbs_category'])) {
            $categories = wp_get_post_terms($page_id, 'mec_category');
            if (!is_array($categories)) $categories = array();

            foreach ($categories as $category) $html .= '<a href="' . esc_url(get_term_link($category)) . '">' . esc_html($category->name) . '</a> <i class="mec-color mec-sl-arrow-right"></i> ';
        }

        /**
         * Current Event
         */
        $html .= '<span class="mec-current">' . get_the_title($page_id) . '</span></div>';

        return $html;
    }

    public function event_gallery($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-gallery', 0, []);
        $content_render = '<div><div class="event-gallery-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_export($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-export', 0, []);
        $content_render = '<div><div class="event-export-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_faq($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-faq', 0, []);
        $content_render = '<div><div class="event-faq-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_locations($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-locations', 0, []);
        $content_render = '<div><div class="event-locations-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_sponsors($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-sponsors', 0, []);
        $content_render = '<div><div class="event-sponsors-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_trailer_url($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('event-trailer-url', 0, []);
        $content_render = '<div><div class="event-trailer-url-gsb">' . $html . '</div></div>';
        return $content_render;
    }

    public function event_rsvp($page_id)
    {
        $html = \MEC\SingleBuilder\SingleBuilder::getInstance()->output('rsvp-form', 0, []);
        $content_render = '<div><div class="event-rsvp-gsb">' . $html . '</div></div>';
        return $content_render;
    }
}

SingleBlocks::get_instance();
