<?php
/**
 * Style Manager base.
 *
 * @package MEC_GSB
 */

namespace MEC_GSB\Core\Gutenberg;

if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

abstract class StyleManagerBase
{

    /**
     * Controls manager instance.
     *
     * @var object
     */
    private $controls_manager;

    /**
     * Constructor.
     *
     * @param string $block Block register handle
     * @since 1.0.0
     *
     */
    public function __construct($block)
    {
        $this->controls_manager = webnus_sm_register_style_for_block($block);
    }

    /**
     * Color control.
     *
     * @param string $id
     * @param string $heading
     * @param string $selector
     * @param string $property
     * @since 1.0.0
     *
     */
    protected function color($id, $heading, $selector, $property)
    {
        $this->controls_manager->add_control(
            array(
                'id' => $id,
                'type' => 'color-picker',
                'label' => esc_html__($heading, 'webnus-style-manager'),
                'css_selector' => array(
                    '{{WRAPPER}} ' . $selector . '' => '' . $property . ': {{VALUE}}',
                ),
            )
        );
    }

    /**
     * Color control.
     *
     * @param string $id
     * @param string $heading
     * @param string $selector
     * @param string $property
     * @since 1.0.0
     *
     */
    protected function range($id, $heading, $selector, $responsive = false, $units = '', $default_unit = '', $property = '')
    {
        $control = $responsive ? 'add_responsive_control' : 'add_control';

        $units_arr = array();

        $px_unit = array(
            'value' => 'px',
            'intervals' => array(
                'step' => 1,
                'min' => 1,
                'max' => 500,
            ),
        );

        $percent_unit = array(
            'value' => '%',
            'intervals' => array(
                'step' => 1,
                'min' => 1,
                'max' => 100,
            ),
        );

        if (in_array('px', $units)) {
            array_push($units_arr, $px_unit);
        }

        if (in_array('%', $units)) {
            array_push($units_arr, $percent_unit);
        }

        $this->controls_manager->$control(
            array(
                'id' => $id,
                'type' => 'range',
                'label' => esc_html__($heading, 'webnus-style-manager'),
                'css_selector' => array(
                    '{{WRAPPER}} ' . $selector . '' => '' . $property . ': {{VALUE}}{{UNIT}};',
                ),
                'attributes' => array(
                    'default' => array(
                        'value' => array(
                            'value' => '',
                            'unit' => $default_unit,
                        ),
                    ),
                ),
                'units' => $units_arr,
            )
        );
    }

    /**
     * Select control.
     *
     * @param string $id
     * @param string $heading
     * @param string $selector
     * @param array $options
     * @param string $property
     * @since 1.0.0
     *
     */
    protected function select($id, $heading, $selector = '', $options = array(), $property='')
    {
        $this->controls_manager->add_control(
            array(
                'id' => $id,
                'type' => 'select',
                'label' => esc_html__($heading, 'webnus-style-manager'),
                'css_selector' => array(
                    '{{WRAPPER}} ' . $selector . '' => '' . $property . ': {{VALUE}}',
                ),
                'options' => $options,
            )
        );
    }

    /**
     * Choose control.
     *
     * @param string $id
     * @param string $heading
     * @param string $selector
     * @param array $options
     * @param string $property
     * @since 1.0.0
     *
     */
    protected function choose($id, $heading, $selector = '', $options = array(), $property='')
    {
        $this->controls_manager->add_control(
            array(
                'id' => $id,
                'type' => 'choose',
                'label' => esc_html__($heading, 'webnus-style-manager'),
                'options' => $options,
                'css_selector' => array(
                    '{{WRAPPER}} ' . $selector . '' => '' . $property . ': {{VALUE}};',
                ),
            )
        );
    }

    /**
     * Dimensions control.
     *
     * @param string $id
     * @param string $heading
     * @param string $selector
     * @param string $property
     * @since 1.0.0
     *
     */
    protected function dimensions($id, $heading, $selector, $property)
    {
        $this->controls_manager->add_responsive_control(
            array(
                'id' => $id,
                'type' => 'dimensions',
                'label' => esc_html__($heading, 'webnus-style-manager'),
                'units' => array('px', '%'),
                'css_selector' => array(
                    '{{WRAPPER}} ' . $selector . '' => '' . $property . ': {{TOP}} {{RIGHT}} {{BOTTOM}} {{LEFT}};',
                ),
            )
        );
    }

    /**
     * Start section.
     *
     * @param string $id
     * @param string $heading
     * @since 1.0.0
     *
     */
    protected function start_section($id, $heading)
    {
        $this->controls_manager->start_section(
            'style_controls',
            array(
                'id' => $id,
                'initialOpen' => true,
                'title' => esc_html__($heading, 'webnus-style-manager'),
            )
        );
    }

    /**
     * End section.
     *
     * @since 1.0.0
     */
    protected function end_section()
    {
        $this->controls_manager->end_section();
    }

    /**
     * Get controls.
     *
     * @since 1.0.0
     */
    protected function get_controls()
    {
        return $this->controls_manager;
    }

    /**
     * Start tabs.
     *
     * @param string $id
     * @since 1.0.0
     *
     */
    protected function start_tabs($id)
    {
        $this->controls_manager->start_tabs(
            'style_controls',
            array(
                'id' => $id,
                'separator' => 'both',
            )
        );
    }

    /**
     * End tabs.
     *
     * @since 1.0.0
     */
    protected function end_tabs()
    {
        $this->controls_manager->end_tabs();
    }

    /**
     * Start tab.
     *
     * @param string $id
     * @param string $heading
     * @since 1.0.0
     *
     */
    protected function start_tab($id, $heading)
    {
        $this->controls_manager->start_tab(
            'style_controls',
            array(
                'id' => $id,
                'title' => esc_html__($heading, 'webnus-style-manager'),
            )
        );
    }

    /**
     * End tab.
     *
     * @since 1.0.0
     */
    protected function end_tab()
    {
        $this->controls_manager->end_tab();
    }
}
