<?php
/**
 * Style Manager.
 *
 * @package MEC_GSB
 */

namespace MEC_GSB\Core\Gutenberg;

if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Class Controls
 */
class StyleManagerControls extends StyleManagerBase
{

	/**
	 * CSS selector
	 *
	 * @var string
	 */
	private $selector;

	/**
	 * Block name
	 *
	 * @var string
	 */
	private $block;

    /**
	 * Block Type name
	 *
	 * @var string
	 */
	private $type;

	/**
	 * Handle prefix
	 *
	 * @var string
	 */
	private $prefix;

	/**
	 * Constructor.
	 *
	 * @param string $block Block register handle
	 * @since 1.0.0
	 *
	 */
	public function __construct($block)
	{
		parent::__construct($block);
		$this->block = $block;
	}

	/**
	 * Text section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function text($selector, $heading,$type)
	{

        $this->type=$type;
		/**
		 * Start section
		 */
		$this->start_section('heading_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('text_tab'.$this->type);

		/**
		 * Start normal tab
		 */
		$this->start_tab('text_normal'.$this->type, 'Normal');

		/**
		 * Text normal
		 */
		$this->text_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('text_hover'.$this->type, 'Hover');

		/**
		 * Text hover
		 */
		$this->text_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();
		/**
		 * End tabs
		 */
		$this->end_tabs();


		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Text normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function text_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Color
		 */
		$this->color($this->prefix . 'heading_color'.$this->type, 'Color', $this->selector, 'color');

		/**
		 * Font size
		 */
		$this->range($this->prefix . 'font_size'.$this->type, 'Font Size', $this->selector, true, array('px', '%'), 'px', 'font-size');

		/**
		 * Line height
		 */
		$this->range($this->prefix . 'line_height'.$this->type, 'Line Height', $this->selector, true, array('px', '%'), 'px', 'line-height');

		/**
		 * Font weight
		 */
		$this->font_weight();

		/**
		 * Font style
		 */
		$this->font_style();

		/**
		 * Text align
		 */
		$this->text_align();

		/**
		 * Text transform
		 */
		$this->text_transform();

		/**
		 * Text decoration
		 */
		$this->text_decoration();

		/**
		 * Letter spacing
		 */
		$this->range($this->prefix . 'letter_spacing'.$this->type, 'Letter Spacing', $this->selector, true, array('px'), 'px', 'letter-spacing');

		/**
		 * Writing mode
		 */
		$this->writing_mode();

		/**
		 * Text orientation
		 */
		$this->text_orientation();
	}

	/**
	 * Text hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function text_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Color hover
		 */
		$this->color($this->prefix . 'heading_color'.$this->type, 'Color', $this->selector, 'color');

		/**
		 * Font size hover
		 */
		$this->range($this->prefix . 'font_size'.$this->type, 'Font Size', $this->selector, true, array('px', '%'), 'px', 'font-size');

		/**
		 * Line height hover
		 */
		$this->range($this->prefix . 'line_height'.$this->type, 'Line Height', $this->selector, true, array('px', '%'), 'px', 'line-height');

		/**
		 * Font weight hover
		 */
		$this->font_weight();

		/**
		 * Font style hover
		 */
		$this->font_style();

		/**
		 * Text align hover
		 */
		$this->text_align();

		/**
		 * Text transform hover
		 */
		$this->text_transform();

		/**
		 * Text decoration hover
		 */
		$this->text_decoration();

		/**
		 * Letter spacing hover
		 */
		$this->range($this->prefix . 'letter_spacing'.$this->type, 'Letter Spacing', $this->selector, true, array('px'), 'px', 'letter-spacing');

		/**
		 * Writing mode hover
		 */
		$this->writing_mode();

		/**
		 * Text orientation hover
		 */
		$this->text_orientation();
	}

	/**
	 * Background section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function background($selector, $heading,$type)
	{
        $this->type=$type;

		/**
		 * Start section
		 */
		$this->start_section('background_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('background_tab'.$this->type);

		/**
		 * Start normal tab
		 */
		$this->start_tab('background_normal'.$this->type, 'Normal');

		/**
		 * Background normal
		 */
		$this->background_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('background_hover'.$this->type, 'Hover');

		/**
		 * Background hover
		 */
		$this->background_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Background normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function background_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Background color
		 */
		$this->color($this->prefix . 'background_color'.$this->type, 'Background Color', $this->selector, 'background-color');
	}

	/**
	 * Background hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function background_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Background color hover
		 */
		$this->color($this->prefix . 'background_color'.$this->type, 'Background Color', $this->selector, 'background-color');
	}


	/**
	 * Display section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function display($selector, $heading,$type)
	{
        $this->type=$type;

//		/**
//		 * Set CSS selector
//		 */
//		$this->set_selector($selector);

		/**
		 * Start section
		 */
		$this->start_section('display_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('display_tab'.$this->type);

		/**
		 * Start normal tab
		 */
		$this->start_tab('display_normal'.$this->type, 'Normal');

		/**
		 * Display normal
		 */
		$this->display_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('display_hover'.$this->type, 'Hover');

		/**
		 * Display hover
		 */
		$this->display_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Display normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function display_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Display
		 */
		$this->display_options();

		/**
		 * Box sizing
		 */
		$this->box_sizing();

		/**
		 * Float
		 */
		$this->float();

		/**
		 * Clear
		 */
		$this->clear();

		/**
		 * Overflow
		 */
		$this->overflow();

		/**
		 * Vertical align
		 */
		$this->vertical_align();

		/**
		 * Direction
		 */
		$this->direction();

		/**
		 * Cursor
		 */
		$this->cursor();

		/**
		 * Object fit
		 */
		$this->object_fit();

		/**
		 * Object position
		 */
		$this->object_position();

		/**
		 * Visibility
		 */
		$this->visibility();

		/**
		 * Opacity
		 */
		$this->opacity();
	}

	/**
	 * Display hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function display_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Display
		 */
		$this->display_options();

		/**
		 * Box sizing
		 */
		$this->box_sizing();

		/**
		 * Float
		 */
		$this->float();

		/**
		 * Clear
		 */
		$this->clear();

		/**
		 * Overflow
		 */
		$this->overflow();

		/**
		 * Vertical align
		 */
		$this->vertical_align();

		/**
		 * Direction
		 */
		$this->direction();

		/**
		 * Cursor
		 */
		$this->cursor();

		/**
		 * Object fit
		 */
		$this->object_fit();

		/**
		 * Object position
		 */
		$this->object_position();

		/**
		 * Visibility
		 */
		$this->visibility();

		/**
		 * Opacity
		 */
		$this->opacity();
	}

	/**
	 * Size section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function size($selector, $heading,$type)
	{
        $this->type=$type;

		/**
		 * Start section
		 */
		$this->start_section('size_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('size_tab'.$this->type);

		/**
		 * Start normal tab
		 */
		$this->start_tab('size_normal'.$this->type, 'Normal');

		/**
		 * Size normal
		 */
		$this->size_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('size_hover'.$this->type, 'Hover');

		/**
		 * Size hover
		 */
		$this->size_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Size normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function size_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Width
		 */
		$this->range($this->prefix . 'width'.$this->type, 'Width', $this->selector, true, array('px', '%'), 'px', 'width');

		/**
		 * Min Width
		 */
		$this->range($this->prefix . 'min_width'.$this->type, 'Min Width', $this->selector, true, array('px', '%'), 'px', 'min-width');

		/**
		 * Max Width
		 */
		$this->range($this->prefix . 'max_width'.$this->type, 'Max Width', $this->selector, true, array('px', '%'), 'px', 'max-width');

		/**
		 * Height
		 */
		$this->range($this->prefix . 'height'.$this->type, 'Height', $this->selector, true, array('px', '%'), 'px', 'height');

		/**
		 * Min Height
		 */
		$this->range($this->prefix . 'min_height'.$this->type, 'Min Height', $this->selector, true, array('px', '%'), 'px', 'min-height');

		/**
		 * Max Height
		 */
		$this->range($this->prefix . 'max_height'.$this->type, 'Max Height', $this->selector, true, array('px', '%'), 'px', 'max-height');
	}

	/**
	 * Size hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function size_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Width
		 */
		$this->range($this->prefix . 'width'.$this->type, 'Width', $this->selector, true, array('px', '%'), 'px', 'width');

		/**
		 * Min Width
		 */
		$this->range($this->prefix . 'min_width'.$this->type, 'Min Width', $this->selector, true, array('px', '%'), 'px', 'min-width');

		/**
		 * Max Width
		 */
		$this->range($this->prefix . 'max_width'.$this->type, 'Max Width', $this->selector, true, array('px', '%'), 'px', 'max-width');

		/**
		 * Height
		 */
		$this->range($this->prefix . 'height'.$this->type, 'Height', $this->selector, true, array('px', '%'), 'px', 'height');

		/**
		 * Min Height
		 */
		$this->range($this->prefix . 'min_height'.$this->type, 'Min Height', $this->selector, true, array('px', '%'), 'px', 'min-height');

		/**
		 * Max Height
		 */
		$this->range($this->prefix . 'max_height'.$this->type, 'Max Height', $this->selector, true, array('px', '%'), 'px', 'max-height');
	}

	/**
	 * Padding and margin section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function padding_margin($selector, $heading,$type)
	{
        $this->type=$type;

		/**
		 * Start section
		 */
		$this->start_section('spacing_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('spacing_tab'.$this->type);

		/**
		 * Padding and margin normal tab
		 */
		$this->start_tab('spacing_normal'.$this->type, 'Normal');

		/**
		 * Padding and margin normal
		 */
		$this->padding_margin_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('spacing_hover'.$this->type, 'Hover');

		/**
		 * Padding and margin hover
		 */
		$this->padding_margin_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Padding and margin normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function padding_margin_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Padding
		 */
		$this->dimensions($this->prefix . 'padding_control'.$this->type, 'Padding', $this->selector, 'padding');

		/**
		 * Margin
		 */
		$this->dimensions($this->prefix . 'margin_control'.$this->type, 'Margin', $this->selector, 'margin');
	}

	/**
	 * Padding and margin hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function padding_margin_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Padding
		 */
		$this->dimensions($this->prefix . 'padding_control'.$this->type, 'Padding', $this->selector, 'padding');

		/**
		 * Margin
		 */
		$this->dimensions($this->prefix . 'margin_control'.$this->type, 'Margin', $this->selector, 'margin');
	}

	/**
	 * Border section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function border($selector, $heading,$type)
	{
        $this->type=$type;

		/**
		 * Start section
		 */
		$this->start_section('border_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('border_tab'.$this->type);

		/**
		 * Border normal tab
		 */
		$this->start_tab('border_normal'.$this->type, 'Normal');

		/**
		 * Border normal
		 */
		$this->border_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('border_hover'.$this->type, 'Hover');

		/**
		 * Border hover
		 */
		$this->border_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Border normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function border_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Border
		 */
		$this->border_control();
	}

	/**
	 * Border hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function border_hover($selector)
	{
		/**
		 * Set CSS selector for hover
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Border
		 */
		$this->border_control();
	}

	/**
	 * Shadow section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function shadow($selector, $heading, $type)
	{
        $this->type=$type;

		/**
		 * Start section
		 */
		$this->start_section('shadow_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('shadow_tab'.$this->type);

		/**
		 * Shadow normal tab
		 */
		$this->start_tab('shadow_normal'.$this->type, 'Normal');

		/**
		 * Shadow normal
		 */
		$this->shadow_box_text_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('shadow_hover'.$this->type, 'Hover');

		/**
		 * Shadow hover
		 */
		$this->shadow_box_text_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Shadow normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function shadow_box_text_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Box Shadow
		 */
		$this->shadow_box_control();

		/**
		 * Text Shadow
		 */
		$this->shadow_text_control();
	}

	/**
	 * Shadow hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function shadow_box_text_hover($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Box Shadow
		 */
		$this->shadow_box_control();

		/**
		 * Text Shadow
		 */
		$this->shadow_text_control();
	}

	/**
	 * Shadow control
	 *
	 * @since 1.0.0
	 */
	private function shadow_box_control()
	{
		$this->get_controls()->add_control(
			array(
				'id' => $this->prefix . 'box-shadow'.$this->type,
				'type' => 'box-shadow',
				'label' => __('Box Shadow', 'webnus-style-manager'),
				'css_selector' => array(
					'{{WRAPPER}} ' . $this->selector . '' => 'box-shadow: {{STYLE}} {{HORIZONTAL}}px {{VERTICAL}}px {{BLUR}}px {{SPREAD}}px {{COLOR}};',
				),
			)
		);
	}

	/**
	 * Shadow control
	 *
	 * @since 1.0.0
	 */
	private function shadow_text_control()
	{
		$this->get_controls()->add_control(
			array(
				'id' => $this->prefix . 'text-shadow'.$this->type,
				'type' => 'text-shadow',
				'label' => __('Text Shadow', 'webnus-style-manager'),
				'css_selector' => array(
					'{{WRAPPER}} ' . $this->selector . '' => 'text-shadow: {{HORIZONTAL}}px {{VERTICAL}}px {{BLUR}}px {{COLOR}};',
				),
			)
		);
	}


	/**
	 * Position section
	 *
	 * @param string $selector CSS selector
	 * @param string $heading section heading
	 * @since 1.0.0
	 *
	 */
	public function position($selector, $heading, $type)
	{
        $this->type=$type;
		/**
		 * Start section
		 */
		$this->start_section('position_section'.$this->type, $heading);

		/**
		 * Start tabs
		 */
		$this->start_tabs('position_tab'.$this->type);

		/**
		 * Position normal tab
		 */
		$this->start_tab('position_normal'.$this->type, 'Normal');

		/**
		 * Position normal
		 */
		$this->position_normal($selector);

		/**
		 * End normal tab
		 */
		$this->end_tab();

		/**
		 * Start hover tab
		 */
		$this->start_tab('position_hover'.$this->type, 'Hover');

		/**
		 * Position hover
		 */
		$this->position_hover($selector);

		/**
		 * End hover tab
		 */
		$this->end_tab();

		/**
		 * End tabs
		 */
		$this->end_tabs();

		/**
		 * End section
		 */
		$this->end_section();
	}

	/**
	 * Position normal
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function position_normal($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector);

		/**
		 * Set prefix
		 */
		$this->set_prefix('normal');

		/**
		 * Position
		 */
		$this->position_control();
	}

	/**
	 * Position hover
	 *
	 * @param string $selector CSS selector
	 * @since 1.0.0
	 *
	 */
	private function position_hover($selector)
	{
		/**
		 * Set CSS selector
		 */
		$this->set_selector($selector . ':hover');

		/**
		 * Set prefix
		 */
		$this->set_prefix('hover');

		/**
		 * Position
		 */
		$this->position_control();
	}

	/**
	 * Position control
	 *
	 * @since 1.0.0
	 */
	private function position_control()
	{
		$this->get_controls()->add_control(
			array(
				'id' => $this->prefix . 'position'.$this->type,
				'type' => 'position',
				'label' => __('Position', 'webnus-style-manager'),
				'css_selector' => array(
					'{{WRAPPER}} ' . $this->selector . '' =>'position: {{STYLE}};top: {{TOP}}px; left:{{LEFT}}px; bottom:{{BOTTOM}}px; right:{{RIGHT}}px;z-index:{{ZINDEX}}',
				),
		));

	}

	/**
	 * Text decoration control
	 *
	 * @since 1.0.0
	 */
	private function text_decoration()
	{
		$options = array(
			'none' => array(
				'shortcut' => esc_html__('None', 'webnus-style-manager'),
				'icon' => 'dashicons-remove',
			),
			'underline' => array(
				'shortcut' => esc_html__('Underline', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-underline',
			),
			'overline' => array(
				'shortcut' => esc_html__('Overline', 'webnus-style-manager'),
				'icon' => 'dashicons-insert-before',
			),
			'line-through' => array(
				'shortcut' => esc_html__('Line-through', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-strikethrough',
			),
		);

		$this->choose($this->prefix . 'text_decoration'.$this->type, 'Text Decoration', $this->selector, $options, 'text-decoration');
	}

	/**
	 * Text transform control
	 *
	 * @since 1.0.0
	 */
	private function text_transform()
	{
		$options = array(
			'none' => array(
				'shortcut' => esc_html__('None', 'webnus-style-manager'),
				'icon' => 'dashicons-remove',
			),
			'uppercase' => array(
				'shortcut' => esc_html__('Uppercase', 'webnus-style-manager'),
				'icon' => 'fa-solid fa-u',
			),
			'lowercase' => array(
				'shortcut' => esc_html__('Lowercase', 'webnus-style-manager'),
				'icon' => 'fa-solid fa-l',
			),
			'capitalize' => array(
				'shortcut' => esc_html__('Capitalize', 'webnus-style-manager'),
				'icon' => 'fa-solid fa-c',
			),
		);

		$this->choose($this->prefix . 'text_transform'.$this->type, 'Text Transform', $this->selector, $options, 'text-transform');
	}

	/**
	 * Text align control
	 *
	 * @since 1.0.0
	 */
	private function text_align()
	{
		$options = array(
			'left' => array(
				'shortcut' => esc_html__('Left', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-alignleft',
			),
			'center' => array(
				'shortcut' => esc_html__('Center', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-aligncenter',
			),
			'right' => array(
				'shortcut' => esc_html__('Right', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-alignright',
			),
			'justify' => array(
				'shortcut' => esc_html__('Justify', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-justify',
			),
		);

		$this->choose($this->prefix . 'text_align'.$this->type, 'Text Align', $this->selector, $options, 'text-align');
	}

	/**
	 * Font style control
	 *
	 * @since 1.0.0
	 */
	private function font_style()
	{
		$options = array(
			'normal' => array(
				'shortcut' => esc_html__('Normal', 'webnus-style-manager'),
				'icon' => 'fa fa-solid fa-t',
			),
			'italic' => array(
				'shortcut' => esc_html__('Italic', 'webnus-style-manager'),
				'icon' => 'dashicons-editor-italic',
			),
//			'oblique' => array(
//				'shortcut' => esc_html__('Oblique', 'webnus-style-manager'),
//				'icon' => 'dashicons-editor-justify',
//			),
		);

		$this->choose($this->prefix . 'font_style'.$this->type, 'Font Style', $this->selector, $options, 'font-style');
	}

	/**
	 * Font weight control
	 *
	 * @since 1.0.0
	 */
	private function font_weight()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => '100',
				'value' => '100',
			),
			array(
				'label' => '200',
				'value' => '200',
			),
			array(
				'label' => '300',
				'value' => '300',
			),
			array(
				'label' => '400',
				'value' => '400',
			),
			array(
				'label' => '500',
				'value' => '500',
			),
			array(
				'label' => '600',
				'value' => '600',
			),
			array(
				'label' => '700',
				'value' => '700',
			),
			array(
				'label' => '800',
				'value' => '800',
			),
			array(
				'label' => '900',
				'value' => '900',
			),
		);

		$this->select($this->prefix . 'font_weight'.$this->type, 'Font Weight', $this->selector, $options, 'font-weight');
	}

	/**
	 * Writing mode control
	 *
	 * @since 1.0.0
	 */
	private function writing_mode()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'horizontal-tb',
				'value' => 'horizontal-tb',
			),
			array(
				'label' => 'vertical-rl',
				'value' => 'vertical-rl',
			),
			array(
				'label' => 'vertical-lr',
				'value' => 'vertical-lr',
			),
		);

		$this->select($this->prefix . 'writing_mode'.$this->type, 'Writing Mode', $this->selector, $options, 'writing-mode');
	}

	/**
	 * Text orientation control
	 *
	 * @since 1.0.0
	 */
	private function text_orientation()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => 'unset',
			),
			array(
				'label' => 'mixed',
				'value' => 'mixed',
			),
			array(
				'label' => 'upright',
				'value' => 'upright',
			),
			array(
				'label' => 'sideways',
				'value' => 'sideways',
			),
			array(
				'label' => 'sideways-right',
				'value' => 'sideways-right',
			),
		);

		$this->select($this->prefix . 'text_orientation'.$this->type, 'Text Orientation', $this->selector, $options, 'text-orientation');
	}

	/**
	 * Display control
	 *
	 * @since 1.0.0
	 */
	private function display_options()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'inline',
				'value' => 'inline',
			),
			array(
				'label' => 'block',
				'value' => 'block',
			),
			array(
				'label' => 'none',
				'value' => 'none',
			),
			array(
				'label' => 'contents',
				'value' => 'contents',
			),
			array(
				'label' => 'flex',
				'value' => 'flex',
			),
			array(
				'label' => 'grid',
				'value' => 'grid',
			),
			array(
				'label' => 'inline-block',
				'value' => 'inline-block',
			),
			array(
				'label' => 'inline-flex',
				'value' => 'inline-flex',
			),
			array(
				'label' => 'inline-grid',
				'value' => 'inline-grid',
			),
			array(
				'label' => 'inline-table',
				'value' => 'inline-table',
			),
			array(
				'label' => 'list-item',
				'value' => 'list-item',
			),
			array(
				'label' => 'run-in',
				'value' => 'run-in',
			),
			array(
				'label' => 'table',
				'value' => 'table',
			),
			array(
				'label' => 'table-caption',
				'value' => 'table-caption',
			),
			array(
				'label' => 'table-column-group',
				'value' => 'table-column-group',
			),
			array(
				'label' => 'table-header-group',
				'value' => 'table-header-group',
			),
			array(
				'label' => 'table-footer-group',
				'value' => 'table-footer-group',
			),
			array(
				'label' => 'table-row-group',
				'value' => 'table-row-group',
			),
			array(
				'label' => 'table-cell',
				'value' => 'table-cell',
			),
			array(
				'label' => 'table-column',
				'value' => 'table-column',
			),
			array(
				'label' => 'table-row',
				'value' => 'table-row',
			),
			array(
				'label' => 'initial',
				'value' => 'initial',
			),
			array(
				'label' => 'inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'display'.$this->type, 'Display', $this->selector, $options, 'display');
	}

	/**
	 * Box sizing control
	 *
	 * @since 1.0.0
	 */
	private function box_sizing()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'content-box',
				'value' => 'content-box',
			),
			array(
				'label' => 'border-box',
				'value' => 'border-box',
			),
			array(
				'label' => 'initial',
				'value' => 'initial',
			),
			array(
				'label' => 'inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'box_sizing'.$this->type, 'Box Sizing', $this->selector, $options, 'box-sizing');
	}

	/**
	 * Float control
	 *
	 * @since 1.0.0
	 */
	private function float()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'Left',
				'value' => 'left',
			),
			array(
				'label' => 'Right',
				'value' => 'right',
			),
			array(
				'label' => 'None',
				'value' => 'none',
			),
			array(
				'label' => 'Initial',
				'value' => 'initial',
			),
			array(
				'label' => 'Inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'float'.$this->type, 'Float', $this->selector, $options, 'float');
	}

	/**
	 * Clear control
	 *
	 * @since 1.0.0
	 */
	private function clear()
	{
		$options = array(
			array(
				'label' => 'Both',
				'value' => 'both',
			),
			array(
				'label' => 'Right',
				'value' => 'right',
			),
			array(
				'label' => 'Left',
				'value' => 'left',
			),
			array(
				'label' => 'None',
				'value' => 'none',
			),
		);

		$this->select($this->prefix . 'clear'.$this->type, 'Clear', $this->selector, $options, 'clear');
	}

	/**
	 * Overflow control
	 *
	 * @since 1.0.0
	 */
	private function overflow()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'Visible',
				'value' => 'visible',
			),
			array(
				'label' => 'Hidden',
				'value' => 'hidden',
			),
			array(
				'label' => 'Scroll',
				'value' => 'scroll',
			),
			array(
				'label' => 'Auto',
				'value' => 'auto',
			),
			array(
				'label' => 'Initial',
				'value' => 'initial',
			),
			array(
				'label' => 'Inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'overflow'.$this->type, 'Overflow', $this->selector, $options, 'overflow');
	}

	/**
	 * Vertical align control
	 *
	 * @since 1.0.0
	 */
	private function vertical_align()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'Baseline',
				'value' => 'baseline',
			),
			array(
				'label' => 'Length',
				'value' => 'length',
			),
			array(
				'label' => 'Sub',
				'value' => 'sub',
			),
			array(
				'label' => 'Super',
				'value' => 'super',
			),
			array(
				'label' => 'Top',
				'value' => 'top',
			),
			array(
				'label' => 'Text-top',
				'value' => 'text-top',
			),
			array(
				'label' => 'Middle',
				'value' => 'middle',
			),
			array(
				'label' => 'Bottom',
				'value' => 'bottom',
			),
			array(
				'label' => 'Text-bottom',
				'value' => 'text-bottom',
			),
			array(
				'label' => 'Initial',
				'value' => 'initial',
			),
			array(
				'label' => 'Inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'vertical_align'.$this->type, 'Vertical Align', $this->selector, $options, 'vertical-align');
	}

	/**
	 * Direction control
	 *
	 * @since 1.0.0
	 */
	private function direction()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'LTR',
				'value' => 'ltr',
			),
			array(
				'label' => 'RTL',
				'value' => 'rtl',
			),
		);

		$this->select($this->prefix . 'direction'.$this->type, 'Direction', $this->selector, $options, 'direction');
	}

	/**
	 * Cursor control
	 *
	 * @since 1.0.0
	 */
	private function cursor()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'alias',
				'value' => 'alias',
			),
			array(
				'label' => 'all-scroll',
				'value' => 'all-scroll',
			),
			array(
				'label' => 'auto',
				'value' => 'auto',
			),
			array(
				'label' => 'cell',
				'value' => 'cell',
			),
			array(
				'label' => 'context-menu',
				'value' => 'context-menu',
			),
			array(
				'label' => 'col-resize',
				'value' => 'col-resize',
			),
			array(
				'label' => 'copy',
				'value' => 'copy',
			),
			array(
				'label' => 'crosshair',
				'value' => 'crosshair',
			),
			array(
				'label' => 'default',
				'value' => 'default',
			),
			array(
				'label' => 'e-resize',
				'value' => 'e-resize',
			),
			array(
				'label' => 'ew-resize',
				'value' => 'ew-resize',
			),
			array(
				'label' => 'grab',
				'value' => 'grab',
			),
			array(
				'label' => 'grabbing',
				'value' => 'grabbing',
			),
			array(
				'label' => 'tab-help',
				'value' => 'tab-help',
			),
			array(
				'label' => 'move',
				'value' => 'move',
			),
			array(
				'label' => 'n-resize',
				'value' => 'n-resize',
			),
			array(
				'label' => 'ne-resize',
				'value' => 'ne-resize',
			),
			array(
				'label' => 'nesw-resize',
				'value' => 'nesw-resize',
			),
			array(
				'label' => 'ns-resize',
				'value' => 'ns-resize',
			),
			array(
				'label' => 'nw-resize',
				'value' => 'nw-resize',
			),
			array(
				'label' => 'nwse-resize',
				'value' => 'nwse-resize',
			),
			array(
				'label' => 'no-drop',
				'value' => 'no-drop',
			),
			array(
				'label' => 'none',
				'value' => 'none',
			),
			array(
				'label' => 'not-allowed',
				'value' => 'not-allowed',
			),
			array(
				'label' => 'pointer',
				'value' => 'pointer',
			),
			array(
				'label' => 'progress',
				'value' => 'progress',
			),
			array(
				'label' => 'row-resize',
				'value' => 'row-resize',
			),
			array(
				'label' => 's-resize',
				'value' => 's-resize',
			),
			array(
				'label' => 'se-resize',
				'value' => 'se-resize',
			),
			array(
				'label' => 'sw-resize',
				'value' => 'sw-resize',
			),
			array(
				'label' => 'text',
				'value' => 'text',
			),
			array(
				'label' => 'URL',
				'value' => 'URL',
			),
			array(
				'label' => 'vertical-text',
				'value' => 'vertical-text',
			),
			array(
				'label' => 'w-resize',
				'value' => 'w-resize',
			),
			array(
				'label' => 'wait',
				'value' => 'wait',
			),
			array(
				'label' => 'zoom-in',
				'value' => 'zoom-in',
			),
			array(
				'label' => 'zoom-out',
				'value' => 'zoom-out',
			),
			array(
				'label' => 'initial',
				'value' => 'initial',
			),
			array(
				'label' => 'inherit',
				'value' => 'inherit',
			),
		);

		$this->select($this->prefix . 'cursor'.$this->type, 'Cursor', $this->selector, $options, 'cursor');
	}

	/**
	 * Object fit control
	 *
	 * @since 1.0.0
	 */
	private function object_fit()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'fill',
				'value' => 'fill',
			),
			array(
				'label' => 'contain',
				'value' => 'contain',
			),
			array(
				'label' => 'cover',
				'value' => 'cover',
			),
			array(
				'label' => 'none',
				'value' => 'none',
			),
			array(
				'label' => 'scale-down',
				'value' => 'scale-down',
			),
		);

		$this->select($this->prefix . 'object_fit'.$this->type, 'Object Fit', $this->selector, $options, 'object-fit');
	}

	/**
	 * Object position control
	 *
	 * @since 1.0.0
	 */
	private function object_position()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'top',
				'value' => 'top',
			),
			array(
				'label' => 'left',
				'value' => 'left',
			),
			array(
				'label' => 'center',
				'value' => 'center',
			),
			array(
				'label' => 'right',
				'value' => 'right',
			),
			array(
				'label' => 'bottom',
				'value' => 'bottom',
			),
			array(
				'label' => 'left top',
				'value' => 'left top',
			),
			array(
				'label' => 'left center',
				'value' => 'left center',
			),
			array(
				'label' => 'left bottom',
				'value' => 'left bottom',
			),
			array(
				'label' => 'right top',
				'value' => 'right top',
			),
			array(
				'label' => 'right center',
				'value' => 'right center',
			),
			array(
				'label' => 'right bottom',
				'value' => 'right bottom',
			),
			array(
				'label' => 'center top',
				'value' => 'center top',
			),
			array(
				'label' => 'center center',
				'value' => 'center center',
			),
			array(
				'label' => 'center bottom',
				'value' => 'center bottom',
			),
		);

		$this->select($this->prefix . 'object_position'.$this->type, 'Object Position', $this->selector, $options, 'object-position');
	}

	/**
	 * Visibility control
	 *
	 * @since 1.0.0
	 */
	private function visibility()
	{
		$options = array(
			array(
				'label' => 'Default',
				'value' => '',
			),
			array(
				'label' => 'visible',
				'value' => 'visible',
			),
			array(
				'label' => 'hidden',
				'value' => 'hidden',
			),
			array(
				'label' => 'collapse',
				'value' => 'collapse',
			),
			array(
				'label' => 'Initial',
				'value' => 'Initial',
			),
			array(
				'label' => 'Inherit',
				'value' => 'Inherit',
			),
		);

		$this->select($this->prefix . 'visibility'.$this->type, 'Visibility', $this->selector, $options, 'visibility');
	}

	/**
	 * Opacity control
	 *
	 * @since 1.0.0
	 */
	private function opacity()
	{
		$this->get_controls()->add_control(
			array(
				'id' => $this->prefix . 'opacity'.$this->type,
				'type' => 'range',
				'label' => __('Opacity', 'webnus-style-manager'),
				'css_selector' => array(
					'{{WRAPPER}} ' . $this->selector . '' => 'opacity: {{VALUE}};',
				),
				'units' => array(
					array(
						'value' => '',
						'intervals' => array(
							'step' => 0.1,
							'min' => 0,
							'max' => 1,
						),
					),
				),
			)
		);
	}

	/**
	 * Border control
	 *
	 * @since 1.0.0
	 */
	private function border_control()
	{
		$this->get_controls()->add_control(
			array(
				'id' => $this->prefix . 'border'.$this->type,
				'type' => 'border',
				'label' => __('Border', 'webnus-style-manager'),
				'css_selector' => array(
					'{{WRAPPER}} ' . $this->selector . '' => 'border-style: {{STYLE}}; border-width: {{WIDTH}}; border-radius: {{RADIUS}}; border-color: {{COLOR}}',
				),
			)
		);
	}

	/**
	 * Set CSS selector
	 *
	 * @param string $selector
	 * @since 1.0.0
	 *
	 */
	private function set_selector($selector)
	{
		$this->selector = $selector;
	}

	/**
	 * Set handle prefix
	 *
	 * @param string $prefix
	 * @since 1.0.0
	 *
	 */
	private function set_prefix($prefix)
	{
		$this->prefix = $prefix;
	}
}
