<?php
/**
 * Class MEC_Seat Core
 *
 * @since           1.0.0
 */

namespace MEC_Seat;

use MEC\Settings\Settings;
use MEC_Seat\Seat\BookSeat;
use MEC_Seat\Seat\EventSeat;
use MEC_Seat\Seat\SeatsTable;
use MEC_Seat\Seat\DisplayForm;
use MEC_Seat\Seat\EventMetaBox;
use MEC_Seat\Seat\SeatLayoutBuilder;

/**
 * Core Class in plugin
 */
final class Base {

	/**
	 * Plugin Version
	 *
	 * @var string
	 */
	public static $version = '1.0.0';

	/**
	 * Session instance
	 *
	 * @var bool
	 */
	protected static $instance;

	/**
	 * Constructor
	 */
	public function __construct() {

		$this->define();
		$this->includes();
		$this->init_hooks();
		$this->admin();
		$this->enqueue_scripts();
		$this->common_enqueue_scripts();
	}

	/**
	 * Instance
	 *
	 * @return self()
	 */
	public static function instance() {

		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Set Constants
	 *
	 * @return void
	 */
	public function define() {

		define( 'MEC_SEAT_PD', plugin_dir_path( MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_PDI', plugin_dir_path( MEC_SEAT_FILE ) . 'src/' );
		define( 'MEC_SEAT_PU_JS', plugins_url( 'assets/js/', MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_PU_ASSETS', plugins_url( 'assets/', MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_PU_CSS', plugins_url( 'assets/css/', MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_PU_IMG', plugins_url( 'assets/img/', MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_PU_FONTS', plugins_url( 'assets/fonts/', MEC_SEAT_FILE ) );
		define( 'MEC_SEAT_TEMPLATES', plugin_dir_path( MEC_SEAT_FILE ) . 'templates/' );

		define( 'MEC_SEAT_VERSION', self::$version );

	}

	/**
	 * Include Files
	 *
	 * @return void
	 */
	public function includes() {

		require_once MEC_SEAT_PDI . 'functions.php';
	}

	/**
	 * Include Files If is Admin
	 *
	 * @return void
	 */
	public function admin() {

		if ( !is_admin() ) {

			return;
		}

		Admin::getInstance()->init();
	}

	/**
	 * Load Plugin Text Domain
	 */
	public function load_plugin_textdomain() {

		load_plugin_textdomain(
			'mec-seat',
			false,
			dirname( plugin_basename( MEC_SEAT_FILE ) ) . '/languages'
		);
	}

	/**
	 * Register actions enqueue scripts
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		// add_action( 'wp_enqueue_scripts', '\MEC_Seat\Scripts::enqueue', 99 );
	}

	/**
	 * Register common actions enqueue scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function common_enqueue_scripts() {

		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'register_common_style_and_script' ), 9 );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'register_common_style_and_script' ), 9 );
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'common_enqueue' ), 10 );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'common_enqueue' ), 10 );
	}

	/**
	 * Register common style and scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function register_common_style_and_script() {

		wp_register_script(
			'iris',
			admin_url( 'js/iris.min.js' ),
			array( 'jquery-ui-draggable', 'jquery-ui-slider', 'jquery-touch-punch' ),
			false,
			1
		);

		wp_register_script(
			'wp-color-picker',
			admin_url( 'js/color-picker.min.js' ),
			array( 'iris' ),
			false,
			1
		);
		$colorpicker_l10n = array(
			'clear' => __( 'Clear' ),
			'defaultString' => __( 'Default' ),
			'pick' => __( 'Select Color' ),
			'current' => __( 'Current Color' ),
		);
		wp_localize_script( 'wp-color-picker', 'wpColorPickerL10n', $colorpicker_l10n );

		wp_register_script( 'mec-seat-layout', MEC_SEAT_PU_ASSETS . 'seatLayout/seatLayout.js', array('jquery'), MECSEATVERSION );
		wp_register_style( 'mec-seat-layout', MEC_SEAT_PU_ASSETS . 'seatLayout/seatLayout.css', null, MECSEATVERSION );
		wp_localize_script( 'mec-seat-layout', 'mec_seat', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'demo_screen_text' => __( 'Scene', MECSEATTEXTDOMAIN ),
			'demo_booked_text' => __( 'Reserved', MECSEATTEXTDOMAIN ),
			'demo_available_text' => __( 'Available', MECSEATTEXTDOMAIN ),
			'demo_selected_text' => __( 'Selected', MECSEATTEXTDOMAIN ),
			'demo_number_of_seats_text' => __( 'Number of seats', MECSEATTEXTDOMAIN ),
			'handicap_accessible_text' => __( 'Handicap accessible', MECSEATTEXTDOMAIN ),
			'demo_available_seats_text' => __( 'Available seats', MECSEATTEXTDOMAIN ),
			'ticket_text' => __( 'Ticket', MECSEATTEXTDOMAIN ),
			'ticket_price_text' => __( 'Price', MECSEATTEXTDOMAIN ),
			'ticket_price_free_text' => __( 'Free', MECSEATTEXTDOMAIN ),
			'seat_number_text' => __( 'Seat', MECSEATTEXTDOMAIN ),
			'row_number_text' => __( 'Row', MECSEATTEXTDOMAIN ),
			'area_text' => __( 'Area', MECSEATTEXTDOMAIN ),
			'seat_number_text2' => __( 'Seat Number', MECSEATTEXTDOMAIN ),
			'display_seat_id_type' => EventSeat::get_display_seat_id_type( get_the_ID() )
		));

		wp_register_style( 'mec-seat-frontend', MEC_SEAT_PU_CSS . 'frontend.css', array( 'mec-lity-style' ) );
		wp_register_script( 'mec-seat-frontend', MEC_SEAT_PU_JS . 'frontend.js', array( 'jquery', 'mec-seat-layout', 'mec-lity-script' ) );

        wp_register_style( 'mec-seat-layout-backend', MEC_SEAT_PU_CSS . 'backendSeatLayout.css', array('wp-color-picker', 'mec-select2-style'), MECSEATVERSION );
        wp_register_script( 'mec-seat-layout-backend', MEC_SEAT_PU_JS . 'backendSeatLayout.js', array('jquery', 'mec-seat-layout', 'wp-color-picker', 'jquery-ui-sortable', 'mec-select2-script'), MECSEATVERSION );

        wp_register_style( 'mec-seat-backend-style', MEC_SEAT_PU_CSS.'/backend.css', ['mec-font-icons','mec-seat-frontend'], '1.0.0' );
        wp_register_script( 'mec-seat-backend-script', MEC_SEAT_PU_JS.'/backend.js', ['jquery','mec-seat-frontend'], '1.0.0' );
        wp_localize_script(
            'mec-seat-backend-script',
            'mec_seat_backend_data',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
            )
        );
	}

	/**
	 * Enqueue common scripts
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function common_enqueue() {

		self::register_common_style_and_script();

		wp_enqueue_script( 'mec-seat-layout' );
		wp_enqueue_style( 'mec-seat-layout' );
	}


	/**
	 * Add Hooks - Actions and Filters
	 *
	 * @return void
	 */
	public function init_hooks(): void {

		add_action( 'init', array( $this, 'init' ) );

		register_activation_hook( MEC_CORE_FILE, __CLASS__ . '::register_activation' );
		$db_version = get_option('mec_seat_db','0.0.0');

		if(version_compare($db_version, MECSEATVERSION, '<')){

			static::register_activation();
		}

		if(mec_seat_is_enabled()){

			add_action( 'mec_booking_end_form_step_2', array( $this, 'display_seat_form' ), 10, 4 );
			add_action( 'mec_validate_booking_form_step_2', array( __CLASS__, 'ajax_validate_seat_numbers' ), 10, 4 );
			add_shortcode( 'mec-seats', array( __CLASS__, 'render_event_seats_shortcode' ) );

			// fes form
			EventMetaBox::getInstance()->fes_init();

			add_action( 'mec_booking_completed', array( __CLASS__, 'hook_create_book_seats') );

			if(wp_doing_ajax()){


			}

			add_action('wp_enqueue_scripts',[ DisplayForm::getInstance(),'enqueue' ]);


			add_action( 'mec_book_invoice_pdf_before_qr_code', array( __CLASS__, 'add_seats_detail_to_mec_invoice_pdf' ), 10, 3 );
			add_filter( 'mec_invoice_more_info', array( __CLASS__, 'add_seats_detail_to_mec_invoice_more_detail' ), 10, 3 );
			add_filter( 'mec_invoice_information', array( __CLASS__, 'add_seat_number_to_ticket' ), 10, 6 );
			add_filter( 'mec_invoice_checkin_data', array( __CLASS__, 'add_seat_number_to_checkin_ticket' ), 10, 4);
			add_action( 'mec_invoice_attendee_more_detail', array( __CLASS__, 'invoice_attendee_more_detail' ), 10, 5);

			// MEC notifications
			add_filter('mec_render_message_email', array( __CLASS__, 'render_email_notification' ), 10, 4 );

			add_filter('mec_profile_event_detail_header', array( __CLASS__, 'add_header_seat_details_to_dashboard' ), 10, 1 );
			add_filter('mec_profile_event_detail', array( __CLASS__, 'add_seat_details_to_dashboard' ), 10, 3 );

			add_action('woocommerce_thankyou', array( __CLASS__, 'display_seat_details_in_wc_thankyou' ), 10, 1 );

			add_action('mec_buddyboss_event_or_booking_detail', array( __CLASS__, 'add_seats_detail_to_buddyboss' ), 10, 3 );

			add_filter( 'mec_zapier_rest_api_data', array( __CLASS__, 'add_seats_detail_to_zapier' ), 10, 4 );

		}

		do_action( 'MEC_Seat_init' );
	}

	/**
	 * Display seat form
	 *
	 * @param int $event_id
	 * @param array $tickets
	 * @param array $all_dates
	 * @param string $date
	 * @param array $selected_seats
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function display_seat_form( $event_id, $tickets, $all_dates, $date, $selected_seats = array() ) {

		if('mec-events' !== get_post_type( $event_id )){

			return;
		}

		$can_display = EventSeat::getInstance()->can_display_seat_form($event_id);
		if(is_wp_error( $can_display )){

            $login_url = wp_login_url();
            $e_code = $can_display->get_error_code();
            return 'not-logged-in' === $e_code ? '<div class="mec-error">'. __('You must be logged in to participate in Seat.',MECSEATTEXTDOMAIN).'  <a href="'.$login_url.'">'.__('Login',MECSEATTEXTDOMAIN).'</a></div>' : '';
        }

        if(!$can_display){

			return;
		}

		$all_dates = !empty( $all_dates ) ? $all_dates : array( $date );

        echo DisplayForm::getInstance()->output( (int)$event_id, $tickets, $all_dates, $selected_seats );
	}

	/**
	 * Init
	 *
	 * @return void
	 */
	public function init(): void {

		$this->load_plugin_textdomain();
	}

	/**
	 * Active Plugin
	 *
	 * @return void
	 */
	public static function register_activation() {

		SeatsTable::create_table();

		update_option('mec_seat_db',MECSEATVERSION);
	}

	/**
	 * Get MEC_Main
	 *
	 * @return \MEC_Main
	 */
	public static function get_main() {

		global $MEC_Main;
		if ( !is_a( $MEC_Main, \MEC_main::class ) ) {

			$MEC_Main = new \MEC_Main();
		}

		return $MEC_Main;
	}

	/**
	 * Add seats after create book
	 *
	 * @param int $book_id
	 *
	 * @hooked mec_booking_completed
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function hook_create_book_seats( $book_id ){

        $transaction_id = get_post_meta( $book_id, 'mec_transaction_id', true );

        $seats_numbers = BookSeat::getInstance()->get_transaction_seats_numbers( $transaction_id );
        foreach( $seats_numbers as $occurrence => $o_seats ){

			$seats = array();
			foreach( $o_seats as $o_seat => $ticket_id ){

				$seats[] = array(
					'seat_number' => trim( $o_seat ),
					'data' => array(
						'ticket_id' => $ticket_id,
					),
				);
			}

			BookSeat::getInstance()->update_book_seats( $book_id, $occurrence, $seats );
        }
    }

	/**
	 * Ajax validate seat numbers
	 *
	 * @param int $book
	 *
	 * @hooked mec_booking_completed
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function ajax_validate_seat_numbers( $book ){

		$all_dates = isset( $book['all_dates'] ) && is_array( $book['all_dates'] ) && !empty( $book['all_dates'] ) ? $book['all_dates'] : array( $book['date'] );
		$event_id = isset( $book['event_id'] ) ? $book['event_id'] : false;
		$seat_numbers = isset( $book['seats'] ) ? $book['seats'] : array();
		$tickets = isset( $book['tickets'] ) ? $book['tickets'] : array();
		$tickets_count = count( $tickets );
		$event_tickets = get_post_meta( $event_id, 'mec_tickets', true );

		$seat_skip_status = \MEC\Settings\Settings::getInstance()->get_settings('seat_skip_status');
		$status = EventSeat::getInstance()->get_seat_status( $event_id );
        if( !$status || $seat_skip_status ){

            return;
        }

		if( empty( $seat_numbers ) ){

			\MEC\Base::get_main()->response(array(
				'success'=> 0,
				'message'=> __( 'Seat(s) not selected.', MECSEATTEXTDOMAIN ),
			));
			die();
		}

		foreach( $all_dates as $date ){

			$ex = explode( ':', $date );
			$start_timestamp = $ex[0];
			$end_timestamp = $ex[1];
			$start_date = date_i18n( 'Y-m-d H:i', $start_timestamp );
			$end_date = date_i18n( 'Y-m-d H:i', $end_timestamp );
			$date_html = "$start_date";

			$o_seats = isset( $seat_numbers[ $start_timestamp ] ) && !empty( $seat_numbers[ $start_timestamp ] ) ? $o_seats = BookSeat::get_sanitized_seats_numbers( $seat_numbers[ $start_timestamp ] ) : array();

			if( count( $o_seats ) !== $tickets_count ){

				$tickets_count_group_by_ticket_id = array();
				foreach( $tickets as $ticket ){

					$t_count = $ticket['count'] ?? 1;
					if( isset( $tickets_count_group_by_ticket_id[ $ticket['id'] ] ) ){

						$tickets_count_group_by_ticket_id[ $ticket['id'] ] += $t_count;
					}else{

						$tickets_count_group_by_ticket_id[ $ticket['id'] ] = $t_count;
					}
				}

				$seats_allowed_ticket = BookSeat::getInstance()->get_seats_allowed_ticket( $event_id, $o_seats );

				foreach( $seats_allowed_ticket as $seat_number => $allowed_ticket_id ){

					if( isset( $tickets_count_group_by_ticket_id[ $allowed_ticket_id ] ) ){

						$tickets_count_group_by_ticket_id[ $allowed_ticket_id ]--;
					}
				}

				foreach( $tickets_count_group_by_ticket_id as $ticket_id => $t_count ){

					if( $t_count > 0 ){

						$message = sprintf(
							__( 'Choose %s other %s seats on "%s".', MECSEATTEXTDOMAIN ),
							$t_count,
							$event_tickets[ $ticket_id ]['name'] ?? '',
							$start_date
						);
					}
				}

				\MEC\Base::get_main()->response(array(
					'success'=> 0,
					'message'=> $message,
				));
				die();
			}

			$can = EventSeat::can_booked_seats_numbers( $event_id, $start_timestamp, $o_seats );

			$booked_seats = array();
			foreach( $can as $seat_number => $s_can ){

				if( ! $s_can ){

					$booked_seats[ $start_timestamp ][ $seat_number ] = $seat_number;
				}
			}

			if( empty( $booked_seats ) ) {
				EventSeat::update_pending_seats_numbers( $event_id, $start_timestamp, array_keys( $o_seats ) );
			}
		}

		if( !empty( $booked_seats ) ){

			$datetime_format = 'Y-m-d H:i';
			$messages = array();
			foreach( $booked_seats as $occurrence => $seats ){

				if( count( $seats ) > 1 ){

					$text = __( 'Seats %1s are already booked on the %2s', MECSEATTEXTDOMAIN );
				}else{

					$text = __( 'Seat %1s is already booked on the %2s', MECSEATTEXTDOMAIN );
				}

				$messages[] = sprintf( $text , implode( ',', $seats ), date_i18n( $datetime_format, $occurrence ) );
			}

			\MEC\Base::get_main()->response(array(
				'success'=> 0,
				'message'=> implode( '<br>', $messages ),
			));
			die();
		}
	}

	/**
	 * Add seats detail to mec invoice pdf
	 *
	 * @param \tFPDF $pdf
	 * @param int $book_id
	 * @param array $transaction
	 *
	 * @hooked mec_book_invoice_pdf_before_qr_code
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function add_seats_detail_to_mec_invoice_pdf( &$pdf, $book_id, $transaction ){

		$seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
        if( empty($seats_numbers) ){

            return;
        }

		$pdf->SetFont('DejaVuBold', '', 16);
		$pdf->Write(20, esc_html__( 'Seat Numbers', MECSEATTEXTDOMAIN ));
		$pdf->Ln();


		foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ){

			$seats_numbers_html = '';
			foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ){

				$numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
				$seats_numbers_html .= $ticket . ': ' . $numbers_html . ' ';
			}

			$pdf->SetFont('DejaVu', '', 12);
			$pdf->Write(6, date_i18n( 'Y-m-d H:i', $timestamp ) .': ');
			$pdf->Write(6, $seats_numbers_html);
			$pdf->Ln();
		}
	}

	/**
	 * Add seats detail to mec-invoice addon more detail
	 *
	 * @param string $more_info
	 * @param int $invoice_id
	 * @param int $book_id
	 *
	 * @hooked mec_invoice_more_info
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function add_seats_detail_to_mec_invoice_more_detail( $more_info, $invoice_id, $book_id ){

		$seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
		if( empty($seats_numbers) ){

			return $more_info;
		}

		ob_start();
		?>
			<tr>
				<td class="mec-invoice-body-info-t" style="font-family: sans-serif;font-size: 14px;vertical-align: top;width: 130px; color: #000; font-family: -apple-system,BlinkMacSystemFont, 'Segoe UI',Roboto, sans-serif; font-size: 14px; font-weight: 500; line-height: 29px;">
					<?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?>
				</td>
				<td class="mec-invoice-body-info" style="font-family: sans-serif;font-size: 14px;vertical-align: top;color: #75787b; font-family: -apple-system,BlinkMacSystemFont, 'Segoe UI',Roboto, sans-serif; font-size: 14px; font-weight: 400; line-height: 25px;">
					<?php
					foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ):

						?>
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?> :</strong>
						<?php foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ):

							$seats_numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
							?>
							<span><?php echo '<strong class="ticket-name">' . $ticket . ': </strong><span class="seat-numbers">' . $seats_numbers_html .'</span>' ?></span>
						<?php endforeach; ?>
						<br>
					<?php endforeach; ?>
				</td>
			</tr>
		<?php
		$html = ob_get_clean();

		return $more_info . $html;
	}

	/**
	 * Replace seats placeholders in mec notifications
	 *
	 * @param string $message
	 * @param int $book_id
	 * @param array $attendee
	 * @param string $timestamps
	 *
	 * @hooked mec_render_message_email
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function render_email_notification( $message, $book_id, $attendee, $timestamps ){

		$seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
		if( empty($seats_numbers) ){

			return $message;
		}

		ob_start();
		?>
            <div class="mec-seat-book-details">
                <?php
                foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ):

					?>
					<div class="mec-row">
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?> :</strong>
						<?php
						foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ){

							$numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
							echo '<span><strong class="ticket-name">' . $ticket . ': </strong><span class="seat-numbers">' . $numbers_html .'</span></span> ';
						}
						?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php
		$seat_numbers_html = ob_get_clean();

		$message = str_replace( '%%seat_numbers%%', $seat_numbers_html, $message );

		return $message;
	}

	/**
	 * Add header seat details to dashboard
	 *
	 * @hooked mec_profile_event_detail_header
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function add_header_seat_details_to_dashboard() {

        echo '<td>'.__( 'Seats', MECSEATTEXTDOMAIN ).'</td>';
    }

	/**
	 * Add seat details to dashboard
	 *
	 * @param int $event_id
	 * @param int $book_id
	 * @param object $event
	 *
	 * @hooked mec_profile_event_detail
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function add_seat_details_to_dashboard( $event_id, $book_id, $event ) {

		$seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
        if( !$event_id || !$book_id || empty($seats_numbers) ){

            echo '<td>-</td>';
            return;
        }

        ob_start();
		?>
            <div class="mec-seat-book-details">
				<?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?>
                <?php

				foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ):

					?>
					<div class="mec-row">
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?> :</strong>
						<?php
						foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ){

							$numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
							echo '<span><strong class="ticket-name">' . $ticket . ': </strong><span class="seat-numbers">' . $numbers_html .'</span></span> ';
						}
						?>
					</div>
				<?php endforeach; ?>
            </div>
        <?php
		$html = ob_get_clean();

        ?>
        <td>
            <?php if( !empty( $html ) ): ?>
                <a class="mec-booking-seat-details-link mec-booking-number-of-attendees"  href="#mec_profile_booking_seat_details_<?php echo $book_id; ?>">
                    <?php echo __('<i class="mec-sl-eye"></i>', MECSEATTEXTDOMAIN ); ?>
                </a>
                <div id="mec_profile_booking_seat_details_<?php echo $book_id; ?>" class="mec-booking-seat-details-content mec-booking-attendees lity-hide">
                    <?php echo $html; ?>
                </div>
            <?php else: ?>
                -
            <?php endif; ?>
        </td>
        <?php
    }

	/**
	 * Add seat details in woocommerce thankyou page
	 *
	 * @param int $order id
	 *
	 * @hooked woocommerce_thankyou
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function display_seat_details_in_wc_thankyou( $order_id ){

		$type = get_post_meta( $order_id, 'mec_order_type', true );
		if( 'mec_ticket' !== $type ){

			return;
		}

		$book_ids = get_posts(array(
			'post_type' => 'mec-books',
			'posts_per_page' => -1,
			'meta_query' => array(
				'' => array(
					'key' => 'mec_order_id',
					'value' => $order_id,
					'compare' => '=',
				),
			),
			'fields' => 'ids',
			// 'meta_key' => 'invoiceID',
		));

		$invoices = array();
		foreach( $book_ids as $book_id ){

			$seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
			if( empty($seats_numbers) ){

				continue;
			}

			$seats[ $book_id ] = $seats_numbers;
		}

		if( empty( $seats ) ){

			return;
		}

		echo '<h2>'. esc_html__( 'Seat Numbers', MECSEATTEXTDOMAIN ) .'</h2>';
		echo '<table class="woocommerce-table woocommerce-table--order-details shop_table order_details">';
		foreach( $seats as $book_id => $seats_numbers ){

				foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ):

					?>
				<tr>
					<td>
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?></strong>
					</td>
					<td>
						<?php
						foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ){

							$numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
							echo '<span><strong class="ticket-name">' . $ticket . ': </strong><span class="seat-numbers">' . $numbers_html .'</span></span> ';
						}
						?>
					</td>
				</tr>
				<?php endforeach; ?>
        <?php
		}
		echo '</table>';
	}


	/**
	 * Add seat details to buddyboss
	 *
	 * @param int $event_id
	 * @param int $book_id
	 * @param object $event
	 *
	 * @hooked mec_buddyboss_event_or_booking_detail
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function add_seats_detail_to_buddyboss( $event_id, $book_or_event_id, $event ) {

		if( !$event_id || !$book_or_event_id || $event_id === $book_or_event_id ){

			return;
		}
		$seats_numbers = BookSeat::getInstance()->get_rendered_book_seats_numbers( $book_or_event_id );

		if( empty($seats_numbers) ){

			return;
		}

		?>
		<div class="single-events-item">
			<div class="events-item-head">
				<?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?>
			</div>
			<div class="events-item-col">
				<?php
				foreach( $seats_numbers as $timestamp => $numbers ):

					$seats_numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
					?>
					<div class="mec-row">
						<?php if( count( $seats_numbers ) > 1 ): ?>
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?> :</strong>
						<?php endif; ?>
						<span><?php echo $seats_numbers_html ?></span>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
        <?php
    }

	/**
	 * Add seat details to zapier
	 *
	 * @param array $data
	 * @param string $task
	 * @param bool $is_set_settings_status
	 * @param \MEC_Zapier\Core\SetRoutes\SetRoutesV2 $class
	 *
	 * @hooked mec_zapier_rest_api_data
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public static function add_seats_detail_to_zapier( $data, $task, $is_set_settings_status, $class ) {

		if( 'new_booking' !== $task ){

			return $data;
		}

		if( !$is_set_settings_status ){

			$book_id = $data['booking_id'] ?? 0;
			if( !$book_id ){

				return $data;
			}
			$seats_numbers = BookSeat::getInstance()->get_rendered_book_seats_numbers( $book_id );

			$start_timestamp = strtotime( $data['event_start_on'] );
			$numbers = isset( $seats_numbers[ $start_timestamp ] ) ? $seats_numbers[ $start_timestamp ] : array();
			$data['seat_numbers'] = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
		}else{

			$data['seat_numbers'] = '1, 2, 3, 4,...';
		}

		return $data;
	}

	/**
	 * Add seat number to ticket
	 *
	 * @param array $InvoiceInformation
	 * @param array $_ticket
	 * @param string $attendee_key
	 * @param int $ticket_event_timestamp
	 * @param int $invoiceID
	 * @param int $book_id
	 *
	 * @since 1.0.1
	 *
	 * @return array
	 */
	public static function add_seat_number_to_ticket( $InvoiceInformation, $_ticket, $attendee_key, $ticket_event_timestamp, $invoiceID, $book_id ){

		$seat_numbers = \MEC_Seat\Seat\BookSeat::getInstance()->get_rendered_book_seat_number_for_attendee( $book_id, $attendee_key, $_ticket, $ticket_event_timestamp );

		if( !empty( $seat_numbers ) ){
			$InvoiceInformation['TicketSeatNumbers'] = $seat_numbers;
		} else {
			$InvoiceInformation['TicketSeatNumbers'] = '-';
		}

		return $InvoiceInformation;
	}

	/**
	 * Add seat number to checkin ticket
	 *
	 * @param array $data
	 * @param int $book_id
	 * @param array $attendee
	 * @param int $occurrence
	 *
	 * @since 1.0.1
	 *
	 * @return array
	 */
	public static function add_seat_number_to_checkin_ticket( $data, $book_id, $attendee, $occurrence ){

		$seat_numbers = \MEC_Seat\Seat\BookSeat::getInstance()->get_rendered_book_seat_number_for_attendee( $book_id, null, $attendee, $occurrence );

		if( !empty( $seat_numbers ) ){

			$html = '<div class="mec-qrcode-ticket-content-information">
						<div class="mec-qrcode-ticket-content-information-title">'. __( 'Seat Number(s)', MECSEATTEXTDOMAIN ) .'</div>
						<div class="mec-qrcode-ticket-content-information-content">'. $seat_numbers .'</div>
					</div>';

			$data['TicketSeatNumbers'] = $html;
		}

		return $data;
	}

	/**
	 * Add seat number to ticket
	 *
	 * @param array $attendee
	 * @param string $attendee_key
	 * @param int $invoice_id
	 * @param int $book_id
	 * @param \MEC\Transactions\Transaction $transactionObject
	 *
	 * @return void
	 */
	public static function invoice_attendee_more_detail( $attendee, $attendee_key, $invoice_id, $book_id, $transactionObject ){

		$dates = $transactionObject->get_dates();
		$rendered_seats = BookSeat::getInstance()->get_rendered_book_seats_numbers( $book_id );
		$seat_numbers = array();
		foreach ( $dates as $occurrence ) {

			$occurrence = explode( ':', $occurrence )[0];
			$_seat_numbers = isset( $attendee[ 'seats_' . $occurrence ] ) ? $attendee[ 'seats_' . $occurrence ] : array();
			if( empty( $_seat_numbers ) ) {

				$attendees = get_post_meta( $book_id, 'mec_attendees', true );
				$_seat_numbers = $attendees[ $attendee_key ][ 'seats_' . $occurrence ] ?? array();
			}
			$rendered_seats = $rendered_seats[ $occurrence ] ?? array();

			foreach( $_seat_numbers as $k => $seat_number ){

				$seat_numbers[ $occurrence ][$k] = $rendered_seats[ $seat_number ] ?? '';
			}
		}

		if( empty( $seat_numbers ) ) {

            return;
        }

		$date_format = get_option('date_format');
        $time_format = get_option('time_format');
        $datetime_format = $date_format . ' ' . $time_format;

        ?>
        <div class="mec-row">
            <strong><?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?>: </strong>
            <?php
            foreach( $seat_numbers as $date => $numbers ){

                $numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
                echo '<br><span><span class="ticket-date">' . date_i18n( $datetime_format, $date ) . ': </span><span class="seat-numbers">' . $numbers_html .'</span></span> ';
            }
            ?>
        </div>
        <?php
	}

	/**
	 * Render Event Seats Shortcode
	 *
	 * @param array $atts
	 *
	 * @since 1.1.4
	 *
	 * @return string
	 */
	public static function render_event_seats_shortcode( $atts ){

		$selected_seats = array();
		$atts = shortcode_atts(
			array(
				'event_id' => 0,
				'start_occurrence' => '',
				'limit' => 1,
			),
			$atts
		);

		$event_id = $atts['event_id'] ?? false;
		if ( ! $event_id ) {
			return;
		}

		$start_occurrence = $atts['start_occurrence'] ?? false;
		$limit = $atts['limit'] ?? 1;
		$start_occurrence = is_string( $start_occurrence ) ? strtotime( $start_occurrence ) : $start_occurrence;
		if ( ! $start_occurrence || ! $limit ) {
			return;
		}

		$event = new \MEC\Events\Event( $event_id );
		$all_dates = array();
		$occurrences = $event->get_occurrences_times( $start_occurrence, $limit );
		foreach ( $occurrences as $occurrence ) {
			$all_dates[] = $occurrence->tstart . ':' . $occurrence->tend;
		}

		if ( empty( $all_dates ) ) {
			return;
		}


		$event_title 	= $event->get_title();
		$single         = new \MEC_skin_single();

		$price_pattern = str_replace( 999.99, '{{PRICE}}', \MEC\Base::get_main()->render_price( 999.99 , $event_id) );
		$event_tickets  = $event->get_tickets();
		$allowed_tickets = array();
		foreach( $event_tickets as $ticket_id => $ticket){

			$t_price = (isset($event_tickets[$ticket_id]) and isset($event_tickets[$ticket_id]['price'])) ? (new \MEC_book())->get_ticket_price($event_tickets[$ticket_id], current_time('Y-m-d'), $event_id) : 0;
            $r_price = (is_numeric($t_price) ? \MEC\Base::get_main()->render_price($t_price, $event_id) : $t_price);


			$event_tickets[ $ticket_id ] = array(
				'name' => $ticket['name'],
				'price' => $r_price,
				'_price' => $t_price,
			);

			$allowed_tickets[ $ticket_id ] = 1;
		}

		$css = '';
		$demo_colors = SeatLayoutBuilder::get_demo_settings_settings( $event_id );
		foreach( $demo_colors as $type => $color ){

			if( empty( $color) ){
			  continue;
			}
			$css .= '.mec-seat-layout{--mec-seat-color-' . $type .': '. $color .';}';
		  }

		ob_start();

			echo '<script>var mec_seat_event_tickets = '. json_encode( $event_tickets ) .'</script>';
			echo '<script>var mec_seat_event_pattern_price = '. json_encode( $price_pattern ) .'</script>';
			echo '<style>'. $css .'</style>';

			$i = 0;
			$js = $contents = $sidebar_contents = $date_options = '';
			foreach( $all_dates as $date ){

				$ex = explode( ':', $date );
				$start_timestamp = $ex[0];
				$active_class = 0 == $i ? 'active' : '';

				$_GET['occurrence'] = $start_timestamp;
				$events = $single->get_event_mec( $event_id );
				$event_detail = current( $events );

				$selected = 0 == $i ? 'selected="selected"' : '';
				$date_options .= '<option value="'. esc_attr( $start_timestamp ) .'" '. $selected .' data-target="#mec-seat-layout-'. esc_attr( $start_timestamp ) .'">'.date_i18n( 'Y-m-d', $start_timestamp ).'</option>';

				$time = '';
				if (isset($event_detail->data->meta['mec_date']['start']) and !empty($event_detail->data->meta['mec_date']['start'])) {
					if (isset($event_detail->data->meta['mec_hide_time']) and $event_detail->data->meta['mec_hide_time'] == '0') {
						$time_comment = isset($event_detail->data->meta['mec_comment']) ? $event_detail->data->meta['mec_comment'] : '';
						$allday = isset($event_detail->data->meta['mec_allday']) ? $event_detail->data->meta['mec_allday'] : 0;

						if ($allday == '0' and isset($event_detail->data->time) and trim($event_detail->data->time['start'])) {

							$time = $event_detail->data->time['start'] . ' ' . (trim($event_detail->data->time['end']) ? esc_html( ' - ' . $event_detail->data->time['end'] ) : '');
						}else{

							$time = \MEC\Base::get_main()->m( 'all_day', esc_html__('All Day' , MECSEATTEXTDOMAIN) );
						}
					}
				}


				$sidebar_contents .= '<div id="mec-seat-layout-'. esc_attr( $start_timestamp ) .'-info" data-occurrence="'. esc_attr( $start_timestamp ) .'" class="'. esc_attr( $active_class ) .'">';

				$sidebar_contents .= '<div class="mec-seat-event-date mec-seat-event-detail">
								<i class="mec-sl-calendar"></i>
								<div class="mec-seat-event-bar-inner">
									<h3 class="mec-date">'. esc_html__( 'Date', MECSEATTEXTDOMAIN ) .'</h3>
									<dd><abbr class="mec-events-abbr"><span class="mec-start-date-label">'. date_i18n( 'Y-m-d', $start_timestamp ) .'</span></abbr></dd>
								</div>
							</div>';

				if( !empty( $time ) ){

					$sidebar_contents .= '<div class="mec-seat-event-time mec-seat-event-detail">
									<i class="mec-sl-clock"></i>
									<div class="mec-seat-event-bar-inner">
										<h3 class="mec-time">'. esc_html__( 'Time', MECSEATMAINFILEPATH ) .'<span></span></h3>
										<dd><abbr class="mec-events-abbr">'. $time .'</abbr></dd>
									</div>
								</div>';
				}

				$sidebar_contents .= '</div>';

				$contents .= '<div id="mec-seat-layout-'. esc_attr( $start_timestamp ) .'" data-occurrence="'. esc_attr( $start_timestamp ) .'" class="mec-seat-layout '. esc_attr( $active_class ) .'"></div>';
				$contents .= '<input type="hidden" style="display:none" class="mec-seat-selected-seats-input mec-seat-layout-data-'. esc_attr( $start_timestamp ) .'" id="mec-seat-layout-data-'. esc_attr( $start_timestamp ) .'" name="book[seats]['. esc_attr( $start_timestamp ) .']" value=""/>';

				$data = EventSeat::getInstance()->get_seat_layout_data_for_render( $event_id, $start_timestamp );
				$areas = EventSeat::getInstance()->get_seat_areas_data_for_render( $event_id, $start_timestamp );

				$booked_seats_numbers = EventSeat::get_disabled_seats_numbers( $event_id, $start_timestamp );
				$tickets = is_array( $allowed_tickets ) ? $allowed_tickets : [];
				$js .= 'mec_seat_layout_frontend["'.$start_timestamp.'"] = {
					"layout": '. wp_unslash( $data ) .',
					"areas": '. json_encode( array_values( $areas ) ).',
					"booked_seats_numbers": '. json_encode( $booked_seats_numbers ).',
					"allowed_tickets": '. json_encode( $tickets ).',
					"number_of_seats": '. (int)array_sum( $tickets ).',
					"selected_seats": '. json_encode( $selected_seats ).',
					"not_allowed_text": '. json_encode( __( 'You can not select this seat.', MECSEATTEXTDOMAIN ) ) .'
				};';

				$i++;
			}

			?>

			<div id="mec-seat-layouts-wrap" class="mec-seat-layouts-wrap show">
				<div class="mec-seat-topbar">
					<div class="">
						<select id="mec-seat-dates" class="mec-seat-dates">
							<?php echo $date_options; ?>
						</select>
					</div>
				</div>
				<div class="mec-seats-wrap">

					<div class="mec-seats-sidebar-wrap">
						<h3 class="mec-seat-event-title"><?php echo esc_html( $event_title ) ?></h3>
						<div class="mec-seat-layouts-tabs-contents">
							<?php echo $sidebar_contents; ?>
						</div>
					</div>
					<div class="mec-seats-content-wrap">
						<div class="mec-seat-layouts-tabs-contents">
							<?php echo $contents ?>
						</div>
					</div>
				</div>
			</div>

			<div id="mec-seat-popup-seat-data" class="mec-seat-popup-seat-data mec-seat-popup-frontend lity-hide">
				<div class="mec-seat-topbar">
					<div class="title"><?php esc_html_e( 'Seat', MECSEATTEXTDOMAIN ); ?></div>
					<div class="mec-seat-close-lity">
					<svg xmlns="http://www.w3.org/2000/svg" width="8" height="8" viewBox="0 0 12 12">
						<path id="close" d="M11.748.252a.86.86,0,0,1,0,1.217l-4.6,4.6,4.462,4.462a.86.86,0,0,1-1.217,1.217L5.93,7.286,1.608,11.609A.86.86,0,0,1,.391,10.392L4.713,6.069.252,1.608A.86.86,0,0,1,1.469.391L5.93,4.852l4.6-4.6A.86.86,0,0,1,11.748.252Z" fill="#939b9e"/>
					</svg>
					</div>
				</div>
				<div class="mec-seat-form-fields">
					<input type="hidden" class="mec-seat-html-id" value="">
					<div class="mec-form-row">
						<select id="mec-seat-tickets" class="mec-seat-tickets"></select>
					</div>
					<div class="mec-form-row">
						<button id="mec-seat-update-seat" class="mec-seat-update-seat button mec-col-12"><?php esc_html_e( 'Select', MECSEATTEXTDOMAIN ); ?></button>
					</div>
				</div>
			</div>

			<script> var mec_seat_layout_frontend = {};<?php echo $js ?></script>
			<script>(function($){mec_seat_module_init($);})(jQuery);</script>
			<script>jQuery(document).ready(function($){$('li.mec-seat').off('click');});</script>
			<style>.mec-seat-layouts-wrap.show{display: block;}</style>
			<?php
		return ob_get_clean();
	}

}
