<?php

namespace MEC_Seat\Seat;

use MEC\Singleton;
use MEC_Seat\Base;

/**
 * @since 1.0.0
 */
class BookMetaBox extends Singleton {

    /**
     * Init
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function init() {

        add_action( 'mec_admin_book_detail', array( __CLASS__, 'render_seat_details' ) );
		add_action( 'mec_admin_book_edit_form_end', [$this,'meta_box_seat_options'], 10, 2 );
        add_action( 'mec_admin_invoice_book_edit_form_end', [$this,'meta_box_seat_options'], 10, 2 );

        add_action( 'save_post', [$this,'save_book'], 999 );
        add_action( 'edit_post', [$this,'save_book'], 999 );
        add_action( 'mec_booking_saved_and_process_completed', [$this,'save_book'], 100 );

        add_action( 'save_post', [$this,'save_invoice'], 9999 );
        add_action( 'edit_post', [$this,'save_invoice'], 9999 );

        add_action( 'mec_admin_book_attendee_details', array( $this, 'attendee_seat_numbers_details' ),10,4);
	}

    /**
     * Save options
     *
     * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function save_seats( $book_id ){

        $transaction_id = get_post_meta( $book_id, 'mec_transaction_id', true );
        if( !$transaction_id ) {

            return;
        }

        $transaction = get_option( $transaction_id, array() );

        $event_id = isset($_POST['mec_event_id']) ? $_POST['mec_event_id'] : get_post_meta( $book_id, 'mec_event_id', true );

        $seat_numbers = isset($_POST['book']['seats']) ? $_POST['book']['seats'] : array();
        foreach( $seat_numbers as $occurrence => $o_seats ){

			$o_seats = BookSeat::get_sanitized_seats_numbers( $o_seats );
            $seat_numbers[ $occurrence ] = $o_seats;
        }
        $transaction['seats'] = $seat_numbers;

        update_option( $transaction_id, $transaction );

        $saved_seat_numbers = BookSeat::getInstance()->get_book_seats_numbers( $book_id );
        foreach( $saved_seat_numbers as $s_occurrence => $s_seat_numbers ){

            if( !in_array( $s_occurrence, array_keys( $seat_numbers ) ) ){

                Seats::getInstance()->delete( $book_id, $event_id, $occurrence = null );
            }
        }

        foreach( $seat_numbers as $occurrence => $o_seats ){

			$seats = array();
			$o_seats = BookSeat::get_sanitized_seats_numbers( $o_seats );
			foreach( $o_seats as $o_seat => $ticket_id ){

				$seats[] = array(
					'seat_number' => trim( $o_seat ),
					'data' => array(
						'ticket_id' => $ticket_id,
					),
				);
			}

			BookSeat::getInstance()->update_book_seats( $book_id, $occurrence, $seats );
        }
    }

    /**
     * Save options
     *
     * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function save_book( $book_id ){

        if(!isset($_POST['mec_seat_nonce'])) return;
        if(!wp_verify_nonce(sanitize_text_field($_POST['mec_seat_nonce']), 'mec_edit_book_seat_data')) return;
        if(defined('DOING_AUTOSAVE') and DOING_AUTOSAVE) return;
        if( 'mec-books' !== get_post_type( $book_id ) ) return;

        $this->save_seats( $book_id );
    }

    /**
     * Save options
     *
     * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function save_invoice( $invoice_id ){

        if(!isset($_POST['mec_seat_nonce'])) return;
        if(!wp_verify_nonce(sanitize_text_field($_POST['mec_seat_nonce']), 'mec_edit_book_seat_data')) return;
        if(defined('DOING_AUTOSAVE') and DOING_AUTOSAVE) return;
        if( 'mec_invoice' !== get_post_type( $invoice_id ) ) return;

        $book_id = get_post_meta( $invoice_id, 'book_id', true );

        $this->save_seats( $book_id );
    }

    /**
     * Display edit seats
     *
     * @param int $event_id
     * @param string $date
     * @param array $_tickets
     * @param array $all_dates
     * @param array $all_seats_numbers
     *
     * @since 1.0.0
     *
     * @return void
     */
    public static function display_edit_seats( $event_id, $date, $_tickets, $all_dates = array(), $all_seats_numbers = array() ){

        $status = EventSeat::getInstance()->get_seat_status( $event_id );

        if( !$status ){

            return;
        }

        $tickets = array();
        foreach( $_tickets as $_ticket_id => $ticket ){

            if( !is_array( $ticket ) ){

                $tickets[ $_ticket_id ] = $ticket;
                continue;
            }

            $ticket_id = $ticket['id'] ?? false;
            $ticket_count = $ticket['count'] ?? 1;
            if( !$ticket_id ){

                continue;
            }

            if ( isset( $tickets[ $ticket_id ] ) ){

                $tickets[ $ticket_id ] += $ticket_count;
            }else{

                $tickets[ $ticket_id ] = $ticket_count;
            }
        }

        Base::instance()->display_seat_form( $event_id, $tickets, $all_dates, $date, $all_seats_numbers );

        wp_nonce_field('mec_edit_book_seat_data', 'mec_seat_nonce');
        ?>
        <script>
            jQuery(document).ready(function($){

                mec_seat_edit_book_init();
            });
        </script>
        <?php
    }

    /**
     * Render seat options
     *
     * @param int $book_id
     * @param array $transaction
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function meta_box_seat_options( $book_id, $transaction ){

        $event_id = get_post_meta( $book_id, 'mec_event_id', true );


        $date = $transaction['date'] ?? '';
        $all_dates = isset( $transaction['all_dates'] ) && is_array( $transaction['all_dates'] ) && !empty( $transaction['all_dates'] ) ? $transaction['all_dates'] : array( $transaction['date'] );
        $all_seats_numbers = BookSeat::getInstance()->get_book_seats_numbers( $book_id );

        $_tickets = $transaction['tickets'] ?? array();

        ?>
        <div id="mec-edit-book-seats">
            <hr>
            <h3><?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?></h3>
            <?php static::display_edit_seats( $event_id, $date, $_tickets, $all_dates, $all_seats_numbers ); ?>
        </div>
        <?php
    }

    /**
     * Get rendered seat options by ajax
     *
     * @param int $event_id
     * @param int $book_id
     * @param array $all_dates
     * @param array $tickets
     *
     * @since 1.0.0
     *
     * @return void
     */
    public static function ajax_meta_box_seat_options( $event_id, $book_id, $all_dates, $tickets ){

        $date = current( $all_dates );
        $all_seats_numbers = BookSeat::getInstance()->get_book_seats_numbers( $book_id );

        ob_start();
        ?>
        <div id="mec-edit-book-seats">
            <hr>
            <h3><?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?></h3>
            <?php static::display_edit_seats( $event_id, $date, $tickets, $all_dates, $all_seats_numbers ); ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render seat details
     *
     * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return void
     */
    public static function render_seat_details( $book_id ){

        $seats_numbers = BookSeat::getInstance()->get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, null, true );
        if( empty($seats_numbers) ){

            return;
        }

		?>
            <hr>
            <div class="mec-seat-book-details">
                <h3><?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?></h3>
                <?php
                foreach( $seats_numbers as $timestamp => $seat_numbers_group_by_tickets ):

                    ?>
					<div class="mec-row">
						<strong><?php echo date_i18n( 'Y-m-d H:i', $timestamp ); ?> :</strong>
						<?php
						foreach( $seat_numbers_group_by_tickets as $ticket => $numbers ){

							$numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
							echo '<span><strong class="ticket-name">' . $ticket . ': </strong><span class="seat-numbers">' . $numbers_html .'</span></span> ';
						}
						?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php
    }

    /**
     * Display attendee seat numbers details
     *
     * @param  array $attendee
     * @param  string $key
     * @param  array $transaction
     * @param  int $book_id
     *
     * @return void
     */
    public function attendee_seat_numbers_details( $attendee, $key, $transaction, $book_id ){

        $date_format = get_option('date_format');
        $time_format = get_option('time_format');
        $datetime_format = $date_format . ' ' . $time_format;
        $attendee_data = $transaction['tickets'][$key] ?? $attendee;
        $all_dates = $transaction['all_dates'] ?? [ $transaction['date'] ];

        $seat_numbers = array();
        foreach( $all_dates as $date ){

            $dates = explode( ':', $date );
            $occurrence = $dates[0];

            $seats = $attendee_data['seats_'.$occurrence] ?? array();
            if( !empty( $seats ) ){

                $numbers = BookSeat::getInstance()->get_rendered_book_seat_number_for_attendee( $book_id, $key, $attendee_data, $occurrence );
                $seat_numbers[ $occurrence ] = $numbers;
            }
        }

        if( empty( $seat_numbers ) ) {

            return;
        }

        ?>
        <div class="mec-row">
            <strong><?php esc_html_e( 'Seat Numbers', MECSEATTEXTDOMAIN ); ?>: </strong>
            <?php
            foreach( $seat_numbers as $date => $numbers ){

                $numbers_html = is_array( $numbers ) ? implode( ', ', $numbers ) : $numbers;
                echo '<br><span><span class="ticket-date">' . date_i18n( $datetime_format, $date ) . ': </span><span class="seat-numbers">' . $numbers_html .'</span></span> ';
            }
            ?>

        </div>
        <?php
    }
}