<?php


namespace MEC_Seat\Seat;

use MEC\Singleton;

/**
 * @since 1.0.0
 */
class BookSeat extends Singleton {

	/**
     * Return book seats numbers
	 *
	 * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_book_seats_numbers( $book_id ){

        $seats_numbers = array();

        $event_id = get_post_meta( $book_id, 'mec_event_id', true );
		$status = EventSeat::getInstance()->get_seat_status( $event_id );
        if( $status ){

            $seats = Seats::getInstance()->get_seats( $event_id, null, $book_id );
            foreach( $seats as $data ){

                $occurrence = $data['occurrence'];
                $seat_number = $data['seat_number'];
                $ticket_id = $data['data']['ticket_id'] ?? 0;
                $seats_numbers[ $occurrence ][ $seat_number ] = $ticket_id;
            }
        }

        return $seats_numbers;
    }

    /**
     * Return alphabet
     *
     * @param int $number
     *
     * @since 1.0.0
     *
     * @return string|false
     */
    public function num_to_alpha($number){

        $number -= 1;
        $alphabet =   array('a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z');
        $alpha_flip = array_flip($alphabet);
        if($number <= 25){
            return $alphabet[$number];
        } elseif($number > 25){
            $dividend = ($number + 1);
            $alpha = '';
            $modulo;
            while ($dividend > 0){
                $modulo = ($dividend - 1) % 26;
                $alpha = $alphabet[$modulo] . $alpha;
                $dividend = floor((($dividend - $modulo) / 26));
            }
            return $alpha;
        }

    }

    /**
     * Return area data from layout data
     *
     * @param  int $occurrence
     * @param  array $layout_data
     *
     * @since 1.1.2
     *
     * @return array
     */
    public function get_area_data_from_layout_data( $occurrence, $layout_data ) {

        $area_key = null;
        $areas_data = $layout_data['data']['seatLayout']['colAreas']['objArea'] ?? array();

        foreach( $areas_data as $a_key => $area_data ) {

            if( $area_data['AreaCode'] == $occurrence ) {

                $area_key = $a_key;
            }
        }

        return $areas_data[ $area_key ] ?? array();
    }

    /**
     * Return seats data
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function get_seats_data( $event_id ){

        $seats_data = array();
        $status = EventSeat::getInstance()->get_seat_status( $event_id );
        if( $status ){

            $layout_data = json_decode(
                EventSeat::getInstance()->get_seat_layout_data( $event_id ),
                true
            );
            $areas_data = EventSeat::getInstance()->get_seat_areas_data( $event_id );

            $seat_number2 = 0;
            $seatNumber = 1;
            $GridRowId = 1;
            foreach( $areas_data as $area_key => $area_data ){

                $id = $area_data['id'] ?? '';
                $number_of_seats = $area_data['number_of_seats'] ?? '';
                $rows = $area_data['rows'] ?? '';
                $title = $area_data['title'] ?? '';

                $layout_area_data = $this->get_area_data_from_layout_data( $area_key, $layout_data );
                for( $j = 0; $j < $rows; $j++ ){

                    $rowSeatNumber = 1;
                    $row_id = strtoupper( $this->num_to_alpha( $GridRowId ) );
                    for( $i = 0; $i < $number_of_seats; $i++ ){

                        $seat_data = $layout_area_data['objRow'][ $j ]['objSeat'][ $i ] ?? array();
                        $seat_status = ! isset( $seat_data['mec_seat_status'] ) || 'yes' === $seat_data['mec_seat_status'] ? true : false;
                        if( $seat_status ) {
                            $seat_number2++;
                        }
                        $seat_data = wp_parse_args(
                            $seat_data,
                            array(
                                'GridSeatNum' => $rowSeatNumber,
                                'SeatStatus' => '0',
                                'seatNumber' =>  $i + 1,
                                'mec_seat_number' => $seatNumber,
                                'GridRowId' => $row_id,
                                'seat_number2' => $seat_number2,
                            )
                        );

                        if( $seat_status ){

                            $seats_data[ $seatNumber ] = $seat_data;
                        }

                        $rowSeatNumber++;
                        $seatNumber++;
                    }

                    $GridRowId++;
                }
            }
        }

        return $seats_data;
    }

    /**
     * Return seats allowed ticket
     *
     * @param int $event_id
     * @param int[]|string|int $seat_numbers
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_seats_allowed_ticket( $event_id, $seat_numbers ){

        $seat_numbers = !is_array( $seat_numbers ) ? explode( ',', $seat_numbers ) : $seat_numbers;

        $data = array();
        $seats_data = $this->get_seats_data( $event_id );
        foreach( $seat_numbers as $seat_number ){

            $seat_data = $seats_data[ $seat_number ] ?? array();
            $allowed_ticket_id =  $seat_data['mec_seat_ticket'] ?? '';

            $data[ $seat_number ] = $allowed_ticket_id;
        }

        return $data;
    }

    /**
     * Return rendered book seats numbers data
	 *
	 * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_rendered_book_seats_numbers_data( $book_id ){

        $seats_numbers = array();

        $event_id = get_post_meta( $book_id, 'mec_event_id', true );
        $event_tickets = get_post_meta( $event_id, 'mec_tickets', true );
        $event_tickets_ids = array_column( $event_tickets, 'id' );
		$seats_data = $this->get_seats_data( $event_id );
        $display_seat_id_type = EventSeat::get_display_seat_id_type( $event_id );

        if( !empty( $seats_data ) ){

            $seats = Seats::getInstance()->get_seats( $event_id, null, $book_id );
            if( empty( $seats ) ){

                return array();
            }

            foreach( $seats as $data ){

                $occurrence = $data['occurrence'];
                $seat_number = $data['seat_number'];
                $seat_ticket_id = $data['data']['ticket_id'] ?? 1;
                $seat_data = $seats_data[ $seat_number ] ?? array();
                $custom_row_id = $seat_data['custom_row_id'] ?? '';
                $seat_number2 = $seat_data['seat_number2'] ?? '';

                $GridRowId = ! empty( $custom_row_id ) ? $custom_row_id : ( $seat_data['GridRowId'] ?? '' );
                $seatNumber = $seat_data['seatNumber'] ?? '';
                $GridSeatNum = $seat_data['GridSeatNum'] ?? '';
                $allowed_ticket_id = isset( $seat_data['mec_seat_ticket'] ) && !empty( $seat_data['mec_seat_ticket'] ) ? $seat_data['mec_seat_ticket'] : $event_tickets_ids;

                $seats_numbers[ $occurrence ][ $seat_number ] = array(
                    'rendered_seat_number' => 'row_number' == $display_seat_id_type && $seat_number2 ? "{$GridRowId}{$GridSeatNum}" : $seat_number2,
                    'seat_number' => $seat_number,
                    'allowed_ticket_ids' => $allowed_ticket_id,
                    'ticket_id' => $seat_ticket_id,
                );
            }
        }

        ksort($seats_numbers);

        return $seats_numbers;
    }

    /**
     * Return rendered book seats numbers
	 *
	 * @param int $book_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_rendered_book_seats_numbers( $book_id ){

        $seats_numbers = $this->get_rendered_book_seats_numbers_data( $book_id );

        foreach( $seats_numbers as $occurrence => $seats ){

            $seats_numbers[ $occurrence ] = array_column( $seats, 'rendered_seat_number', 'seat_number' );
        }

        return $seats_numbers;
    }

    /**
     * Return sanitized seats numbers
     *
     * @param  string|array $seat_numbers
     *
     * @return array
     */
    public static function get_sanitized_seats_numbers( $seat_numbers ) {

        if( is_array( $seat_numbers ) ) {

            return $seat_numbers;
        }

        $seat_numbers = wp_unslash( $seat_numbers );

        if( false !== strpos( $seat_numbers,'{' ) ) {

            $seat_numbers = json_decode( $seat_numbers, true );
        } else {

            $seat_numbers = explode( ',', $seat_numbers );
        }

        return is_array( $seat_numbers ) ? $seat_numbers : array();
    }

    /**
	 * Return transaction seats numbers for all occurrences
	 *
	 * @param  string $transaction_id
	 *
	 * @return array
	 */
	public function get_transaction_seats_numbers( $transaction_id ) {

		$transaction = get_option( $transaction_id );

		$seats_numbers = isset( $transaction['seats'] ) ? $transaction['seats'] : array();
		foreach( $seats_numbers as $occurrence => $o_seats ){

            $o_seats = static::get_sanitized_seats_numbers( $o_seats );

			$seats_numbers[ $occurrence ] = $o_seats;
		}

		return $seats_numbers;
	}

    /**
     * Return rendered book seat numbers data group by ticket id
	 *
	 * @param int $book_id
     * @param int $occurrence
     *
     * @since 1.0.1
     *
     * @return array
     */
    public function get_rendered_book_seat_numbers_data_group_by_ticket_id( $book_id, $occurrence = null ){

        $transaction_id = get_post_meta( $book_id, 'mec_transaction_id', true );

        $seats_numbers_group_by_ticket_id = array();
        $seats_numbers = $this->get_rendered_book_seats_numbers_data( $book_id );

        $transaction_seats_numbers = $this->get_transaction_seats_numbers( $transaction_id );

        foreach( $seats_numbers as $s_occurrence => $seats ){

            $t_seats_numbers = $transaction_seats_numbers[ $s_occurrence ] ?? array();
            foreach( $seats as $seat_number => $seat_data ){

                $selected_seat_ticket_id = $t_seats_numbers[ $seat_number ] ?? '';
                $allowed_ticket_ids = $seat_data['allowed_ticket_ids'] ?? [];
                if( empty( $allowed_ticket_ids ) || in_array($selected_seat_ticket_id, (array)$allowed_ticket_ids) ) {

                    $allowed_ticket_id = $selected_seat_ticket_id;
                }

                $seats_numbers_group_by_ticket_id[ $s_occurrence ][ $allowed_ticket_id ][] = $seat_data;
            }
        }

        if( !is_null( $occurrence ) ){

            return $seats_numbers_group_by_ticket_id[ $occurrence ] ?? array();
        }else{

            return $seats_numbers_group_by_ticket_id;
        }
    }

    /**
     * Return rendered book seat numbers group by ticket id
	 *
	 * @param int $book_id
     * @param int $occurrence
     *
     * @since 1.0.1
     *
     * @return array
     */
    public function get_rendered_book_seat_numbers_group_by_ticket_id( $book_id, $occurrence = null, $replace_ticket_id_with_name = false ){

        if( $replace_ticket_id_with_name ) {

            $event_id = get_post_meta( $book_id, 'mec_event_id', true );
            $event_tickets = get_post_meta( $event_id, 'mec_tickets', true );
        }

        $seats_numbers_data_group_by_ticket_id = $this->get_rendered_book_seat_numbers_data_group_by_ticket_id( $book_id );

        $seats_numbers_group_by_ticket_id = array();
        foreach( $seats_numbers_data_group_by_ticket_id as $s_occurrence => $seats_numbers_data ) {

            foreach( $seats_numbers_data as $s_ticket_id => $seats ) {

                if( $replace_ticket_id_with_name ) {

                    $s_ticket_id = $event_tickets[ $s_ticket_id ]['name'] ?? $s_ticket_id;
                }

                $seats_numbers_group_by_ticket_id[ $s_occurrence ][ $s_ticket_id ] = array_column( $seats, 'rendered_seat_number', 'seat_number' );
            }
        }

        if( !is_null( $occurrence ) ){

            return $seats_numbers_group_by_ticket_id[ $occurrence ] ?? array();
        }else{

            return $seats_numbers_group_by_ticket_id;
        }
    }

    /**
     * Return rendered book seat number for the attendee
	 *
	 * @param int $book_id
     * @param string|int $attendee_key
     * @param array $attendee attendee
     * @param int $occurrence
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_rendered_book_seat_number_for_attendee( $book_id, $attendee_key, $attendee, $occurrence ){

        $seat_numbers = isset( $attendee[ 'seats_' . $occurrence ] ) ? $attendee[ 'seats_' . $occurrence ] : array();
        if( empty( $seat_numbers ) ) {

            $attendees = get_post_meta( $book_id, 'mec_attendees', true );
            $seat_numbers = $attendees[ $attendee_key ][ 'seats_' . $occurrence ] ?? array();
        }

        $rendered_seats = $this->get_rendered_book_seats_numbers( $book_id );
        $rendered_seats = $rendered_seats[ $occurrence ] ?? array();

        foreach( $seat_numbers as $k => $seat_number ){

            $seat_numbers[$k] = $rendered_seats[ $seat_number ] ?? '';
        }

        return implode( ',', $seat_numbers );
    }

    /**
     * Update book seats numbers
	 *
	 * @param int $book_id
     * @param int $occurrence
     * @param array $seats [['seat_number' => int, 'data' => array],..]
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public function update_book_seats( $book_id, $occurrence, $seats ){

        $event_id = get_post_meta( $book_id, 'mec_event_id', true );

        $success = Seats::getInstance()->update_seats( $book_id, $event_id, $occurrence, $seats );

        $transaction_id = get_post_meta( $book_id, 'mec_transaction_id', true );
        $transaction = get_option( $transaction_id );

        $seats_numbers_group_by_ticket_id = $this->get_rendered_book_seat_numbers_data_group_by_ticket_id( $book_id, $occurrence );
        $attendees = $transaction['tickets'] ?? array();
        foreach( $attendees as $k => $attendee ){

            $attendee_ticket_id = $attendee['id'] ?? '';
            $total_tickets = $attendee['count'] ?? 1;

            $attendees[ $k ]['seats_' . $occurrence ] = array();
            for( $i = 1; $i<= $total_tickets; $i++ ){

                if ( !isset( $seats_numbers_group_by_ticket_id[ $attendee_ticket_id ] ) ) {

                    continue;
                }

                $seat = current( $seats_numbers_group_by_ticket_id[ $attendee_ticket_id ] );
                $seat_key = key( $seats_numbers_group_by_ticket_id[ $attendee_ticket_id ] );
                unset( $seats_numbers_group_by_ticket_id[ $attendee_ticket_id ][ $seat_key ] );
                $seat_number = $seat['seat_number'] ?? '';
                $attendees[ $k ]['seats_' . $occurrence ][ $seat_number ] = $seat_number;
            }
        }

        $transaction['tickets'] = $attendees;
        update_option( $transaction_id, $transaction );

        return $success;
    }

}