<?php


namespace MEC_Seat\Seat;

use MEC\Settings\Settings;
use MEC\Singleton;

/**
 * @since 1.0.0
 */
class EventSeat extends Singleton {

    /**
     * Return options
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function get_seat_options( $event_id ){

        $seat_options = get_post_meta($event_id, 'mec_seat', true);

        return is_array($seat_options) ? $seat_options : [];
    }

    /**
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return boolean
     */
    public function can_display_seat_form( $event_id ){

        $can = $this->get_seat_status( $event_id );

        return $can;
    }

    /**
     * Return event seat status
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public function get_seat_status( $event_id ){

        return (bool)get_post_meta( $event_id, 'mec_seat_status', true );
    }

    /**
     * Return event seat custom layout status
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public function get_seat_custom_layout_status( $event_id ){

        $inherit_from_global = (bool)get_post_meta( $event_id, 'mec_seat_custom_seat_layout_status', true );

        return $inherit_from_global ? false : true;
    }

    /**
     * Return seat layout data
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return string
     */
    public function get_seat_layout_data( $event_id ){

        $data = '';

        $status = $this->get_seat_custom_layout_status( $event_id );
        if( $status ){

            $data = get_post_meta( $event_id, 'mec_seat_layout_data', true );
        }else{

            $data = \MEC\Settings\Settings::getInstance()->get_settings('seat_layout_data');
            $data = wp_unslash( $data );//TODO: check
        }

        return is_array( $data ) ? json_encode( $data ) : $data;
    }

    /**
     * Return seat areas
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_seat_areas_data( $event_id ){

        $data = array();
        $status = $this->get_seat_custom_layout_status( $event_id );
        if( $status ){

            $data = get_post_meta( $event_id, 'mec_seat_areas_data', true );
        }else{

            $data = \MEC\Settings\Settings::getInstance()->get_settings('seat_areas_data');
        }

        $data = is_array( $data ) ? $data : array();
        unset( $data['_fi_'] );
        unset( $data['{{id}}'] );

        return $data;
    }

    /**
     * Return seat layout data for render
     *
     * @param int $event_id
     * @param int $occurrence
     *
     * @since 1.0.0
     *
     * @return string
     */
    public function get_seat_layout_data_for_render( $event_id, $occurrence ){

        $data = $this->get_seat_layout_data( $event_id );

        return $data;
    }

    /**
     * Return seat layout data for render
     *
     * @param int $event_id
     * @param int $occurrence
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function get_seat_areas_data_for_render( $event_id, $occurrence ){

        $data = $this->get_seat_areas_data( $event_id );

        return $data;
    }

    /**
     * Return event booked seats numbers
     *
     * @param int $event_id
     * @param int $occurrence
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function get_booked_seats_numbers( $event_id, $occurrence ){

        $seats = Seats::getInstance()->get_seats( $event_id, $occurrence );

        return array_column( $seats, 'seat_number' );
    }

    /**
     * Return event disabled seats numbers
     *
     * @param int $event_id
     * @param int $occurrence
     *
     * @since 1.1.4
     *
     * @return array
     */
    public static function get_disabled_seats_numbers( $event_id, $occurrence ){

        $booked_seat_numbers = self::get_booked_seats_numbers( $event_id, $occurrence );
        $pending_seat_numbers = array_keys(
            self::get_pending_seats_numbers( $event_id, $occurrence )
        );

        return array_merge( $booked_seat_numbers, $pending_seat_numbers );
    }

    /**
     * Return event all occurrences pending seats numbers
     *
     * @param int $event_id
     *
     * @since 1.1.4
     *
     * @return array
     */
    public static function get_pending_seats_time( $event_id ){

        $time = 0;
        $seat_options = self::get_seat_options( $event_id );
        $global_status = $seat_options['seat_pending_time_global_status'] ?? false;
        if( ! $global_status  ) {

            return $seat_options['seat_pending_time'] ?? 0;
        }

        return (int)\MEC\Settings\Settings::getInstance()->get_settings('seat_pending_time');
    }

    /**
     * Return display seat id type
     *
     * @param int $event_id
     *
     * @since 1.2.1
     *
     * @return array
     */
    public static function get_display_seat_id_type( $event_id ){

        $seat_options = self::get_seat_options( $event_id );
        $display_seat_id = $seat_options['display_seat_id'] ?? false;
        if(  'global' !== $display_seat_id && $display_seat_id  ) {

            return $seat_options['display_seat_id'] ?? 'row_number';
        }

        $display_seat_id = \MEC\Settings\Settings::getInstance()->get_settings('display_seat_id');

        return $display_seat_id ? $display_seat_id : 'row_number';
    }

    /**
     * Return event all occurrences pending seats numbers
     *
     * @param int $event_id
     *
     * @since 1.1.4
     *
     * @return array
     */
    public static function _get_pending_seats_numbers( $event_id ){

        $user_ip = \MEC\Base::get_main()->get_client_ip();
        $seat_numbers = get_post_meta( $event_id, 'mec_pending_seat_numbers', true );

        foreach( $seat_numbers as $s_occurrence => $seats ) {

            foreach( $seats as $seat_number => $seat_args ) {

                $due = $seat_args['due'];
                $ip = $seat_args['ip'];

                if( $due < time() || $ip === $user_ip ) {
                    unset( $seat_numbers[ $s_occurrence ][ $seat_number ] );
                }
            }
        }

        return is_array( $seat_numbers ) ? $seat_numbers : array();
    }

    /**
     * Return event pending seats numbers
     *
     * @param int $event_id
     * @param int $occurrence
     *
     * @since 1.1.4
     *
     * @return array
     */
    public static function get_pending_seats_numbers( $event_id, $occurrence ){

        $seat_numbers = self::_get_pending_seats_numbers( $event_id );
        $seat_numbers = isset( $seat_numbers[ $occurrence ] ) && is_array( $seat_numbers[ $occurrence ] ) ? $seat_numbers[ $occurrence ] : array();

        return is_array( $seat_numbers ) ? $seat_numbers : array();
    }

    /**
     * Update event pending seats numbers
     *
     * @param int $event_id
     * @param int $occurrence
     * @param int[] $seat_numbers
     *
     * @since 1.1.4
     *
     * @return array
     */
    public static function update_pending_seats_numbers( $event_id, $occurrence, $seat_numbers ){

        $pending_time = self::get_pending_seats_time( $event_id );
        if( ! $pending_time ) {
            return;
        }

        $pending_seat_numbers = self::_get_pending_seats_numbers( $event_id );
        foreach( $seat_numbers as $seat_number ) {

            $pending_seat_numbers[ $occurrence ][ $seat_number ]['due'] = time() + $pending_time;
            $pending_seat_numbers[ $occurrence ][ $seat_number ]['ip'] = \MEC\Base::get_main()->get_client_ip();
        }

        update_post_meta( $event_id, 'mec_pending_seat_numbers', $pending_seat_numbers );
    }

    /**
     * Return event pending seats numbers
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function is_pending_seats_numbers( $event_id, $occurrence, $seat_numbers ){

        $pending_seats_numbers = array_keys(
            self::get_pending_seats_numbers( $event_id, $occurrence )
        );

        $is_pending = array();
        foreach( $seat_numbers as $seat_number => $value ){

            $is_pending[ $seat_number ] = in_array( $seat_number, $pending_seats_numbers, false );
        }

        return $is_pending;
    }

    /**
     * Return event booked seats numbers
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function is_booked_seats_numbers( $event_id, $occurrence, $seat_numbers ){

        $booked_seats_numbers = self::get_booked_seats_numbers( $event_id, $occurrence );

        $is_booked = array();
        foreach( $seat_numbers as $seat_number => $value ){

            $is_booked[ $seat_number ] = in_array( $seat_number, $booked_seats_numbers, false );
        }

        return $is_booked;
    }

    /**
     * Return event booked seats numbers
     *
     * @param int $event_id
     *
     * @since 1.0.0
     *
     * @return array
     */
    public static function can_booked_seats_numbers( $event_id, $occurrence, $seat_numbers ){

        $booked_seat_numbers = self::get_booked_seats_numbers( $event_id, $occurrence );
        $pending_seat_numbers = array_keys(
            self::get_pending_seats_numbers( $event_id, $occurrence )
        );
        $can = array();
        foreach( $seat_numbers as $seat_number => $value ){

            $can[ $seat_number ] = ! (
                in_array( $seat_number, $booked_seat_numbers, false )
                ||
                in_array( $seat_number, $pending_seat_numbers, false )
            );
        }

        return $can;
    }

}
