<?php
/**
 * Plugin Name: MEC Advanced Reports
 * Plugin URI: http://webnus.net/modern-events-calendar/
 * Description: This plugin makes it possible to users to have accurate and practical statistics of their ticketing system.
 * Author: Webnus
 * Version: 1.2.0
 * Text Domain: mec-ars
 * Domain Path: /languages
 * Author URI: http://webnus.net
 **/

namespace MECAdvancedReport;

use MECAdvancedReport\Core\Loader;
use MECAdvancedReport\App\Libraries\Charts;
use MECAdvancedReport\App\Libraries\Bookings;

// Don't load directly.
if (! defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Plugin.
 *
 * @author  Webnus Team
 * @package MECAdvancedReport
 * @since   1.0.0
 */
class Plugin
{
    /**
     * Modern Events Calendar Options
     *
     * @since   1.0.0
     */
    public $mecOptions;

    /**
     * Instance of this class.
     *
     * @since   1.0.0
     * @access  public
     * @var     MECAdvancedReport
     */
    public static $instance;

    /**
     * Provides access to a single instance of a module using the singleton pattern.
     *
     * @since   1.0.0
     * @return  object
     */
    public static function instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }


    public function __construct()
    {
        if (defined('MECARSVERSION')) {
            return;
        }
        self::settingUp($this);
        self::preLoad();
        self::setHooks($this);
        do_action('mec_advanced_report_init');
    }

    /**
     * Global Variables.
     *
     * @since   1.0.0
     */
    public static function settingUp($This)
    {
        define('MECARSVERSION', '1.2.0');
        define('MECARSDIR', plugin_dir_path(__FILE__));
        define('MECARSURL', plugin_dir_url(__FILE__));
        define('MECARSASSETS', MECARSURL . 'assets/');
        define('MECARSNAME', 'Advanced Reports');
        define('MECARSSLUG', 'mec-advanced-reports');
        define('MECARSOPTIONS', 'mec_advanced_reports');
        define('MECARSTEXTDOMAIN', 'mec-ars');
        define('MECARSMAINFILEPATH' ,__FILE__);
        define('MECARSABSPATH', dirname(__FILE__));
        if (! defined('DS')) {
            define('DS', DIRECTORY_SEPARATOR);
        }
        $This->mecOptions = get_option('mec_options');
    }

    /**
     * Preload
     *
     * @since   1.0.0
     */
    public static function preload()
    {
        if (static::checkPlugins()) {
            include_once MECARSDIR . DS . 'core' . DS . 'Patterns' . DS . 'SingletonPattern.php';
            include_once MECARSDIR . DS . 'core' . DS . 'Autoloader.php';
            Loader::instance();
        }
    }

    /**
     * Set Hooks
     *
     * @since   1.0.0
     */
    public static function setHooks($This)
    {
        add_action('admin_enqueue_scripts', [$This, 'scripts'], 99);
        add_action('init', [$This, 'loadLanguages']);
    }

    public function loadLanguages()
    {
        load_plugin_textdomain('mec-ars', false, 'mec-advanced-reports/languages');
    }

    /**
     * Enqueue Scripts
     *
     * @since 1.0.0
     */
    public function scripts()
    {
        if (isset($_GET['page']) && $_GET['page'] === 'mec-advanced-report') {
            // Plugins Scripts
            wp_enqueue_style('spinkit', MECARSASSETS . 'plugins/spinkit.min.css', [], '1.0.0', 'all');
            wp_enqueue_style('daterangepicker', MECARSASSETS . 'plugins/daterangepicker/daterangepicker.css', [], '3.0.5', 'all');
            wp_enqueue_script('moment', MECARSASSETS . 'plugins/daterangepicker/moment.min.js', ['jquery'], '3.0.5', true);
            wp_enqueue_script('daterangepicker', MECARSASSETS . 'plugins/daterangepicker/daterangepicker.js', ['jquery'], '3.0.5', true);
            wp_enqueue_style('Chart.js', MECARSASSETS . 'plugins/Chart.js/Chart.min.css', [], '2.9.3', 'all');
            wp_enqueue_script('Chart.js', MECARSASSETS . 'plugins/Chart.js/Chart.min.js', [], '2.9.3', false);
            wp_enqueue_style('mec-select2-style');
            wp_enqueue_script('mec-select2-script');
            // Modern Events Calendar Advanced Reports Scripts
            wp_enqueue_style('mec-advanced-report', MECARSASSETS . 'mec-advanced-report.css', [], MECARSVERSION, 'all');
            wp_enqueue_script('mec-advanced-report', MECARSASSETS . 'mec-advanced-report.min.js', ['jquery'], MECARSVERSION, true);
            $this->localizeScripts();
        }
    }

    /**
     * Localize Scripts
     *
     * @since   1.0.0
     */
    public function localizeScripts()
    {
        $termsData = [];
        $gatewaysData = [];
        $allAttendees = Bookings::instance()->getAttendees('all');
        $confirmedAttendees = Bookings::instance()->getAttendees('confirmed');
        $verifiedAttendees = Bookings::instance()->getAttendees('verified');
        $terms = get_terms([
            'taxonomy' => ['mec_category', 'mec_location', 'mec_organizer'],
            'hide_empty' => false,
        ]);
        foreach ($terms as $term) {
            $termsData[$term->taxonomy][] = ['id' => $term->term_id, 'text' => $term->name];
        }
        wp_localize_script('mec-advanced-report', 'mecARS', [
            'nonce' => wp_create_nonce('mecARS'),
            'ajaxurl' => admin_url('admin-ajax.php'),
            'lineCharts' => [
                'attendeesLabel' => esc_html__('Attendees', 'mec-ars'),
                'salesLabel' => esc_html__('Sales', 'mec-ars'),
                'salesCurrencySing' => \MEC\Base::get_main()->get_currency_sign(),
                'data' => Charts::instance()->getLineChartData('locations', $confirmedAttendees),
            ],
            'doughnutChart' => Charts::instance()->getDoughnutChartData($confirmedAttendees),
            'terms' => $termsData,
            'attendeesQueryInit' => [
                'allMaxChunkNumZero' => count($allAttendees) >= 15 ? false : true,
                'confirmedMaxChunkNumZero' => count($confirmedAttendees) >= 15 ? false : true,
                'verifiedMaxChunkNumZero' => count($verifiedAttendees) >= 15 ? false : true,
            ],
        ]);
    }

    /**
     * Plugin Requirements Check
     *
     * @since 1.0.0
     */
    public static function checkPlugins()
    {
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        if (!self::$instance) {
            self::$instance = static::instance();
        }

        if (!is_plugin_active('modern-events-calendar/mec.php')) {
            add_action('admin_notices', [self::$instance, 'MECNotice']);
            return false;
        }

        return true;
    }

    /**
     ** Send Admin Notice (MEC)
     **
     ** @since 1.0.0
     **/
    public function MECNotice($type = false)
    {
        $screen = get_current_screen();
        if (isset($screen->parent_file) && 'plugins.php' === $screen->parent_file && 'update' === $screen->id) {
            return;
        }

        $plugin = 'modern-events-calendar/mec.php';
        if ($this->is_mec_installed()) {
            if (!current_user_can('activate_plugins')) {
                return;
            }
            $activation_url = wp_nonce_url('plugins.php?action=activate&amp;plugin=' . $plugin . '&amp;plugin_status=all&amp;paged=1&amp;s', 'activate-plugin_' . $plugin);
            $message        = '<h3>' . __( 'Thanks for installing the Advanced Reports!', 'mec-ars' ) . '</h3><p>' . __('Advanced Reports requires the Modern Events Calendar plugin to work. Please activate it to continue.', 'mec-ars') . '</p>';
            $message       .= '<p>' . sprintf('<a href="%s" class="button-primary">%s</a>', $activation_url, __('Activate Modern Events Calendar', 'mec-ars')) . '</p>';
        } else {
            if (!current_user_can('install_plugins')) {
                return;
            }
            $install_url = 'https://webnus.net/modern-events-calendar/';
            $message     = '<h3>' . __( 'Thanks for installing the Advanced Reports!', 'mec-ars' ) . '</h3><p>' . __('Advanced Reports requires the Modern Events Calendar plugin to work. Please install it to continue.', 'mec-ars') . '</p>';
            $message    .= '<p>' . sprintf('<a href="%s" class="button-primary">%s</a>', $install_url, __('Install Modern Events Calendar', 'mec-ars')) . '</p>';
        }
        ?>
        <div class="notice notice-error is-dismissible mec-notice-content">
            <p><?php echo $message; ?></p>
        </div>
        <?php
    }


    /**
     * Is MEC installed?
     *
     * @since     1.0.0
     */
    public function is_mec_installed()
    {
        $file_path         = 'modern-events-calendar/mec.php';
        $installed_plugins = get_plugins();
        return isset($installed_plugins[$file_path]);
    }
}

Plugin::instance();
