<?php

namespace MECAdvancedReport\App\Libraries;

use MEC\Events\EventsQuery;
use MECAdvancedReport\Plugin;
use MECAdvancedReport\App\Libraries\Events;
use MECAdvancedReport\Core\Patterns\SingletonPattern;

// Don't load directly.
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Bookings.
 *
 * @author  Webnus Team
 * @package MECAdvancedReport
 * @since   1.0.0
 */
class Bookings extends SingletonPattern
{
    /**
     * Bookings Properties
     *
     * @since   1.0.0
     */
    private $bookings;
    public $MECMainInstance;
    public $MECBookingsInstance;
    public $pagination;

    /**
     * Constructor
     *
     * @since   1.0.0
     */
    public function __construct()
    {
        add_action('init', function () {
            $this->setBookings();
            $this->MECMainInstance = \MEC::getInstance('app.libraries.main');
            $this->MECBookingsInstance = \MEC::getInstance('app.libraries.book');
        });
        add_action('init', function () {
            $this->setBookings();
            $this->MECMainInstance = \MEC::getInstance('app.libraries.main');
            $this->MECBookingsInstance = \MEC::getInstance('app.libraries.book');
            if (!isset($this->main) || !$this->main) {
                $this->main = \MEC::getInstance('app.libraries.main');
            }
        });
    }

    /**
     * Set Bookings Property
     *
     * @since   1.0.0
     */

    public function setBookings()
    {
        if (isset($_GET['page']) && $_GET['page'] == 'mec-advanced-report') {
            $this->bookings = $this->getBookings();
        }
    }

    /**
     * Get Bookings Posts
     *
     * @since   1.0.0
     */
    public function getBookings($metaQuery = [], $numberPosts = 20)
    {
        $postType = apply_filters('mec_book_post_type_name', 'mec-books');
        $metaQuery['eventIDs'] = isset($metaQuery['eventIDs']) && $metaQuery['eventIDs'] ? $metaQuery['eventIDs'] : [];
        $metaQuery['bookingTime'] = isset($metaQuery['bookingTime']) && $metaQuery['bookingTime'] ? $metaQuery['bookingTime'] : [];
        $paged = isset($_REQUEST['paged']) && $_REQUEST['paged'] > 0 ? (int)$_REQUEST['paged'] : 1;

        $args = [
            'post_type' => $postType,
            'numberposts' => $numberPosts,
            'posts_per_page' => $numberPosts,
            'paged'          => $paged,
            'meta_query' => [
                $metaQuery['eventIDs'],
                $metaQuery['bookingTime'],
            ],
            'meta_key' => $metaQuery['meta_key'] ?? 'mec_booking_time',
            'orderby' => $metaQuery['orderby'] ?? 'meta_value',
            'order' => $metaQuery['order'] ?? 'DESC',
            'post_status' => 'publish',
        ];
        $bookings = get_posts($args);
        $this->bookingsData = [];

        foreach ($bookings as $booking) {
            $booking_id = $booking->ID;
            $event_id = get_post_meta($booking_id, 'mec_event_id', true);

            if (!$event_id) {
                continue;
            }

            $ticketData = get_post_meta($event_id, 'mec_tickets', true);

            if (!is_array($ticketData) || empty($ticketData)) {
                continue;
            }

            $attendees = get_post_meta($booking_id, 'mec_attendees', true);

            if (!is_array($attendees) || empty($attendees)) {
                continue;
            }

            foreach ($attendees as $attendee) {
                if (!isset($attendee['id']) || empty($attendee['id'])) {
                    continue;
                }

                $attendee_id = $attendee['id'];
                $ticket_name = isset($ticketData[$attendee_id]['name']) ? $ticketData[$attendee_id]['name'] : 'Unknown';

                $this->bookingsData[$booking_id][$attendee_id] = $ticket_name;
            }
        }

        return $bookings;
    }


    /**
     * Get Bookings Statistics
     *
     * @since   1.0.0
     */
    public function getBookingsStatistics($bookingQuery = [])
    {
        $bookingsStatistics = [
            'allAttendees' => 0,
            'income' => 0,
            'confirmed' => 0,
            'verified' => 0,
            'all' => 0,
        ];

        $bookings = $this->getBookings($bookingQuery, -1);
        if ($bookings) {
            foreach ($bookings as $booking) {
                $transactionID = get_post_meta($booking->ID, 'mec_transaction_id', true);
                $transaction = get_option($transactionID);

                $attendeesMeta = get_post_meta($booking->ID, 'mec_attendees', true) ?: [];

                $confirmedStatus = get_post_meta($booking->ID, 'mec_confirmed', true);
                if ($confirmedStatus == '1') {
                    $bookingsStatistics['income'] += isset($transaction['price']) ? $transaction['price'] : get_post_meta($booking->ID, 'mec_price', true);
                    $bookingsStatistics['confirmed']++;
                }

                $verifiedStatus = get_post_meta($booking->ID, 'mec_verified', true);
                if ($verifiedStatus == '1') {
                    $bookingsStatistics['verified']++;
                }

                $bookingsStatistics['all']++;
                $bookingsStatistics['allAttendees'] += count($attendeesMeta);
            }
        }

        return $bookingsStatistics;
    }


    /**
     * Get Fees Price
     *
     * @since   1.0.0
     */
    public function getFeesAmount($eventID = null, $tickets = [])
    {
        $total = 0;
        $totalFeesAmount = 0;
        $totalTicketsCount = 0;
        foreach ($tickets as $ticketID => $ticket) {
            $total += (float)$ticket['price'];
            ++$totalTicketsCount;
        }
        if (isset(Plugin::instance()->mecOptions['settings']['taxes_fees_status']) and Plugin::instance()->mecOptions['settings']['taxes_fees_status']) {
            $fees = $this->MECBookingsInstance->get_fees($eventID);
            foreach ($fees as $key => $fee) {
                if (!is_numeric($key)) {
                    continue;
                }
                if ($fee['type'] == 'percent') {
                    $feeAmount = (($total) * $fee['amount']) / 100;
                } elseif ($fee['type'] == 'amount') {
                    $feeAmount = ($totalTicketsCount * $fee['amount']);
                } elseif ($fee['type'] == 'amount_per_booking') {
                    $feeAmount = $fee['amount'];
                }
                $totalFeesAmount += $feeAmount;
            }
        }
        return $totalFeesAmount;
    }

    public function get_event_ids($eventQuery)
    {

        $q_args = [
            'posts_per_page' => -1,
            'category' => isset($eventQuery['categoryIDs']) && $eventQuery['categoryIDs'] ? $eventQuery['categoryIDs'] : [],
            'location' => isset($eventQuery['locationIDs']) && $eventQuery['locationIDs'] ? $eventQuery['locationIDs'] : [],
            'organizer' => isset($eventQuery['organizerIDs']) && $eventQuery['organizerIDs'] ? $eventQuery['organizerIDs'] : [],
        ];

        return EventsQuery::getInstance()->get_events_ids($q_args);
    }
    private function formatFieldsForCSV($fields)
    {
        if (!is_array($fields) || empty($fields)) {
            return 'No Data';
        }

        $formatted = [];
        foreach ($fields as $key => $value) {
            if (is_array($value)) {
                if (isset($value['label']) && isset($value['value'])) {
                    $formatted[] = "{$value['label']}: {$value['value']}";
                } elseif (isset($value['label'])) {
                    $formatted[] = "{$value['label']}";
                }
            } else {
                $formatted[] = "{$key}: {$value}";
            }
        }

        return implode(' | ', $formatted);
    }

    function getTicketName($event_id, $attendee_id)
    {
        if (!$event_id || !$attendee_id) {
            return esc_html__('Unknown', 'mec');
        }

        $tickets = get_post_meta($event_id, 'mec_tickets', true);
        if (!is_array($tickets)) {
            return esc_html__('Unknown', 'mec');
        }

        return isset($tickets[$attendee_id]['name']) ? $tickets[$attendee_id]['name'] : esc_html__('Unknown', 'mec');
    }

    /**
     * Get Attendees
     *
     * @since   1.0.0
     */
    public function getAttendees($attendeesType = 'all', $bookingQuery = [], $eventQuery = [])
    {
        $php_order_by = '';
        $php_order = '';
        $meta_key = $bookingQuery['meta_key'] ?? false;
        if ('sort_php' === $meta_key) {
            $php_order_by = $bookingQuery['orderby'];
            $php_order = $bookingQuery['order'];

            unset($bookingQuery['meta_key']);
            unset($bookingQuery['orderby']);
            unset($bookingQuery['order']);
            $attendeesType = 'all';
        }

        $attendees = [];
        if ($attendeesType == 'allAttendees' || $attendeesType == 'all') {
            $bookings = $bookingQuery ? $this->getBookings($bookingQuery, -1) : $this->bookings;
        } else {
            $bookings = $bookingQuery ? $this->getBookings($bookingQuery) : $this->bookings;
        }

        if ($bookings) {
            $i = 0;
            $book_object = new \MEC_book();
            foreach ($bookings as $booking) {
                $confirmedStatus = get_post_meta($booking->ID, 'mec_confirmed', true);
                if ($attendeesType == 'confirmed' && $confirmedStatus != '1') {
                    continue;
                }
                $verifiedStatus = get_post_meta($booking->ID, 'mec_verified', true);
                if ($attendeesType == 'verified' && $verifiedStatus != '1') {
                    continue;
                }

                $eventID = $booking->mec_event_id;
                $transactionID = get_post_meta($booking->ID, 'mec_transaction_id', true);
                $transaction = get_option($transactionID, []);
                $orderTime = get_post_meta($booking->ID, 'mec_booking_time', true);
                $bookDate = date('Y-m-d', strtotime(get_the_date('Y-m-d', $booking->ID)));
                $gateway = get_post_meta($booking->ID, 'mec_gateway', true);

                $locationID = get_post_meta($eventID, 'mec_location_id', true);
                $locationName = $locationID ? get_term($locationID)->name : 'Unknown Location';

                $organizerID = get_post_meta($eventID, 'mec_organizer_id', true);
                $organizerName = $organizerID ? get_term($organizerID)->name : 'Unknown Organizer';

                $attendeesMeta = get_post_meta($booking->ID, 'mec_attendees', true) ?: [];
                if ($attendeesMeta) {
                    foreach ($attendeesMeta as $attendeeMetaKey => $attendeeMeta) {
                        $attendeePrice = $book_object->get_ticket_total_price($transaction, $attendeeMeta, $booking->ID);

                        $reg_fields = $this->main->get_reg_fields($eventID);
                        if (is_array($reg_fields)) {
                            unset($reg_fields[':i:'], $reg_fields[':fi:']);
                        }

                        $per_attendee_fields = isset($attendeeMeta['reg']) ? $attendeeMeta['reg'] : [];
                        $formatted_fields = [];

                        foreach ($per_attendee_fields as $key => $value) {
                            $label = isset($reg_fields[$key]['label']) ? $reg_fields[$key]['label'] : "Field $key";
                            $formatted_fields[$label] = is_array($value) ? implode(', ', $value) : $value;
                        }

                        if (isset($this->main) && method_exists($this->main, 'get_bfixed_fields')) {
                            $fixed_fields_raw = $this->main->get_bfixed_fields($eventID);
                        } else {
                            $fixed_fields_raw = get_post_meta($eventID, 'mec_fixed_fields', true);
                        }

                        if (is_serialized($fixed_fields_raw)) {
                            $fixed_fields = maybe_unserialize($fixed_fields_raw);
                        } elseif (is_string($fixed_fields_raw) && json_decode($fixed_fields_raw, true) !== null) {
                            $fixed_fields = json_decode($fixed_fields_raw, true);
                        } elseif (is_array($fixed_fields_raw)) {
                            $fixed_fields = $fixed_fields_raw;
                        } else {
                            $fixed_fields = [];
                        }

                        $fixed_fields = array_filter($fixed_fields, function ($value) {
                            return !empty($value);
                        });

                        $fixed_fields_filled = [];
                        if (!empty($fixed_fields) && isset($transaction['fields']) && is_array($transaction['fields'])) {
                            foreach ($fixed_fields as $field_id => $field_data) {
                                if (!is_numeric($field_id)) continue;
                                $field_label = $field_data['label'] ?? "Fixed Field $field_id";
                                $field_value = $transaction['fields'][$field_id] ?? '---';
                                $fixed_fields_filled[$field_label] = $field_value;
                            }
                        }

                        $attendeeID = $attendeeMeta['id'] ?? null;
                        $ticketName = isset($this->bookingsData[$booking->ID][$attendeeID]) ? $this->bookingsData[$booking->ID][$attendeeID] : 'Unknown';
                        $ticket_variations_output = '';
                        if (isset($attendeeMeta['variations']) && is_array($attendeeMeta['variations']) && count($attendeeMeta['variations'])) {
                            $ticket_variations = \MEC\Base::get_main()->ticket_variations($eventID, $attendeeID);
                            foreach ($attendeeMeta['variations'] as $variation_id => $variation_count) {
                                if ((int) $variation_count > 0) {
                                    $ticket_variations_output .= (isset($ticket_variations[$variation_id]) ? $ticket_variations[$variation_id]['title'] : 'N/A') . ": (" . $variation_count . "), ";
                                }
                            }
                        }
                        $ticket_variations_output = html_entity_decode(trim($ticket_variations_output, ', '), ENT_QUOTES | ENT_HTML5);

                        $attendee_data = [
                            'booking' => [
                                'id' => $booking->ID,
                                'link' => get_edit_post_link($booking->ID),
                            ],
                            'invoice' => [
                                'id' => $transactionID,
                                'link' => $book_object->get_invoice_link($transactionID),
                            ],
                            'name' => $attendeeMeta['name'],
                            'email' => $attendeeMeta['email'],
                            'orderTime' => $orderTime,
                            'price' => $attendeePrice,
                            'event' => [
                                'id' => $eventID,
                                'name' => get_the_title($eventID),
                                'link' => get_permalink($eventID),
                            ],
                            'bookDate' => $bookDate,
                            'location_name' => $locationName,
                            'organizer_name' => $organizerName,
                            'ticket_name' => $ticketName,
                            'ticketVariation' => $ticket_variations_output,
                            'status' => [
                                'verified' => [
                                    'boolean' => $verifiedStatus,
                                    'text' => $verifiedStatus == '1' ? esc_html__('Verified', 'mec-ars') : esc_html__('Waiting', 'mec-ars'),
                                ],
                                'confirmed' => [
                                    'boolean' => $confirmedStatus,
                                    'text' => $confirmedStatus == '1' ? esc_html__('Confirmed', 'mec-ars') : esc_html__('Pending', 'mec-ars'),
                                ],
                            ],
                            'categories' => wp_get_post_terms($eventID, 'mec_category', ['fields' => 'names']),
                            'locations' => wp_get_post_terms($eventID, 'mec_location', ['fields' => 'names']),
                            'organizers' => wp_get_post_terms($eventID, 'mec_organizer', ['fields' => 'names']),
                            'gateway' => $gateway,
                        ];

                        foreach ($formatted_fields as $field_label => $field_value) {
                            $attendee_data[$field_label] = $field_value;
                        }

                        foreach ($fixed_fields_filled as $field_label => $field_value) {
                            $attendee_data[$field_label] = $field_value;
                        }

                        $attendees[$i] = $attendee_data;
                        $i++;
                    }
                }
            }
        }

        return $attendees;
    }




    /**
     * Render Attendees
     *
     * @since   1.0.0
     */
    public function renderAttendees($attendeesType = 'all', $filter = true, $bookingQuery = [], $eventQuery = [])
    {
        $attendees = $this->getAttendees($attendeesType, $bookingQuery, $eventQuery);
        $chunks = is_array($attendees) ? array_chunk($attendees, 15) : false;
        $attendeesFirstChunk = $attendees ? $chunks[0] : [];
?>
        <div class="mec-ars-attendees">
            <div class="mec-ars-flex mec-ars-attendees-export">
                <img src="<?php echo esc_url(MECARSASSETS . 'images/export.svg') ?>" alt="">
                <div class="mec-ars-flex mec-ars-attendees-export-inner">
                    <span data-export="json"><?php esc_html_e('JSON', 'mec-ars'); ?></span>
                    <span data-export="csv"><?php esc_html_e('CSV', 'mec-ars'); ?></span>
                    <span data-export="xml"><?php esc_html_e('XML', 'mec-ars'); ?></span>
                </div>
                <div class="mec-ars-flex mec-ars-attendees-export current-filter-attendees">
                    <div class="mec-ars-flex mec-ars-attendees-export-inner">
                        <span data-export="json"><?php esc_html_e('JSON', 'mec-ars'); ?></span>
                        <span data-export="csv"><?php esc_html_e('CSV', 'mec-ars'); ?></span>
                        <span data-export="xml"><?php esc_html_e('XML', 'mec-ars'); ?></span>
                    </div>
                </div>
                <div class="mec-ars-flex mec-ars-attendees-export all-attendees">
                    <div class="mec-ars-flex mec-ars-attendees-export-inner">
                        <span data-export="json"><?php esc_html_e('JSON', 'mec-ars'); ?></span>
                        <span data-export="csv"><?php esc_html_e('CSV', 'mec-ars'); ?></span>
                        <span data-export="xml"><?php esc_html_e('XML', 'mec-ars'); ?></span>
                    </div>
                </div>
                <span class="mec-ars-attendees-export-message"><?php esc_html_e('Please select your desired attendees', 'mec-ars'); ?></span>
            </div>
            <?php if ($filter) { ?>
                <?php $this->renderAttendeesFilter(); ?>
            <?php } ?>
            <?php $this->renderAttendeesList($attendeesFirstChunk); ?>
        </div>
    <?php
    }

    /**
     * Render Attendees Filter
     *
     * @since   1.0.0
     */
    public function renderAttendeesFilter()
    {
    ?>
        <div class="mec-ars-attendees-filter">
            <form>
                <div class="mec-ars-flex">
                    <div class="mec-ars-flex-item">
                        <input class="mec-ars-form-control" type="text" name="mec-ars-daterange" placeholder="<?php esc_attr_e('Pick a Date Range', 'mec-ars'); ?>" autocomplete="off">
                    </div>
                    <div class="mec-ars-flex-item">
                        <?php Events::instance()->renderEventsDropdown(); ?>
                    </div>
                    <div class="mec-ars-flex-item">
                        <?php Events::instance()->renderCategoriesDropdown(); ?>
                    </div>
                    <div class="mec-ars-flex-item">
                        <?php Events::instance()->renderLocationsDropdown(); ?>
                    </div>
                    <div class="mec-ars-flex-item">
                        <?php Events::instance()->renderOrganizersDropdown(); ?>
                    </div>
                </div>
            </form>
        </div>
    <?php
    }

    /**
     * Render Attendees List
     *
     * @since   1.0.0
     */
    public function renderAttendeesList($attendees = [], $attendeesOrder = 0)
    {
    ?>
        <div class="mec-ars-attendees-list mec-ars-flex mec-ars-flex-column">
            <div class="mec-ars-attendees-list-header mec-ars-flex">
                <div class="mec-ars-flex-item mec-ars-sm">
                    <input class="mec-ars-checkbox mec-ars-checkbox-all" type="checkbox" value="">
                </div>
                <div class="mec-ars-flex-item mec-ars-smd">
                    <span><?php esc_html_e('ID', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="bookingID">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-md">
                    <span><?php esc_html_e('Invoice', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="invoiceID">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-lg">
                    <span><?php esc_html_e('Assigned to', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="assignedTo">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-md">
                    <span><?php esc_html_e('Order Time', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="orderTime">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-smd">
                    <span><?php esc_html_e('Price', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="price">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-lg">
                    <span><?php esc_html_e('Event', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="event">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-md">
                    <span><?php esc_html_e('Event Date', 'mec-ars'); ?></span>
                    <span class="mec-ars-sort-column" data-sort="bookDate">
                        <i class="mec-sl-arrow-up" data-sort-up></i>
                        <i class="mec-sl-arrow-down" data-sort-down></i>
                    </span>
                </div>
                <div class="mec-ars-flex-item mec-ars-slg mec-ars-attendee-status">
                    <span><?php esc_html_e('Status', 'mec-ars'); ?></span>
                    <span>(<?php esc_html_e('Confirmation, Verification', 'mec-ars'); ?>)</span>
                </div>
            </div>
            <?php if ($attendees) { ?>
                <?php foreach ($attendees as $key => $attendee) { ?>
                    <div class="mec-ars-attendee mec-ars-flex" style="order: <?php echo esc_attr($attendeesOrder); ?>">
                        <div class="mec-ars-flex-item mec-ars-sm">
                            <input class="mec-ars-checkbox" type="checkbox" value="<?php echo esc_attr($key); ?>">
                        </div>
                        <div class="mec-ars-flex-item mec-ars-smd" data-sort="bookingID" data-sort-content="<?php echo esc_attr($attendee['booking']['id']); ?>">
                            <a href="<?php echo esc_url($attendee['booking']['link']); ?>" class="mec-ars-attendee-booking" target="_blank"><?php echo esc_html($attendee['booking']['id']); ?></a>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-md" data-sort="invoiceID" data-sort-content="<?php echo esc_attr($attendee['invoice']['id']); ?>">
                            <a href="<?php echo esc_url($attendee['invoice']['link']); ?>" class="mec-ars-attendee-invoice" target="_blank"><?php echo esc_html($attendee['invoice']['id']); ?></a>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-lg" data-sort="assignedTo" data-sort-content="<?php echo esc_attr(strtolower($attendee['name'])); ?>">
                            <div class="mec-ars-attendee-info">
                                <span class="mec-ars-attendee-avatar"><?php echo get_avatar($attendee['email'], 35); ?></span>
                                <div class="mec-ars-attendee-info-inner">
                                    <span class="mec-ars-attendee-name"><?php echo esc_html($attendee['name']); ?></span>
                                    <span class="mec-ars-attendee-email"><?php echo esc_html($attendee['email']); ?></span>
                                </div>
                            </div>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-md" data-sort="orderTime" data-sort-content="<?php echo esc_attr(strtolower($attendee['orderTime'])); ?>">
                            <span class="mec-ars-attendee-order-date"><?php echo esc_html($attendee['orderTime']); ?></span>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-smd" data-sort="price" data-sort-content="<?php echo esc_attr(round($attendee['price'], 2)); ?>">
                            <span class="mec-ars-attendee-price"><?php echo esc_html(\MEC\Base::get_main()->render_price(round($attendee['price'], 2))); ?></span>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-lg" data-sort="event" data-sort-content="<?php echo esc_attr(strtolower($attendee['event']['name'])); ?>">
                            <a href="<?php echo esc_url($attendee['event']['link']); ?>" class="mec-ars-attendee-event" target="_blank"><?php echo esc_html($attendee['event']['name']); ?></a>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-md" data-sort="bookDate" data-sort-content="<?php echo esc_attr(strtolower($attendee['bookDate'])); ?>">
                            <span class="mec-ars-attendee-book-date"><?php echo esc_html($attendee['bookDate']); ?></span>
                        </div>
                        <div class="mec-ars-flex-item mec-ars-slg">
                            <span class="mec-ars-attendee-status-confirmed<?php echo $attendee['status']['confirmed']['boolean'] == '1' ? ' mec-ars-attendee-confirmed' : ($attendee['status']['confirmed']['boolean'] == '-1' ? ' mec-ars-attendee-rejected' : ' mec-ars-attendee-pending'); ?>"><?php echo esc_html($attendee['status']['confirmed']['text']); ?></span>
                            <span class="mec-ars-separator">,</span>
                            <span class="mec-ars-attendee-status-verified<?php echo $attendee['status']['verified']['boolean'] == '1' ? ' mec-ars-attendee-verified' : ($attendee['status']['verified']['boolean'] == '-1' ? ' mec-ars-attendee-canceled' : ' mec-ars-attendee-waiting'); ?>"><?php echo esc_html($attendee['status']['verified']['text']); ?></span>
                        </div>
                    </div>
                    <?php ++$attendeesOrder; ?>
                <?php } ?>
            <?php } else { ?>
                <div class="mec-ars-flex">
                    <div class="mec-ars-flex-item mec-ars-equal mec-ars-align-center">
                        <span class="mec-ars-no-attendees"><?php esc_html_e('No Attendees', 'mec-ars'); ?></span>
                    </div>
                </div>
            <?php } ?>
        </div>
    <?php
    }


    /**
     * Render All Attendees Number
     *
     * @since   1.0.0
     */
    public function renderAllAttendeesNumber($number = 0)
    {
    ?>
        <div class="mec-ars-all-attendees-number mec-ars-statistics-item">
            <span class="mec-ars-number"><?php echo esc_html($number); ?></span>
            <span class="mec-ars-title"><?php esc_html_e('All Attendees', 'mec-ars'); ?></span>
        </div>
    <?php
    }

    /**
     * Render All Bookings Number
     *
     * @since   1.0.0
     */
    public function renderAllBookingsNumber($number = 0, $attendeesNumber = 0)
    {
    ?>
        <div class="mec-ars-all-bookings-number mec-ars-statistics-item">
            <span class="mec-ars-number"><?php echo esc_html($number); ?></span>
            <span class="mec-ars-title"><?php esc_html_e('All Bookings', 'mec-ars'); ?></span>
        </div>
    <?php
    }

    /**
     * Render Confirmed Bookings Number
     *
     * @since   1.0.0
     */
    public function renderConfirmedBookingsNumber($number = 0, $attendeesNumber = 0)
    {
    ?>
        <div class="mec-ars-confirmed-bookings-number mec-ars-statistics-item">
            <span class="mec-ars-number"><?php echo esc_html($number); ?></span>
            <span class="mec-ars-title"><?php esc_html_e('Confirmed Bookings', 'mec-ars'); ?></span>
        </div>
    <?php
    }

    /**
     * Render Verified Bookings Number
     *
     * @since   1.0.0
     */
    public function renderVerifiedBookingsNumber($number = 0, $attendeesNumber = 0)
    {
    ?>
        <div class="mec-ars-verified-bookings-number mec-ars-statistics-item">
            <span class="mec-ars-number"><?php echo esc_html($number); ?></span>
            <span class="mec-ars-title"><?php esc_html_e('Verified Bookings', 'mec-ars'); ?></span>
        </div>
    <?php
    }

    /**
     * Render Total Income Bookings Number
     *
     * @since   1.0.0
     */
    public function renderTotalIncome($number = 0)
    {
        $price = $this->MECMainInstance->render_price($number);
    ?>
        <div class="mec-ars-total-income mec-ars-statistics-item">
            <span class="mec-ars-number"><?php echo esc_html($price); ?></span>
            <span class="mec-ars-title"><?php esc_html_e('Total Income', 'mec-ars'); ?></span>
        </div>
<?php
    }
}
