<?php
namespace MECAdvancedReport\App\Libraries;

use MECAdvancedReport\Plugin;
use MECAdvancedReport\Core\Patterns\SingletonPattern;
use MECAdvancedReport\App\Libraries\Bookings;

// Don't load directly.
if (!defined('ABSPATH')) {
    header('Status: 403 Forbidden');
    header('HTTP/1.1 403 Forbidden');
    exit;
}

/**
 * Charts.
 *
 * @author  Webnus Team
 * @package MECAdvancedReport
 * @since   1.0.0
 */
class Charts extends SingletonPattern
{
    /**
     * Constructor
     *
     * @since   1.0.0
     */
    public function __construct()
    {
    }

    /**
     * Get Line Chart Data
     *
     * @since   1.0.0
     */
    public function getLineChartData($type = 'locations', $attendees = [], $termIDs = [])
    {
        $chartData['labels'] = $chartData['attendees'] = $chartData['sales'] = $chartData = $tempData = [];
        if ($attendees) {
            foreach ($attendees as $attendee) {
                if ($attendee['status']['confirmed']['boolean'] != '1') {
                    continue;
                }
                if ($termIDs) {
                    $eventID = $attendee['event']['id'];
                    if ($type == 'locations') {
                        $postTermIDs = wp_get_post_terms($eventID, 'mec_location', ['fields' => 'ids']);
                    } elseif ($type == 'organizers') {
                        $postTermIDs = wp_get_post_terms($eventID, 'mec_organizer', ['fields' => 'ids']);
                    } elseif ($type == 'categories') {
                        $postTermIDs = wp_get_post_terms($eventID, 'mec_category', ['fields' => 'ids']);
                    }
                    if (count(array_intersect($termIDs, $postTermIDs)) === 0) {
                        continue;
                    }
                }
                $orderTime = date_i18n('d F', strtotime($attendee['orderTime']));
                $slug = sanitize_title($orderTime);
                $tempData[$slug]['label'] = $orderTime;
                $tempData[$slug]['attendees'] = isset($tempData[$slug]['attendees']) ? ++$tempData[$slug]['attendees'] : 1;
                $tempData[$slug]['sales'] = isset($tempData[$slug]['sales']) ? $tempData[$slug]['sales'] : 0;
                $tempData[$slug]['sales'] += $attendee['price'];
            }
        }
        if ($tempData) {
            foreach (array_reverse($tempData) as $data) {
                array_push($chartData['labels'], $data['label']);
                array_push($chartData['attendees'], $data['attendees']);
                array_push($chartData['sales'], $data['sales']);
            }
        }
        return $chartData;
    }

    /**
     * Get Doughnut Chart Data
     *
     * @since   1.0.0
     */
    public function getDoughnutChartData($attendees = [])
    {
        $chartData['labels'] = $chartData['sales'] = $chartData = $tempData = [];
        $enableGateways = $this->getEnableGateways();
        if ($attendees) {
            foreach ($attendees as $attendee) {
                if ($attendee['status']['confirmed']['boolean'] != '1') {
                    continue;
                }
                $slug = str_replace('MEC_gateway_', '', $attendee['gateway']);
                $tempData[$slug]['sales'] = isset($tempData[$slug]['sales']) ? $tempData[$slug]['sales'] : 0;
                $tempData[$slug]['sales'] += $attendee['price'];
            }
        }
        if ($tempData) {
            foreach ($tempData as $key => $data) {
                if (in_array($key, ['pay_locally', 'paypal_express', 'paypal_credit_card', 'stripe', 'woocommerce', 'stripe_connect'])) {
                    if (isset($enableGateways[$key])) {
                        $enableGateways[$key]['sales'] = $data['sales'];
                    }
                } else {
                    $enableGateways[$key]['sales'] = $data['sales'];
                    $enableGateways[$key]['label'] = ucwords(str_replace('_', ' ', $key));
                }
            }
        }
        if ($enableGateways) {
            foreach ($enableGateways as $data) {
                array_push($chartData['labels'], $data['label']);
                array_push($chartData['sales'], $data['sales']);
            }
        }
        if (array_unique($chartData['sales']) == [0 => 0]) {
            array_unshift($chartData['sales'], 1);
            array_unshift($chartData['labels'], 'mecArsEmptyLabel');
            $chartData['empty'] = true;
        }
        return $chartData;
    }

    /**
     * Render Line Charts
     *
     * @since   1.0.0
     */
    public function renderLineCharts()
    {
        ?>
        <div class="mec-ars-title">
            <h2><?php esc_html_e('Sales / Attendees', 'mec-ars'); ?></h2>
        </div>
        <div class="mec-ars-tabs">
            <ul class="mec-ars-tabs-nav">
                <li class="mec-ars-active">
                    <a href="#mec-ars-locations-chart"><?php esc_html_e('Locations', 'mec-ars'); ?></a>
                </li>
                <li>
                    <a href="#mec-ars-organizers-chart"><?php esc_html_e('Organizers', 'mec-ars'); ?></a>
                </li>
                <li>
                    <a href="#mec-ars-categories-chart"><?php esc_html_e('Categories', 'mec-ars'); ?></a>
                </li>
            </ul>
            <div class="mec-ars-tabs-content">
                <!-- Preloader -->
                <div class="mec-ars-flex mec-ars-justify-content-center mec-ars-preloader">
                    <div class="sk-pulse"></div>
                </div>
                <div id="mec-ars-locations-chart" class="mec-ars-active">
                    <?php Events::instance()->renderLocationsDropdown(); ?>
                    <canvas id="mec-ars-locations-chart-canvas"></canvas>
                </div>
                <div id="mec-ars-organizers-chart">
                    <?php Events::instance()->renderOrganizersDropdown(); ?>
                    <canvas id="mec-ars-organizers-chart-canvas"></canvas>
                </div>
                <div id="mec-ars-categories-chart">
                    <?php Events::instance()->renderCategoriesDropdown(); ?>
                    <canvas id="mec-ars-categories-chart-canvas"></canvas>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Doughnut Chart
     *
     * @since   1.0.0
     */
    public function renderDoughnutChart()
    {
        ?>
        <div class="mec-ars-title">
            <h2><?php esc_html_e('Gateways', 'mec-ars'); ?></h2>
        </div>
        <canvas id="mec-ars-gateways-chart-canvas"></canvas>
        <ul class="mec-ars-flex mec-ars-flex-column" id="mec-ars-gateways-chart-legends"></ul>
        <?php
    }

    /**
     * Get Enable Gateways
     *
     * @since   1.0.0
     */
    public function getEnableGateways()
    {
        $enableGateways = [];
        $mecGateways = Plugin::instance()->mecOptions['gateways'];
        foreach ($mecGateways as $key => $gateway) {
            if (isset($gateway['status']) && $gateway['status'] != '1') {
                continue;
            }
            switch ($key) {
                case '1':
                    $enableGateways['pay_locally']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('Pay Locally', 'mec-ars');
                    $enableGateways['pay_locally']['sales'] = 0;
                    break;
                case '2':
                    $enableGateways['paypal_express']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('PayPal Express', 'mec-ars');
                    $enableGateways['paypal_express']['sales'] = 0;
                    break;
                case '3':
                    $enableGateways['paypal_credit_card']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('PayPal Credit Card', 'mec-ars');
                    $enableGateways['paypal_credit_card']['sales'] = 0;
                    break;
                case '5':
                    $enableGateways['stripe']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('Stripe', 'mec-ars');
                    $enableGateways['stripe']['sales'] = 0;
                    break;
                case '6':
                    $enableGateways['woocommerce']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('Pay by WooCommerce', 'mec-ars');
                    $enableGateways['woocommerce']['sales'] = 0;
                    break;
                case '7':
                    $enableGateways['stripe_connect']['label'] = isset($gateway['title']) && $gateway['title'] ? $gateway['title'] : esc_html__('Stripe Connect', 'mec-ars');
                    $enableGateways['stripe_connect']['sales'] = 0;
                    break;
            }
        }
        return $enableGateways;
    }
}
