<?php

namespace MEC_Virtual_Events\Single;

use MEC_Virtual_Events\Core\Books\BooksQuery;

/**
 * @since 1.1.1
 */
class SinglePageAddon {

	private $event_type = 'virtual';
	private $event;
	private static $instance;

	/**
	 * Get Instance
	 *
	 * @return self $instance
	 */
	public static function getInstance() {

		if ( is_null( static::$instance ) ) {

			static::$instance = new static();
		}

		return static::$instance;
	}

	/**
	 * Init
	 *
	 * @since 1.1.1
	 * @return void
	 */
	public function init() {
		$this->event_type = 'virtual';
		add_action( 'mec_single_after_content', array( $this, 'display_in_single_page' ) );
	}

	/**
	 * @param int $name
	 * @since 1.1.1
	 * @return bool
	 */
	public function is_virtual_event( $event_id ) {

		return '1' === get_post_meta( $event_id, 'mec_' . $this->event_type . '_event', true );
	}

	/**
	 * @param object   $event
	 * @param MEC_Main $main
	 * @param int      $timestamp
	 * @since 1.1.1
	 * @return boolean
	 */
	public function can_display_access_info( $event, &$main, $timestamp ) {

		$event_id   = $event->data->ID;
		$event_date = date( 'Y-m-d h:i a', $timestamp );

		$can_check_going = isset( $event->date['start']['timestamp'] ) && $timestamp <= $event->date['start']['timestamp'];

		$hide_info_before_start = get_post_meta( $event_id, 'mec_' . $this->event_type . '_hide_info_before_start', true );
		$virtual_show_info_time = get_post_meta( $event_id, 'mec_' . $this->event_type . '_show_info_time', true );
		$virtual_show_info_hm   = get_post_meta( $event_id, 'mec_' . $this->event_type . '_show_info_hm', true );

		$hide_info = get_post_meta( $event_id, 'mec_' . $this->event_type . '_hide_info', true );

		$check_datetime = false;
		if ( $hide_info_before_start && $virtual_show_info_time > 0 ) {

			$check_datetime           = $main->check_date_time_validation( 'Y-m-d h:i a', strtolower( $event_date ) );
			$datetime_to_hide_earlier = strtotime( "-{$virtual_show_info_time}{$virtual_show_info_hm}", strtotime( $event_date ) );
		}

		$can_display =
			( ! $hide_info && $check_datetime && $datetime_to_hide_earlier > current_time( 'timestamp', 0 ) )
			||
			( $can_check_going && $hide_info and ! $main->is_ongoing( $event ) )
			||
			( ! $hide_info and ! $hide_info_before_start );

		return $can_display;
	}

	/**
	 * @param int $event_id
	 * @since 1.1.1
	 * @return boolean
	 */
	public function can_display_user_book_history( $event_id ) {

		return 'yes' === get_post_meta( $event_id, 'mec_' . $this->event_type . '_display_user_booking_history', true );
	}

	/**
	 * Get global settings for virtual event link,password,embed
	 *
	 * @param int $event_id
	 * @since 1.1.1
	 * @return array
	 */
	public function get_global_access_info_for_event( $event_id ) {

		// link
		$info['link_url']                = get_post_meta( $event_id, 'mec_' . $this->event_type . '_link_url', true );
		$info['link_title']              = get_post_meta( $event_id, 'mec_' . $this->event_type . '_link_title', true );
		$info['link_target']             = get_post_meta( $event_id, 'mec_' . $this->event_type . '_link_target', true );
		$info['display_link_in_booking'] = get_post_meta( $event_id, 'mec_' . $this->event_type . '_display_link_in_booking', true );

		// password
		$info['display_password_in_booking'] = get_post_meta( $event_id, 'mec_' . $this->event_type . '_display_password_in_booking', true );
		$info['password']                    = get_post_meta( $event_id, 'mec_' . $this->event_type . '_password', true );

		// embed
		$info['display_embed_in_booking'] = get_post_meta( $event_id, 'mec_' . $this->event_type . '_display_embed_in_booking', true );
		$info['embed'] = get_post_meta( $event_id, 'mec_' . $this->event_type . '_embed', true );

		return $info;
	}

	/**
	 * Display link,password,embed  in single event page
	 *
	 * @param object $event
	 * @since 1.1.1
	 * @return void
	 */
	public function display_in_single_page( $event ) {

		$this->event = $event;
		$main        = new \MEC_main();
		$event_id = isset( $event->data->ID ) ? $event->data->ID : $event->ID;
		$timestamp   = isset( $event->data->time['start_timestamp'] ) ? $event->data->time['start_timestamp'] : false;
		if( !$timestamp ){

			return;
		}
		$timestamp   = $main->get_start_time_of_multiple_days( $event_id, $timestamp );

		$ex        = explode( ':', $timestamp );
		$timestamp = (int) $ex[0];

		if ( ! $this->is_virtual_event( $event_id ) ) {

			return;
		}

		if ( ! $this->can_display_access_info( $event, $main, $timestamp ) ) {

			return;
		}

		$info = $this->get_global_access_info_for_event( $event_id );
		// replace values from custom access

		$custom_event_access = $this->get_custom_event_access( $event_id, $timestamp );
		if ( $custom_event_access ) {

			$info          = $custom_event_access;
			$info['embed'] = '';
		}

		echo $this->display_event_access_html( $info );
		$this->display_user_book_history( $event_id, $main, $timestamp );
	}

	/**
	 * Display book history for current user
	 *
	 * @param int      $event_id
	 * @param MEC_Main $main
	 * @param int      $timestamp
	 * @since 1.1.1
	 * @return void
	 */
	public function display_user_book_history( $event_id, &$main, $timestamp ) {

		if ( ! $this->can_display_user_book_history( $event_id ) ) {

			return;
		}
		$date_format = get_option( 'date_format' );
		$books_times = $this->get_user_books_times_for_event( $event_id );
		if ( ! $books_times ) {

			return;
		}
		arsort( $books_times );
		$books_times = array_unique( $books_times );

		$inside_html = '';
		$global_info = $this->get_global_access_info_for_event( $event_id );
		$html        = '<div class="mec-virtual-event-history">';
		$html        .= '<h3>' . __( 'History', 'mec-virtual' ) . '</h3>';
		$html       .= '<div class="mec-user-booking-history">';
		foreach ( $books_times as $book_id => $b_timestamp ) {

			// must be used for each occurrence
			// $can_display = $this->can_display_access_info($this->event,$main,$b_timestamp);
			// if(!$can_display){
			// continue;
			// }

			$class = $timestamp === $b_timestamp ? 'mec-box-marker' : '';

			$custom_event_access = $this->get_custom_event_access( $event_id, $b_timestamp );
			if ( $custom_event_access ) {
				$info = $custom_event_access;
			} else {
				$info = $global_info;
			}

			$info['booked'] = true;

			$html_event_access = $this->display_event_access_html( $info, $class );
			if ( ! empty( $html_event_access ) ) {

				$inside_html .= '<span class="mec-book-date">'.$main->date_i18n( $date_format, $b_timestamp ).'</span>';
				$inside_html .= $html_event_access;
			}
		}

		if ( ! empty( $inside_html ) ) {

			$html .= $inside_html;
		}

		$html .= '</div>';
		$html .= '</div>';

		if ( ! empty( $inside_html ) ) {

			echo $html;
		}
	}

	/**
	 * @param int $event_id
	 * @since 1.1.1
	 * @return array
	 */
	public function get_user_books_times_for_event( $event_id ) {

		if ( ! is_user_logged_in() ) {

			return false;
		}

		$user_data = wp_get_current_user();

		$email = $user_data->user_email;
		if ( empty( $email ) ) {

			return false;
		}

		$books = BooksQuery::getInstance()->get_books_ids(
			array(
				'attendee_email' => $email,
				'event_id'       => $event_id,
			)
		);

		$books_times = array();
		if ( is_array( $books ) ) {

			foreach ( $books as $book_id ) {

				$books_times[ $book_id ] = get_post_meta( $book_id, 'mec_attention_time_start', true );
			}
		}

		return $books_times;
	}

	/**
	 * Get custom event access if exists for $timestamp "start time"
	 *
	 * @param int $event_id
	 * @param int $timestamp
	 * @since 1.1.1
	 * @return array|bool
	 */
	public function get_custom_event_access( $event_id, $timestamp ) {

		$custom_event_access = get_post_meta( $event_id, '_mec_' . $this->event_type . '_event_access', true );
		$occurrence          = $timestamp;
		if ( isset( $custom_event_access[ $occurrence ] ) ) {
			$is_enable_custom_event_access = $custom_event_access[ $occurrence ]['custom_event_access'] ? $custom_event_access[ $occurrence ]['custom_event_access'] : false;

			if ( $is_enable_custom_event_access ) {
				return $custom_event_access[ $occurrence ];
			}
		}

		return false;
	}

	/**
	 * @param array  $access_info link_title,link_url,link_target,display_link_in_booking,password,password,display_password_in_booking,embed
	 * @param string $class
	 * @since 1.1.1
	 * @return string
	 */
	public function display_event_access_html( $access_info, $class = '' ) {

		$is_booked = isset( $access_info['booked'] ) ? $access_info['booked'] : false;
		// generate html start
		$link_html = $password_html = $embed_html = '';

		$display_link_in_booking = isset( $access_info['display_link_in_booking'] ) ? $access_info['display_link_in_booking'] == '1' : false;
		if ( $access_info['link_title'] && ( !$display_link_in_booking || $is_booked ) ) {

			$link_html = '<a class="mec-event-virtual-link" href="' . $access_info['link_url'] . '" target="' . $access_info['link_target'] . '">' . $access_info['link_title'] . '</a>';
		}

		$display_password_in_booking = isset( $access_info['display_password_in_booking'] ) ? $access_info['display_password_in_booking'] == '1' : false;
		if ( $access_info['password'] && ( !$display_password_in_booking || $is_booked ) ) {

			$password_html = '<div class="mec-virtual-password"><strong>' . esc_html__( 'Password:', 'mec-virtual' ) . '</strong> <span>' . $access_info['password'] . '</span></div>';
		}

		$display_embed_in_booking = isset( $access_info['display_embed_in_booking'] ) ? $access_info['display_embed_in_booking'] == '1' : false;
		if ( isset( $access_info['embed'] ) && !empty( $access_info['embed'] ) && ( !$display_embed_in_booking || $is_booked ) ) {

			$embed_html = '<div class="mec-event-virtual-embed">' . $access_info['embed'] . '</div>';
		}
		// generate html end

		if ( ! empty( $embed_html ) or ! empty( $link_html ) or ! empty( $password_html ) ) {

			return '<div class="mec-event-data-fields mec-virtual-box-under-content mec-frontbox ' . $class . '">'
					. $embed_html
					. $link_html
					. $password_html
				. '</div>';
		}

		return '';
	}

	public function get_event_access_info( $event_id, $timestamp ){

		$info = $this->get_global_access_info_for_event( $event_id );
		$custom_event_access = $this->get_custom_event_access( $event_id, $timestamp );
		if ( $custom_event_access ) {

			$info          = $custom_event_access;
			$info['embed'] = '';
		}

		return $info;
	}
}
